 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Geometry Application Services
   >>>>              Primitive List Routines 
   >>>>
   >>>>   Static:
   >>>>			_kgeom_construct_component()
   >>>>			_kgeom_free_component()
   >>>>			_kgeom_construct_primitive()
   >>>>			_kgeom_free_primitive()
   >>>>			_kgeom_construct_manifest()
   >>>>			_kgeom_free_manifest()
   >>>>  Private:
   >>>>			kgeom_establish_manifest_primitive()
   >>>>			kgeom_verify_manifest_primitive()
   >>>>			kgeom_get_manifest_primitive_list()
   >>>>			kgeom_set_manifest_primitive_list()
   >>>>			kgeom_get_manifest_number_primitives()
   >>>>			kgeom_set_manifest_number_primitives()
   >>>>			kgeom_write_transcript()
   >>>>			kgeom_write_transcript()
   >>>>			kgeom_get_manifest()
   >>>>			kgeom_destroy_manifest()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "geom_internals.h"

/* 
 *    ================================================================== 
 *    Component Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_construct_component
|
|       Purpose: This simple routine will construct a
|                component.  
|
|         Input: present     - TRUE if the component is present
|		 begin_point - beginning index into segment
|		 size[2]     - array indicating size of component
|
|        Output: none
|
|       Returns: the constructed component structure
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1995 15:29
| Modifications:
|
------------------------------------------------------------*/
kgeom_component *
kgeom_construct_component(
   int   present,
   int   begin_point,
   int  *size)
{
   kgeom_component *new = NULL;
   
   new = (kgeom_component *) kcalloc(sizeof(kgeom_component),1);
   
   new->present     = present;
   new->begin_point = begin_point;
   new->size[0]     = size[0];
   new->size[1]     = size[1];
   
   return new;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_free_component
|
|       Purpose: This simple routine will free a component
|                structure.
|
|         Input: comp - pointer to the component definition 
|			to be freed
|
|        Output: none
|
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 16:24
| Modifications:
|
------------------------------------------------------------*/
static void 
_kgeom_free_component(kgeom_component *comp)
{
   /* -- trivial free -- */
   if (comp == NULL) return;
   
   kfree(comp);
   return;
}

/* 
 *    ================================================================== 
 *    Primitive Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_construct_primitive
|
|       Purpose: This simple routine will construct a primitive
|                structure.  The components will be initialized
|		 to be NULL.
|
|         Input: primitive      - primitive identifier
|		 num_components - number of empty components to 
|				  allocate
|
|        Output: none
|
|       Returns: the constructed primitive.
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
static kgeom_primitive *
_kgeom_construct_primitive(
   int   primitive,
   int   num_components)
{
   kgeom_primitive *new = NULL;
   
   new = (kgeom_primitive *) kmalloc(sizeof(kgeom_primitive));
   
   /* -- assign the primitive type -- */
   new->primitive = primitive;

   /* -- initially has no vertices -- */
   new->number  = 0;

   /* -- initially all per-primitive attributes -- */
   new->texture = NULL;
   new->type    = KGEOM_LINE_TYPE_DEFAULT;
   new->width   = KGEOM_LINE_WIDTH_DEFAULT;
   
   /* -- allocate space for the given num_components -- */
   new->num_components = num_components;
   new->components = (kgeom_component **) 
      kcalloc(sizeof(kgeom_component *),num_components);

   return new;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_free_primitive
|
|       Purpose: This simple routine will free a primitive 
|                structure along with all of its component structures.
|
|         Input: prim - pointer to the primitive structure to be freed
|
|        Output: none
|
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 16:24
| Modifications:
|
------------------------------------------------------------*/
static void 
_kgeom_free_primitive(kgeom_primitive *prim)
{
   int i;
   
   /* -- trivial free -- */
   if (prim == NULL) return;
   
   for( i = 0; i < prim->num_components; i++)
      _kgeom_free_component(prim->components[i]);
   
   kfree(prim->components);
   kfree(prim->texture);
   kfree(prim);
   return;
}

/* 
 *    ================================================================== 
 *    Manifest Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_construct_manifest
|
|       Purpose: This simple routine will construct a primitive
|                manifest structure.  The initial number of primitives
|		 will be zero.
|
|         Input: none
|
|        Output: none
|
|       Returns: the constructed manifest.
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
static kgeom_manifest *
_kgeom_construct_manifest(void)
{
   kgeom_manifest *new = NULL;
   
   new = (kgeom_manifest *) kmalloc(sizeof(kgeom_manifest));
   
   new->num_primitives = 0;
   new->primitives     = NULL;
   
   return new;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kgeom_free_manifest
|
|       Purpose: This simple routine will free a primitive manifest
|                structure along with all of its primitive and
|		 component structures.
|
|         Input: manifest - pointer to the manifest structure to be freed
|
|        Output: none
|
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 16:24
| Modifications:
|
------------------------------------------------------------*/
static void 
_kgeom_free_manifest(kgeom_manifest *manifest)
{
   int i;
   
   /* -- trivial free -- */
   if (manifest == NULL) return;
   
   for (i = 0; i < manifest->num_primitives; i++)
      _kgeom_free_primitive(manifest->primitives[i]);
   
   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_establish_manifest_primitive
|
|       Purpose: This routine will establish a primitive within
| 		 the primitive manifest at the given primitive 
|		 position according to the given primitive definition.
|
|         Input: manifest   - the manifest from which to establish
|			      the primitive
|		 pos        - the position at which to establish 
|			      the primitive
|		 definition - the definition of the primitive to 
|		 	      be established
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
int 
kgeom_establish_manifest_primitive(
   kgeom_manifest 	 *manifest,
   int            	  pos,
   kgeom_primitive_defin *defin)
{
   /* -- sanity check -- */
   if (manifest == NULL || defin == NULL)
      return FALSE;

   if (pos > manifest->num_primitives)
      return FALSE;

   /* -- if there's something there already, get rid of it -- */
   if (manifest->primitives[pos] == NULL)
      _kgeom_free_primitive(manifest->primitives[pos]);

   /* -- construct new primitive according to given defin -- */
   manifest->primitives[pos] = _kgeom_construct_primitive(defin->primitive, 
							defin->num_components);
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_verify_manifest_primitive
|
|       Purpose: This routine will verify that the primitive at 
|		 a given position matches the specified primitive.
|
|         Input: manifest   - the manifest from which to verify
|			      the primitive
|		 primitive  - the primitive to verify	
|		 pos        - the position at which to verify
|			      the primitive
|		 definition - the definition of the primitive to 
|		 	      be established.  if null, then the
|			      definition will be found within this routine
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
int 
kgeom_verify_manifest_primitive(
   kgeom_manifest 	  *manifest,
   int            	   primitive,
   int            	   pos,   
   kgeom_primitive_defin *defin)
{
   /* -- sanity check -- */
   if (manifest == NULL)
      return FALSE;

   if (pos > manifest->num_primitives)
      return FALSE;

   if (defin == NULL)
      defin = kgeom_locate_primitive_defin(primitive);
   
   /* -- if they match, then all is well -- */
   if (manifest->primitives[pos]->primitive == primitive)
      return TRUE;

   /* -- if no primitive has yet been set, then establish this one -- */
   if (manifest->primitives[pos]->primitive == KNULL_PRIMITIVE)
      return kgeom_establish_manifest_primitive(manifest, pos, defin);
   
   /* -- primitive doesn't match -- */
   kgeom_error(KGEOM_EPRIMITIVE_NO_MATCH_POSITION);

   return FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_get_manifest_primitive_list
|
|       Purpose: This routine will return a static array of 
|                containing the primitive identifiers contained
|		 within the specified geometry manifest.
|
|         Input: manifest - the manifest from which to get 
|			    the primitive list
|        Output: none
|
|       Returns: the primitive list
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
int *
kgeom_get_manifest_primitive_list(
   kgeom_manifest *manifest)
{
   static int *list = NULL;
   static int  list_size;
   int         i;
   
   /* -- sanity check -- */
   if (manifest == NULL)
      return NULL;

   /* -- make sure static primitive list can hack the number of prims -- */
   if (manifest->num_primitives > list_size)
      list = krealloc(list, (list_size += KLENGTH) * sizeof(int));

   /* -- traverse the primitives in the manifest and fill out the list -- */
   for (i = 0; i < manifest->num_primitives; i++)
      list[i] = manifest->primitives[i]->primitive;

   return list;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_set_manifest_primitive_list
|
|       Purpose: This routine will take a new primitive
|		 list, traverse the primitives in the 
|		 manifest, destroy any primitives in the
|		 manifest that do not match the new list,
|		 and establish any primitives in the
|		 new list which have not yet been established.
|
|		 After this routine, the primitives in the
|		 manifest will correctly represent the 
|		 primitive identifiers in the list that
|		 was provided.
|
|         Input: manifest - the manifest in which to set 
|			    the new primitive list
|		 list     - the new list of primitives
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
int 
kgeom_set_manifest_primitive_list(
   kgeom_manifest *manifest,
   int            *list)
{
   int i;

   /* -- sanity checks -- */
   if (manifest == NULL)
      return FALSE;

   if ((manifest->num_primitives != 0) && list == NULL)
      return FALSE;
   
   /* -- traverse the primitives in the manifest and fill out the list -- */
   for (i = 0; i < manifest->num_primitives; i++)
   {
      /* -- check if new primitive matches what is on old manifest -- */
      if (manifest->primitives[i]->primitive != list[i])  
      {
	 /* -- destroy the old one -- */
	 _kgeom_free_primitive(manifest->primitives[i]);

	 /* -- construct the new one -- */
	 if (list[i] == KNULL_PRIMITIVE)
	    manifest->primitives[i] = _kgeom_construct_primitive(list[i], 0);
	 else
	    kgeom_establish_manifest_primitive(manifest, i, 
				        kgeom_locate_primitive_defin(list[i]));
      }
   }
   
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_get_manifest_number_primitives
|
|       Purpose: This routine will return the number of 
|		 primitives containined within a manifest.
|
|         Input: manifest - the manifest from which to get 
|			    the number of primitives
|        Output: none
|
|       Returns: the number of primitives
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
int 
kgeom_get_manifest_number_primitives(
   kgeom_manifest *manifest)
{
   /* -- sanity check -- */
   if (manifest == NULL)
      return NULL;

   return manifest->num_primitives;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_set_manifest_number_primitives
|
|       Purpose: This routine will change the number of
|		 primitives in a given manifest to match
|		 the new specified number.  If the new
|		 number necessitates the list to grow,
|		 the list will be filled out with 
|	     	 null primitives.
|
|         Input: manifest - the manifest in which to set 
|			    the new number of primitives
|		 num      - the new number of primitives
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
int 
kgeom_set_manifest_number_primitives(
   kgeom_manifest *manifest,
   int             num)
{
   int i;

   /* -- sanity checks -- */
   if (manifest == NULL)
      return FALSE;

   /* -- make sure this number isn't something outrageous -- */
   if (num < 0) 
      kgeom_fail_error(KGEOM_EINVALID_NUMBER_PRIMITIVES);

   /* -- that was sure easy -- */
   if (manifest->num_primitives == num)
      return TRUE;

   /* -- if shrinking, free the excess primitives the list -- */
   if (manifest->num_primitives > num)
   {
      for (i = num; i < manifest->num_primitives; i++)
	 _kgeom_free_primitive(manifest->primitives[i]);
      manifest->primitives = krealloc(manifest->primitives, 
				      num * sizeof(kgeom_primitive *));
   }
   

   /* -- if growing, fill out the new primitives with null primitives -- */
   else 
   {
      manifest->primitives = krealloc(manifest->primitives, 
				      num * sizeof(kgeom_primitive *));

      for (i = manifest->num_primitives; i < num; i++)
	 manifest->primitives[i] = _kgeom_construct_primitive(KNULL_PRIMITIVE,
							      0);
   }

   /* -- make the list equal to the new size -- */
   manifest->num_primitives = num;


   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_write_transcript
|
|       Purpose: This routine will store a given manifest
|		 into the permanent attribute 
|		 KGEOM_MANIFEST_TRANSCRIPT.
|
|		 The manifest structure is basically unrolled
|		 in a deterministic manner into a huge array
|		 of integers and stored in a single attribute.		
|
|	         Additionally, any texture strings which 
|	         may be stored per primitive are stored 
|                here in a separate string attribute.
|
|         Input: obj      - the object in which to store
|			    the manifest
|		 manifest - the manifest to be stored
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) on failure
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
int 
kgeom_write_transcript(
   kobject         obj,
   kgeom_manifest *manifest)
{
   char **textures = NULL;
   int    arg_size = 1;
   int   *transcript;
   int   *ltypes;
   int   *lwidths;
   int   *p;
   int    number;
   int    i;
   int    j;

   /* -- sanity check -- */
   if (manifest == NULL)
      return NULL;
   
   /* -- if any transcript attributes exist already, then blast it -- */
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TRANSCRIPT, 
			    NULL, NULL, NULL, NULL))
      kdms_destroy_attribute(obj, NULL, KGEOM_MANIFEST_TRANSCRIPT);
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TEXTURE, 
			    NULL, NULL, NULL, NULL))
      kdms_destroy_attribute(obj, NULL, KGEOM_MANIFEST_TEXTURE);
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TYPE, 
			    NULL, NULL, NULL, NULL))
      kdms_destroy_attribute(obj, NULL, KGEOM_MANIFEST_TYPE);
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_WIDTH, 
			    NULL, NULL, NULL, NULL))
      kdms_destroy_attribute(obj, NULL, KGEOM_MANIFEST_WIDTH);
   
   /* 
    *    create transcript of proper size and data type :
    *      there will be 1 int for the number of primitives 
    * 	   followed by 3 ints/primitive and 4 ints/component
    *
    *    additional transcript information will be a list of 
    *    strings with one per number of primitives as well as
    *    two integer arrays.  we have to add one to the array
    *    to cope with the single primitive case, as data 
    *    services does not know how to handle arrays of size 1.
    *
    */
   for (i = 0; i < manifest->num_primitives; i++, arg_size += 3)
      arg_size += manifest->primitives[i]->num_components * 4;

   /* -- allocate the transcript -- */
   transcript = kmalloc(arg_size * sizeof(int));
   textures   = kcalloc((manifest->num_primitives+1), sizeof(char *));
   ltypes     = kcalloc((manifest->num_primitives+1), sizeof(int));
   lwidths    = kcalloc((manifest->num_primitives+1), sizeof(int));

   /* -- fill out transcript information -- */
   transcript[0] = manifest->num_primitives;
   for (i = 0, p = &(transcript[1]); i < manifest->num_primitives; i++)
   {
      *p++ = manifest->primitives[i]->primitive;
      *p++ = manifest->primitives[i]->num_components;
      *p++ = manifest->primitives[i]->number;

      for (j = 0; j < manifest->primitives[i]->num_components; j++)
      {
	 *p++ = manifest->primitives[i]->components[j]->present;
	 *p++ = manifest->primitives[i]->components[j]->begin_point;
	 *p++ = manifest->primitives[i]->components[j]->size[0];
	 *p++ = manifest->primitives[i]->components[j]->size[1];
      }

      /* -- per-primitive attributes -- */
      textures[i] = manifest->primitives[i]->texture;
      ltypes[i]   = manifest->primitives[i]->type;
      lwidths[i]  = manifest->primitives[i]->width;
   }
   
   /* -- store the transcript -- */
   if (!kdms_create_attribute(obj, NULL, KGEOM_MANIFEST_TRANSCRIPT, 1,
			      arg_size, KINT, TRUE, TRUE))
      return FALSE;

   if (!kdms_set_attribute(obj, NULL, KGEOM_MANIFEST_TRANSCRIPT, transcript))
      return FALSE;

   /* -- store the per-primitive attributes -- */
   number = manifest->num_primitives+1;

   /* -- per-primitive textures -- */
   if (!kdms_create_attribute(obj, NULL, KGEOM_MANIFEST_TEXTURE, 1,
			      number, KSTRING, TRUE, TRUE))
      return FALSE;
   if (!kdms_set_attribute(obj, NULL, KGEOM_MANIFEST_TEXTURE, textures))
      return FALSE;

   /* -- per-primitive types -- */
   if (!kdms_create_attribute(obj, NULL, KGEOM_MANIFEST_TYPE, 1,
			      number, KINT, TRUE, TRUE))
      return FALSE;
   if (!kdms_set_attribute(obj, NULL, KGEOM_MANIFEST_TYPE, ltypes))
      return FALSE;

   /* -- per-primitive width -- */
   if (!kdms_create_attribute(obj, NULL, KGEOM_MANIFEST_WIDTH, 1,
			      number, KINT, TRUE, TRUE))
      return FALSE;
   if (!kdms_set_attribute(obj, NULL, KGEOM_MANIFEST_WIDTH, lwidths))
      return FALSE;

   kfree(transcript);
   kfree(textures);
   kfree(ltypes);
   kfree(lwidths);

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_read_transcript
|
|       Purpose: This routine will retrieve a given manifest
|		 from the permanent attribute 
|		 KGEOM_MANIFEST_TRANSCRIPT.
|
|         Input: obj - the object from which to get the manifest
|			    
|        Output: none
|
|       Returns: the manifest
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
kgeom_manifest *
kgeom_read_transcript(kobject obj)
{
   kgeom_manifest  *manifest;
   kgeom_primitive *primitive;
   char **textures = NULL;
   int 	 *transcript;
   int   *ltypes   = NULL;
   int   *lwidths  = NULL;
   int 	 *p;
   int 	  present;
   int 	  begin_point;
   int    primitive_id;
   int    num_components;
   int 	  size[2];
   int 	  i;
   int 	  j;
   
   /* -- get the attribute which is the transcript of the manifest  -- */
   if (!kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TRANSCRIPT, 
			     NULL, NULL, NULL, NULL))
      return NULL;

   if (!kdms_get_attribute(obj, NULL, KGEOM_MANIFEST_TRANSCRIPT, &transcript))
      return NULL;

   /* -- get the per-primitive attributes -- */

   /* -- per-primitive textures -- */
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TEXTURE,
			    NULL, NULL, NULL, NULL))
      if (!kdms_get_attribute(obj, NULL, KGEOM_MANIFEST_TEXTURE, &textures))
	 return NULL;

   /* -- per-primitive types -- */
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TYPE,
			    NULL, NULL, NULL, NULL))
      if (!kdms_get_attribute(obj, NULL, KGEOM_MANIFEST_TYPE, &ltypes))
	 return NULL;
   
   /* -- per-primitive widths -- */
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_WIDTH,
			    NULL, NULL, NULL, NULL))
      if (!kdms_get_attribute(obj, NULL, KGEOM_MANIFEST_WIDTH, &lwidths))
	 return NULL;


   p = transcript;

   /* -- start with the manifest -- */
   manifest = _kgeom_construct_manifest();

   /* -- allocate pointers to the number of primitives needed -- */
   manifest->num_primitives = *p++;
   manifest->primitives = kmalloc(manifest->num_primitives *
				  sizeof(kgeom_primitive *));

   /* -- get each primitive -- */
   for (i = 0; i < manifest->num_primitives; i++)
   {
      primitive_id   = *p++;
      num_components = *p++;
      
      primitive = _kgeom_construct_primitive(primitive_id, num_components);

      primitive->number  = *p++;

      /* -- per primitive attributes -- */
      primitive->texture = (textures == NULL) ? NULL : textures[i];
      primitive->width   = (lwidths  == NULL) ? 1 : lwidths[i];
      primitive->type    = (ltypes   == NULL) ? 1 : ltypes[i];

      /* -- get each component within that primitive -- */
      for (j = 0; j < primitive->num_components; j++)
      {
	 present     = *p++;
	 begin_point = *p++;
	 size[0]     = *p++;
	 size[1]     = *p++;
	 
	 primitive->components[j] = kgeom_construct_component(present, 
							      begin_point, 
							      size);
      }
      manifest->primitives[i] = primitive;
   }

   return manifest;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_get_manifest
|
|       Purpose: This routine will get the primitive manifest
|		 from the specified object.  If the object does
|		 not have a manifest, one will be retored from
|		 the manifest transcript.  If there is no manifest
|		 transcript, then there is no manifest to return.
|
|         Input: obj - the object from which to get the manifest
|			    
|        Output: none
|
|       Returns: the manifest
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
kgeom_manifest *kgeom_get_manifest(kobject obj)
{
   kgeom_manifest *manifest;
   
   /* -- do we have the manifest already figured out? -- */
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST, 
			    NULL, NULL, NULL, NULL))
   {
      if (!kdms_get_attribute(obj, NULL, KGEOM_MANIFEST, &manifest))
	 return NULL;
      else
	 return manifest;
   }

   /* -- see if there's a transcript of earlier manifest -- */
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TRANSCRIPT,
			    NULL, NULL, NULL, NULL))
   {
      manifest = kgeom_read_transcript(obj);
      
      /* -- we want to keep this manifest around in memory... -- */
      kdms_create_attribute(obj, NULL, KGEOM_MANIFEST, 1, 1, KSTRUCT, 
			    FALSE, TRUE);

      /* -- ... so we associate it with the object -- */
      kdms_set_attribute(obj, NULL, KGEOM_MANIFEST, manifest);

      return manifest;
   }

   /* -- no manifest, or transcript of there ever having been one -- */
   kgeom_error(KGEOM_ENO_PRIMITIVE_LIST);
   
   return NULL;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_destroy_manifest
|
|       Purpose: This routine will destroy the primitive manifest
|		 and the manifest transcipt from the specified object.
|
|         Input: obj - the object from which to destroy
|			    
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
int kgeom_destroy_manifest(kobject obj)
{
   kgeom_manifest *manifest;
   
   /* -- free the manifest if we already have it in memory -- */
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST, 
			    NULL, NULL, NULL, NULL))
   {
      kdms_get_attribute(obj, NULL, KGEOM_MANIFEST, &manifest);
      _kgeom_free_manifest(manifest);

      kdms_destroy_attribute(obj, NULL, KGEOM_MANIFEST);
   }

   /* -- blast the manifest transcript if we have one -- */
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TRANSCRIPT,
			    NULL, NULL, NULL, NULL))
      kdms_destroy_attribute(obj, NULL, KGEOM_MANIFEST_TRANSCRIPT);

   /* -- blast all per-primitive attributes if we have them -- */
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TEXTURE,
			    NULL, NULL, NULL, NULL))
      kdms_destroy_attribute(obj, NULL, KGEOM_MANIFEST_TEXTURE);
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_TYPE,
			    NULL, NULL, NULL, NULL))
      kdms_destroy_attribute(obj, NULL, KGEOM_MANIFEST_TYPE);
   if (kdms_query_attribute(obj, NULL, KGEOM_MANIFEST_WIDTH,
			    NULL, NULL, NULL, NULL))
      kdms_destroy_attribute(obj, NULL, KGEOM_MANIFEST_WIDTH);
   
   return TRUE;
}
