 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Color Compression Utilities
   >>>> 
   >>>>   Static: 
   >>>>  Private: 
   >>>>   Public: 
   >>>>			kcolor_gamut_object()
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/*
 *              This routine uses a modification of Heckbert's
 *              median cut algorithm: Ref: P. Heckbert,
 *              "Color Image Quantization for Frame Buffer Display",
 *              Computer Graphics, Vol. 16, No. 3, July 1982,
 *              p297.
 */

struct color
  {
    unsigned char vec[4];                 /* Color vector */
    int count;                            /* Pixel count */
    struct color *prev;                   /* Ptrs to previous & next color */
    struct color *next;

    struct pixel *pixlist;                /* List of pixels of this color */
    struct pixel *pixtail;                /* End of pixel list */
  };

struct pixel
  {
    unsigned short x,y;                   /* Location of pixel */
    struct pixel *next;                   /* Pointer to next one */
  };

#define MAX_CHANLS 262144                 /* Max number of hash indices */

struct color *chead,*ctail;               /* Composite color list */
struct color **ch,**ct;                   /* Hash index color lists */

struct box
  {
    unsigned char vmin[4],vmax[4];	  /* Bounds on box */
    struct color *colors;                 /* List of colors in box */
    struct box *next;                     /* Next box */
    int count;                            /* Number of colors in box */
  };

struct box *boxhead,*boxtail;

struct pixel *pixpile,*pixels;            /* Pointers to pixel area */
#define pmalloc() pixpile++               /* Pixel handout function */

#define MAX_CBLOCKS 16384                 /* Max number of color blocks */
#define CNUM 2048                         /* Number of colors/block */
struct color *cblock[MAX_CBLOCKS];        /* Color block pointer array */
int cbcount,ccount;                       /* Color block count, colors/block*/

int hash;                                 /* Global hash index */
int vlen;                                 /* Global vector length */

unsigned char rshift[256];                /* Shift tables */
unsigned char lshift[256];                /* Shift tables */

int height,width,elements,depth,ttime;	  /* Image component counts */

/*-- prototypes for static functions used by kcolor_gamut_object --*/

static int		load_color_list		PROTO((kobject));
static struct color *	cmalloc			PROTO((void));
static void		gamut_head_swap		PROTO((struct color *));
static int		compress_colors		PROTO((int *,int *,float));
static void		box_move_color		PROTO((	struct box *,	\
							struct box *,	\
							struct color **,\
							struct color **));
struct box *		find_biggest_box1	PROTO((void));
struct box *		find_biggest_box2	PROTO((void));
static void		gamut_set_box_bounds	PROTO((struct box *));
static int		gamut_adjust_image	PROTO((kobject,int,int));
static void		free_colors		PROTO((void));
static int		sort_cmaps		PROTO((kobject,int,int));

int kcolor_gamut_object_free(klist *);
klist *objlist=NULL;

/* Fix up a wrapper to use for calling routines so that we don't have to
   explicitly check for errors, free object, etc after each call. */
/* SPECIAL NOTE: THIS SHOULD NOT BE REPLACED WITH THE KCALL FROM
   DATASERV/KAPPUTILS! This version is modified to include a call to
   free_colors. */
#undef KCALL		/* Undefine the original KCALL in kapputils */
#define KCALL(func) \
if (!func) \
  { \
    if (kstrstr(#func,"_set_attribute") != NULL) \
      kerror(lib,rtn,"Unable to set object attributes"); \
    else if (kstrstr(#func,"_get_attribute") != NULL) \
      kerror(lib,rtn,"Unable to get object attributes"); \
    else if (kstrstr(#func,"_get_data") != NULL) \
      kerror(lib,rtn,"Unable to get data from object"); \
    else if (kstrstr(#func,"_put_data") != NULL) \
      kerror(lib,rtn,"Unable to put data from object"); \
    else if (kstrstr(#func,"_reference_object") != NULL) \
      kerror(lib,rtn,"Unable to reference_object"); \
    else if (kstrstr(#func,"_create_") != NULL) \
      kerror(lib,rtn,"Unable to create segment or attribute"); \
    else if (kstrstr(#func,"_destroy_") != NULL) \
      kerror(lib,rtn,"Unable to destroy segment or attribute"); \
    else if (kstrstr(#func,"_open_") != NULL) \
      kerror(lib,rtn,"Unable to open object"); \
    else if (kstrstr(#func,"_close_") != NULL) \
      kerror(lib,rtn,"Unable to close object"); \
    else if (kstrstr(#func,"_copy_") != NULL) \
      kerror(lib,rtn,"Unable to copy segment, attribute, data, or object"); \
    else \
      kerror(lib,rtn,"Got an error from subroutine call:\n"#func,lib,rtn); \
    kinfo(KSYSLIB,"%s:%s: Offending subroutine call:\n"#func,lib,rtn); \
    free_colors(); \
    (void)klist_free(objlist, (kfunc_void)lgamut_free); \
    return(0); \
  }

/* This code stolen from KAPPUTILS/utilities.c. It's duplicated here because
   the KAPPSERV lib is loaded AFTER the KAPPUTILS lib, so those routines are
   not available to this library. ligamut_free is really just lkcall_free
   copied to this file to make the KCALL macro work. */
int lgamut_free(klist *p)
  {
    char *lib="kappserv";
    char *rtn="lgamut_free";

    if (kstrcmp(p->client_data,"KOBJECT")==0)
      {
        if (!kpds_close_object((kobject)(p->identifier)))
          kerror(lib,rtn,"Unable to close object");
      }
    else if (kstrcmp(p->client_data,"KMALLOC")==0)
      kfree(p->identifier);
    else
        (void)kinfo(KSYSLIB,"lgamut_free: Unrecognized item in list: %s\n",
                             p->client_data);
    return(TRUE);
  }

/****************************************************************
* 
*  Routine Name: kcolor_gamut_object - perform color quantization of
*  				       1..4 plane images
* 
*       Purpose: kcolor_gamut_object uses a variation of Paul
*		 Heckbert's median cut algorithm to perform color
*		 quantization of one, two, three, or four plane images
*		 producing a single-plane image with color map.
*
*		 The quantization is performed by isolating clusters
*		 of "neighboring" color vectors in a three dimensional
*		 histogram, with each axis being one of the color
*		 components.  The clusters are obtained using a
*		 modified version of Heckbert's median cut.  The true
*		 colors are then matched to the closest cluster, and
*		 the input vector is then re-mapped to an n-color
*		 pseudo color image.
*
*		 To keep the histogram from becoming exceedingly large
*		 (max of around 2^24 bytes), one may need to quantize
*		 the grey levels of the input bands to less than 8
*		 bits. 6 bits (64 levels) gives results that are
*		 reasonable in a short amount of time. The number of
*		 bits that are kept is called the color precision,
*		 which can be specified at execution time.  The
*		 general tradeoff is that smaller precision is faster
*		 and takes less memory, but it looks worse too. High
*		 precision takes longer and great gobs of memory, but
*		 looks decent, provided that a reasonable number (say
*		 128 or more) colors is specified.  The execution time
*		 is very dependent on the image statistics. In
*		 general, a small number of colors is faster than a
*		 large number of colors.  In either case, if the image
*		 has good spatial color coherence, execution time is
*		 greatly reduced.
*
*		 The allocation fraction controls how large areas
*		 of nearly the same color are handled. An allocation
*		 fraction of 0.0 will cause the large areas to be
*		 broken into as many colors as possible with the
*		 largest areas of a particular color range being
*		 broken first. An allocation fraction of 1.0 will
*		 attempt to preserve the detail in the image by
*		 preserving the color range of all parts of the image
*		 at the expense of smooth coloring of the larger
*		 areas. An allocation fraction of around 0.2 to 0.5
*		 gives very good results on most images.
*
*		 If the input image contains less than the number of *
*		 colors requested then the output image will contain *
*		 only the number of colors present in the input *
*		 image. The color map will contain the number of *
*		 entries requested (meaning all colors in the image) *
*		 with any extra entries zero padded.
*
*		 Multiple plane images are processed by quantizing
*		 each plane independently, generating a corresponding
*		 plane of colors in the map. Thus an input object with
*		 (w,h,d,t,e)=(512,480,10,10,4) will result in an
*		 output object with a value segment with dimensions
*		 (512,480,10,10,1) and a map segment with dimensions
*		 (4,10,10,10,1).
*
*         Input: src - object to be color quanitized
*                ncolors - desired number of colors for result image, 
*                          should be (1..65535)
*                bits - number of bits of resolution to keep from each
*                       color plane. Should be(1..8). Higher values give
*                       better results but take longer and require more
*                       memory.
*                fraction - fraction of color splits to be base on the
*                           span of the color space versus the population 
*                           count in the color space. 1.0 means split only
*                           on span. 0.0 means split only on population.
*                           0.0 is effectively a popularity contest. You will
*                           usually get best results somewhere around 0.5.
*
*        Output: dest - object to hold quantized result
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Scott Wilson
*          Date: Jan 01, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 27-Dec-94 Scott Wilson: Major revisions.
*                          1. Added support for more than 256 output colors
*                          2. Cleaned up much error handling and memory
*                             resource recovery code.
*                          3. Added support for 1..4 bands of color data,
*                             which means that you can handle RGB alpha.
*                          4. Use better hasher now, added hash table
*                             instrumentation and statistics.
*			   5. Force a map size of ncolors regardless of
*                             the number actually present.
*                1-Jan-95 Scott Wilson:
*			   1. Added support for multi-plane processing.
*   Declaration: int kcolor_gamut_object (
*                !   kobject src,
*                !   int ncolors,
*	         !   int bits,
*                !   float fraction,
*                !   kobject dest )
****************************************************************/
int kcolor_gamut_object (
	kobject	src,
	int	ncolors,
	int	bits,
	float	fraction,
	kobject	dest )
{
    char *lib = "kappserv", *rtn = "kcolor_gamut_object";
    int shift;                            /* Shift distance for precision */
    int i;
    int dtype;
    int status = TRUE;
    int otype;
    int nimages;
    kobject tmpsrc,tmpdest;
    unsigned char *data;
    int nactual; 			/* Actual number of colors found */
    /* See if we have valid src and dest objects */
    if (src == KOBJECT_INVALID)
      {
        kerror(lib,rtn,"Invalid input object.");
        return(FALSE);
      }
    if (dest == KOBJECT_INVALID)
      {
        kerror(lib,rtn,"Invalid output object.");
        return(FALSE);
      }

    /* Make sure we have a value segment */
    if (!kpds_query_value(src))
      {
        kerror(lib, rtn, "Source object does not contain value data");
        return(FALSE);
      }

    /* Reference the input object to avoid side effects, then add the
       reference to the list of goodies to be autmatically free'd on
       error. */
    KCALL(((src = kpds_reference_object(src)) != NULL));
    objlist = klist_add(objlist,src,"KOBJECT");

    /* Check out the data type of the value segment, but turn on mapping
       first. */
    KCALL(kpds_set_attribute(src,KPDS_MAPPING_MODE,KMAPPED));
    KCALL(kpds_get_attribute(src,KPDS_VALUE_DATA_TYPE,&dtype));
    if (dtype != KBYTE && dtype != KUBYTE)
      {
        kerror(lib, rtn, "Source object must be KBYTE or KUBYTE");
        return(FALSE);
      }

    /* Find out how big the input image is, set global size variables */
    KCALL(kpds_get_attribute(src, KPDS_VALUE_SIZE, 
			&width, &height, &depth, &ttime, &elements));
    nimages = depth*ttime;
    vlen = elements;

    /* Fix up the destination image to have a map type of KUBYTE,
       with a data type consistent with the number of colors requested. */
    if (ncolors >255) otype = KUSHORT;
    else otype = KUBYTE;
    if (!kpds_query_value(dest)) KCALL(kpds_create_value(dest));
    KCALL(kpds_set_attribute(dest,KPDS_VALUE_DATA_TYPE,otype));
    KCALL(kpds_set_attribute(dest,KPDS_VALUE_SIZE,width,height,depth,ttime,1));
    KCALL(kpds_set_attribute(dest,KPDS_VALUE_POSITION,0,0,0,0,0));
    if (!kpds_query_map(dest)) KCALL(kpds_create_map(dest));
    KCALL(kpds_set_attribute(dest,KPDS_MAP_DATA_TYPE,KUBYTE));
    KCALL(kpds_set_attribute(dest,KPDS_MAP_SIZE,vlen,ncolors,depth,ttime,1));
    KCALL(kpds_set_attribute(dest,KPDS_MAP_POSITION,0,0,0,0,0));

    /* Make sure the color attributes copy across */
    KCALL(kcolor_copy_attribute(src,dest,KCOLOR_COLORSPACE));
    KCALL(kcolor_copy_attribute(src,dest,KCOLOR_HAS_ALPHA));

    /* Build shift tables */
    shift = 8-bits;                   /* Get shift distance */
    for (i=0; i<256; i++)
      {
        lshift[i] = i << shift;
        rshift[i] = i >> shift;
      }

    if ((tmpsrc = kpds_create_object()) == KOBJECT_INVALID)
      {
        (void)klist_free(objlist, (kfunc_void)lgamut_free);
        kerror(lib, rtn, "Unable to create temp src object");
        return(FALSE);
      }
    objlist = klist_add(objlist,tmpsrc,"KOBJECT");
    if ((tmpdest = kpds_create_object()) == KOBJECT_INVALID)
      {
        (void)klist_free(objlist, (kfunc_void)lgamut_free);
        kerror(lib, rtn, "Unable to create temp src object");
        return(FALSE);
      }
    objlist = klist_add(objlist,tmpdest,"KOBJECT");

    /* Fix up the source object */
    KCALL(kpds_set_attribute(src,KPDS_VALUE_POSITION,0,0,0,0,0));
    KCALL(kpds_set_attribute(src,
                          KPDS_VALUE_REGION_SIZE,width,height,1,1,vlen));

    /* Fix up the temp src object */
    KCALL(kpds_copy_object_attr(src,tmpsrc));
    KCALL(kpds_set_attribute(tmpsrc,KPDS_VALUE_SIZE,width,height,1,1,vlen));
    KCALL(kpds_set_attribute(tmpsrc,
                          KPDS_VALUE_REGION_SIZE,width,height,1,1,vlen));

    /* Fix up the temp dest object */
    KCALL(kpds_copy_object_attr(dest,tmpdest));
    KCALL(kpds_set_attribute(tmpdest,KPDS_VALUE_DATA_TYPE,otype));
    KCALL(kpds_set_attribute(tmpdest,KPDS_VALUE_SIZE,width,height,1,1,1));
    KCALL(kpds_set_attribute(tmpdest,KPDS_MAP_DATA_TYPE,KUBYTE));
    KCALL(kpds_set_attribute(tmpdest,KPDS_MAP_SIZE,vlen,ncolors,1,1,1));

    /* Allocate space for the input object data buffer. The size is taken
       as unsigned int in case that is what ends up getting shoved in. */
    KCALL(!((data=(unsigned char *)kmalloc(width*height*vlen*sizeof(unsigned int))) == NULL));
    objlist = klist_add(objlist,data,"KMALLOC");

    for (i=0; i<nimages; i++)
      {
        /* Initialize variables used for fast dynamic memory handouts */
        cbcount = 0;
        ccount = 0;
        pixels = (struct pixel *)kmalloc(height*width*sizeof(struct pixel));
        /* Note: pixels will be free'd by the free_color routine */
        if (pixels == NULL)
          {
            kfprintf(kstderr,"kcolor_gamut_object: cannot allocate pixel space!\n");
            return FALSE;
          }
        kbzero(pixels,height*width*sizeof(struct pixel));
        pixpile = pixels;

        /* Pull out one vector-plane of value data and make a
           new object out of it. Quantize that object into
           a new one with a scalar-plane value segment and it's
           associated map. Slap the new value data sequentially
           into the output value segment and the new map data 
           sequentially into the output map segment. */
        data = (unsigned char *)kpds_get_data(src,KPDS_VALUE_REGION,(kaddr *)data);
        KCALL(kpds_set_attribute(tmpsrc,
                          KPDS_VALUE_REGION_SIZE,width,height,1,1,vlen));
        KCALL(kpds_set_attribute(tmpsrc,KPDS_VALUE_POSITION,0,0,0,0,0));
        KCALL(kpds_put_data(tmpsrc,KPDS_VALUE_REGION,(kaddr)data));
        KCALL(kpds_set_attribute(tmpsrc,KPDS_VALUE_POSITION,0,0,0,0,0));
        KCALL(kpds_set_attribute(tmpdest,KPDS_VALUE_POSITION,0,0,0,0,0));
        KCALL(kpds_set_attribute(tmpdest,KPDS_MAP_POSITION,0,0,0,0,0));

        /* Do the quantization */
        status =load_color_list(tmpsrc)			&&
		compress_colors(&ncolors, &nactual, fraction)	&&
		gamut_adjust_image(tmpdest, ncolors, nactual);

        /* Handle the value data */
        KCALL(kpds_set_attribute(tmpdest,KPDS_VALUE_POSITION,0,0,0,0,0));
        data = (unsigned char *)kpds_get_data(tmpdest,KPDS_VALUE_PLANE,(kaddr *)data);
        KCALL(kpds_put_data(dest,KPDS_VALUE_PLANE,(kaddr)data));

        /* Handle the map data */
        KCALL(kpds_set_attribute(tmpdest,KPDS_MAP_POSITION,0,0,0,0,0));
        data = (unsigned char *)kpds_get_data(tmpdest,KPDS_MAP_PLANE,(kaddr *)data);
        KCALL(kpds_put_data(dest,KPDS_MAP_PLANE,(kaddr)data));
        free_colors();	/* Give back allocated memory */
      }

    (void)klist_free(objlist, (kfunc_void)lgamut_free);

    return status;
}

/*-----------------------------------------------------------
|  Routine Name: load_color_list - loads colors :-)
|
|       Purpose:
|
|         Input:	src	- the source data object
|       Returns: TRUE on success, FALSE otherwise.
|    Written By: Scott Wilson 
|          Date: 22-mar-94
------------------------------------------------------------*/
static int
load_color_list(kobject	src)
{
    char *lib = "kappserv", *rtn = "kcolor_gamut_object:load_color_list";
    register struct color *pc;
    register unsigned char *data;
    register int i,j,k;
    unsigned char v[4];
    int found;
    int nzacc,nzcount,cccount; /* For hash table statistics */

    chead = NULL;                         /* Initialize color list */
    ctail = NULL;

    /* Allocate space for hash index lists */
    ch = (struct color **)kmalloc(MAX_CHANLS*sizeof(struct color *));
    ct = (struct color **)kmalloc(MAX_CHANLS*sizeof(struct color *));
    if (ch == NULL || ct == NULL)
      {
        kerror("kappserv",rtn,"Unable to allocate hash lists!\n");
        return FALSE;
      }
    kbzero(ch,MAX_CHANLS*sizeof(struct color *));
    kbzero(ct,MAX_CHANLS*sizeof(struct color *));

    KCALL(kpds_set_attribute(src,KPDS_VALUE_REGION_SIZE,width,1,1,1,vlen));
    data = NULL;
    for (i=0; i<height; i++)
      {
        if ((data = (unsigned char *)kpds_get_data(src,KPDS_VALUE_REGION,data)) == NULL)
          {
            kerror("kappserv",rtn,"kpds_get_data failed");
            kfree(ch); kfree(ct); kfree(data);
            return FALSE;
          }
        for (j=0; j<width; j++)
          {
            for (k=0; k<vlen; k++) 
              v[k] = rshift[(int)(data[j+k*width])];  /* Quantize */

            /* Go update the clist */
            hash = khash((char *)v,vlen) & 0x3ffff;
            pc = ch[hash];
            while (pc != NULL)
              {
                found = 1;
                for (k=0; k<vlen; k++)
                  {
                    if (pc->vec[k] != v[k])
                      {
                        found = 0;
                        break;
                      }
                  }
                if (found == 1) break;
                else pc = pc->next;
              }     
            if (pc == NULL)
              {
                /* Add a color with the current values */
                pc = cmalloc();
                if (pc == NULL)
                  {
                    kfree(ch); kfree(ct); kfree(data);
                    return FALSE;
                  }
                if (ch[hash] == NULL && ct[hash] == NULL)
                  {
		    /* Empty color list */
                    ch[hash] = pc;
                    ct[hash] = pc;
                    pc->prev = NULL;
                    pc->next = NULL;
                  }
                else
                  {
                    /* Not empty, add to head */
                    pc->prev = NULL;
                    pc->next = ch[hash];
                    ch[hash]->prev = pc;
                    ch[hash] = pc;
                  }
                for (k=0; k<vlen; k++) pc->vec[k] = v[k];
                pc->count = 1;
                pc->pixlist = pmalloc();
                pc->pixlist->next = NULL;
                pc->pixlist->x = (unsigned short) j;
                pc->pixlist->y = (unsigned short) i;
                pc->pixtail = pc->pixlist;
              }
            else
              {
                /* Update the pixel list of this color */
                pc->count++;
                pc->pixtail->next = pmalloc();
                pc->pixtail = pc->pixtail->next;
                pc->pixtail->next = NULL;
                pc->pixtail->x = (unsigned short) j;
                pc->pixtail->y = (unsigned short) i;
                if (pc != ch[hash]) gamut_head_swap(pc);
              }
          }
      }
    kfree(data);

    /* Do some hash table analysis if the notify level is
       kVERBOSE */
    if (kget_notify() == KSYSLIB)
      {
        nzcount = 0;
        nzacc = 0;
        for (i=0; i<MAX_CHANLS-1; i++)
          {
            cccount = 0;
            if (ch[i] != NULL)
              {
                nzcount++;
                pc = ch[i];
                while (pc != NULL)
                  {
                    cccount++;
                    pc = pc->next;
                  }
              }
            nzacc += cccount;
          }
        kinfo(KSYSLIB,"kcolor_gamut_object: Hash table information:\n");
        kinfo(KSYSLIB,"kcolor_gamut_object:   Hash table length:%d\n",MAX_CHANLS);
        kinfo(KSYSLIB,"kcolor_gamut_object:   Used table entries:%d\n",nzcount);
        kinfo(KSYSLIB,"kcolor_gamut_object:   Avg used entry occupancy :%f\n",
                                          (float)nzacc/(float)nzcount);
      }

    /* Now append all color lists together */
    for (i=0; i<MAX_CHANLS; i++)
      {
        if (ch[i] != NULL)     /* If have something to add */
          {
            if (ctail != NULL) /* If main list not empty */
              {
                ctail->next = ch[i];
                ch[i]->prev = ctail;
                ctail = ct[i];
              }
            else               /* Main list is empty */
              {
                chead = ch[i];
                ctail = ct[i];
              }
          }
      }
    kfree(ch);
    kfree(ct);

    return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: cmalloc - allocate a color structure
|       Purpose:
|
|       Returns: a pointer to a color structure, or NULL on failure.
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
static struct color *
cmalloc(void)
  {
    char *lib = "kappserv", *rtn = "kcolor_gamut_object:cmalloc";
    /* Maintain a list of blocks of CNUM colors, allocated in chunks and then
       doled out as needed. */
    if (cbcount == 0 || ccount >= CNUM)
      {
        if (cbcount >= MAX_CBLOCKS)
          {
            kerror(lib,rtn,
		"Too many color blocks!\n Recompile with larger MAX_CBLOCKS");
            return NULL;
          }
        cblock[cbcount]=(struct color *)kmalloc(CNUM*sizeof(struct color));
        kbzero(cblock[cbcount],CNUM*sizeof(struct color));
        if (cblock[cbcount] == NULL)
          {
            kerror(lib,rtn,"Unable to allocate color block!");
            return NULL;
          }
        cbcount++;
        ccount = 0;
      }
    return(cblock[cbcount-1]+(ccount++));
  }

/*-----------------------------------------------------------
|  Routine Name: gamut_head_swap - move color to head of list
|       Purpose:
|
|         Input:
|        Output:
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
static void
gamut_head_swap(
	register struct color *p)
  {
    register int k;

    k = khash((char *)(p->vec),vlen) & 0x3ffff;

    /* Delete current color from list */
    if (p == ct[k])
      {
        p->prev->next = p->next;
        ct[k] = p->prev;
      }
    else
      {
        p->prev->next = p->next;
        p->next->prev = p->prev;
      }

    /* Re-insert color at head of list*/
    p->next = ch[k];
    p->prev = NULL;
    ch[k]->prev = p;
    ch[k] = p;
  }
 
/*-----------------------------------------------------------
|  Routine Name: compress_colors - fill in the blank :-)
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
static int
compress_colors(
	int	*ncolors,
	int	*nactual,
	float	fraction)
{
    char *lib = "kappserv", *rtn = "kcolor_gamut_object:compress_colors";
    int i,k,n,count,popok;
    unsigned char vdiff[4],cmax;
    struct color *p1,*p2;
    register struct box *b1,*b2;
    
    /* Go find out how many colors are present in the image */
    count = 0;
    p1 = chead;
    while (p1 != NULL)
      {
        count++;
        p1 = p1->next;
      }

    /* If verbose is turned on tell how many colors were found. */
    kinfo(KSYSLIB,"kcolor_gamut_object: Image contains %d colors at the requested color precision.\n",count);

    /* Now see if the user has requested more colors than exist in the
       image. Spit out a message if verbose is true. Return only the
       number of colors that actually exist. */
    if (*ncolors > count)
      {
        kinfo(KSYSLIB,"kcolor_gamut_object: Found %d colors, user requested %d\n",count,*ncolors);
        kinfo(KSYSLIB,"kcolor_gamut_object: Returning image with %d colors\n",count);
        *nactual = count;
      }
    else
      *nactual = *ncolors;

    /* Initialize box structure */
    boxhead = NULL;
    boxtail = NULL;

    /* Make first box and attach all colors to it */
    b1 = (struct box *)kmalloc(sizeof(struct box));
    kbzero(b1,sizeof(struct box));
    if (b1 == NULL)
      {
        kerror(lib,rtn,"not enough memory for root box!");
        return FALSE;
      }
    boxhead = b1;
    boxtail = b1;
    b1->colors = chead;
    gamut_set_box_bounds(b1);
    b1->next = NULL;
    
    /* Now go through and perform a median cut on the box list */
    /* until the desired number of boxes (colors) are formed. */
    popok = TRUE;
    for (i=0; i<(*ncolors)-1; i++)
      {
        /* Hunt down the biggest box */
        if (i < (*ncolors)*(1.0-fraction) && popok == TRUE) 
          {
            b1 = find_biggest_box1();  /* Population */
            if (b1 == NULL) return TRUE; /* No more colors! */
          }
        else 
          {
retry:      b1 = find_biggest_box2();                /* 2-norm */
            if (b1 == NULL) return TRUE; /* No more colors! */
          }

        /* First make new box */
        b2 = (struct box *)kmalloc(sizeof(struct box));
        kbzero(b2,sizeof(struct box));
        if (b2 == NULL)
          {
            kerror(lib,rtn,"Not enough memory for additional box!");
            return FALSE;
          }
        b2->next = NULL;
        b2->colors = NULL;

        /* Copy the color bounds to the new box. */
        for (k=0; k<vlen; k++)
          {
            b2->vmax[k] = b1->vmax[k];
            b2->vmin[k] = b1->vmin[k];
          }

        /* Compute boundary for splitting the color space */
        for (k=0; k<vlen; k++) vdiff[k] = b1->vmax[k] - b1->vmin[k];
        cmax = vdiff[0]; n = 0;
        for (k=1; k<vlen; k++)
          {
            if (vdiff[k] > cmax)
              {
                cmax = vdiff[k];
                n = k;
              }
          }

        /* Check to make sure that the axis being split has a legal span */
        if (popok == TRUE) /* Splitting on population */
          {
            if (vdiff[n] < 2)
              {
                popok = FALSE;
                kfree(b2);
                goto retry;
              }
          }
        else     /* Splitting on 2-norm */
          {
            if (vdiff[n] < 1) /* Check for a degenerate box */
              {
                kerror(lib,rtn,
                        "Couldn't find a splittable color subspace!\n"
                        "Use more precision, or request fewer colors.");
                return FALSE;
              }
          }

        /* Update the global boxtail pointer */
        boxtail->next = b2;
        boxtail = boxtail->next;

        /* Modify the color bounds for the original and new boxes */
        b1->vmax[n] = b1->vmin[n]+vdiff[n]/2;
        b1->vmin[n] = b1->vmax[n]-vdiff[n]/2;

        /* Partition the color space of the first box */
        p1 = b1->colors;           /* Color list in original box */
        p2 = b2->colors;           /* End of color list in new box */
        while (p1 != NULL)
          {
            if (p1->vec[n] > b1->vmax[n]) box_move_color(b1,b2,&p1,&p2);
            else p1 = p1->next;
          }

        /* Re-compute the color bounds for both boxes */
        gamut_set_box_bounds(b1);
        gamut_set_box_bounds(b2);
      }

    return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: box_move_color - move a box's color
|       Purpose:
|
|         Input:
|        Output:
|       Returns:
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
static void
box_move_color(
	register struct box	*b1,
	register struct box	*b2,
	register struct color	**p1,
	register struct color	**p2)
  {
    /* Move the color p1 from box b1's color list to box b2's color list,
       which has the tail of its color list marked by p2. p1 must be left
       pointing to the next color in b1's color list, while p2 must be left
       pointing at the last color in b2's color list. */

    register struct color *c1;

    c1 = (*p1)->next;           /* Remember where "next" is. */

    /* Unlink color */
    if (b1->colors == *p1 && (*p1)->next != NULL) /* p1 is head of color list */
      {
        b1->colors = (*p1)->next;
        (*p1)->next->prev = NULL;
      }
    else if (b1->colors == *p1 && (*p1)->next == NULL) /* p1 is only color */
      {
        b1->colors = NULL;
      }
    else if (b1->colors != *p1 && (*p1)->next == NULL) /* p1 is tail of list */
      {
        (*p1)->prev->next = NULL;
      }
    else
      {
        (*p1)->next->prev = (*p1)->prev;      /* p1 in middle of list */
        (*p1)->prev->next = (*p1)->next;
      }
    (*p1)->next = NULL;
    (*p1)->prev = NULL;

    /* Append the color to the correct place */
    if (b2->colors == NULL)                  /* Empty color list */
       {
         b2->colors = *p1;
         *p2 = *p1;
       }
    else                                     /* Non-empty color list */
      {
        (*p2)->next = *p1;
        (*p1)->prev = *p2;
        *p2 = *p1;
      }

    *p1 = c1;                   /* Regurgitate "next". */
  }

/*-----------------------------------------------------------
|  Routine Name: find_biggest_box1 - find box with largest population
|       Purpose:
|
|       Returns: pointer to box structure with largest population
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
struct box *
find_biggest_box1(void)
 {
   /* Find the box with the largest population */
   register struct box *b,*b1;
   register int max;

   b = boxhead;
   b1 = NULL;
   max = 0;
   while (b != NULL)
     {
       if (b->count > max)
         {
           max = b->count;
           b1 = b;
         }
       b = b->next;
     }
   return(b1);
 }

/*-----------------------------------------------------------
|  Routine Name: find_biggest_box2 - find box with largest 2-norm
|       Purpose:
|
|       Returns: pointer to box structure with largest 2-norm
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
struct box *
find_biggest_box2(void)
 {
   /* Find the box with the largest 2-norm */
   register struct box *b,*b1;
   register int k,maxnorm,norm,diff;

   b = boxhead;
   b1 = NULL;
   maxnorm = 0;
   while (b != NULL)
     {
       norm = 0;
       for (k=0; k<vlen; k++)
         {
           diff = (int)(b->vmax[k])-(int)(b->vmin[k]);
           norm += diff*diff;
         }
       if (norm > maxnorm)
         {
           maxnorm = norm;
           b1 = b;
         }
       b = b->next;
     }
   return(b1);
 }

/*-----------------------------------------------------------
|  Routine Name: gamut_set_box_bounds - set box bounds
|       Purpose:
|
|         Input: b - pointer to box struct
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
static void
gamut_set_box_bounds(struct box *b)
  {
    register struct color *c;
    int min[4],max[4];
    register int n,k;

    for (k=0; k<vlen; k++)
      {
        min[k] = 255;
        max[k] = 0;
      }

    n = 0;
    c = b->colors;
    while (c != NULL)
      {
        for (k=0; k<vlen; k++)
          {
            if ((int)(c->vec[k]) > max[k]) max[k] = (int)(c->vec[k]);
            if ((int)(c->vec[k]) < min[k]) min[k] = (int)(c->vec[k]);
          }
        c = c->next;
        n++;
      }

    for (k=0; k<vlen; k++)
      {
        b->vmax[k] = (unsigned char)max[k];
        b->vmin[k] = (unsigned char)min[k];
      }
    b->count = n;
  }

/*-----------------------------------------------------------
|  Routine Name: gamut_adjust_image - adjust image in kobject
|       Purpose:
|
|         Input:	dest	-	the kobject to gamut
|			ncolors	-	the number of colors
|       Returns: TRUE if gamut is successful, FALSE otherwise
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
static int
gamut_adjust_image(kobject dest, int ncolors, int nactual)
  {
    char *lib = "kappserv", *rtn = "kcolor_gamut_object:gamut_adjust_image";
    register struct color *c;
    register struct pixel *p;
    int v[4];
    register struct box *bp;
    register unsigned short i;
    unsigned short *d,*dptr;
    unsigned char *m,*mptr;
    kobject destref;
    int k;

    /* Allocate space for the new image data */
    dptr = (unsigned short *)kmalloc(height*width*sizeof(unsigned short));
    if (dptr == NULL)
      {
        kerror(lib,rtn,"Not enough memory for new image data!");
        return FALSE;
      }
    d = dptr;
    KCALL(((destref = kpds_reference_object(dest)) != NULL));
    KCALL(kpds_set_attribute(destref,KPDS_VALUE_DATA_TYPE,KUSHORT));

    /* Read thru the box list stepping thru its color list and placing
       the pixels in the output image */
    i = 0;
    bp = boxhead;
    while (bp != NULL)
      {
        c = bp->colors;
        while (c != NULL)
          {
            p = c->pixlist;
            while (p != NULL)
              {
                d[p->y*width+p->x] = i;
                p = p->next;
              }
            c = c->next;
          }
        bp = bp->next;
        i++;
      }

    /* Put the new image data to the output object */
    if (!kpds_put_data(destref,KPDS_VALUE_PLANE,dptr))
      {
        kerror("kappserv","kcolor_gamut_object","Unable to write output image data to output object!");
        return FALSE;
      }
    kfree(dptr);
    KCALL(kpds_close_object(destref));
 
    /* Compute the new color for each box as the weighted average of all
       colors inside the box */
    bp = boxhead;
    while (bp != NULL)
      {
        i = 0;
        for (k=0; k<vlen; k++) v[k] =  0;
        c = bp->colors;
        while (c != NULL)
          {
            for (k=0; k<vlen; k++) v[k] += c->count*(int)(c->vec[k]);
            i += c->count;
            c = c->next;
          }
        for (k=0; k<vlen; k++) bp->vmin[k] = (unsigned char)(v[k]/(float)i);
        bp = bp->next;
      }

    /* Build new color map */
    mptr = (unsigned char *)kmalloc(vlen*ncolors*sizeof(unsigned char));
    if (mptr == NULL)
      {
        kerror(lib,rtn,"Not enough memory for RGB color maps!");
        return FALSE;
      }
    kbzero(mptr,vlen*ncolors*sizeof(unsigned char));
    m = mptr;
    bp = boxhead;
    while (bp != NULL)
      {
        for (k=0; k<vlen; k++) 
          *m++ = lshift[bp->vmin[k]]; /* Code back to 8 bits */
        bp = bp->next;
      }

    /* Set up the map part of the output object and write the new image data 
       to it */ 
    KCALL(kpds_set_attribute(dest,KPDS_MAP_SIZE,vlen,ncolors,1,1,1));
    if (!kpds_put_data(dest,KPDS_MAP_PLANE,mptr))
      {
        kerror(lib,rtn,"Unable to write output image mapdata to output object!");
        kfree(mptr);
        return FALSE;
      }
    kfree(mptr);

    /* Sort the color maps and remap the data */
    return sort_cmaps(dest,ncolors,nactual);
  }

/*-----------------------------------------------------------
|  Routine Name: free_colors - free all allocated data, including
|                the color and box lists. Locally malloc'd
|                stuff is taken care of inside each routine.
|       Purpose:
|
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
static void
free_colors(void)
  {
    /* Release all malloc()'ed items except the new colormap. */
    struct box *b,*b1;
    int i;

    kfree(pixels);
    for (i=0; i<cbcount; i++) kfree(cblock[i]);

    b = boxhead;
    while (b != NULL)
      {
        b1 = b->next;
        kfree(b);
        b = b1;
      }
  }

/*-----------------------------------------------------------
|  Routine Name: sort_cmaps - sort colormaps
|       Purpose: Employ a very ad-hoc color map sorting
|                scheme to try and keep related colors
|                near to each other in the colormap. Only
|                The first three bands are used in the sort -
|                additional bands are assumed not to be
|                of interest for sorting purposes.
|                If less than three bands are present, then
|                replication is used to get three to work
|                with, simplifying the code. This whole
|                subterfuge should be redone sometime.
!
!                Special note: Only the colors actually used
!                are sorted; the remainder of the output map
!                is padded with zeros.
|
|         Input:	dest	-	kobject containing image
|			ncolors	-	number of colors
|       Returns: TRUE on success, FALSE otherwise
|    Written By: Scott Wilson
|          Date: 22-mar-94
------------------------------------------------------------*/
static int
sort_cmaps(kobject dest, int ncolors, int nactual)
  {
    char *lib = "kappserv", *rtn = "kcolor_gamut_object:sort_cmaps";
    int i,j,k,jold;
    unsigned int *mapd,*r,*g,*b,dr,dg,db,er,eb,eg,trail;
    unsigned char data[4];
    unsigned short *d;
    unsigned char *m;
    unsigned int *used,*trans,v,dark,nearby;
    kobject destref;

    mapd = (unsigned int *)kmalloc(4*nactual*sizeof(unsigned int));
    kbzero(mapd,4*nactual*sizeof(unsigned int));
    r = mapd+0;
    g = mapd+ncolors;
    b = mapd+2*ncolors;
    used = (unsigned int *)kmalloc(nactual*sizeof(unsigned int));
    kbzero(used,nactual*sizeof(int));
    trans = (unsigned int *)kmalloc(nactual*sizeof(unsigned int));

    destref = kpds_reference_object(dest);

    /* Read all of the map into a temp storage area */
    KCALL(kpds_set_attribute(destref,KPDS_MAP_POSITION,0,0,0,0,0));
    for (i=0; i<nactual; i++)
      {
        kpds_get_data(destref,KPDS_MAP_LINE,data);
        for (k=0; k<vlen; k++) mapd[i+k*nactual] = data[k];
        /* Replicate last band into empty bands to get three total */
        for (k=vlen-1; k<3; k++) mapd[i+(k+1)*nactual] = data[vlen-1];
      }

    j = 0;
    dark = r[0]*r[0]+g[0]*g[0]+b[0]*b[0];
    for (i=1; i<nactual; i++)
      {
        v = r[i]*r[i]+g[i]*g[i]+b[i]*b[i];
        if (v < dark)
          {
            j = i;
            dark = v;
          }
      }
    if (r[j] < g[j]) trail = 1;
    else if (g[j] < b[j]) trail = 2;
    else trail=3;

    /* Begin filling in the output list */
    trans[j] = 0;
    used[j] = 1;
    jold = j;
    for (i=1; i<nactual; i++)
      {
        nearby = KMAXLINT;
        for (k=0; k<nactual; k++)
          {
            if (used[k] == 0)
              {
                dr = abs((int)r[jold]-(int)r[k]);
                dg = abs((int)g[jold]-(int)g[k]);
                db = abs((int)b[jold]-(int)b[k]);
                if (trail == 1) dr *= 0.2;
                else if (trail == 2) dg *= 0.2;
                else db *= 0.2;
                er = dr+0.1*r[k]*(i/(float)nactual);
                eg = dg+0.1*g[k];
                eb = db+0.1*b[k]*((nactual-i)/(float)nactual);
                v = er+eg+eb;
                if (v < nearby)
                  {
                    nearby = v;
                    j = k;
                  }
              }
          }
        dr = abs((int)r[jold]-(int)r[j]);
        dg = abs((int)g[jold]-(int)g[j]);
        db = abs((int)b[jold]-(int)b[j]);
        if (trail == 1) dr *= 0.2;
        else if (trail == 2) dg *= 0.2;
        else db *= 0.2;
        er = dr+0.1*r[j]*(i/(float)nactual);
        eg = dg+0.1*g[j];
        eb = db+0.1*b[j]*((nactual-i)/(float)nactual);
        if (er < eg) trail = 1;
        else if (eg < eb) trail = 2;
        else trail=3;
        trans[j] = i;
        used[j] = 1;
        jold = j;
      }

    /* Remap the data */
    KCALL(kpds_set_attribute(destref,KPDS_VALUE_DATA_TYPE,KUSHORT));
    d = NULL;
    for (j=0; j<height; j++)
      {
        KCALL(kpds_set_attribute(destref,KPDS_VALUE_POSITION,0,j,0,0,0));
        if ((d=(unsigned short *)kpds_get_data(destref,KPDS_VALUE_LINE,d)) == NULL)
          {
            kerror(lib,rtn,"kpds_get_data failed");
            kfree(used); kfree(trans); kfree(mapd);
            return FALSE;
          }
        for (i=0; i<width; i++) d[i] = trans[d[i]];
        KCALL(kpds_set_attribute(destref,KPDS_VALUE_POSITION,0,j,0,0,0));
        if (!kpds_put_data(destref,KPDS_VALUE_LINE,d))
          {
            kerror(lib,rtn,"kpds_put_data failed");
            kfree(used); kfree(trans); kfree(mapd);
            return FALSE;
          }
      }
    kfree(d);

    /* Install the sorted maps */
    KCALL(kpds_set_attribute(destref,KPDS_MAP_POSITION,0,0,0,0,0));
    if ((m = (unsigned char *)kpds_get_data(destref,KPDS_MAP_PLANE,d)) == NULL)
      {
        kerror(lib,rtn,"Unable to read color maps from dest!");
        return FALSE;
      }
    /* Remap all actual colors */
    for (i=0; i<nactual; i++)
      {
        for (k=0; k<vlen; k++) m[vlen*trans[i]+k] = mapd[i+k*nactual];
      }
    /* Set all unused map entries to zero */
    for (i=nactual; i<ncolors; i++)
      {
        for (k=0; k<vlen; k++) m[vlen*i+k] = 0;
      }
    if (!kpds_put_data(destref,KPDS_MAP_PLANE,m))
      {
        kerror(lib,rtn,"Unable to write output image mapdata to output object!");
        return FALSE;
      }
    kfree(m);

    KCALL(kpds_close_object(destref));

    kfree(used); kfree(trans); kfree(mapd);

    return TRUE;
  }


