 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for ksegcmp
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lksegcmp
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include <float.h>

static char *_print_direction_str PROTO((int));

static int _compare_segment PROTO((kobject, kobject, char *, char *, char *,
   				   char *, int, int, int *, int, int, int, int,
				   int, int, int, int, double, kfile *));

#define lkseg_token(a) ((int)((long)klist_identifier(a)))
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lksegcmp - compare two data segments
* 
*       Purpose: This routine will compare the data contained
*		 in two data segments point by point, allowing
*		 for precise comparisons of data objects independent 
*		 of file format or machine architecture.
*
*		 The segments will first be compared to see if they
*		 are of the same size and dimensionality.  If the
*		 data types are different, a comparison can still be
*		 done in the higher of the two data types if the
*		 cast argument is set.
*
*		 A +/- comparison tolerance to use when comparing each
*		 data point may also be passed in.  In cases where a 
*		 complex comparison is performed, this tolerance will
*		 be applied to both the real and imaginary parts of the 
*		 values.
*
*		 A summary of the differences found between the segments
*		 can be printed to a given summary file.  This file will
*		 contain the names of the two segments, and will specify
*		 if the segments are the same or different.  If the 
*	         sizes, dimensionality, or data types are different, then 
*		 the differing attributes are printed.  If the sizes
*		 and dimensionality match, then the number of differing
*		 points is printed. The segment level attributes may 
*		 optionally be compared and printed as part of the summary.
*
*		 Specific differences found between the segments can
*		 be printed to a difference file.  This file will contain
*		 an entry for each differing point printed in the following 
*		 format:
*
*		!	[POSITION]
*		!	< Input #1 Value (Input #1 Data Type)
*		!	> Input #2 Value (Input #2 Data Type)
*
*	 	 Each component of this data point difference may optionally
*		 be left out by passing in the appropriate arguments.
*
*		 Note that the specific differences are generated and
*		 printed to the difference file first.  The summary
*		 is then generated and printed.
*
*		 If no output is desired in either file, this
*		 routine can be told to run silent by setting the
*		 run_silent argument.
*
*	 	 This routine will compare data "intelligently", never
*		 reading in more than a line at a time from both data
*		 objects.  This allows very large data objects to be 
*		 compared on machines with limited virtual memory.
*
*         Input: obj1  - object containing the first segment for 
*			 comparison.
*
*		 name1 - name of the first object to use in all
*			 summary reports.
*
*		 obj2  - object containing the second segment for
*			 comparison.
*
*		 name2 - name of the second object to use in all
*			 summary reports.
*		
*		 segment1 - name of the segment from the first
*		  	    object to compare.
*
*		 segment2 - name of the segment from the second
*			    object to compare.
*		
*		 cast     - TRUE if segments of differing data
*			    types should be casted to the higher
*			    data type before comparison.  If this
*			    is FALSE and the data types differ,
*			    the segments are considered different.
*
*		 tolerance  - comparison tolerance to use when comparing
*			      each point.
*
*		 run_silent - TRUE implies that nothing should be printed
*			      during the entire comparison to either of the
*			      summary or the difference files - only the
*		              return status of the program is needed
*			      as output.
*
*		 compare_attributes - TRUE if the attributes for the
*			 	      given segment should be compared
*
*		 print_position - TRUE if the position of differing points
*				  should be printed to the difference file.
*		 print_value1   - TRUE if the value from the first segment
*				  should be printed to the difference file.
*		 print_type1    - TRUE if the data type from the first segment
*				  should be printed to the difference file.
*		 print_value2   - TRUE if the value from the second segment
*				  should be printed to the difference file.
*		 print_type2    - TRUE if the data type from the second segment
*				  should be printed to the difference file.
*
*
*        Output: summary_file 	  - An open kfile in which to print
*			            the summary information.  Note that
*				    this summary will be printed after the 
*			            specific differences are printed
*				    to the difference file.  If NULL,
*				    then nothing will be printed.
*
*  		  difference_file - An open kfile in which to print
*				    the specific data differences.
*				    Note that this will be printed before
*				    the summaries are printed to the 
*				    summary file.  If NULL, then nothing
*				    will be printed.
*
*       Returns: TRUE (1) if the objects are tolerably identical,
*                FALSE (0) otherwise
*
*  Restrictions: Data types can not be printed to the difference
*		 file without their corresponding values.  
*
*    Written By: Steve Kubica
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int 
lksegcmp(
   kobject obj1,
   char *name1,
   kobject obj2,
   char *name2,
   char *segment1,
   char *segment2,
   int cast,
   double tolerance,
   int run_silent,
   int print_summary,
   int compare_attributes,
   int print_position,
   int print_value1,
   int print_type1,
   int print_value2,
   int print_type2,
   kfile * summary_file,
   kfile * difference_file)
/* -library_def_end */

/* -library_code */
{
   kobject ref1 = NULL;
   kobject ref2 = NULL;

   int i;
   int j;
   int dim1;
   int dim2;
   int type1;
   int type2;
   int *size1 = NULL;
   int *size2 = NULL;
   int *ord1  = NULL;
   int *ord2  = NULL;

   int keep_comparing = TRUE;	/* if things get too different, set this! */

   int type_used = 0;
   int order_found = FALSE;
   int attributes_same = TRUE;  /* assume that they're the same unless 
				   we explciitly find out that they are
				   different */

   /* comparison results for the summary */

   int obj1_segment_present = FALSE;
   int obj2_segment_present = FALSE;
   int dimensions_match = TRUE;
   int orders_found = TRUE;
   int sizes_match = TRUE;
   int types_match = TRUE;
   int casting_done = FALSE;
   int number_different = 0;
   int total_number_points = 1;


   /*
    *   Check for NULL objects 
    *  ............................................................
    */

   if (obj1 == NULL)
   {
      kerror("kdatamanip", "lksegcmp", "specified object 1 is NULL");
      return (FALSE);
   }

   if (obj2 == NULL)
   {
      kerror("kdatamanip", "lksegcmp", "specified object 2 is NULL");
      return (FALSE);
   }

   /*
    *   Check for existence of requested segments.
    *   If the second segment is given to be NULL, then use
    *   the first segment for both.
    *  ............................................................
    */

   if (segment1 == NULL)
   {
      kerror("kdatamanip", "lksegcmp", "segment name must not be NULL");
      return (FALSE);
   }

   if (segment2 == NULL)
      segment2 = segment1;


   if ((obj1_segment_present = kdms_query_segment(obj1, segment1)) == FALSE)
      keep_comparing = FALSE;

   if ((obj2_segment_present = kdms_query_segment(obj2, segment2)) == FALSE)
      keep_comparing = FALSE;


   /*
    *   We first want to compare the dimensionality of the segments
    *   to make sure that they match -- if they are both zero, then
    *   we don't want to keep comparing data (because there isn't any).
    *  ............................................................
    */
   if (keep_comparing)
   {
      if (!kdms_get_attribute(obj1, segment1, KDMS_DIMENSION, &dim1))
      {
	 kerror("kdatamanip", "lksegcmp",
		"unable to retreive the dimensionality of the '%s' "
		"\nsegment from object '%s'\n", segment1, name1);
	 return (FALSE);
      }

      if (!kdms_get_attribute(obj2, segment2, KDMS_DIMENSION, &dim2))
      {
	 kerror("kdatamanip", "lksegcmp",
		"unable to retreive the dimensionality of the '%s' "
		"\nsegment from object '%s'\n", segment1, name1);
	 return (FALSE);
      }

      /* if the dimensionality doen't fail, then continue */
      if ((dimensions_match = (dim1 == dim2)) == FALSE)
         keep_comparing = FALSE;

      if (dimensions_match && dim1 == 0)
         keep_comparing = FALSE;
   }

   /*
    *   At this point, we know if we have valid objects and if the
    *   segments actually exist and have dimensionality.  The next step 
    *   is to compare the data attributes of the segment to verify that 
    *   the size and the data type are the same.
    *  ............................................................
    */
   if (keep_comparing)
   {
      if (!kdms_get_attributes(obj1, segment1,
			       KDMS_SIZE, &size1,
			       KDMS_INDEX_ORDER, &ord1,
			       KDMS_DATA_TYPE, &type1,
			       NULL))
      {
	 kerror("kdatamanip", "lksegcmp",
		"unable to retreive the size, index order, or data type, "
		"of the '%s' \nsegment from object '%s'\n",
		segment1, name1);
	 return (FALSE);
      }

      if (!kdms_get_attributes(obj2, segment2,
			       KDMS_SIZE, &size2,
			       KDMS_INDEX_ORDER, &ord2,
			       KDMS_DATA_TYPE, &type2,
			       NULL))
      {
	 kerror("kdatamanip", "lksegcmp",
		"unable to retreive the size, index order, or data type, "
		"of the '%s' \nsegment from object '%s'\n",
		segment2, name2);
	 return (FALSE);
      }
   }

   /*
    *   At this point, we have valid all the segment information
    *   from both objects, now need to compare the index orders and sizes.
    *  ............................................................
    */

   /* 
    *   Make sure that the both segments contain the same "directions" in
    *   their index orders - the ordering isn't important, we just want
    *   to make sure that they both encompass the same space.
    */
   if (keep_comparing)
   {
      for (i = 0; i < dim1; i++)
      {
	 /* For each direction in ord1, make sure it exists in ord2 */
	 order_found = FALSE;
	 j = 0;
	 do
	 {
	    order_found = (ord1[i] == ord2[j]);
	    j++;
	 }
	 while (!order_found && j < dim2);

	 orders_found &= order_found;
      }

      /* need to make the index orders match */
      if (orders_found)
      {
	 if (!kdms_set_attribute(obj2, segment2,
				 KDMS_INDEX_ORDER, ord1))
	 {
	    kerror("kdatamanip", "lksegcmp",
		   "unable to set the presentation index order "
		   "of the '%s' \nsegment from object '%s'\n",
		   segment1, name2);
	    return (FALSE);
	 }
      }
      else
	 keep_comparing = FALSE;
   }

   /* 
    *   Make sure the sizes match.  
    *   We can do this because we know the index orders are in the
    *   same space.
    */
   if (keep_comparing)
   {
      for (i = 0; i < dim1; i++)
      {
	 sizes_match &= (size1[i] == size2[i]);

	 if (!sizes_match)
	    keep_comparing = FALSE;
      }
   }

   /*
    *   Check the data types.  We should only fail if we can't cast later.
    */
   if (keep_comparing && ((types_match = (type1 == type2)) == FALSE) && !cast)
      keep_comparing = FALSE;


   /*
    *   Now we know if the sizes are the same and the index orders
    *   in both objects span the same directional space.  
    *   Let's do the preliminary work for doing the data comparison.
    */
   if (keep_comparing)
   {
      /* Create reference objects so we can avoid side effects */
      ref1 = kdms_reference(obj1);
      if (ref1 == NULL)
      {
	 kerror("kdatamanip", "lksegcmp", "Unable to reference object '%s'",
		name1);
	 return (FALSE);
      }

      ref2 = kdms_reference(obj2);
      if (ref2 == NULL)
      {
	 kerror("kdatamanip", "lksegcmp", "Unable to reference object %s",
		name2);
	 return (FALSE);
      }

      /* If the data types are identical, then casting is no problem */
      type_used = type1;

      /* If the data types are different, then cast to the bigger one */
      if ((type1 != type2) && cast)
      {
	 if (type1 > type2)
	 {
	    if (!kdms_set_attribute(obj2, segment2, KDMS_DATA_TYPE, type1))
	    {
	       kerror("kdatamanip", "lksegcmp",
		      "Unable to set data type on object %s", name2);
	       return (FALSE);
	    }
	    type_used = type1;
	    casting_done = TRUE;
	 }
	 else
	 {
	    if (!kdms_set_attribute(obj1, segment1, KDMS_DATA_TYPE, type2))
	    {
	       kerror("kdatamanip", "lksegcmp",
		      "Unable to set data type on object %s", name1);
	       return (FALSE);
	    }
	    type_used = type1;
	    casting_done = TRUE;
	 }
      }
      else if (type1 != type2)
	 keep_comparing = FALSE;
   }

   /* If we are doing bit data, then make it byte because it's easier */
   if (keep_comparing)
   {
      if (type_used == KBIT)
      {
	 if (!kdms_set_attribute(obj1, segment1, KDMS_DATA_TYPE, KBYTE))
	 {
	    kerror("kdatamanip", "lksegcmp",
		   "Unable to set data type on object %s", name1);
	    return (FALSE);
	 }

	 if (!kdms_set_attribute(obj2, segment2, KDMS_DATA_TYPE, KBYTE))
	 {
	    kerror("kdatamanip", "lksegcmp",
		   "Unable to set data type on object %s", name1);
	    return (FALSE);
	 }
	 type_used = KBYTE;
	 casting_done = TRUE;
      }
   }

   /* Present the second object with the first object's index order */
/*
   This is confusing -- I would think I would need to set this,
   but identical data sets to pds compare differently here...

   <shrug>              SK

   if (keep_comparing)
   if (!kdms_set_attribute(obj2, segment2, KDMS_INDEX_ORDER, ord1))
   {
   kerror( "kdatamanip","lksegcmp", 
   "Unable to set index order on object %s", name2);       
   return(FALSE);
   }   
 */


   /*
    *   Finally go and compare the data.  Do the comparison
    *   for lines along the lowest index order so the data will
    *   tile properly for large data sets.  Do the comparision 
    *   in the proper data type.
    */

   if (keep_comparing)
      number_different = _compare_segment(obj1, obj2, segment1, segment2,
				       name1, name2, dim1, type_used, size1,
					  type1, type2,
					  run_silent, print_position,
					  print_value1, print_type1,
					  print_value2, print_type2,
					  tolerance, difference_file);

   if (keep_comparing)
      for (i = 0; i < dim1; i++)
	 total_number_points *= _kdms_get_order(ord1[i], ord1, size1, dim1, 1);

   /*
    *   Print the data summary, if requested.
    */

   if (!run_silent && print_summary && (summary_file != NULL))
   {
      if (kstrcmp(segment1, segment2) == 0)
	 kfprintf(summary_file, "\n# -- '%s' data differences\n#", segment1);
      else
	 kfprintf(summary_file,
		  "\n# -- differences between '%s' and '%s' data\n#",
		  segment1, segment2);

      if (!obj1_segment_present)
	 kfprintf(summary_file, "\n#     '%s' data is not present in '%s' ",
		  segment1, name1);

      if (!obj2_segment_present)
	 kfprintf(summary_file, "\n#     '%s' data is not present in '%s' ",
		  segment2, name2);

      if (!obj1_segment_present || !obj2_segment_present)
	 goto report_finished;

      if (!dimensions_match)
      {
	 kfprintf(summary_file, "\n#     dimensions differ...");
	 kfprintf(summary_file, "\n#      '%s' in '%s' dimension : %d",
		  segment1, name1, dim1);
	 kfprintf(summary_file, "\n#      '%s' in '%s' dimension : %d",
		  segment2, name2, dim2);
	 goto report_finished;
      }


      if (!orders_found)
      {
	 kfprintf(summary_file, "\n#     index order directions differ...");
	 kfprintf(summary_file, "\n#       index order of '%s' in '%s' :"
		  "\n#       ",
		  segment1, name1);

	 for (i = 0; i < dim1; i++)
	    kfprintf(summary_file, "%s (%d) ",
		     _print_direction_str(ord1[i]), ord1[i]);

	 kfprintf(summary_file, "\n#       index order of '%s' in '%s' :"
		  "\n#       ",
		  segment2, name2);

	 for (i = 0; i < dim2; i++)
	    kfprintf(summary_file, "%s (%d) ",
		     _print_direction_str(ord2[i]), ord2[i]);

	 goto report_finished;
      }


      if (!sizes_match)
      {
	 kfprintf(summary_file, "\n#     sizes differ...");
	 kfprintf(summary_file, "\n#      size of '%s' in '%s' : ",
		  segment1, name1);

	 for (i = 0; i < dim1; i++)
	    kfprintf(summary_file, "%d ",
		     _kdms_get_order(ord1[i], ord1, size1, dim1, 1));

	 kfprintf(summary_file, "\n#      size of '%s' in '%s' : ",
		  segment2, name2);

	 for (i = 0; i < dim2; i++)
	    kfprintf(summary_file, "%d ",
		     _kdms_get_order(ord2[i], ord2, size2, dim2, 1));

	 goto report_finished;
      }

      if (!types_match)
      {
	 kfprintf(summary_file, "\n#     data types differ... ");
	 kfprintf(summary_file, "\n#      data type of '%s' in '%s' : %s (%d) ",
		  segment1, name1, kdefine_to_datatype(type1), type1);
	 kfprintf(summary_file, "\n#      data type of '%s' in '%s' : %s (%d) ",
		  segment2, name2, kdefine_to_datatype(type2), type2);

	 if (casting_done)
	    kfprintf(summary_file, "\n#      data cast to %s (%d) "
		     "for comparison\n#",
		     kdefine_to_datatype(type_used), type_used);
	 else
	    goto report_finished;
      }

      if (number_different == 0)
      {
	 kfprintf(summary_file, "\n#    no data points are different");
	 if (tolerance != 0.0)
	    kfprintf(summary_file, " within +/- %g", tolerance);
      }
      else
      {
	 kfprintf(summary_file, "\n#    %d data points out of %d are different",
		  number_different, total_number_points);
	 if (tolerance != 0.0)
	    kfprintf(summary_file, " within +/- %g", tolerance);
      }

    report_finished:

      kfprintf(summary_file, "\n#\n");
   }


   /*
    *   Compare the attributes and summarize those differences
    */

   if (compare_attributes)
      attributes_same = lksegcmp_attributes(obj1, obj2, segment1, 
		 			    summary_file,
					    !(print_summary && (!run_silent)));

   /*
    *   Return TRUE if things are the same, FALSE otherwise
    */

   if (!obj1_segment_present ||
       !obj2_segment_present ||
       !dimensions_match ||
       !orders_found ||
       !sizes_match ||
       (!types_match && !casting_done) ||
       (number_different != 0) ||
       !attributes_same)
      return (FALSE);
   else
      return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _print_direction_str 
|
|       Purpose: This routine will return a pointer to 
|                a static string representing the given
|                direction.
|
|         Input: direction - index order direction
|
|        Output: none
|
|       Returns: string containing the direction
|
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
| Modifications:
|
------------------------------------------------------------*/
static char *
_print_direction_str(int direction)
{
   switch (direction)
   {
      case KWIDTH:
	 return ("Width");
      case KHEIGHT:
	 return ("Height");
      case KDEPTH:
	 return ("Depth");
      case KTIME:
	 return ("Time");
      case KELEMENTS:
	 return ("Elements");
   }
   return (" ");		/* Just unknown to polymorphic services */
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_pos
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_pos(kfile * out, int print_pos, int *pos, int dim)
{
   int i;
   if (print_pos)
   {
      kfprintf(out, "[ ");
      for (i = 0; i < dim; i++)
	 kfprintf(out, " %d ", pos[i]);
      kfprintf(out, "]\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_byte
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_byte(kfile * out, int print_val1, int print_type1,
	    int print_val2, int print_type2,
	    char *d1, char *d2,
	    int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %d", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> %d", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_ubyte
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_ubyte(kfile * out, int print_val1, int print_type1,
	     int print_val2, int print_type2,
	     unsigned char *d1, unsigned char *d2,
	     int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %d", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> %d", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_short
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_short(kfile * out, int print_val1, int print_type1,
	     int print_val2, int print_type2,
	     short *d1, short *d2,
	     int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %d", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {

      kfprintf(out, "> %d", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_ushort
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_ushort(kfile * out, int print_val1, int print_type1,
	      int print_val2, int print_type2,
	      unsigned short *d1, unsigned short *d2,
	      int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %d", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {

      kfprintf(out, "> %d", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_int
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_int(kfile * out, int print_val1, int print_type1,
	   int print_val2, int print_type2,
	   int *d1, int *d2,
	   int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %d", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> %d", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_uint
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_uint(kfile * out, int print_val1, int print_type1,
	    int print_val2, int print_type2,
	    unsigned int *d1, unsigned int *d2,
	    int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %d", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> %d", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}


/*-----------------------------------------------------------
|  Routine Name: (static) _print_long
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_long(kfile * out, int print_val1, int print_type1,
	    int print_val2, int print_type2,
	    long *d1, long *d2,
	    int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %d", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> %d", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_ulong
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_ulong(kfile * out, int print_val1, int print_type1,
	     int print_val2, int print_type2,
	     unsigned long *d1, unsigned long *d2,
	     int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %d", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> %d", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_float
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_float(kfile * out, int print_val1, int print_type1,
	     int print_val2, int print_type2,
	     float *d1, float *d2,
	     int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %g", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> %g", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_double
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_double(kfile * out, int print_val1, int print_type1,
	      int print_val2, int print_type2,
	      double *d1, double *d2,
	      int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< %g", d1[ind]);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> %g", d2[ind]);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _print_complex
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_complex(kfile * out, int print_val1, int print_type1,
	       int print_val2, int print_type2,
	       kcomplex *d1, kcomplex *d2,
	       int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< (%g,%g)", d1[ind].r, d1[ind].i);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> (%g,%g)", d2[ind].r, d2[ind].i);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}
/*-----------------------------------------------------------
|  Routine Name: (static) _print_dcomplex
|    Written By: Steve Kubica
|          Date: Mar 18, 1994 10:50
------------------------------------------------------------*/
static void 
_print_dcomplex(kfile * out, int print_val1, int print_type1,
	        int print_val2, int print_type2,
	        kdcomplex *d1, kdcomplex *d2,
	        int typ1, int typ2, int ind)
{
   if (print_val1)
   {
      kfprintf(out, "< (%g,%g)", d1[ind].r, d1[ind].i);
      if (print_type1)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ1));
      kfprintf(out, "\n");
   }

   if (print_val2)
   {
      kfprintf(out, "> (%g,%g)", d2[ind].r, d2[ind].i);
      if (print_type2)
	 kfprintf(out, "  (%s)", kdefine_to_datatype(typ2));
      kfprintf(out, "\n");
   }
   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: _compare_segment
|
|       Purpose: This routine will compare two data segments in 
|                two different objects, count the number of points
|                that are different, and print a detailed difference
|                report into an output file.
|
|         Input: obj1 - object containing the first segment to be
|		        compared.
|
|		 obj2 - object containing the second segment to be
|		        compared.
|	
|		 segment1 - name of the segment to be compared 
|			    from the first object.
|
|		 segment2 - name of the segment to be compared
|			    from the second object.
|
|		 name1    - the name to use in the printed summary
|			    when refering to the first object.
|
|		 name2    - the name to use in the printed summary
|			    when refering to second object.
|	
|		 dim       - the dimensionality of the segments
|
|		 type_used - the data type to use for the comparison
|
|		 size      - the size of the data segments
|
|	         type1     - the actual data type of the first segment
|			     to be used when printing the report
|			 
|	         type2     - the actual data type of the second segment
|			     to be used when printing the report
|
|		 run_silent - TRUE if nothing should be printed
|
|
|		 print_position - TRUE if the position of the differing
|			          points should be printed.
|
|		 print_value1   - TRUE if the value from the first
|				  object should be printed
|
|		 print_type1    - TRUE if the data type from the first
|				  object should be printed
|
|		 print_value2   - TRUE if the value from the first
|				  object should be printed
|
|		 print_type2    - TRUE if the data type from the first
|				  object should be printed 
|
|
|        Output: outfile   - open kfile in which to print the specific
|			     differences
|
|       Returns: the number of points that are different
|
|    Written By: Steve Kubica
|          Date: Mar 02, 1993
| Modifications:
|
------------------------------------------------------------*/
static int 
_compare_segment(
   kobject obj1,
   kobject obj2,
   char *segment1,
   char *segment2,
   char *name1,
   char *name2,
   int dim,
   int type_used,
   int *size,
   int type1,
   int type2,
   int run_silent,
   int print_position,
   int print_value1,
   int print_type1,
   int print_value2,
   int print_type2,
   double tolerance,
   kfile * outfile)
{
   int begin[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int end[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int pos[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};

   int i;
   int j;
   int k;
   int line_size;
   int num_different = 0;

   kaddr data1 = NULL;
   kaddr data2 = NULL;

   /* Print a specific difference header */
   if (!run_silent && (print_position ||
		print_value1 || print_type1 || print_value2 || print_type2))
   {
      kfprintf(outfile, "\n#     -- specific differences between : ");
      kfprintf(outfile, "\n#     --  '%s' data in object '%s' ",
	       segment1, name1);
      kfprintf(outfile, "\n#     --  '%s' data in object '%s' ",
	       segment2, name2);
      kfprintf(outfile, "\n#\n");
   }

   /* The line size is simply the size along the lowest index order */
   line_size = size[0];
   end[0] = line_size - 1;

 the_beginning:

   /* Get the data */
   if ((data1 = kdms_get_data(obj1, segment1, begin, end, data1)) == NULL ||
       (data2 = kdms_get_data(obj2, segment2, begin, end, data2)) == NULL)
   {
      kerror("kdatamanip", "lksegcmp", "Unable to access data for comparison");
      return (FALSE);
   }

   /* 
    * Compare according to type - all comparisons are really done
    * as double, but we need to index according to the actual datatype,
    * hence this huge case statement.
    */
   for (j = 0; j < line_size; j++)
   {
      pos[0] = j;
      for (k = 1; k < dim; k++)
	 pos[k] = begin[k];

      switch (type_used)
      {
	 case KBYTE:
	    {
	       char *d1 = (char *)data1;
	       char *d2 = (char *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if ((val1 < (val2 - kabs(tolerance)) ||
		    (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_byte(outfile, print_value1, print_type1,
				 print_value2, print_type2,
				 d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KUBYTE:
	    {
	       unsigned char *d1 = (unsigned char *)data1;
	       unsigned char *d2 = (unsigned char *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if ((val1 < (val2 - kabs(tolerance)) ||
		    (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_ubyte(outfile, print_value1, print_type1,
				  print_value2, print_type2,
				  d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KSHORT:
	    {
	       short *d1 = (short *)data1;
	       short *d2 = (short *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if ((val1 < (val2 - kabs(tolerance)) ||
		    (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_short(outfile, print_value1, print_type1,
				  print_value2, print_type2,
				  d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KUSHORT:
	    {
	       unsigned short *d1 = (unsigned short *)data1;
	       unsigned short *d2 = (unsigned short *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if ((val1 < (val2 - kabs(tolerance)) ||
		    (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_ushort(outfile, print_value1, print_type1,
				   print_value2, print_type2,
				   d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KINT:
	    {
	       int *d1 = (int *)data1;
	       int *d2 = (int *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if ((val1 < (val2 - kabs(tolerance)) ||
		    (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_int(outfile, print_value1, print_type1,
				print_value2, print_type2,
				d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KUINT:
	    {
	       unsigned int *d1 = (unsigned int *)data1;
	       unsigned int *d2 = (unsigned int *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if ((val1 < (val2 - kabs(tolerance)) ||
		    (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_uint(outfile, print_value1, print_type1,
				 print_value2, print_type2,
				 d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KLONG:
	    {
	       long *d1 = (long *)data1;
	       long *d2 = (long *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if ((val1 < (val2 - kabs(tolerance)) ||
		    (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_long(outfile, print_value1, print_type1,
				 print_value2, print_type2,
				 d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KULONG:
	    {
	       unsigned long *d1 = (unsigned long *)data1;
	       unsigned long *d2 = (unsigned long *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if ((val1 < (val2 - kabs(tolerance)) ||
		    (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_ulong(outfile, print_value1, print_type1,
				  print_value2, print_type2,
				  d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KFLOAT:
	    {
	       float *d1 = (float *)data1;
	       float *d2 = (float *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if (tolerance == 0.0) tolerance = FLT_EPSILON;

	       if ((val1 < (val2 - kabs(tolerance)) ||
		    (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_float(outfile, print_value1, print_type1,
				  print_value2, print_type2,
				  d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KDOUBLE:
	    {
	       double *d1 = (double *)data1;
	       double *d2 = (double *)data2;
	       double val1 = (double)d1[j];
	       double val2 = (double)d2[j];

	       if (tolerance == 0.0) tolerance = DBL_EPSILON;

	       if ((val1 < (val2 - kabs(tolerance)) ||
		   (val1 > (val2 + kabs(tolerance)))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_double(outfile, print_value1, print_type1,
				   print_value2, print_type2,
				   d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KCOMPLEX:
	    {
	       kcomplex *d1 = (kcomplex *)data1;
	       kcomplex *d2 = (kcomplex *)data2;
	       kcomplex val1;
	       kcomplex val2;
	       val1 = d1[j];
	       val2 = d2[j];

	       if (tolerance == 0.0) tolerance = FLT_EPSILON;

	       if ((val1.r < (val2.r - (float) kabs(tolerance)) ||
		   (val1.i < (val2.i - (float) kabs(tolerance)) ||
		   (val1.r > (val2.r + (float) kabs(tolerance)) ||
		   (val1.i > (val2.i + (float) kabs(tolerance)))))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_complex(outfile, print_value1, print_type1,
				   print_value2, print_type2,
				   d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;
	 case KDCOMPLEX:
	    {
	       kdcomplex *d1 = (kdcomplex *)data1;
	       kdcomplex *d2 = (kdcomplex *)data2;
	       kdcomplex val1;
	       kdcomplex val2;
	       val1 = d1[j];
	       val2 = d2[j];

	       if (tolerance == 0.0) tolerance = DBL_EPSILON;

	       if ((val1.r < (val2.r - (double) kabs(tolerance)) ||
		   (val1.i < (val2.i - (double) kabs(tolerance)) ||
		   (val1.r > (val2.r + (double) kabs(tolerance)) ||
		   (val1.i > (val2.i + (double) kabs(tolerance)))))))
	       {
		  num_different++;
		  if (!run_silent)
		  {
		     _print_pos(outfile, print_position, pos, dim);
		     _print_dcomplex(outfile, print_value1, print_type1,
				   print_value2, print_type2,
				   d1, d2, type1, type2, j);
		  }
	       }
	    }
	    break;

	 default:
	    {
	       kerror("kdatamanip", "lksegcmp",
		      "Attempt to use invalid data type %d for comparison",
		      type_used);
	       kfree(data1);
	       kfree(data2);
	       return (0);
	    }
      }
   }

   /* This is Jeremy's Magic Incrementing Code */
   for (i = 1; i < dim - 1; i++)
   {
      begin[i] += 1;
      end[i] += 1;
      if (begin[i] < size[i])
	 goto the_beginning;
      else
      {
          begin[i] = 0;	 
	  end[i] = 0;
      }
   }
   begin[dim - 1] += 1;
   end[dim - 1] += 1;

   if (begin[dim - 1] < size[dim - 1] && dim > 1)
      goto the_beginning;

   kfree(data1);
   kfree(data2);

   if (num_different == 0)
      kfprintf(outfile, "#     --  no differences found\n");

   return (num_different);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _is_special
|
|       Purpose: This routine checks a given attribute name
|		 and verifies that it is not one of the special
|		 attributes which is not to be considered during
|		 an attribute comparison.  Specifically, these 
|		 special attributes are :
|
|		 	KDMS_SIZE,
|		 	KDMS_DATA_TYPE,
|		 	KDMS_INDEX_ORDER,
|		 	KDMS_DIMENSION
|		 	KDMS_COUPLING
|
|		 The date attribute is also considered special
|		 and not up for comparison because it is nearly
|		 always different.  Similarly for the history
|		 attribute.
|
|         Input: attribute
|
|        Output: none
|
|       Returns: TRUE (1) if attribute is special, 
|		 FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 02, 1993
| Modifications:
|
------------------------------------------------------------*/
static int
_is_special(char *attribute)
{
   if (kstrcmp(attribute, KDMS_SIZE) == 0)
      return(TRUE);
   else if (kstrcmp(attribute, KDMS_DATA_TYPE) == 0)
      return(TRUE);   
   else if (kstrcmp(attribute, KDMS_INDEX_ORDER) == 0)
      return(TRUE);
   else if (kstrcmp(attribute, KDMS_DIMENSION) == 0)
      return(TRUE);
   else if (kstrcmp(attribute, KDMS_DATE) == 0)
      return(TRUE);
   else if (kstrcmp(attribute, KDMS_COUPLING) == 0)
      return(TRUE);
   else if (kstrcmp(attribute, KPDS_HISTORY) == 0)
      return(TRUE);
   
   return(FALSE);
}

/****************************************************************
* 
*  Routine Name: lksegcmp_attributes - compare segment and/or global 
*				       attributes of two objects
* 
*       Purpose: This routine compares attributes between two data 
*                objects.  If a segment is specified, then the attributes 
*		 for that segment is compared across the two data objects.
*		 The size, data type, dimensionality, and index order
*		 attributes are ignored.  If no segment is specified, then
*		 the object-level attributes are compared.
*
*		 The attribute differences are printed to a specified
*		 output file, unless the run_silent argument is set,
*		 in which case nothing is printed.
*
*         Input: obj1   - object containing first segment with attributes
*			  to compare.
*		 obj2   - object containing second segment with attributes
*			  to compare.
*
*		 segment - segment name to compare the attributes for.
*			   If NULL, then the object level segments
*			   between the two objects are compared.
*	
*		 run_silent - TRUE if no output should be generated
*			      to the outfile.
*
*        Output: outfile    - An open kfile in which to print the
*			      attribute differences.  
*
*       Returns: TRUE (1) if the attributes all match, FALSE (0) otherwise
*
*  Restrictions: Only defined attributes are compared.  User created
*		 attributes are not considered in the comparison.
*
*    Written By: Steve Kubica
*          Date: May 15, 1994
*      Verified: 
*  Side Effects: 
* Modifications: 
*   Declaration: int lksegcmp_attributes(
*		 !    kobject obj1,
*	         !    kobject obj2,
*	         !    char   *segment,
*		 !    kfile  *outfile,
*	         !    int     run_silent)
*
****************************************************************/
int 
lksegcmp_attributes(
   kobject obj1,
   kobject obj2,
   char *segment,
   kfile *outfile,
   int run_silent)
{
   int status = TRUE;
   char *name1;
   char *name2;
   char **atr = NULL;
   int i;
   int natr;
   int perm1, perm2;

   /* Check for NULL objects */
   if (obj1 == NULL)
   {
      kerror("kdatamanip", "lksegcmp", "Specified Input Object 1 is NULL");
      return (FALSE);
   }
   else
      kdms_get_attributes(obj1, NULL, KDMS_NAME, &name1, NULL);

   if (obj2 == NULL)
   {
      kerror("kdatamanip", "lksegcmp", "Specified Input Object 2 is NULL");
      return (FALSE);
   }
   else
      kdms_get_attributes(obj1, NULL, KDMS_NAME, &name2, NULL);


   /* Check for existence of segment in both objects */
   if (segment != NULL)
   {
      if (!kdms_query_segment(obj1, segment))
      {
	 if (!run_silent)
            kfprintf(outfile, "Specified Segment %s does not exist in "
			      "Object %s", segment, name1);
	 return (FALSE);
      }

      if (!kdms_query_segment(obj2, segment))
      {
	 if (!run_silent)
	    kfprintf(outfile, "Specified Segment %s does not exist in "
			      "Object %s", segment, name2);
	 return (FALSE);
      }
   }

   /* 
    *   Get the list of all the defined attributes 
    *   This will give us all of the quasi-attributes *and* the 
    *   generic attributes.   We can then go and query for the 
    *   existence of this attribute on the segments in both of
    *   the objects.  If they exist in both, then we can do the
    *   comparison.
    *
    *   The reason this is being done this way is because 
    *   the function to get the attribute list from a given
    *   segment does not include the quasi-attributes, which 
    *   we want to compare.  This method, while more confusing, is
    *   better.  But, it undoubtedly still isn't foolproof.
    *
    *   It will miss, for example, attributes which have been 
    *   created but not defined.
    */

   /* get the defined attributes that are not internal */

   /* -- get only the stored attributes from obj1 -- this needs
         to be one day changed to get obj1 & obj2 and merge the
	 two lists .... -- */
   atr = kdms_get_stored_attribute_names(obj1, segment, &natr);
#if 0
   atr = kdms_get_defined_attribute_names(NULL, &natr);
#endif

   /* Loop through the attributes in the first object */
   for (i = 0; i < natr; i++)
   {
      /* Does this attribute even exist in the first object? */
      if (kdms_query_attribute(obj1, segment, atr[i], NULL, NULL, NULL, 
			       &perm1))
      {
	 /* Does this attribute even exist in the second object? */
	 if (kdms_query_attribute(obj2, segment, atr[i], NULL, NULL, NULL, 
				  &perm2))
	 {
	    /* If yes, then verify that they are both permanent 
	     * and make sure it isn't a special attribute such as
             * size, or data type, etc.
             */
  	    if (perm1 && perm2 && !_is_special(atr[i]))
	    {
   	       if (!kdms_match_attribute(obj1, obj2, segment, atr[i]))
	       {
	          status = FALSE;
	          if (!run_silent)
	          {
		     kfprintf(outfile, "\n#     --  attribute %s differs", 
			      atr[i]);

		     kfprintf(outfile, "\n#     --  %s in %s : ", atr[i], 
			      name1);
		     kdms_print_attribute(obj1, segment, atr[i], outfile);

		     kfprintf(outfile, "\n#     --  %s in %s : ", atr[i], 
			      name2);
		     kdms_print_attribute(obj2, segment, atr[i], outfile);

		     kfprintf(outfile, "\n");
	          }
	       }
	    }
	 }
         else /* This attribute exists in only object 1 */
	 {
	    status = FALSE;
	    if (!run_silent)
	       kfprintf(outfile, "\n#     --  attribute %s exists only "
		                 "in object %s.\n", atr[i], name1);
	 }
      }
      else if (kdms_query_attribute(obj2, segment, atr[i], NULL, NULL, NULL, 
				    NULL))
      {
	 status = FALSE;
	 if (!run_silent)
	    kfprintf(outfile, "\n#     --  attribute %s exists only "
		     "in object %s.\n", atr[i], name2);
      }
   }

   karray_free(atr, natr, NULL);

   return (status);
}
/* -library_code_end */
