 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kimportasc
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkimportasc
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkimportasc - import ASCII data into specified segment of data object
* 
*       Purpose: This routine will import an ASCII data file,
*		 ascii_file, and insert it into a data
*		 object, source_obj, if it is specified.
*		 If there is no source object to insert
*		 the new segment into, pass NULL for source_obj and
*		 the data object specified by the parameter 
*		 destination_object, will be created.
*		 The new segment specified by segment_name will be
*		 inserted into this data object.  If a source_obj
*		 was specified, then all of its data segments and
*		 the global object attributes will be copied to 
*		 the destination.
*
*		 If the segment_name is one of the Polymorphic
*		 segments, the correct dimensionality must be provided :
*
*                .TS H
*                center tab(:) ;
*                lfB  lfB  
*                lf(CW) l .
*        	 Segment Name:Dimension
*                =
*                .TH
*        	 KDMS_SEGMENT_VALUE     :  5
*        	 KDMS_SEGMENT_MAP       :  5
*        	 KDMS_SEGMENT_MASK      :  5
*        	 KDMS_SEGMENT_LOCATION  :  4
*        	 KDMS_SEGMENT_TIME      :  1
*		 .TE
*
*		 There are four attributes of the new segment that
*		 must be passed in : dimension, index order, size and
*		 data type.  The number of indicies and sizes
*		 specified must be equal to the dimension.
*
*		 The parameter dimension must be between 0 and 5.
*
*       	 The parameter index_order is a string that contains the
*		 indicies of the segment separated by spaces, "elements
*		 width height depth time" or "e w h d t".  The only allowed
*		 values are :
*
*                .TS
*                center tab(:) ;
*                l l .
*        	 width (w)      :  mapwidth (mw)
*	 	 height (h)     :  mapheight (mh)
*        	 depth (d)      :  mapdepth (md)
*	 	 time (t)       :  maptime (mt)
*        	 elements (e)   :  mapelements (me)
*	 	 dimension (n)  :
*		 .TE
*
*                where the characters in the ()'s are accepted
*                abbreviations.
*
*		 The parameter size_vector contains the size of the
*		 segment to create.  All the values must be > 1.
*
*		 The parameter data_type_string must contain one of
*		 the following strings to specify the data type of
*		 the segment:
*
*                .TS H
*                center tab(:) ;
*                lfB  lfB  lfB
*                l    l    lf(CW) .
*                Data Type:Abbreviation:Data Type
*                =
*                .TH
*                bit              :bi     : KBIT
*                byte             :by     : KBYTE
*                unsigned byte    :un by  : KUBYTE
*                ubyte            :uby    : KUBYTE
*                short            :sh     : KSHORT
*                unsigned short   :un sh  : KUSHORT
*                ushort           :ush    : KUSHORT
*                integer          :in     : KINT
*                unsigned integer :un in  : KUINT
*                uint             :ui     : KUINT
*                long             :lon    : KLONG
*                unsigned long    :un lon : KULONG
*                ulong            :ul     : KULONG
*                float            :fl     : KFLOAT
*                double           :do     : KDOUBLE
*                complex          :co     : KCOMPLEX
*                double complex   :do co  : KDCOMPLEX
*                dcomplex         :dc     : KDCOMPLEX
*                .T&
*                l s s.
*                Propagate Input Type - will not change the data type
*                .TE
*
*		 The parameter start_offset is the point in the ASCII
*		 source at which to start importing in the ASCII data.
*		 All points before this one are ignored.
*
*		 The parameter skip_factor is the number of points to
*		 skip in the ASCII data between reads.  If the size
*		 of the new segment's lowest index is 3 and the skip
*		 factor is 5, then this routine will read in 3 points
*		 skip 5 and read in 3 more, etc. until the entire ASCII
*		 file read in.
*
*         Input: ascii_file       - file name of ASCII file 
*                source_obj       - source data object to insert segment in
*                segment_name     - name of segment to create
*                dimension        - dimensionality of segment being created
*                index_order      - index order of segment being created
*                size_vector      - size of segment being created
*                data_type_string - data type of segment being created
*                start_offset     - starting point in ascii data to start from
*                skip_factor      - skip factor to use while importing ascii 
*				    data
*
*        Output: destination_object - data object to create containing 
*				      new segment
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: John M. Salas, Donna Koechner, Jeremy Worley & John Rasure
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int 
lkimportasc(
   char   *ascii_file,
   kobject source_obj,
   char   *segment_name,
   int     dimension,
   char   *index_order,
   int    *size_vector,
   char   *data_type_string,
   int     start_offset,
   int     skip_factor,
   kobject destination_object)
/* -library_def_end */

/* -library_code */
{
   char  **index_strings;
   int    *index_order_tmp;
   int     i;
   int     count;
   int     data_type;
   int     success = FALSE;
   int    *begin = NULL;
   int    *end = NULL;
   int    *nbegin = NULL;
   int    *nend = NULL;
   int     size;
   int    *orig_size_vector;
   int     ascii_data_type;
   int    *ascii_size_vector;
   int     ascii_dimension;
   int    *ascii_index_order;
   int     offset;
   int     check_ascii_size;
   int     check_size;
   kobject source_object1;
   kobject tmp_srcobject;
   kobject tmp_object;
   kobject reference_object;
   kaddr   data = NULL;


   /* 
    *   open the specified ASCII input ascii file
    *      we will be using the ASCII transport available with data 
    *      services to read this data, hence the kdms_open.
    */
   if ((source_object1 = kdms_open(ascii_file, KOBJ_READ)) == NULL)
   {
      kerror("ksegops", "lkimportasc", 
	     "Failed to open ASCII input file");
      return (FALSE);
   }

   /* was an index order even provided? */
   if (!*index_order)
   {
      kerror("ksegops", "lkimportasc", "No index order specified.");
      return (FALSE);
   }

   /* was a data type even provided? */
   if (!*data_type_string)
   {
      kerror("ksegops", "lkimportasc", "No data type specified.");
      return (FALSE);
   }

   /* was a size even provided? */
   if (!*size_vector)
   {
      kerror("ksegops", "lkimportasc", "No size specified.");
      return (FALSE);
   }

   /* check to make sure the dimensionality is reasonable for data services */
   if (dimension > 5 || dimension < 1)
   {
      kerror("ksegops", "lkimportasc", 
	     "Dimension must be between 1 and 5.");
      return (FALSE);
   }

   /* check the start offset to make sure it isn't unreasonably small */
   if (start_offset < 0)
   {
      kerror("ksegops", "lkimportasc", "Start offset must be > 0.");
      return (FALSE);
   }

   /* check the skip factor to make sure it isn't unreasonably small */
   if (skip_factor < 0)
   {
      kerror("ksegops", "lkimportasc", "Skip factor must be > 0.");
      return (FALSE);
   }


   /* 
    *   check the given data segment name -- make sure that the specified
    *   dimension is in-line with the expectcted dimension sizes for the
    *   polymorphic segments.   if it's not a polymorphic segment, then
    *   they are calling the shots and there's nothing we can check.
    */
   if (!*segment_name || !kstrcmp(segment_name, ""))
   {
      kerror("ksegops", "lkimportasc", "No segment specified");
      return (FALSE);
   }
   else
   {
      if ((kstrcmp(KDMS_SEGMENT_VALUE, segment_name) == 0) && 
	   dimension != 5)
      {
	 kerror("ksegops", "lkimportasc", 
		"The value segment must be 5D, not '%d'D.", dimension);
	 return (FALSE);
      }
      else if ((kstrcmp(KDMS_SEGMENT_MASK, segment_name) == 0) && 
	        dimension != 5)
      {
	 kerror("ksegops", "lkimportasc", 
		"The mask segment must be 5D, not '%d'D.", dimension);
	 return (FALSE);
      }
      else if ((kstrcmp(KDMS_SEGMENT_MAP, segment_name) == 0) &&
	        dimension != 5)
      {
	 kerror("ksegops", "lkimportasc", 
		"The map segment must be 5D, not '%d'D.", dimension);
	 return (FALSE);
      }
      else if ((kstrcmp(KDMS_SEGMENT_LOCATION, segment_name) == 0) &&
	        dimension != 4)
      {
	 kerror("ksegops", "lkimportasc", 
		"The location segment must be 4D, not '%d'D.", dimension);
	 return (FALSE);
      }
      else if ((kstrcmp(KDMS_SEGMENT_TIME, segment_name) == 0) &&
	        dimension != 1)
      {
	 kerror("ksegops", "lkimportasc", 
		"The time segment must be 1D, not '%d'D.", dimension);
	 return (FALSE);
      }


      /* break up the specified index order intro separate index strings */
      index_strings = kparse_string_delimit(index_order, " ", 
					    KDELIM_CLEAN, &count);

      /* did they specify the correct number of indices? */
      if (count != dimension)
      {
	 kerror("ksegops", "lkimportasc", 
		"You specified %d indicies for the index order, but the "
		"specified dimensionality is %d.  You must specify %d "
		"indicies.", count, dimension, dimension);
	 return (FALSE);
      }

      if ((index_order_tmp = (int *)kcalloc(kmax(dimension, 5),
					    sizeof(int))) == NULL)
      {
	 kerror("ksegops", "lkimportasc", 
		"Could not allocate memory for temporary variable.");
	 return (FALSE);
      }

      /* determine the index 'values' from the string index orders */
      for (i = 0; i < count; i++)
      {
	 if ((kstrcmp(index_strings[i], "width") == 0) ||
	     (index_strings[i][0] == 'w' && 
	      kstrlen(index_strings[i]) == 1))
	 {
	    index_order_tmp[i] = KWIDTH;
	 }
	 else if (!kstrcmp(index_strings[i], "height") ||
		  (index_strings[i][0] == 'h'
		   && kstrlen(index_strings[i]) == 1))
	 {
	    index_order_tmp[i] = KHEIGHT;
	 }
	 else if (!kstrcmp(index_strings[i], "depth") ||
		  (index_strings[i][0] == 'd'
		   && kstrlen(index_strings[i]) == 1))
	 {
	    index_order_tmp[i] = KDEPTH;
	 }
	 else if (!kstrcmp(index_strings[i], "elements") ||
		  (index_strings[i][0] == 'e'
		   && kstrlen(index_strings[i]) == 1))
	 {
	    index_order_tmp[i] = KELEMENTS;
	 }
	 else if (!kstrcmp(index_strings[i], "time") ||
		  (index_strings[i][0] == 't'
		   && kstrlen(index_strings[i]) == 1))
	 {
	    index_order_tmp[i] = KTIME;
	 }
	 else if (!kstrcmp(index_strings[i], "dimension") ||
		  (index_strings[i][0] == 'n'
		   && kstrlen(index_strings[i]) == 1))
	 {
	    index_order_tmp[i] = KDIMENSION;
	 }
	 else if (!kstrcmp(index_strings[i], "mapwidth") ||
		  !kstrcmp(index_strings[i], "mw"))
	 {
	    index_order_tmp[i] = KMAP_WIDTH;
	 }
	 else if (!kstrcmp(index_strings[i], "mapheight") ||
		  !kstrcmp(index_strings[i], "mh"))
	 {
	    index_order_tmp[i] = KMAP_HEIGHT;
	 }
	 else if (!kstrcmp(index_strings[i], "mapelements") ||
		  !kstrcmp(index_strings[i], "me"))
	 {
	    index_order_tmp[i] = KMAP_ELEMENTS;
	 }
	 else if (!kstrcmp(index_strings[i], "mapdepth") ||
		  !kstrcmp(index_strings[i], "md"))
	 {
	    index_order_tmp[i] = KMAP_DEPTH;
	 }
	 else if (!kstrcmp(index_strings[i], "maptime") ||
		  !kstrcmp(index_strings[i], "mt"))
	 {
	    index_order_tmp[i] = KMAP_TIME;
	 }
	 else
	 {
	    kerror("ksegops", "lkimportasc", 
		   "The only indicies supported are : width (w), height (h), "
		   "depth (d), time (t), elements (e), dimension (n), "
		   "mapwidth (mw), mapheight (mh), mapelements (me), "
		   "mapdepth (md) and maptime (mt).");
	    return (FALSE);
	 }
	 kfree(index_strings[i]);
      }

      /* null out any empty indices above the specified dimensionality */
      for (i = count; i < 5; i++)
	 index_order_tmp[i] = KNONE;
   }

   /*
    *   we have all the arguments we need and they all seem to be okay
    *   -- let's go check out the input ASCII file.  The ASCII transport 
    *      places everything in the value segment, so we look there.
    */
   if (!kdms_get_attributes(source_object1, KDMS_SEGMENT_VALUE,
			    KDMS_DIMENSION,   &ascii_dimension,
			    KDMS_SIZE,        &ascii_size_vector,
			    KDMS_DATA_TYPE,   &ascii_data_type,
			    KDMS_INDEX_ORDER, &ascii_index_order,
			    NULL))
   {
      kerror("ksegops", "lkimportasc", 
	     "Can not determine attributes of the ASCII source object.");
      kfree(index_order_tmp);
      return (FALSE);
   }

   /*
    * make sure the size of the output file matches the input file or
    * at least is not greater than it.
    */
   for (check_ascii_size = ascii_size_vector[0], i = 1;
	i < ascii_dimension; i++)
   {
      check_ascii_size *= ascii_size_vector[i];
   }

   for (check_size = (size_vector[0] + skip_factor), i = 1;
	i < dimension; i++)
   {
      check_size *= size_vector[i];
   }

   if ((check_size + start_offset - 1) > check_ascii_size)
   {
      kerror("ksegops", "lkimportasc", 
	     "There are not enough data points in the ascii file to fill "
	     "out a segment of the specified size in the output file.");
      kfree(index_order_tmp);
      return (FALSE);
   }

   /*
    * if input source viff exists then copy it to the output before 
    * inserting the new segment.
    */
   if (source_obj &&
       !kdms_copy_object(source_obj, destination_object, TRUE, TRUE))
   {
      kerror("ksegops", "lkimportasc", 
	     "Could not copy source data object to destination object.");
      kfree(index_order_tmp);
      return (FALSE);
   }

   /* does this thing already have a segment of the given name? */
   if (!kdms_query_segment(destination_object, segment_name) &&
       !kdms_create_segment(destination_object, segment_name))
   {
      kerror("ksegops", "lkimportasc", 
	     "Can not create the segment '%s' in destination object.", 
	     segment_name);
      kfree(index_order_tmp);
      return (FALSE);
   }

   /* is the given data type legal? */
   if ((data_type = kdatatype_to_define(data_type_string)) == 0)
   {
      kerror("ksegops", "lkimportasc",
	     "Invalid data type '%s' was specified.", data_type_string);
      kfree(index_order_tmp);
      return (FALSE);
   }

   /* set up the new segment attributes in the destination object */
   if (!kdms_set_attributes(destination_object, segment_name,
			    KDMS_DIMENSION,     dimension,
			    KDMS_INDEX_ORDER,   index_order_tmp,
			    KDMS_DATA_TYPE,     data_type,
			    KDMS_SIZE,          size_vector,
			    NULL))
   {
      kerror("ksegops", "lkimportasc", 
	     "Can not set attributes on the segment '%s' in "
	     "destination object.", segment_name);
      kfree(index_order_tmp);
      return (FALSE);
   }
   
   /* 
    *  create a reference on the destination so we can deal with 
    *  the data in a friendly data type, yet still store the data in 
    *  the desired data type 
    */
   reference_object = kdms_reference(destination_object);

   kdms_set_attribute(reference_object, segment_name,
		      KDMS_DATA_TYPE, ascii_data_type);

   /*
    * copy the data with kdms_get_data() and then put it with kdms_put_data().
    * First figure at the begin and end region first. 
    */
   if (start_offset == 0 && skip_factor == 0 &&
       (ascii_dimension != dimension ||
	ascii_data_type != data_type || !success))
   {

      begin  = (int *)kmalloc(ascii_dimension * sizeof(int *));
      end    = (int *)kmalloc(ascii_dimension * sizeof(int *));
      nbegin = (int *)kmalloc(kmax(dimension, 5) * sizeof(int *));
      nend   = (int *)kmalloc(kmax(dimension, 5) * sizeof(int *));

      if (!begin || !end || !nbegin || !nend)
      {
	 kerror("ksegops", "lkimportasc", 
		"Could not allocate memory for temporary variables.");
	 kfree(index_order_tmp);
	 kfree(begin);
	 kfree(end);
	 kfree(nbegin);
	 kfree(nend);
	 return (FALSE);
      }

      for (i = 0; i < ascii_dimension; i++)
      {
	 begin[i] = 0;
	 end[i]   = ascii_size_vector[i] - 1;
      }

      for (i = 0; i < dimension; i++)
      {
	 nbegin[i] = 0;
	 nend[i]   = size_vector[i] - 1;
      }
      for (i = dimension; i < 5; i++)
      {
	 nbegin[i] = 0;
	 nend[i]   = 0;
      }

      data = kdms_get_data(source_object1, KDMS_SEGMENT_VALUE,
			   begin, end, NULL);
      kdms_put_data(reference_object, segment_name,
		    nbegin, nend, data);

   }
   else if (start_offset != 0 || skip_factor != 0 || !success)
   {

      orig_size_vector = (int *)kmalloc(kmax(dimension, 5) * sizeof(int *));
      begin  = (int *)kmalloc(ascii_dimension * sizeof(int *));
      end    = (int *)kmalloc(ascii_dimension * sizeof(int *));
      nbegin = (int *)kmalloc(kmax(dimension, 5) * sizeof(int *));
      nend   = (int *)kmalloc(kmax(dimension, 5) * sizeof(int *));

      if (!begin || !end || !nbegin || !nend)
      {
	 kerror("ksegops", "lkimportasc", 
		"Could not allocate memory for temporary variables.");
	 kfree(index_order_tmp);
	 kfree(orig_size_vector);
	 kfree(begin);
	 kfree(end);
	 kfree(nbegin);
	 kfree(nend);
	 return (FALSE);
      }

      for (i = 0, size = 1.0; i < ascii_dimension; i++)
      {
	 size *= ascii_size_vector[i];
	 begin[i] = 0.0;
	 end[i] = ascii_size_vector[i] - 1;

	 orig_size_vector[i] = 1.0;
	 nbegin[i] = 0;
	 nend[i] = 0;
      }

      nend[0] = size - 1;
      orig_size_vector[0] = size;

      tmp_srcobject = kdms_create();

      kdms_create_segment(tmp_srcobject, "tempsegment");

      kdms_set_attributes(tmp_srcobject, "tempsegment",
			  KDMS_DIMENSION, 5,
			  KDMS_INDEX_ORDER, ascii_index_order,
			  KDMS_DATA_TYPE, ascii_data_type,
			  KDMS_SIZE, orig_size_vector,
			  NULL);

      data = kdms_get_data(source_object1, KDMS_SEGMENT_VALUE,
			   begin, end, data);
      kdms_put_data(tmp_srcobject, "tempsegment",
		    nbegin, nend, data);

      kfree(data);
      data = NULL;

      for (i = 0; i < ascii_dimension; i++)
      {
	 begin[i] = 0.0;
	 end[i]   = 0.0;
      }

      for (i = 0, size = 1.0; i < dimension; i++)
      {
	 orig_size_vector[i] = size_vector[i];
	 size *= size_vector[i];
	 size_vector[i] = 1.0;
	 nbegin[i] = 0;
	 nend[i] = 0;
      }
      size_vector[0] = size;

      for (i = dimension; i < 5; i++)
      {
	 nbegin[i] = 0;
	 nend[i]   = 0;
      }


      tmp_object = kdms_create();

      kdms_create_segment(tmp_object, segment_name);

      kdms_copy_segment_attributes(destination_object, segment_name,
				   tmp_object, segment_name);

      kdms_set_attributes(tmp_object, segment_name,
			  KDMS_DATA_TYPE, ascii_data_type,
			  KDMS_SIZE, size_vector,
			  NULL);

      offset = 0;
      nbegin[0] = 0;

      while (size > nbegin[0])
      {
	 begin[0] = start_offset + offset;
	 end[0] = begin[0] + orig_size_vector[0] - 1;
	 nend[0] = nbegin[0] + orig_size_vector[0] - 1;

	 data = kdms_get_data(tmp_srcobject, "tempsegment",
			      begin, end, data);
	 kdms_put_data(tmp_object, segment_name,
		       nbegin, nend, data);

	 offset += skip_factor + orig_size_vector[0];
	 nbegin[0] = nend[0] + 1;
      }

      nbegin[0] = 0.0;
      for (i = 0, size = 1.0; i < dimension; i++)
      {
	 begin[i] = 0;
	 end[i] = orig_size_vector[i] - 1;
      }

      for (i = dimension; i < 5; i++)
      {
	 begin[i] = 0;
	 end[i] = 0;
      }

      kfree(data);
      data = NULL;

      data = kdms_get_data(tmp_object, segment_name,
			   nbegin, nend, data);
      kdms_put_data(reference_object, segment_name,
		    begin, end, data);

      kfree(orig_size_vector);

      kdms_close(tmp_srcobject);
      kdms_close(tmp_object);
   }

   kfree(index_order_tmp);
   kfree(begin);
   kfree(end);
   kfree(nbegin);
   kfree(nend);
   kdms_close(reference_object);

   return (TRUE);
}
/* -library_code_end */
