 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kbitwise
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkbitwise
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/*
 * Operation structure declaration
 */
struct _operation
{
        char    *name;
	int     routine;
};

/*
 * Function declaration list
 */
static struct _operation operations[] =
{
#define KAND    0
    { "and", KAND },
#define KOR     1
    { "or",  KOR  },
#define KXOR    2
    { "xor", KXOR },
#define KNAND   3
    { "nand", KNAND },
#define KNOR    4
    { "nor",  KNOR },
#define KANDREV 5
    { "andrev",  KANDREV },
#define KANDINV 6
    { "andinv",  KANDINV },
#define KORREV  7
    { "orrev",  KORREV },
#define KORINV  8
    { "orinv",  KORINV },
#define KRS     9
    { "rs",  KRS },
#define KLS     10
    { "ls",  KLS },
    { "srs",  ERROR },
    { "sls",  ERROR },
    { "rr",  ERROR },
    { "lr",  ERROR },
};
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkbitwise - perform dual operand bitwise operations on data object(s)
* 
*       Purpose: The following is the library routine that performs
*		 a binary bitwise operation on two data objects.
*		 The routine takes two input data object (or a single data
*		 object and a constant), a function name, and an output
*		 data object in which to store the result of the
*		 function.
*
*		 The supported functions are:
*                .TS
*                center tab(:) ;
*                l l .
*		 and    :  o = i1 & (i2 or sval)
*		 or     :  o = i1 | (i2 or sval)
*		 xor    :  o = i1 ^ (i2 or sval)
*		 nand   :  o = (~i1) | (~(i2 or sval))
*		 nor    :  o = (~i1) & (~(i2 or sval))
*		 andrev :  o = i1 & (~(i2 or sval))
*		 andinv :  o = (~i1) & (i2 or sval)
*		 orrev  :  o = i1 | (~(i2 or sval))
*		 orinv  :  o = (~i1) | (i2 or sval)
*		 rs     :  o = i1 >> by (i2 or sval)
*		 ls     :  o = i1 << by (i2 or sval)
*		 srs    :  Signed Right Shift by (i2 or sval)
*		 sls    :  Signed Left Shift by (i2 or sval)
*		 rr     :  Right Rotate by (i2 or sval)
*		 lr     :  Left Rotate by (i2 or sval)
*                .TE
*
*		 Therefore if lkbitwise() were called with two input data
*		 objects:
*
*		 !    lkbitwise(i1, i2, 0, "and", o);
*
*		 then the following would be performed:
*
*		 !    o = i1 & i2
*
*		 If lkbitwise were called with a single data object and
*		 a constant:
*
*		 !    lkbitwise(i1, NULL, 10, "and", o);
*
*		 then the following would be performed:
*
*		 !    o = i1 & 10
*
*         Input: i1   - the input object to be processed
*		 i2   - the second input object (if NULL then uses the
*			 constant "sval")
*		 sval  - the constant to be instead of the second data object
*		 function - the function to be applied
*        Output: o  - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions: 
*    Written By: Donna Koechner & Mark Young
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkbitwise (
   kobject i1,
   kobject i2,
   long    sval,
   char    *function,
   kobject o)
/* -library_def_end */

/* -library_code */
{
	int routine;
	unsigned long *idata1 = NULL; 
        unsigned long *idata2 = NULL; 
        unsigned long *odata  = NULL, val = (unsigned long)sval;
	int  j, num, indx;
        int  type1, type2;
        int  num_lines, line_size, error = FALSE;
        int w1, w2, h1, h2, d1, d2, t1, t2, e1, e2;


	if ((i1 = kpds_reference_object(i1)) == KOBJECT_INVALID)
	{
	   kerror("kdatamanip", "lkbitwise",
		  "failed to duplicate input object 1");
	   return(FALSE);
	}

	if (i2 != NULL && ((i2 = kpds_reference_object(i2)) == KOBJECT_INVALID))
	{
	   (void) kpds_close_object(i1);
	   kerror("kdatamanip", "lkbitwise",
		  "failed to duplicate input object 2");
	   return(FALSE);
	}

        /*
         *  Get the maximum size of the value segment for i1 & i2 and make
         *  sure that i1, i2, o are the same.
         */
        kpds_get_attribute(i1, KPDS_VALUE_SIZE, &w1, &h1, &d1, &t1, &e1);

        if (i2 != NULL)
        {
           kpds_get_attribute(i2, KPDS_VALUE_SIZE, &w2, &h2, &d2, &t2, &e2);
           w1 = kmax(w1, w2); h1 = kmax(h1, h2); d1 = kmax(d1, d2);
           t1 = kmax(t1, t2); e1 = kmax(e1, e2);

           kpds_set_attribute(i2, KPDS_VALUE_SIZE, w1, h1, d1, t1, e1);

           kpds_set_attribute(i1, KPDS_VALUE_SIZE, w1, h1, d1, t1, e1);
        }

        kpds_set_attribute(o, KPDS_VALUE_SIZE, w1, h1, d1, t1, e1);

        kpds_get_attribute(i1, KPDS_VALUE_DATA_TYPE, &type1);

        if (i2 != NULL)
        {
           kpds_get_attribute(i2, KPDS_VALUE_DATA_TYPE, &type2);
           type1 = kdatatype_cast_output(type1,type2);
        }
        
        kpds_set_attribute(o, KPDS_VALUE_DATA_TYPE, type1);
        
	if ((o = kpds_reference_object(o)) == KOBJECT_INVALID)
        {
           (void) kpds_close_object(i1);
           if (i2) (void) kpds_close_object(i2);
           kerror("kdatamanip", "lkbitwise",
                  "failed to duplicate output object");
           return(FALSE);
        } 

	for (num = 0, indx = -1; num < knumber(operations); num++)
	{
	   if (kstrcmp(function, operations[num].name) == 0)
	   {
	      indx = num;
	      break;
	   }
	}

	if (indx == -1)
	{
	   (void) kpds_close_object(i1);
	   if (i2) (void) kpds_close_object(i2);
	   (void) kpds_close_object(o);
	   kerror("kdatamanip", "lkbitwise", 
                  "Sorry, but the function \
'%s' is not in the list of callable operations", function);
	   return(FALSE);
	}
	else if (operations[indx].routine == ERROR)
	{
	   (void) kpds_close_object(i1);
	   if (i2) (void) kpds_close_object(i2);
	   (void) kpds_close_object(o);
	   kerror("kdatamanip", "lkbitwise", "Sorry, but i am unable to \
run the operation '%s' on data services objects at this time.", function);
	   return(FALSE);
	}
	else
	   routine = (int) operations[indx].routine;

	/*
	 *  Decide the data type to be used in processing and set it for
	 *  i1, i2, and o.
	 */ 
	kpds_set_attribute(i1, KPDS_VALUE_DATA_TYPE, KULONG);
	if (i2 != NULL)
	{
	   kpds_set_attribute(i2, KPDS_VALUE_DATA_TYPE, KULONG);
	}
	kpds_set_attribute(o,  KPDS_VALUE_DATA_TYPE, KULONG);

	/*
	 *  Do the processing by simply requesting the lines of data from
	 *  i1 & i2 and then evaluate the symbol list and then store the
	 *  result into the output. 
	 */

        kpds_get_attribute(i1, KPDS_VALUE_LINE_INFO, &line_size, &num_lines);

	odata  = (unsigned long *) kmalloc(w1 * sizeof(unsigned long));

	for (num = 0; num < num_lines; num++)
	{

	   idata1 = (unsigned long *) kpds_get_data(i1, KPDS_VALUE_LINE, (kaddr) idata1);

	   if (i2 != NULL)
	   {
	     idata2 = (unsigned long *) kpds_get_data(i2, KPDS_VALUE_LINE,
					     (kaddr) idata2);
	   }

	   for (j = 0; j < line_size; j++)
	   {
	      switch (routine)
	      {
		  case KAND:
		       odata[j] = idata1[j] & ((!i2) ? val : idata2[j]);
		       break;

		  case KOR:
		       odata[j] = idata1[j] | ((!i2) ? val : idata2[j]);
		       break;

		  case KXOR:
		       odata[j] = idata1[j] ^ ((!i2) ? val : idata2[j]);
		       break;

		  case KNAND:
		       odata[j] = ~idata1[j] | ~((!i2) ? val : idata2[j]);
		       break;

		  case KNOR:
		       odata[j] = ~idata1[j] & ~((!i2) ? val : idata2[j]);
		       break;

		  case KANDREV:
		       odata[j] = idata1[j] & ~((!i2) ? val : idata2[j]);
		       break;

		  case KANDINV:
		       odata[j] = ~idata1[j] & ((!i2) ? val : idata2[j]);
		       break;

		  case KORREV:
		       odata[j] = idata1[j] | ~((!i2) ? val : idata2[j]);
		       break;

		  case KORINV:
		       odata[j] = ~idata1[j] | ((!i2) ? val : idata2[j]);
		       break;

		  case KRS:
		       odata[j] = idata1[j] >> ((!i2) ? val : idata2[j]);
		       break;

		  case KLS:
		       odata[j] = idata1[j] << ((!i2) ? val : idata2[j]);
		       break;
	      }
	   }
           /* Mask off extraneous bits of the output data type is KBIT */
           if (type1 == KBIT)
             {
	       for (j = 0; j < line_size; j++)
                 {
                   odata[j] = odata[j] & 0x1;
                 }
             }

	   if (!kpds_put_data(o, KPDS_VALUE_LINE, (kaddr) odata))
	   {
	      kerror("kdatamanip", "lkbitwise",
		     "failed to put line in output object");
              error = TRUE;
              break;
	   }
	}
	(void) kpds_close_object(i1);
	if (i2) (void) kpds_close_object(i2);
	(void) kpds_close_object(o);
	kfree(idata1);
	kfree(idata2);
	kfree(odata);
        if (error)
           return(FALSE);
        else
	   return(TRUE);
}
/* -library_code_end */
