/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Token Utilities
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>			ktoken_find()
   >>>>   Public:
   >>>>                 kstring_to_token()
   >>>>                 ktoken_to_string()
   >>>>                 ktoken_check()
   >>>>                 ktoken_delete()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


#define TABLESIZE 1024

/*
 *  This is the next available token, hash table, string table
 *
 *  NOTE:  token_next must be initially greater than 0.
 */
static int   token_last = -1;
static int   token_next = 1;
static int   token_tablesize = 0;
static char  **token_table = NULL;
static klist **hash_table = NULL;

/*-----------------------------------------------------------
|
|  Routine Name: ktoken_find - static routine to find a token
|
|       Purpose: This routine checks to see if the string has been
|		 token'ized.
|
|         Input: string - the string to check if token'ized
|         	 hash   - if -1 then re-compute the hash from the string
|        Output: indx   - the hash index into the hash_table
|       Returns: the token or 0 if it doesn't exist
|
|    Written By: Mark Young
|          Date: Nov 16, 1993
| Modifications:
|
------------------------------------------------------------*/

int ktoken_find(
   char *string,
   int  hash,
   int  *indx)
{
	klist    *list;


	/*
	 *  Sanity check to make sure that string is not NULL
	 */
	if (!string)
	   return(0);

	/*
	 *  Pre-check
	 */
	if (token_last != -1 && strcmp(string, token_table[token_last]) == 0)
	   return(token_last);

	/*
	 *  First check to make sure that hash table has been initialized.
	 *  ie) not NULL.
	 */
	if (!hash_table)
	{
	   hash_table = (klist **) kcalloc(TABLESIZE, sizeof(klist *));
	   token_table = (char **) kcalloc(TABLESIZE, sizeof(char *));
	   token_tablesize = TABLESIZE;
	}

	/*
	 *  Compute the index into the hash table
	 */
	if (hash == -1)
	   hash = khash(string, -1);

	hash = kabs(hash) % TABLESIZE;
	if (indx) *indx = hash;

	for (list = hash_table[hash]; list != NULL; list = list->next)
	{
	   if (strcmp(string, list->client_data) == 0)
	   {
	      token_last = (int) list->identifier;
	      return(token_last);
	   }
	}
	return(0);
}


/************************************************************
*
*  Routine Name: kstring_to_token - return the token that is associated with
*				 the specified string
*
*       Purpose: Returns the token that is associated with the supplied
*		 token.  The idea is that given a string a unique token
*		 is returned so if an application is being slowed down
*		 by many string comparisons, or searching thru many string
*		 attributes as is the case with the xvwidgets and kdataserv
*		 libraries, the use of tokens will help to greatly reduce
*		 the comparison time.
*
*		 kstring_to_token() routine allows a routine to perform string
*		 comparisons using "token1 == token2" instead of the expensive
*		 kstrcmp(str1, str2) function call.  For example to instatiate
*		 a string, using the function call kstring_to_token(), the
*		 programmer uses the following call:
*
*		 !	token1 = kstring_to_token("test string1");
*		 !	token2 = kstring_to_token("test string2");
*
*		 This should yield two totally distinct tokens, which when
*		 compared (token1 == token2) will result that the two strings
*		 are not the same.
*
*		 If the string is NULL or we are unable to allocate the
*		 space to copy the string, then 0 is returned.  A call to
*		 ktoken_to_string() of 0 will result in NULL being returned.
*
*	  Input: string - the string which we will be returning the token for
*
*       Returns: the token or 0 upon failure 
*
*    Written By: Mark Young
*          Date: Nov 16, 1992
*************************************************************/

int kstring_to_token(
   char *string)
{
	int indx, token;


	/*
	 *  First try to see if the token already exists...
	 */
	if ((token = ktoken_find(string, -1, &indx)) != 0 || !string)
	   return(token);

	if ((token = token_next++) == token_tablesize)
	{
	   token_tablesize += TABLESIZE;
	   token_table = (char **) krealloc(token_table, token_tablesize *
				sizeof(char *));
	}
	token_last = token;
	token_table[token] = kstrdup(string);
	hash_table[indx] = klist_add(hash_table[indx], (kaddr) token,
				(kaddr) token_table[token]);
	return(token);
}


/************************************************************
*
*  Routine Name: ktoken_to_string - return the string associated with
*				 the specified token
*
*       Purpose: Returns the string that is associated with the supplied
*		 token.  The idea is that given a unique token for each
*		 string an application uses ktoken_to_string(token) in order
*		 to get the unique string back.  The kstring_to_token() routine
*		 should be used to create the unique token for the specified
*		 string.
*
*	  Input: token - the token which we will be returning the string for
*        Output:
*
*       Returns: the string or NULL if the token does not exist
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 16, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char *ktoken_to_string(
   int token)
{
	return((token < 0 || token >= token_next) ? NULL : token_table[token]);
}


/************************************************************
*
*  Routine Name: ktoken_check - check to see if a string has been token'ized
*
*       Purpose: Checks to see if the string has a token representation.
*
*	  Input: string - the string which we will be checking for
*       Returns: If the string is NULL or if the string has not been
*		 token'ized 0 is returned, otherwise the token is returned.
*    Written By: Mark Young
*          Date: Nov 16, 1993
*************************************************************/

int ktoken_check(
   char *string)
{
	/*
	 *  return whether a token already exists...
	 */
	return(ktoken_find(string, -1, NULL));
}


/************************************************************
*
*  Routine Name: ktoken_delete - delete the token'ized string from the list
*				 of tokens
*
*       Purpose: Deletes the token'ized representation of a string from the
*		 token list.  If the string has a token representation then
*		 it is removed from the list.
*
*		 If the string is NULL or the string or the token'ized
*		 string cannot be deleted then FALSE is returned.
*
*	  Input: string - the string which we will be deleting
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 16, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int ktoken_delete(
   char *string)
{
	return(FALSE);
}
