/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Program Utility Routines
   >>>>
   >>>>  Private:
   >>>>                 kprog_set_argc()
   >>>>                 kprog_set_argv()
   >>>>                 kprog_set_envp()
   >>>>                 kprog_set_program()
   >>>>                 kprog_set_toolbox()
   >>>>
   >>>>                 kprog_get_argc()
   >>>>                 kprog_get_argv()
   >>>>                 kprog_get_envp()
   >>>>                 kprog_get_program()
   >>>>                 kprog_get_toolbox()
   >>>>                 kprog_get_command()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static char  *program_name    = "Unknown_Program_Name";
static char  *program_toolbox = "BOOTSTRAP";
static char  *program_command = NULL;

static int    program_argc   = 0;
static char   **program_argv = NULL;
#if (KOPSYS_LOCAL == KOPSYS_IRIX)
extern char   **_environ;
#define kenviron _environ
#else
extern char   **environ;
#define kenviron environ
#endif


/************************************************************
*
*  Routine Name: kprog_set_argc - set the number of commandline parameters
*       Purpose: Sets the number of arguments in the program argv structure.
*	  Input: argc - The current list in which we will be adding the
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young
*          Date: Sep 26, 1992 13:07
*
*************************************************************/

int kprog_set_argc(
   int argc)
{
	if (argc < 0)
	   return(FALSE);

	program_argc = argc;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kprog_set_argv - set the command line argument array
*
*       Purpose: Set the argv list that the program uses.
*	  Input: argv - The array to set the program argv list to.
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young
*          Date: Sep 26, 1992 13:07
*************************************************************/

int kprog_set_argv(
   char *argv[])
{
	if (argv == NULL)
	   return(FALSE);

	program_argv = argv;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kprog_set_envp - set the number of environment variable
*				  parameters
*       Purpose: Sets the number of environment variable parameters structure
*		 from the program envp structure.
*	  Input: argc - The current list in which we will be adding the
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young
*          Date: Sep 26, 1992 13:07
*************************************************************/

int kprog_set_envp(
   char *envp[])
{
	if (envp == NULL)
	   return(FALSE);

	kenviron = (char **) envp;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kprog_set_program - set the name of the program
*
*       Purpose: Sets the name of the program.  This is usually the
*		 basename of the first argument in the argv structure.
*
*	  Input: program - The current name of this program
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young & Steve Kubica
*          Date: Jun 23, 1994
*************************************************************/
int kprog_set_program(
   char *program)
{
	char *arguments;

	if (program == NULL)
	   return(FALSE);

	program_name = kbasename(program, NULL);
	if (program_argv != NULL && program_argc > 1)
	{
	   arguments = karray_to_string(&program_argv[1], program_argc -1, " ");
	   program_command = kstring_3cat(program_name, " ", arguments, NULL);
	   kfree(arguments);
	}
	else
	   program_command = kstrdup(program_name);

	return(TRUE);
}

/************************************************************
*
*  Routine Name: kprog_set_toolbox - set the toolbox in which this
*				     software object belongs.
*       Purpose: Sets the name of the toolbox in where the software
*		 belongs.
*
*	  Input: toolbox - toolbox name in which the program exists
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young
*          Date: Sep 26, 1992 13:07
*  Side Effects: the toolbox name is also placed into the environment
*		 variable list so that "$TOOLBOX" will be defined.
*************************************************************/

int kprog_set_toolbox(
   char *toolbox)
{
	char temp[KLENGTH], *name;


	if (toolbox == NULL)
	   return(FALSE);

	/*
	 *  Stick the program toolbox in to the environment variable list
	 *  so that "$TOOLBOX" is defined.
	 */
	name = kstring_upper(toolbox, NULL);
	(void) ksprintf(temp,"TOOLBOX=$%s", name);
	kfree(name);
	kputenv(temp);

	program_toolbox = toolbox;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kprog_get_argc - get the number of arguments
*				  in the argv structure
*
*       Purpose: Gets the number of arguments in the program argv
*		 structure.
*	  Input: argc - The current list in which we will be adding the
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young
*          Date: Sep 26, 1992 13:07
*************************************************************/

int kprog_get_argc(void)
{
	return(program_argc);
}

/************************************************************
*
*  Routine Name: kprog_get_argv - get the arguments in the argv structure
*       Purpose: returns the array of commandline arguments.
*	  Input: argc - The current list in which we will be adding the
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions: This routine returns a pointer to the internal argv array.
*		 Do not change or free any data in this array, use for
*		 read only purposes only.
*    Written By: Mark Young
*          Date: Sep 26, 1992 13:07
*************************************************************/

char **kprog_get_argv(void)
{
	return(program_argv);
}

/************************************************************
*
*  Routine Name: kprog_get_envp - gets the environment variable parameter
*				  structure
*       Purpose: Gets the environment variable parameters structure, which
*		 is the envp.
*       Returns: returns the envp structure or NULL upon failure
*  Restrictions: This routine returns a pointer to the internal environment
*		 array pointer.  Do not change or free any data in this array,
*		 Use as a read only resource only.
*    Written By: Mark Young
*          Date: Sep 26, 1992 13:07
*************************************************************/

char **kprog_get_envp(void)
{
	return((char **) kenviron);
}

/************************************************************
*
*  Routine Name: kprog_get_program - gets the name of the program
*
*       Purpose: Gets the name of the program.  This is usually the
*		 basename of the first argument in the argv structure.
*
*	  Input: program - The current name of this program.
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: This routine returns a pointer to the private string
*		 used to store the program variable.  Use this as
*		 a read only resource; do NOT change or free this pointer.
*    Written By: Mark Young
*          Date: Sep 26, 1992 13:07
*************************************************************/

char *kprog_get_program(void)
{
	return(program_name);
}

/************************************************************
*
*  Routine Name: kprog_get_toolbox - gets the toolbox in which this
*				     program belongs.
*
*       Purpose: Gets the name of the toolbox in which this program
*		 belongs.
*       Returns: the current toolbox or NULL if not initialized
*  Restrictions: This routine returns a pointer to the private string
*		 used to store the toolbox variable.  Use this as
*		 a read only resource; do NOT change or free this pointer.
*    Written By: Mark Young
*          Date: Sep 26, 1992
*
*************************************************************/

char *kprog_get_toolbox(void)
{
	return(program_toolbox);
}

/************************************************************
*
*  Routine Name: kprog_get_command - gets the command string in which this
*				     program was executed with.
*       Purpose: Gets the command used to originally excute the current program.
*  Restrictions: This routine returns a pointer to the private string
*		 used to store the toolbox variable.  Use this as
*		 a read only resource; do NOT change or free this pointer.
*    Written By: Mark Young & Steve Kubica
*          Date: Jun 23, 1994
*************************************************************/

char *kprog_get_command(void)
{
	return(program_command);
}
