/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Sun Virtual Memory Transport Routines
   >>>>
   >>>>  Private:
   >>>>			mmap_tempnam
   >>>>			mmap_open
   >>>>			mmap_close
   >>>>			mmap_read
   >>>>			mmap_write
   >>>>			mmap_lseek
   >>>>			mmap_access
   >>>>			mmap_unlink
   >>>>			mmap_lock
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


#if !defined(KMMAP_DEF)
TransportInformation mmap_transport[] = {NULL};
#else

/* memory mapped transport */
#include <sys/mman.h>

static int mmap_tempnam	PROTO((char *, char *, char *));
static int mmap_open	PROTO((char *, int, int, kfile *));
static int mmap_close	PROTO((kfile *));
static int mmap_read	PROTO((kfile *, char *, int));
static int mmap_write	PROTO((kfile *, char *, int));
static int mmap_lseek	PROTO((kfile *, int, int));
static int mmap_access	PROTO((char *, char *, int));
static int mmap_unlink	PROTO((char *, char *));
static int mmap_lock	PROTO((kfile *, int));

TransportInformation mmap_transport[] =
{
    {
	"Memory Mapped Management",
	"mmap",
	TRUE,
	mmap_tempnam,
	mmap_open,
	mmap_close,
	mmap_read,
	mmap_write,
	mmap_lseek,
	mmap_access,
	mmap_unlink,
	mmap_lock,
	NULL,
    }
};

typedef struct
{
	int	fid;
	char	*addr;
	int	size;
	int	offset;
	int	prot;
	int	map;

} ResourceStruct;


#if !defined(mmap)  && !defined(__sgi)
extern char *mmap();
#endif

/*-------------------------------------------------------------------*
|  
|		  Private Sun Virtual Mempory Routines
|  
--------------------------------------------------------------------*/

static void mmap_reopen(
   kfile *file,
   int   num_bytes)
{
	char	*addr;
	ResourceStruct *resources = (ResourceStruct *) file->resources;
	int	map = resources->map;


	/*
	 *  Go ahead and seek to the position in which we are know that
	 *  data will be written.  This extends the file so that the size
	 *  of the memory map will be valid to this point.
	 */
	lseek(resources->fid, num_bytes-1, 0);
	if (write(resources->fid, "", 1) != 1)
	{
	   kinfo(KSYSLIB, "mmap_reopen:  Write failure, before mmap call. \
Unable to remap file '%s'", file->path);
	   return;
	}

	/*
	 *  Attach to file the appropriate memory map space...
	 */
	if ((addr = mmap(resources->addr, num_bytes, resources->prot, map,
			 resources->fid, 0)) == (char *) -1)
	{
	   kinfo(KSYSLIB, "mmap_reopen:  Memory Map (mmap) failure.  Unable \
to remap file '%s'", file->path);
	   return;
	}
	resources->addr = addr;
	resources->size = num_bytes;
	/*ksetbuffer(file, addr, num_bytes);*/
}

/*-------------------------------------------------------------------*
|  
|		  Sun Virtual Mempory Routines
|  
--------------------------------------------------------------------*/

/*-------------------------------------------------------------
|  
|  Routine Name: mmap_tempnam
|  
|       Purpose: This function initializes a tempnam for a "mmap"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"mmap=XXXXXXXXXX"
|		  	     ...etc....
|  
|         Input:  identifier - the transport identifier
|		  template   - the filename template to be modeled
|        Output:  result     - the newly created transport name
|   Called From:  internal routine called from ktempnam()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:13
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int mmap_tempnam(
   char *identifier,
   char *template,
   char *result)
{
	int	fid;
	char	*temp, filename[KLENGTH];


	(void) kstrcpy(filename, template);
	if ((temp = ktmpnam(filename)) == NULL)
	   return(-1);


	if ((fid = open(temp, KOPEN_WRONLY | KOPEN_CREAT | KOPEN_TRUNC, 0666)) <= 0)
	{
	   kinfo(KSYSLIB, "file_tempnam: Unable to create temporary file '%s'",
			temp);
	   return(-1);
	}
	close(fid);

	if (identifier == NULL)
	   (void) kstrcpy(result, temp);
	else
	   (void) sprintf(result,"%s=%s", identifier, temp);

	kfree(temp);
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: mmap_open
|  
|       Purpose: This function opens a "mmap" connection.  It is the
|	         internal driver to open a file, which is called by
|	         the user when they call kopen().  kopen() calls when
|	         it gets an open request with a filename of the following
|	         syntax:
|  
|		  	"mmap=XXXXXXXXX"
|         Input: 
|        Output:  returns 0 or -1 if an error occurs
|   Called From:  internal routine called from kopen()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:13
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int mmap_open(
   char  *path,
   int   flags,
   int   mode,
   kfile *file)
{
	char	*addr;
	struct  stat buf;
	ResourceStruct *resources;
	int	size, fid, map, prot, flg = flags;



	/*
	 *  Get memory map (mmap) protections from the open permission
	 *  flags...
	 */
	if ((flags & KOPEN_ACCMODE) == KOPEN_RDONLY)
	{
	   map = MAP_SHARED; 
	   prot = PROT_READ;
	}
	else
	{
	   map = MAP_SHARED; 
	   prot = PROT_WRITE | PROT_READ;
	   if (flags & KOPEN_WRONLY)
	   {
	      flg ^= KOPEN_WRONLY;
	      flg |= KOPEN_RDWR;
	   }
	}

	/*
	 *  Need to modify the flags so that we never open a file using
	 *  append.  This really screws us up when we try to close the
	 *  memory map.  Instead we will set the offset to initially be
	 *  the size of the file.  This should achieve the same result...
	 */
	if (flags & KOPEN_APPEND)
	   flg ^= KOPEN_APPEND;

	/*
	 *  Open the file for the memory map (mmap) call.
	 */
	if ((fid = open(path, flg, mode)) < 0)
	{
	   kinfo(KSYSLIB, "mmap_open:  Unable to open file '%s'", path);
	   return(-1);
	}

	/*
	 *  Get the size of the file
	 */
	if (stat(path, &buf) == -1 || (size = buf.st_size) < 0)
	{
	   kinfo(KSYSLIB, "mmap_open:  Unable to stat file '%s'", path);
	   return(-1);
	}

	if (size == 0 && write(fid, "", 1) != 1)
        {
           kinfo(KSYSLIB, "mmap_open:  Write failure, before mmap call. \
Unable to map file '%s'", path);
#if 0
kprintf("FAILED! unable to mmap file '%s'\n", path);
#endif
	   return(kfile_rename(file, "file="));
        }


	/*
	 *  Attach to file the appropriate memory map space...
	 */
	if ((addr = mmap(NULL, kmax(size,1), prot, map, fid, 0)) == (char *) -1)
	{
	   /*
	    *  Since we failed to memory map the device revert to using
	    *  the UNIX file transport.
	    */
	   kinfo(KSYSLIB, "mmap_open:  Unable to memory map file '%s'", path);
#if 0
kprintf("FAILED! unable to mmap file '%s'\n", path);
#endif
	   return(kfile_rename(file, "file="));
	}

	if ((resources = (ResourceStruct *) kcalloc(1,
			sizeof(ResourceStruct))) == NULL)
	{
           kinfo(KSYSLIB, "mmap_open: Not enough memory to kcalloc (%d) bytes \
for the khoros mmap structure.\n", sizeof(ResourceStruct));
	   return(-1);
	}
	resources->fid    = fid;
	resources->addr   = addr;
	resources->size   = size;
	resources->prot   = prot;
	resources->map    = map;

        if (resources->size > 0 && !kfile_iswrite(file))
	{
	   ksetbuffer(file, resources->addr, resources->size);
	   resources->offset = size;
	}
	else
	   resources->offset = (flags & KOPEN_APPEND) ? size : 0;

	file->resources = (kaddr) resources;
#if 0
kprintf("successfully opened mmap file '%s'\n", path);
#endif
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: mmap_close
|  
|       Purpose: This function closes a "mmap".  It is the internal driver
|	         to close a data mmap, which is called by the user
|	         when they call kclose().  kclose() calls the "mmap" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the initial kopen().
|  
|         Input:  file - the kfile structure.
|        Output:  returns whether we were able to close the file
|   Called From:  internal routine called from kclose()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:13
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int mmap_close(
   kfile *file)
{
	char	*addr;
	int	fid, size, offset;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Call munmap() and close() since "kfile" is really a memory mapped
	 *  file.
	 */
	fid    = resources->fid;
	size   = resources->size;
	addr   = resources->addr;
	offset = resources->offset;

	/*
	 *  If this data file is opened for writing and data has been written
	 *  then we better make sure that the extent of the file matches that
	 *  of the memory map.  We do this by seeking to the offset and writing
	 *  the last byte into the file.  This guarantees that at least this
	 *  amount of the memory map will be written into the file.
	 */
	if (kfile_iswrite(file) && addr != NULL && offset > 0)
	{
	   lseek(resources->fid, offset-1, 0);
	   if (write(resources->fid, &addr[offset-1], 1) != 1)
	   {
	      kinfo(KSYSLIB, "mmap_close: Write failure in close call for \
file '%s'.  Ignoring and flushing as much of the memory map as possible.",
		   file->path);
	   }
	}
	return(munmap(addr, size) | close(fid));
}

/*-------------------------------------------------------------
|  
|  Routine Name: mmap_read
|  
|       Purpose: This function reads a "mmap".  It is the internal driver
|	         to read data from a mmap, which is called by the user
|	         when they call kread().  kread() calls the "mmap" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr    - the pointer to store the data into.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes read from the mmap
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:13
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int mmap_read(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	char	*addr;
	int	size, offset;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call kmemcpy() since "kfile" is just a regular 
	 *  memory pointer.
	 */
	addr   = resources->addr;
	size   = resources->size;
	offset = file->position;

	if (offset >= size)
	{
	   return(EOF);
	}
	else if (addr == ptr || addr+offset == ptr)
	{
	   if (offset+nbytes > size)
	      nbytes = size-offset;
	}
	else if (offset+nbytes > size)
	{
	   kmemcpy(ptr, addr+offset, size-offset);
	   nbytes = size-offset;
	}
	else
	   kmemcpy(ptr, addr+offset, nbytes);

	resources->offset = offset+nbytes;
	return(nbytes);
}

/*-------------------------------------------------------------
|  
|  Routine Name: mmap_write
|  
|       Purpose: This function writes to a "mmap".  It is the internal driver
|	         to write data from the supplied data array to a mmap, which
|	         is called by the user when they call kwrite().  kwrite() calls
|	         the "mmap" internal drivers by the nature of the transport,
|	         which is dictated by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr  - the pointer to write the data from.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes written to the kfile
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:13
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int mmap_write(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	char	*addr;
	int	size, offset;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call kmemcpy() since "kfile" is just a regular 
	 *  memory pointer.
	 */
	addr   = resources->addr;
	size   = resources->size;
	offset = resources->offset;

	/*
	 *  Make sure that we aren't trying to read more bytes than we
	 *  currently have.  If so then try re-opening the file to see
	 *  if it's gotten bigger.
	 */
	if (offset+nbytes > size)
	{
	   mmap_reopen(file, offset+nbytes);
	   addr   = resources->addr;
	   size   = resources->size;
	}

	if (offset == size)
	{
	   return(EOF);
	}
	else if (addr == ptr || addr+offset == ptr)
	{
	   if (offset+nbytes > size)
	      nbytes = size-offset;
	}
	else if (offset+nbytes > size)
	{
	   kmemcpy(addr+offset, ptr, size-offset);
	   nbytes = size-offset;
	}
	else
	   kmemcpy(addr+offset, ptr, nbytes);

	resources->offset = offset+nbytes;
	return(nbytes);
}

/*-------------------------------------------------------------
|  
|  Routine Name: mmap_lseek
|  
|       Purpose: This function is used to do a "lseek".  It is the internal
|	         driver to rewind to a specific point so that the data can be
|	         skipped or re-read.  This is called when the user calls
|	         klseek().  klseek() calls the "mmap" internal drivers by the
|	         nature of the transport, which is dictated by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  offset - the offset in which to seek
|		  whence - the control of how the offset will be applied
|        Output:  returns the number of bytes written to the kfile
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:13
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int mmap_lseek(
   kfile *file,
   int   offset,
   int   whence)
{
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call lseek() since "kfile" is just a regular mmap
	 */
	if (whence == KSEEK_SET)
	   resources->offset = offset;
	else if (whence == KSEEK_CUR)
	   resources->offset += offset;
	else if (whence == KSEEK_END)
	   resources->offset = resources->size + offset;
	else
	{
	   errno = EINVAL;
	   kinfo(KSYSLIB, "mmap_lseek:  Invalid whence argument value for \
file '%s'", file->path);
	   return(-1);
	}

	if (resources->offset > resources->size)
	/*EMPTY*/
	{
	   /* not sure as of yet */
	}
	return(resources->offset);
}

/*-------------------------------------------------------------
|  
|  Routine Name: mmap_access
|  
|       Purpose: This function is used to do see if a file is accessable.
|	         This is called when the user calls kaccess().  kaccess()
|	         calls the "mmap" internal drivers by the nature of the
|	         transport, which is dictated by the ktransport_identifier().
|  
|         Input:  path     - the initial path.
|		  filename - the filename that they wish to access
|		  mode     - the access mode information
|        Output:  returns the result
|   Called From:  internal routine called from kaccess()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:13
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int mmap_access(
   char *path,
   char *filename,
   int  mode)
{
	int	   status;


	/*
	 *  Simply call access() since "filename" is just a regular filename
	 */
	status = access(filename, mode);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: mmap_unlink
|  
|       Purpose: This function initializes a tempnam for a "mmap"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"sunmvm=XXXXXXXXXX"
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kunlink()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:13
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int mmap_unlink(
   char *identifier,
   char *filename)
{
	int	status;


	status = unlink(filename);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: mmap_lock
|  
|       Purpose: This function locks a "mmap" transport.  Given
|	         a request with a filename of the following syntax:
|  
|		  	"mmap=filename"
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kflock()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:13
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int mmap_lock(
   kfile *file,
   int   operation)
{
	int	status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	status = flock(resources->fid, operation);
	return(status);
}

#endif  /* KMMAP_DEF */
/* don't add after the endif */
