/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>             Signal Generation Routines
   >>>>
   >>>>  Private:
   >>>>
   >>>>   Static:
   >>>>
   >>>>   Public:
   >>>>             kgen_sine()
   >>>>             kgen_linear()
   >>>>             kgen_sinec()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: kgen_sine() - generates a sinusoid data set
*
*       Purpose: kgen_sine() generates a one dimensional sinusoidal data set.
*
*         Input: num      - number of elements in vector
*                sample   - sampling frequency
*                amp      - amplitude of data set
*                freq     - frequency of data set 
*                phase    - phase of data set 
*
*        Output: vect     - a vector containing the generated sinusoidal
*			    data set.
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jul 10, 1992 13:52
*      Verified:
*  Side Effects:
* Modifications: 
*
*************************************************************/

int kgen_sine(
   int    num,
   double sample,
   double amp,
   double freq,
   double phase,
   double *vect)
{
    int i;

    if(vect==NULL){
       errno = KINVALID_PARAMETER;
       kerror("kmath","kgen_sine", "The output vector must not be NULL.");
       return(FALSE);
    }

    if(sample==0.0){
       errno = KINVALID_PARAMETER;
       kerror("kmath","kgen_sine", "The sampling frequency must not be zero.");
       return(FALSE);
    }

    for(i=0;i<num;i++)
            *vect++ = amp*sin((double)i*freq/sample + phase);

    return(TRUE);
}


/************************************************************
*
*  Routine Name: kgen_linear() - generate a piecewise linear data
*                          set.
*
*       Purpose: Generates a one dimensional "piecewise linear" data 
*		 set.  A piecewise linear data set is a data set composed
*		 of connected end-to-end lines.
*                By properly setting arguments to this 
*                routine, you can generate impulse data 
*                (spikes), triangular waves, sawtooth waves,
*                reverse sawtooth waves, and square waves.
*
*         Input: num      -  number of elements in vector
*                sample   -  sampling frequency
*                minimum  -  minimum value of data set.
*                maximum  -  maximum value of the data set.
*                period   -  period of function 
*                rise     -  rise time of function 
*                fall     -  fall time of function 
*                width    -  width (when high) of pulse
*
*        Output: vect     -  a vector containing the generated set of
*			     piecewise linear numbers.
*       RETURNS: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jul 10, 1992 13:53
*      Verified:
*  Side Effects:
* Modifications: 
*
*************************************************************/

int kgen_linear(
   int    num,
   double sample,
   double minimum,
   double maximum,
   double period,
   double rise,
   double fall,
   double width,
   double *vect)
{
    int i;
    double amp,section,part;

    if(vect==NULL){
       errno = KINVALID_PARAMETER;
       kerror("kmath","kgen_linear", "The output vector must not be NULL.");
       return(FALSE);
    }

    if(sample==0.0){
       errno = KINVALID_PARAMETER;
       kerror("kmath","kgen_linear","The sampling frequency must not be zero.");
       return(FALSE);
    }

    amp = maximum - minimum;

    for(i=0;i<num;i++){
        section = (kfmod((double)i,(double)((period*sample))/sample));
        if(section<rise && rise!=0.0){
               part = section/rise;
        }else if(section<(rise + width) && width!=0.0){
               part = 1;
        }else if(section<(rise+fall+width) && fall!=0.0){
               part = 1.0 - (section-width-rise)/fall; 
        }else{
               part = 0.0;
        }

        *vect++ = amp*part+minimum;
    }

    return(TRUE);
}

/************************************************************
*
*  Routine Name: kgen_sinec() - generate a sinc data set.
*
*       Purpose: kgen_sinec() generates a one dimensional sinc data set.  The
*                sinc function is defined as sin(x)/x.
*
*         Input: num      - number of elements in vector
*                sample   - sampling frequency
*                amp      - amplitude of signal
*                freq     - frequency of signal 
*                center   - centering option: 
*                             0 - center the sync on the data set 
*                                 (i.e. the zero point of the 
*                                 data is at the midpoint of 
*                                 vect)
*                             1 - left justify the sync (the center 
*                                 of the data is at vect[0], 
*                                 and you only get the  right 
*                                 hand side of the sinc)
*
*        Output: vect     - a vector containing the generated sinc data set.
*                RETURNS: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jul 10, 1992 14:00
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kgen_sinec(
   int    num,
   double sample,
   double amp,
   double freq,
   int    center,
   double *vect)
{
    int i;
    double den;

    if(vect==NULL){
       errno = KINVALID_PARAMETER;
       kerror("kmath","kgen_sinec", "The output vector must not be NULL.");
       return(FALSE);
    }

    if(sample==0.0){
       errno = KINVALID_PARAMETER;
       kerror("kmath","kgen_sinec", "The sampling frequency must not be zero.");
       return(FALSE);
    }

    if(!kgen_sine(num,sample,amp,freq,0.0,vect)){
       errno = KINTERNAL;
       kerror("kmath","kgen_sinec", "An error occured during an internal call to kgen_sine.");
       return(FALSE);
    }

    for(i=0;i<num;i++){
         *vect=((den = ((double)i*freq/sample) + (num>>1)*center) != 0)
			? (*vect/den) : amp;
         vect++;
    }

    return(TRUE);
}
