/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Noise Generation Routines
   >>>>
   >>>>  Private:
   >>>>
   >>>>   Static:
   >>>>
   >>>>   Public:
   >>>>             kurng()
   >>>>             kgen_gauss()
   >>>>             kgen_expon()
   >>>>             kgen_unif()
   >>>>             kgen_rayleigh()
   >>>>             kgen_poisson()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: kurng() - generate a uniform random
*                          number in the range [0:1].
*
*       Purpose: The function kurng() generates a double precision 
*		 uniform random number between 0 and 1.
*
*         Input: 
*
*        Output: 
*                RETURNS: A double precision random number between 0 and 1.
*
*  Restrictions: 
*    Written By: Ramiro Jordan and Marcelo Teran
*          Date: Jul 10, 1992 14:50
*      Verified:
*  Side Effects:
* Modifications: Converted from urng in Khoros 1.0 (JW)
*
*************************************************************/

double kurng(void)
{
  static int iseed = 0;

/*
 * use time as seed 
 */

  if (iseed == 0) {
      ksrandom( (int) time(NULL)); 
      iseed = 1;
  } 

/*
 * generate random number and scale it down to range [0,1]
 */

  return((double)((double)krandom()/(double)KMAXLINT));
}

/******************************************************************
*
*  Routine Name: kgen_gauss() - generate a vector of gaussian
*                            random numbers.
*        
*       Purpose: kgen_gauss() generates a one dimensional gaussian random 
*		 noise data set.
*
*         Input: num      - number of elements in vector
*                mean     - mean of the data set
*                variance - variance of the data set 
*
*        Output: vect     - a vector containing the generated set of
*			    gaussian random numbers.
*                
*       RETURNS:  TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley  
*          Date: Jul 10, 1992 14:34
*      Verified:
*  Side Effects: 
* Modifications: Converted from ggauss in Khoros 1.0 (JW)
*                13-Aug-94 Scott Wilson - fixed bogus u1 generation,
*                          corrected code to use ln instead of log10,
*                          which got rid of Ramiros booger factor.
*
***********************************************************************/

int kgen_gauss(
   int    num,
   double mean,
   double variance,
   double *vect)
{
    int i;
    double temp,u1,u2;
    
   /* 
    * generate the noise sequence 
    */

    /* There is a game here with u1. The idea is that we need to make
       sure that we never get u1 exactly equal to zero or the log10
       will barf, and then the sqrt will barf. So, the while loop on
       u1==0 is there to make sure this never happens, and the reset
       of u1 to 0 at the bottom of the loop makes sure we get a new
       u1 next time around. */

    u1=0.0;
    for(i = 0; i < num/2; i++){
        while (u1 == 0.0) u1 = kurng(); 
        u2 = kurng();
        temp = sqrt(-2.0*variance*klog(u1));
        vect[2*i] = temp * cos(K2PI*u2) + mean;
        vect[2*i+1] = temp * sin(K2PI*u2) + mean;
        u1=0.0;
    }
 
    u1=0.0;
    if(num & 1){
       while (u1 == 0.0) u1 = kurng();
       u2 = kurng();
       temp = sqrt(-2.0*variance*klog(u1));
       vect[num-1] = temp * cos(K2PI*u2) + mean;
       u1=0.0;
    }

    return(TRUE);
}

/******************************************************************
*
*  Routine Name: kgen_expon() - generate a vector of exponential
*                            random numbers.
*
*       Purpose: kgen_expon() generates a 1d exponential random noise 
*                data set 
*
*         Input: num      - number of elements in data set.
*                variance - variance of the data set 
*
*        Output: vect     - a vector containing the generated set of
*			    exponential random numbers.
*                
*       RETURNS: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley  
*          Date: Jul 10, 1992 14:36
*      Verified:
*  Side Effects: 
* Modifications: Converted from gexpon in Khoros 1.0 (JW)
*
******************************************************************/

int kgen_expon(
   int    num,
   double variance,
   double *vect)
{
    int i;
    
    for(i = 0; i < num; i++)
       vect[i] = -1.0*sqrt(variance)*KLN10*log10(kurng());
 
    return(TRUE);
}

/*******************************************************************
*
*  Routine Name: kgen_unif() - generate a vector of uniform random 
*                           numbers.
*
*       Purpose: kgen_unif() generates a one dimensional uniform random 
*		 noise data set.
*
*         Input: num      - number of points
*                minimum  - minimum of the data set
*                maximum  - maximum of the data set 
*
*        Output: vect     - a vector containing the generated set of 
*		            uniform random numbers.
*       RETURNS: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley  
*          Date: Jul 10, 1992 14:37
*      Verified:
*  Side Effects: 
* Modifications: Converted from gunif in Khoros 1.0 (JW)
*
*******************************************************************/

int kgen_unif(
   int    num,
   double minimum,
   double maximum,
   double *vect)
{
    int i;
    double amp;

    amp = maximum - minimum;

    for(i = 0; i < num; i++)
       vect[i] = amp * kurng() + minimum;

    return(TRUE);
}

/*******************************************************************
*
*  Routine Name: kgen_rayleigh() - generate a vector of Rayleigh random
*                             numbers.
*
*       Purpose: kgen_rayleigh() generates a one dimensional Rayleigh random 
*		 noise data set.
*
*         Input: num      - number of elements in input vector
*                variance - variance of the data set 
*
*        Output: vect     - a vector containing the generated set of Rayleigh
*			    random numbers.
*                
*       RETURNS: TRUE (1) on success, otherwise it returns FALSE (0)
*		 if it is unable to allocate sufficient memory or the
*		 internal call to kgen_gauss fails.
*
*  Restrictions: 
*    Written By: Jeremy Worley  
*          Date: Jul 10, 1992 14:38
*      Verified:
*  Side Effects: 
* Modifications: Converted from grleigh in Khoros 1.0 (JW)
*
*******************************************************************/

int kgen_rayleigh(
   int    num,
   double variance,
   double *vect)
{
    int junk1,junk2,i;
    double *t1,*t2;

/*
 * allocate some space for gaussian temporary random variables
 */

    if((t1 = (double *)kmalloc(num*sizeof(double)))==NULL){
       kerror("kmath","kgen_rayleigh",
           "Unable to allocate sufficient memory for temporary space.");
       return(FALSE);
    }

    
    if((t2 = (double *)kmalloc(num*sizeof(double)))==NULL){
       kerror("kmath","kgen_rayleigh",
           "Unable to allocate sufficient memory for temporary space.");
       kfree(t1);
       return(FALSE);
    }

/*
 * generate the noise sequence
 */

    junk1 = kgen_gauss(num,(double)0,(double)(variance/(2.0 - K2PI/4)),t1);
    junk2 = kgen_gauss(num,(double)0,(double)(variance/(2.0 - K2PI/4)),t2);

    if(junk1 == FALSE || junk2 == FALSE){
       kfree(t1);
       kfree(t2);
       errno = KINTERNAL;
       kerror("kmath","kgen_rayleigh",
              "An internal call to kgen_gauss() failed.");
       return(FALSE);
    }

/*
 * rayleigh is the square root of the sum of the squares of two gaussian
 * random variables
 */

    for(i = 0; i < num; i++)
        *vect++ = sqrt((double)(t1[i]*t1[i] + t2[i]*t2[i]));

/*
 * free up temporary space
 */

    kfree(t1);
    kfree(t2);
 
    return(TRUE);
}

/***********************************************************************
*
*  Routine Name: kgen_poisson() - generate a vector of Poisson random
*                            numbers.
*
*       Purpose: kgen_poisson() generates a one dimensional Poisson random 
*		 noise data set.
*
*         Input: num      - number of elements in data set.
*                variance - variance of the data set 
*                atime    - amount of time
*
*        Output: vect     - a vector containing the generated set of Poisson
*			    random numbers.
*                
*       RETURNS: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley  
*          Date: Jul 10, 1992 14:44
*      Verified:
*  Side Effects: 
* Modifications: Converted from gpoiss in Khoros 1.0 (JW)
*
***********************************************************************/

int kgen_poisson(
   int    num,
   double variance,
   double atime,
   double *vect)
{
    int i;
    double t,tsum,p;

    for(i=0;i<num;i++){
        tsum = p = 0;
        do{
           t  = (-1.0*atime/variance)*KLN10*log10(1.0 - kurng());
           tsum += t;
           if(tsum <= atime) p += 1.0;
        }while(tsum <= atime);
       *vect++ = p;
    }

   return(TRUE);
}
