/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>             Complex Arithmetic Routines
   >>>>
   >>>>  Private:
   >>>>             
   >>>>   Public:
   >>>>             kdcadd()
   >>>>             kdcsub()
   >>>>             kdcmult()
   >>>>             kdcang()
   >>>>             kdcdiv()
   >>>>             kdcmagsq()
   >>>>             kdcmag()
   >>>>             kdclogmagp1()
   >>>>		    kdclogmagsq()
   >>>>             kdcconj()
   >>>>
   >>>>             kcadd()
   >>>>             kcsub()
   >>>>             kcmult()
   >>>>             kcdiv()
   >>>>             kcmagsq()
   >>>>             kcmag()
   >>>>             kclogmagp1()
   >>>>             kcconj()
   >>>>
   >>>>             kdcomplex_to_arrays()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: kdcadd() - add two double precision complex 
*                           numbers.
*
*       Purpose: kdcadd() adds two double precision complex 
*                numbers and returns the result.  The two 
*                inputs are of type kdcomplex. The typedef 
*                for kdcomplex can be found in $BOOTSTRAP/machine/cdefs.h.
*
*         Input: a - first operand of the dcomplex addition
*                b - second operand of the dcomplex addition
*
*        Output: 
*                
*       Returns: The result of the kdcomplex addition.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems.
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kdcomplex kdcadd(
   kdcomplex a,
   kdcomplex b)
{
   kdcomplex c;

   c.r = a.r + b.r;
   c.i = a.i + b.i;
   return(c);
}
   
/************************************************************
*
*  Routine Name: kdcsub() - subtract one double precision 
*                           complex number from another.
*
*       Purpose: kdcsub() subtracts the second double 
*                precision complex number from the first and 
*                returns the result.  The two inputs are of 
*                type kdcomplex. The typedef for kdcomplex can 
*                be found in $BOOTSTRAP/machine/cdefs.h.
*
*         Input: a - first operand of the kdcomplex 
*                    subtraction
*                b - second operand of the kdcomplex 
*                    subtraction
*
*        Output: 
*                
*       Returns: The result of the kdcomplex subtraction.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems.
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kdcomplex kdcsub(
   kdcomplex a,
   kdcomplex b)
{
   kdcomplex c;

   c.r = a.r - b.r;
   c.i = a.i - b.i;
   return(c);
}

/************************************************************
*
*  Routine Name: kdcmult() - multiply two double precision 
*                complex numbers.
*
*       Purpose: kdcsub() multiplies two double precision 
*                complex numbers and returns the result.  The 
*                two inputs are of type kdcomplex. The typedef 
*                for kdcomplex can be found in $BOOTSTRAP/machine/cdefs.h.
*
*         Input: a - first operand of the kdcomplex 
*                    subtraction
*                b - second operand of the dcomplex 
*                    subtraction
*
*        Output: 
*                
*       Returns: The result of the kdcomplex multiplication.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems. 
*
*    Written By: Jeremy Worley
*  
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kdcomplex kdcmult(
   kdcomplex a,
   kdcomplex b)
{
   kdcomplex c;

   c.r = a.r*b.r - a.i*b.i;
   c.i = a.r*b.i + a.i*b.r;
   return(c);
}

/************************************************************
*
*  Routine Name: kdcang() - compute the radian angle of a complex
*                          number.
*
*       Purpose: kdcang() returns the radian angle of the input
*                complex number.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The angle in radians of the complex input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

double kdcang(
   kdcomplex a)
{
   if(a.i==0.0 && a.r==0.0)
      return(0.0);
   else
      return((double)atan2((double)a.i,(double)a.r));
}

/************************************************************
*
*  Routine Name: kdcdiv() - divide one double precision complex 
*                           number by another.
*
*       Purpose: kdcdiv() divides the second double precision 
*                complex number into the first and returns 
*                the result.  The two inputs are of type 
*                kdcomplex. The typedef for kdcomplex can be 
*                found in $BOOTSTRAP/machine/cdefs.h.
*
*         Input: a - first operand of the kdcomplex 
*                    subtraction
*                b - second operand of the kdcomplex 
*                    subtraction
*
*        Output: 
*                
*       Returns: The result of the kdcomplex division.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems. 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects: This routine will issue warning messages via kinfo if
*                a division by zero occurs or if b has a
*                magnitude of 0 (the call to atan2() with
*                b = 0 will cause problems).  The result
*                will not be accurate.
*
* Modifications:
*
*************************************************************/

kdcomplex kdcdiv(
   kdcomplex a,
   kdcomplex b)
{
   kdcomplex c;
   double asmag,bsmag,asang,bsang;

   asmag = ksqrt(a.r*a.r+a.i*a.i);
   bsmag = ksqrt(b.r*b.r+b.i*b.i);
   if(bsmag == 0.0) {
      kinfo(KSTANDARD, 
	    "A division by zero was attempted.  Result is not accurate.");
      bsmag = KMAXFLOAT;
   }

   asang = kdcang(a);
   bsang = kdcang(b);

   c.r = (asmag/bsmag)*cos(asang-bsang);
   c.i = (asmag/bsmag)*sin(asang-bsang);

   return(c);
}  

/************************************************************
*
*  Routine Name: kdcmagsq() - calculate the squared magnitude
*                             of a double precision complex number.
*
*       Purpose: kdcmagsq() returns the squarred magnitude of
*                the input argument.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: The magnitude squared of the input argument.
*
*  Restrictions:
* 
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992 11:05
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

double kdcmagsq(
   kdcomplex a)
{
  return(a.r*a.r + a.i*a.i);
}

/************************************************************
*
*  Routine Name: kdcmag() - compute the magnitude of a double 
*                           precision complex number.
*
*       Purpose: kdcmag() returns the magnitude of the input
*                double precision complex number.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: The magnitude of the kdcomplex input 
*                argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

double kdcmag(
   kdcomplex a)
{
  return(ksqrt(kdcmagsq(a)));
}

/************************************************************
*
*  Routine Name: kdclogmagp1() - compute the log magnitude of a double 
*                                precision complex number plus one.
*
*       Purpose: kdclogmag() returns the log magnitude of the input
*                double precision complex number.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: The magnitude of the kdcomplex input 
*                argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

double kdclogmagp1(
   kdcomplex a)
{
  return(klog10(kdcmag(a)+1.0));
}

/************************************************************
*
*  Routine Name: kdclogmag() - compute the log magnitude of a double 
*                                precision complex number.
*
*       Purpose: kdclogmag() returns the log magnitude of the input
*                double precision complex number.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: The magnitude of the kdcomplex input 
*                argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

double kdclogmag(
   kdcomplex a)
{
  return(klog10(kdcmag(a)));
}

/************************************************************
*
*  Routine Name: kdclogmagsq() - compute the log magnitude squared of a double 
*                                precision complex number.
*
*       Purpose: kdclogmag() returns the log magnitude of the input
*                double precision complex number.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: The magnitude of the kdcomplex input 
*                argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

double kdclogmagsq(
   kdcomplex a)
{
  return(klog10(kdcmagsq(a)));
}

/************************************************************
*
*  Routine Name: kdclogmagsqp1() - compute the log magnitude squared of a double 
*                                precision complex number plus one.
*
*       Purpose: kdclogmag() returns the log magnitude of the input
*                double precision complex number.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: The magnitude of the kdcomplex input 
*                argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

double kdclogmagsqp1(
   kdcomplex a)
{
  return(klog10(kdcmagsq(a)+1.0));
}

/************************************************************
*
*  Routine Name: kdcconj() - compute the conjugate of a double 
*                            precision complex number.
*
*       Purpose: kdcconj() conjugates the double precision 
*                complex input argument.
*
*         Input: a - kdcomplex number to be conjugated.
*
*        Output: 
*                
*       Returns: The kdcomplex conjugate of the input
*                argument.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems. 
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992  10:18:03
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kdcomplex kdcconj(
   kdcomplex a)
{
   kdcomplex c;

   c.r = a.r;
   c.i = -a.i;
   return(c);
}

/************************************************************
*
*  Routine Name: kcadd() - add two complex numbers.
*
*       Purpose: kcadd() adds two complex numbers and returns
*                the result.  The two inputs are of type 
*                kcomplex. The typedef for kcomplex can be 
*                found in $BOOTSTRAP/machine/cdefs.h.
*
*         Input: a - first operand of the complex addition
*                b - second operand of the complex addition
*
*        Output: 
*                
*       Returns: The result of the complex addition.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems. 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kcomplex kcadd(
   kcomplex a,
   kcomplex b)
{
   kcomplex c;

   c.r = (float)((double)a.r + (double)b.r);
   c.i = (float)((double)a.i + (double)b.i);
   return(c);
}
   
/************************************************************
*
*  Routine Name: kcsub() - subtract one kcomplex number from
*                          another.
*
*       Purpose: kcsub() subtracts the second complex number 
*                from the first and returns the result.  The 
*                two inputs are of type kcomplex. The typedef
*                for kcomplex can be found in $BOOTSTRAP/machine/cdefs.h.
*
*         Input: a - first operand of the complex subtraction
*                b - second operand of the complex subtraction
*
*        Output: 
*
*       Returns: The result of the complex subtraction.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems. 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kcomplex kcsub(
   kcomplex a,
   kcomplex b)
{
   kcomplex c;

   c.r = (float) ((double)a.r - (double)b.r);
   c.i = (float) ((double)a.i - (double)b.i);
   return(c);
}

/************************************************************
*
*  Routine Name: kcmult() - multiply two complex numbers.
*
*       Purpose: kcsub() multiplies two complex numbers and
*                returns the result.  The two inputs are of 
*                type kcomplex. The typedef for kcomplex can 
*                be found in $BOOTSTRAP/machine/cdefs.h.
*
*         Input: a - first operand of the complex subtraction
*                b - second operand of the complex subtraction
*
*        Output: 
*                
*       Returns: The result of the complex multiplication.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems. 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kcomplex kcmult(
   kcomplex a,
   kcomplex b)
{
   kcomplex c;

   c.r = (float) ((double)a.r*(double)b.r - (double)a.i*(double)b.i);
   c.i = (float) ((double)a.r*(double)b.i + (double)a.i*(double)b.r);
   return(c);
}

/************************************************************
*
*  Routine Name: kcang() - compute the radian angle of a complex
*                          number.
*
*       Purpose: kcang() returns the radian angle of the input
*                complex number.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The angle in radians of the complex input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

float kcang(
   kcomplex a)
{
   if(a.i==0.0 && a.r==0.0)
      return(0.0);
   else
      return((float)atan2((double)a.i,(double)a.r));
}

/************************************************************
*
*  Routine Name: kcdiv() - divide one complex number by
*                          another.
*
*       Purpose: kcdiv() divides the second complex number 
*                into the first and returns the result.  The 
*                two inputs are of type kcomplex. The typedef
*                for kcomplex can be found in $BOOTSTRAP/machine/cdefs.h.
*
*         Input: a - first operand of the complex subtraction
*                b - second operand of the complex subtraction
*
*        Output: 
*                
*       Returns: The result of the complex division.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems. 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects: This routine will issue warning messages if
*                a division by zero occurs or if b has a
*                magnitude of 0 (the call to atan2() with
*                b = 0 will cause problems).  The result
*                will not be accurate.
*
* Modifications:
*
*************************************************************/

kcomplex kcdiv(
   kcomplex a,
   kcomplex b)
{
   kcomplex c;
   float asmag,bsmag,asang,bsang;

   asmag = (float)ksqrt((double)a.r*(double)a.r+(double)a.i*(double)a.i);
   bsmag = (float)ksqrt((double)b.r*(double)b.r+(double)b.i*(double)b.i);
   if(bsmag == 0.0){
      kinfo(KSTANDARD, 
		 "A division by zero was attempted.  Result is approximate.");
      bsmag = KMAXFLOAT;
   }

   asang = kcang(a);
   bsang = kcang(b);

   c.r = (float) ((double)asmag/(double)bsmag)*
                 (double)cos((double)((double)asang-(double)bsang));
   c.i = (float) ((double)asmag/(double)bsmag)*
                 (double)sin((double)((double)asang-(double)bsang));

   return(c);
}  

/************************************************************
*
*  Routine Name: kcmagsq() - calculate the squared magnitude
*                            of a kcomplex number.
*
*       Purpose: kcmagsq() returns the squarred magnitude of
*                the input argument.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The magnitude squared of the input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992 11:05
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

float kcmagsq(
    kcomplex a)
{
  return((float)((double)a.r*(double)a.r + (double)a.i*(double)a.i));
}

/************************************************************
*
*  Routine Name: kcmag() - compute the magnitude of a complex
*                          number.
*
*       Purpose: kcmag() returns the magnitude of the input
*                complex number.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The magnitude of the complex input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

float kcmag(
   kcomplex a)
{
  return((float)ksqrt((double)kcmagsq(a)));
}

/************************************************************
*
*  Routine Name: kclogmagp1() - compute the log magnitude of a complex
*                               number plus one.
*
*       Purpose: kclogmag() returns the log magnitude of the input
*                complex number plus one.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The magnitude of the complex input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

float kclogmagp1(
   kcomplex a)
{
  return((float)klog10((double)kcmag(a)+1.0));
}

/************************************************************
*
*  Routine Name: kclogmag() - compute the log magnitude of a complex
*                               number.
*
*       Purpose: kclogmag() returns the log magnitude of the input
*                complex number.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The magnitude of the complex input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

float kclogmag(
   kcomplex a)
{
  return((float)klog10((double)kcmag(a)));
}

/************************************************************
*
*  Routine Name: kclogmagsq() - compute the log magnitude squared of
*  a complex number.
*
*       Purpose: kclogmag() returns the log magnitude of the input
*                complex number.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The magnitude of the complex input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

float kclogmagsq(
   kcomplex a)
{
  return((float)klog10((double)kcmagsq(a)));
}

/************************************************************
*
*  Routine Name: kclogmagsqp1() - compute the log magnitude squared of
*  a complex number plus one.
*
*       Purpose: kclogmag() returns the log magnitude of the input
*                complex number plus one.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The magnitude of the complex input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jun 11, 1992 16:39
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

float kclogmagsqp1(
   kcomplex a)
{
  return((float)klog10((double)kcmagsq(a)+1.0));
}

/************************************************************
*
*  Routine Name: kcconj() - compute the conjugate of a complex
*                           number.
*
*       Purpose: kcconj() conjugates the complex input 
*                argument.
*
*         Input: a - complex number to be conjugated.
*
*        Output: 
*                
*       Returns: The complex conjugate of the input
*                argument.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems. 
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992  10:18:03
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kcomplex kcconj(
   kcomplex a)
{
   kcomplex c;

   c.r = a.r;
   c.i = -a.i;
   return(c);
}

/************************************************************
*
*  Routine Name: kdcomplex_to_arrays - separate array of double complex into real and imaginary arrays
*
*       Purpose: kdcomplex_to_arrays returns the real and imaginary portion
*                of a kdcomplex array in two double arrays.
*
*         Input: c - a dcomplex array to be split into real & imag parts 
*
*        Output: r - an array of real numbers (double *) derived from the real 
*                    component of the input array of double complex pairs
*                i - an array of real numbers (double *) derived from the
*                    imaginary component of the input array of double 
*                    complex pairs.
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Ashish Malhotra & Jeremy Worley
*          Date: Nov 02, 1993 14:03
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kdcomplex_to_arrays(kdcomplex *c, int num, double **r, double **i)
{
   int j;

   if (c == NULL) 
      return(FALSE);

   if (r != NULL && *r == NULL) {
      if ((*r = (double *)kmalloc(num * sizeof (double))) == NULL)
         return(FALSE);
   }
   if (i != NULL && *i == NULL) {
      if ((*i = (double *)kmalloc(num * sizeof (double))) == NULL)
         return(FALSE);
   }

   for (j = 0; j < num; j++) {
       if (r) (*r)[j] = kdcreal(c[j]);
       if (i) (*i)[j] = kdcimag(c[j]);
   }

   return(TRUE);
}
