/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>        External Selection Creation Utilities          <<<<
   >>>>      (for use by guise and the xvlang library)        <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>             kvf_create_new_selection                  <<<<
   >>>>             kvf_create_new_toggle_member              <<<<
   >>>>             kvf_link_extension                        <<<<
   >>>>             kvf_get_extensions                        <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
static int set_extension_array PROTO((kselection *, kform_struct ***, int *));


/*-----------------------------------------------------------
|
|  Routine Name: kvf_create_new_selection
|
|       Purpose: creates a new selection node in the form tree
|                based on the UIS line passed in.
|
|         Input: control   - master, guidepane, or pane in which
|                            to create the selection.
|                line      - UIS line describing selection
|        Output:
|       Returns:
|    Written By: Danielle Argiro
|          Date: July 05, 1994
| Modifications:
|
------------------------------------------------------------*/
kselection *kvf_create_new_selection(
    kcontrol *control,
    char     *line)
{
        kselection *selection;
	int        typeflag;
	Line_Info  line_info;

	if (control == NULL)
	{
	    kerror("kforms", "kvf_create_new_selection",
		   "control passed in is NULL");
	    return(NULL);
	}

        /*
         * allocate selection structure
         */
	selection = (kselection *) kcalloc(1,sizeof(kselection));
        if (selection == NULL)
        {
            kerror(NULL, "create_int_sel",
                   "Cannot allocate room for kselection struct");
            return(NULL);
        }

        /*
         * make sure UIS line is ok
         */
	typeflag = kvf_get_line_type(line);
	if (typeflag == -1)
	{
	    kerror("kforms", "kvf_create_new_selection",
		   "UIS passed in is invalid");
	    return(NULL);
	}
	
	kvf_clear_line_info(&line_info);
	kvf_gen_parse(line, &line_info);

        if (!(kvf_set_selection_values(selection, line, typeflag,
                                       control->back_form,
                                       control->back_subform,
                                       control->back_guide,
                                       control, NULL, line_info.variable, 
				       NULL)))
            return(NULL);

	kvf_free_line_info_strings(&line_info);

        /*
         * link new selection into selection list
         */
        kvf_link_selection(control, selection);

	return(selection);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_create_new_toggle_member
|
|       Purpose: creates a new toggle member in a toggle selection
|                based on the UIS line passed in.
|
|         Input: toggle -  toggle selection to which to add new member
|                line   - UIS line describing toggle member
|        Output:
|       Returns:
|    Written By: Danielle Argiro
|          Date: July 05, 1994
| Modifications:
|
------------------------------------------------------------*/
kselection *kvf_create_new_toggle_member(
    kselection *toggle_start,
    char       *line)
{
	kselection *toggle_member, *toggleptr;
	int        toggle_type, typeflag, member_num;
	int        toggle_count = 1;
	Line_Info  line_info;

	/*
	 *  if there are already members in the toggle, make sure we are
         *  not attempting to add a toggle member with a clashing data type
	 */
	typeflag = kvf_get_line_type(line);
	kvf_get_attribute(toggle_start->back_kformstruct, 
			  KVF_TOGGLE_SIZE, &member_num);
	if (member_num > 1)
	{
	    kvf_get_attribute(toggle_start->back_kformstruct, 
			      KVF_TOGGLE_TYPE, &toggle_type);
	    if (toggle_type != typeflag)
	    {
	        kerror("kforms", "kvf_create_new_toggle_member",
		       "Attempt to add toggle member of type %s to toggle of type %s; inconsistency not allowed", kvf_ascii_typeflag(typeflag), kvf_ascii_typeflag(toggle_type));
	        return(NULL);
	    }
	}

	/*
	 *  allocate the kselection for the toggle member
	 */
        toggle_member = (kselection *) kcalloc(1,sizeof(kselection));
        if (toggle_member == NULL)
        {
            kerror(NULL, "kvf_create_new_selection",
                   "Cannot allocate room for kselection struct");
            return(NULL);
        }

	/*
	 * to to parse line to get variable to send to kvf_set_selection_values
	 */
	kvf_clear_line_info(&line_info);
	kvf_gen_parse(line, &line_info);


        /*
         * set the selection values
         */
        if (!(kvf_set_selection_values(toggle_member, line, line_info.typeflag,
                                       toggle_start->back_form,
                                       toggle_start->back_subform,
                                       toggle_start->back_guide,
                                       toggle_start->back_control,
                                       NULL, line_info.variable, NULL)))
        {
            kvf_free_line_info_strings(&line_info);
            return(NULL);
        }
	kvf_free_line_info_strings(&line_info);

	if (line_info.opt_sel) toggle_member->opt_selected = TRUE;

        /*
         *  link new toggle member to the end of the toggle list
         */
        if (toggle_start->toggle_next == NULL)
            toggle_start->toggle_next = toggle_member;
        else
        {
            toggleptr = toggle_start->toggle_next;
	    toggle_count++;
            while (toggleptr->next != NULL)
	    {
                toggleptr = toggleptr->next;
		toggle_count++;
	    }
            toggleptr->next = toggle_member;
        }

	/*
	 *  set toggle_num and backlink
	 */
	toggle_member->toggle_num =  toggle_count;
	toggle_member->back_toggle = toggle_start;
	toggle_member->next = NULL;

	return(toggle_member);

}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_link_extension
|
|       Purpose: For use with cantata's workspace GUI this routine links
|                two selections together using the extensions array.  After
|                this is done, changing a "value" attribute on one of the
|                selections will cause them both to be changed.  For example,
|                setting XVF_FILE_NAME on one will update the filename on both.
|
|         Input: kformstruct1 - kformstruct corresponding to 1st selection
|                kformstruct2 - kformstruct corresponding to 2nd selection
|        Output: TRUE on success, FALSE on failure
|       Returns:
|    Written By: Danielle Argiro
|          Date: July 4, 1994
| Modifications:
|
------------------------------------------------------------*/
int kvf_link_extension(
    kform_struct *kformstruct1,
    kform_struct *kformstruct2)
{
	int indx;

        if (kformstruct1->type != KSELECTION)
        {
            kerror("kforms", "kvf_link_extension",
                   "First kformstruct passed in not a selection");
            return(FALSE);
        }

        if (kformstruct2->type != KSELECTION)
        {
            kerror("kforms", "kvf_link_extension",
                   "Second kformstruct passed in not a selection");
            return(FALSE);
        }

	/* 
	 *  see if the second selection is already in the extensions array
	 *  of the first selection;  if not, add it in
	 */
	indx = karray_locate((kaddr)kformstruct1->Selptr->extensions, 
			     kformstruct2->Selptr, 
			     kformstruct1->Selptr->extension_num);
	if (indx == -1)
	{
	    kformstruct1->Selptr->extensions = (kaddr) karray_insert(
		          (kaddr)kformstruct1->Selptr->extensions, 
                          kformstruct2->Selptr, 
                          kformstruct1->Selptr->extension_num,
			  KLIST_TAIL, FALSE);
	    kformstruct1->Selptr->extension_num++;
	}

        /* 
         *  see if the first selection is already in the extensions array
         *  of the second selection;  if not, add it in
         */
        indx = karray_locate((kaddr)kformstruct2->Selptr->extensions, 
                             kformstruct1->Selptr, 
                             kformstruct2->Selptr->extension_num);
        if (indx == -1)
        {
	    kformstruct2->Selptr->extensions = (kaddr) karray_insert(
            		  (kaddr)kformstruct2->Selptr->extensions, 
                          kformstruct1->Selptr, 
                          kformstruct2->Selptr->extension_num,
                          KLIST_TAIL, FALSE);
            kformstruct2->Selptr->extension_num++;
        }
        return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_get_extensions
|
|       Purpose: For use with deleting selections from cantata's front
|                panel when a glyph is destroyed, this routine takes
|                a form tree (the one associated with the glyph) and
|                returns an array of the kformstructs associated with
|                each selection that has an extension (ie, a corresponding
|                selection on the front panel).
|
|         Input: form - form tree defining GUI of glyph
|        Output: array of kformstructs on success, NULL on failure
|       Returns: num - number of kformstructs returned in array
|    Written By: Danielle Argiro
|          Date: July 7, 1994
| Modifications:
|
------------------------------------------------------------*/
kform_struct **kvf_get_extensions(
    kform *form,
    int   *num)
{
	kform_struct **kformstruct_array = NULL;
	int         size = 0;

	*num = 0;

	/* sanity check */
	if (form == NULL)
	    return(NULL);

	/* set_extension_array() assumes a single pane on a single subform */
	if (!(kvf_check_for_single_pane(form, "kvf_get_extensions")))
	    return(NULL);

	/* create array of extensions */ 
	if (!(set_extension_array(form->subform->guide->pane->sel_list,
			          &kformstruct_array, &size)))
	    return(NULL);

	*num = size;
	return(kformstruct_array);
}

static int set_extension_array(
	kselection   *sel_list,
	kform_struct ***kformstruct_array,
	int          *size)
{
	int        i;
	kselection *selection;

	selection = sel_list;
	while (selection != NULL)
        {
            if (selection->extensions != NULL)
            {
	        for (i = 0; i < selection->extension_num; i++)
		{
		    *kformstruct_array = (kform_struct **)
			karray_add((char **) *kformstruct_array,
				selection->extensions[i]->back_kformstruct,
				(*size)++);
	        }
	   }

	   if (selection->group_next != NULL)
		set_extension_array(selection->group_next, 
				    kformstruct_array, size);

	   selection = selection->next;
	}
	return(TRUE);
}
