/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>	     form-to-command and command-to-form conversion   <<<<
   >>>>                                                       <<<<
   >>>>  Private:					      <<<<
   >>>>                kvf_form_to_cmd()	              <<<<
   >>>>                kvf_modify_form_from_cmd()	      <<<<
   >>>>   Static:					      <<<<
   >>>>   Public:					      <<<<
   >>>>		                                              <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*
 *  helper routines for kvf_form_to_cmd()
 */
static int kvf_constr_cmd_line    PROTO((kselection *, char *, char *,
				         char *, char *, char *, int, int));
static int kvf_constr_file_cmd    PROTO((kselection *, char *, int));
static int kvf_constr_str_cmd     PROTO((kselection *, char *, int));
static int kvf_constr_std_cmd     PROTO((kselection *, char *, int));
static int kvf_constr_int_cmd     PROTO((kselection *, char *, int));
static int kvf_constr_flt_cmd     PROTO((kselection *, char *, int));
static int kvf_constr_dbl_cmd     PROTO((kselection *, char *, int));
static int kvf_constr_misc_cmd    PROTO((kselection *, char *, int));
static int kvf_constr_toggle_cmd  PROTO((kselection *, char *, int));
static int kvf_constr_routine_cmd PROTO((kselection *, char *, char *, int));


/*
 *  helper routines for kvf_modify_form_from_cmd()
 */
static int modify_selections_from_cmd PROTO((kselection *, char *, int));
static int modify_file_from_cmd       PROTO((kselection *, char *, int));
static int modify_int_from_cmd        PROTO((kselection *, char *, int));
static int modify_float_from_cmd      PROTO((kselection *, char *, int));
static int modify_double_from_cmd     PROTO((kselection *, char *, int));
static int modify_logical_from_cmd    PROTO((kselection *, char *, int));
static int modify_list_from_cmd       PROTO((kselection *, char *, int));
static int modify_flag_from_cmd       PROTO((kselection *, char *, int));
static int modify_string_from_cmd     PROTO((kselection *, char *, int));
static int modify_toggle_from_cmd     PROTO((kselection *, char *, int));

/************************************************************
*
*  Routine Name: kvf_form_to_cmd - translate form tree to command line string
*
*       Purpose: Gathers information from the form tree representing
*                the command line arguments specified in a *.pane file
*                and creates the corresponding command line representation 
*                according to the current selection values and the program 
*                named by the routine button on the form.  NOTE: this
*                routine is primarily used in the kcodegen library, and by
*                cantata.
*
*         Input: form    - pointer to the form tree
*                subform - pointer to a subform tree.
*                machine - pointer to name of machine on which to exec command
*		 expand  - TRUE if filenames are to be expanded, and
*			   expressions are to be evaluated; FALSE otherwise
*        Output:
*       Returns: Allocates & returns a string representing the command 
*                represented by the current configuration of the form tree.
*  Restrictions:
*    Written By: Danielle Argiro
*          Date:  March 24, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char *kvf_form_to_cmd(
   kform    *form,
   ksubform *subform,
   char     *machine,
   int      expand)
{
   	kguide  *guide;
	char    *cmd_line;
	char    command[5*KLENGTH];      /* the complete command line string */
	char    cmd_args[KLENGTH];       /* cat'ed cmd string containing all
                                            args represented by selections */
        char    stdin_segment[KLENGTH];  /* "kcp -i infile -o - |" must
                                             precede program name & args */
        char    stdout_segment[KLENGTH]; /* "kcp -i - -o outfile" must follow
                                             program name & args */
        char    program_call[KLENGTH];   /*  program name as specified on the
                                            -R UIS line */
	command[0]        = '\0';
        stdin_segment[0]  = '\0';
        program_call[0]   = '\0';
        cmd_args[0]       = '\0';
        stdout_segment[0] = '\0';

	/*
	 * sanity check
	 */
	if (form == NULL)
	{
	    kerror("kforms", "kvf_form_to_cmd", 
		   "form passed in is NULL");
	    return(NULL);
	}
	if (subform == NULL)
        {
            kerror("kforms", "kvf_form_to_cmd", 
                   "subform passed in is NULL");
            return(NULL);
        }

	guide = kvf_search_sel_guide(subform);

        kvf_constr_cmd_line(guide->pane->sel_list, machine, stdin_segment,
	                    program_call, cmd_args, stdout_segment, expand,
			    FALSE);

	/* first in the command string is the stdin segment (if any) */
        if (kstrlen(stdin_segment) > 0)
            kstrcat(command, stdin_segment);

        /* then is the program call itself */
	if (kstrlen(program_call) == 0)
	    kstrcat(command, " ");
	else kstrcat(command, program_call);

        /* followed by all the program arguments & their values */
        kstrcat(command, cmd_args);

        /* last is the stdout segment (if any) */
        if (kstrlen(stdout_segment) > 0)
            kstrcat(command, stdout_segment);

   	cmd_line = kstrdup(command);

   	return(cmd_line);
}

/*--------------------------------------------------------------
|
|  Routine Name: kvf_construct_partial_cmd
|
|       Purpose: Creates a partial command line for use by cantata, 
|		 including everything on the form except input files,
|		 Note the GLARING SIMILARITY to kvf_form_to_cmd().
|
|         Input: form    - pointer to the form tree
|                subform - pointer to a subform tree.
|                machine - pointer to name of machine on which to exec command
|                expand  - TRUE if filenames are to be expanded, and
|                          expressions are to be evaluated; FALSE otherwise
|
|       Returns: Allocates & returns a string representing the command
|                represented by the current configuration of the form tree.
|          Date: Aug 26, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/

char *kvf_construct_partial_cmd(
   kform    *form,
   ksubform *subform,
   char     *machine,
   int      expand)
{
        kguide  *guide;
        char    *cmd_line;
        char    command[5*KLENGTH];      /* the complete command line string */
        char    cmd_args[KLENGTH];       /* cat'ed cmd string containing all
                                            args represented by selections */
        char    stdin_segment[KLENGTH];  /* "kcp -i infile -o - |" must
                                             precede program name & args */
        char    stdout_segment[KLENGTH]; /* "kcp -i - -o outfile" must follow
                                             program name & args */
	char    program_call[KLENGTH];   /*  program name as specified on the
                                            -R UIS line */

        command[0]        = '\0';
        stdin_segment[0]  = '\0';
        program_call[0]   = '\0';
        cmd_args[0]       = '\0';
        stdout_segment[0] = '\0';

        /*
         * sanity check
         */
        if (form == NULL)
        {
            kerror("kforms", "kvf_construct_partial_cmd",
                   "form passed in is NULL");
            return(NULL);
        }
        if (subform == NULL)
        {
            kerror("kforms", "kvf_construct_partial_cmd",
                   "subform passed in is NULL");
            return(NULL);
        }

        guide = kvf_search_sel_guide(subform);

        kvf_constr_cmd_line(guide->pane->sel_list, machine, stdin_segment,
                            program_call, cmd_args, stdout_segment, expand,
			    TRUE);

        /* first in the command string is the stdin segment (if any) */
        if (kstrlen(stdin_segment) > 0)
            kstrcat(command, stdin_segment);

        /* then is the program call itself */
        if (kstrlen(program_call) == 0)
            kstrcat(command, " ");
        else kstrcat(command, program_call);

        /* followed by all the program arguments & their values */
        kstrcat(command, cmd_args);

        /* last is the stdout segment (if any) */
        if (kstrlen(stdout_segment) > 0)
            kstrcat(command, stdout_segment);

        cmd_line = kstrdup(command);

        return(cmd_line);
}

/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_cmd_line
|
|       Purpose: Creates a command line, according to the values
|                of a particular selection list.
|
|         Input: sel_list - header of the selection list
|		 machine  - pointer to machine on which to exec command
|                expand   - TRUE if filenames are to be expanded, and
|                           expressions are to be evaluated; FALSE otherwise
|		 exclude  - TRUE to exclude input files, outout files, 
|                           stdin, stdout, and routine buttons (this is used 
|                           in cantata to restore workspaces)
|
|        Output: command  - buffer w/ command to execute 
|   
|       Returns: none
|          Date: March 24, 1992
|    Written By: Danielle Argiro
| Modifications: 
|
----------------------------------------------------------------*/

static int kvf_constr_cmd_line(
    kselection *sel_list,
    char       *machine,
    char       *stdin_segment,
    char       *program_call,
    char       *cmd_args,
    char       *stdout_segment,
    int        expand,
    int        exclude)
{
	kselection *selection;
	int        flag, routine;
	char       cmd_segment[KLENGTH];  /* segment of cmd string generated 
			                     by a single selection (argument) */
	cmd_segment[0] = '\0';
	routine = FALSE;
	selection = sel_list;
	while (selection != NULL)
	{
	    flag = kvf_get_line_type(selection->line);
	    switch (flag)
	    {
	      case KUIS_STDIN:
		   if (exclude) break;
	      	   kvf_constr_std_cmd(selection, stdin_segment, expand);
		   break;

	      case KUIS_STDOUT:
		   if (exclude) break;
	      	   kvf_constr_std_cmd(selection, stdout_segment, expand);
		   break;

	      case KUIS_INPUTFILE:
	      case KUIS_OUTPUTFILE:
		   if (exclude) break;
		   if (kvf_constr_file_cmd(selection, cmd_segment, expand))
                       kstrcat(cmd_args, cmd_segment);
                   break;


	      case KUIS_STRING:
	      case KUIS_STRINGLIST:
	      	   if (kvf_constr_str_cmd(selection, cmd_segment, expand))
                       kstrcat(cmd_args, cmd_segment);
	           break;

	      case KUIS_INTEGER:
	           if (kvf_constr_int_cmd(selection, cmd_segment, expand))
                       kstrcat(cmd_args, cmd_segment);
	           break;

	      case KUIS_FLOAT:
	           if (kvf_constr_flt_cmd(selection, cmd_segment, expand))
                       kstrcat(cmd_args, cmd_segment);
	           break;

	      case KUIS_DOUBLE:
	           if (kvf_constr_dbl_cmd(selection, cmd_segment, expand))
                       kstrcat(cmd_args, cmd_segment);
	           break;

	      case KUIS_LOGICAL:
	      case KUIS_FLAG:
	      case KUIS_CYCLE:
	      case KUIS_LIST:
	      case KUIS_DISPLAYLIST:
	           if (kvf_constr_misc_cmd(selection, cmd_segment, expand))
                       kstrcat(cmd_args, cmd_segment);
	           break;

	      case KUIS_TOGGLE:
		   if (exclude)
	           {
		   	int toggle_type;
		        kvf_get_attribute(selection->back_kformstruct,
			   	          KVF_TOGGLE_TYPE, &toggle_type);
			if ((toggle_type == KUIS_INPUTFILE) ||
			    (toggle_type == KUIS_OUTPUTFILE))
			    break;
		   }
	           if (kvf_constr_toggle_cmd(selection, cmd_segment, expand))
                       kstrcat(cmd_args, cmd_segment);
  	           break;

	      case KUIS_ROUTINE:
		   if (exclude) break;
		   if (routine)
		   {
		       errno = KINTERNAL;
		       kerror("kforms", "kvf_constr_cmd_line",
			      "getting a second routine name in command line");
		       return(FALSE);
	           }
	           if (!kvf_constr_routine_cmd(selection, program_call, 
			                       machine, expand))
		       return(FALSE);
		   routine = TRUE;
		   break;

	      case KUIS_MUTEXCL:
	      case KUIS_MUTINCL:
	      case KUIS_GROUP:
	           kvf_constr_cmd_line(selection->group_next, machine, 
				       stdin_segment, program_call, cmd_args,
			               stdout_segment, expand, exclude);
	           break;

	      default:
	           break;
              }
              selection = selection->next;
   	}
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_file_cmd
|
|       Purpose: Constructs the appropriate command line segment
|                for an InputFile or OutputFile selection according 
|                to its variable name and value.  For example, 
|                "-O ~/data/images/junk.xv"
|
|         Input: selection - selection for which to constr CLUI segment
|                expand    - TRUE if filenames are to be expanded, and
|                            expressions are to be evaluated; FALSE otherwise
|
|        Output: cmd_segment - stores the command line segment
|       Returns: TRUE if a correct command line segment is created,
|                FALSE otherwise
|          Date: March 24, 1993
|    Written By: Danielle Argiro & Mark Young
| Modifications: 
|
----------------------------------------------------------------*/
static int kvf_constr_file_cmd(
    kselection *selection,
    char       *cmd_segment,
    int        expand)
{
	Line_Info line_info;
	char      temp[KLENGTH];
	char      buffer[KLENGTH];
	char      error[KLENGTH];
	char      *filename;
	
	cmd_segment[0] = '\0';

	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	
	if ((line_info.typeflag != KUIS_INPUTFILE)  &&
	    (line_info.typeflag != KUIS_OUTPUTFILE))
	{
	    errno = KCALL;
	    kerror("kforms", "kvf_constr_file_cmd", 
		   "selection of incorrect type passed in");
	    return(FALSE);
	}

	if ((line_info.optional) && (!line_info.opt_sel)) 
	    return(FALSE);

	if ((line_info.filename == NULL) && (line_info.literal == NULL))
	    return(FALSE);

	if (line_info.literal == NULL)
	    line_info.literal = kstrdup(line_info.filename);

	if (expand)
	{
	    filename = kfullpath(line_info.literal, NULL, temp); 
	    if ((kstrchr(filename,'$')) != NULL)
            {
                if (!(kvf_call_expr_handler(selection->back_form,
                                           filename, KSTRING,
                                           (kaddr) &buffer, error)))
	        {
	            errno = KINVALID_INPUT;
                    kerror("kforms", "kvf_form_to_cmd", "'%s' %s\n is not a proper input file expression\n\n", line_info.title, line_info.literal);
                    return(FALSE);
	        }
    	    }
	    else kstrcpy(buffer, filename);

	    ksprintf(cmd_segment, "-%s %s ", line_info.variable, buffer);
	}
	else 
	{
	    kstrcpy(buffer, line_info.literal);
	    ksprintf(cmd_segment, "-%s %s ", line_info.variable, buffer);
	}
	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}
/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_str_cmd
|
|       Purpose: Constructs the appropriate command line segment
|                for a String, or StringList selection according 
|                to its variable name and value.
|                For example, "-O ~/data/images/junk.xv"
|
|         Input: selection - selection for which to constr CLUI segment
|                expand    - TRUE if filenames are to be expanded, and
|                            expressions are to be evaluated; FALSE otherwise
|
|        Output: cmd_segment - stores the command line segment
|       Returns: TRUE if a correct command line segment is created,
|                FALSE otherwise
|          Date: March 24, 1993
|    Written By: Danielle Argiro & Mark Young
| Modifications: 
|
----------------------------------------------------------------*/
static int kvf_constr_str_cmd(
    kselection *selection,
    char       *cmd_segment,
    int        expand)
{
	Line_Info line_info;
	char      *filename = NULL;
	char      buffer[KLENGTH];
	
	cmd_segment[0] = '\0';

	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	
	if ((line_info.typeflag != KUIS_STRING)     &&
	    (line_info.typeflag != KUIS_STRINGLIST))
	{
	    errno = KCALL;
	    kerror("kforms", "kvf_constr_str_cmd", 
		   "selection of incorrect type passed in");
	    return(FALSE);
	}

	if ((line_info.optional) && (!line_info.opt_sel)) 
	    return(FALSE);

	kstrcpy(buffer, line_info.literal);
	ksprintf(cmd_segment, "-%s \'%s\' ", line_info.variable, buffer);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_std_cmd
|
|       Purpose: Constructs the appropriate command line segment
|                for a Stdin or Stdout selection according to its 
|                variable name and filename value (if any).
|                stdin:  kcp -i infile -o - | {command string}
|                stdout: {command string} | kcp -i - -o outfile
|                stdin + stdout: 
|                    kcp -i infile -o - | {command string}| kcp -i - -o outfile
|
|         Input: selection - selection for which to constr CLUI segment
|                expand    - TRUE if filenames are to be expanded, and
|                            expressions are to be evaluated; FALSE otherwise
|
|        Output: cmd_segment - stores the command line segment
|       Returns: TRUE if a correct command line segment is created,
|                FALSE otherwise
|          Date: April 27, 1994
|    Written By: Danielle Argiro
| Modifications: 
|
----------------------------------------------------------------*/
static int kvf_constr_std_cmd(
    kselection *selection,
    char       *cmd_segment,
    int        expand)
{
	Line_Info line_info;
	char      *filename = NULL;
	char      buffer[KLENGTH];
	
	cmd_segment[0] = '\0';

	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	
	if ((line_info.typeflag != KUIS_STDIN)      &&
	    (line_info.typeflag != KUIS_STDOUT))
	{
	    errno = KCALL;
	    kerror("kforms", "kvf_constr_std_cmd", 
		   "selection of incorrect type passed in");
	    return(FALSE);
	}

	if ((line_info.optional) && (!line_info.opt_sel))
	    return(FALSE);

        if (line_info.filename == NULL) 
            return(FALSE);

	if (expand)
	{
	    filename = kfullpath(line_info.filename, NULL, NULL); 
	    if (filename == NULL)
	    {
	        errno = KINVALID_INPUT;
                kerror("kforms", "kvf_form_to_cmd", "'%s' %s\n is not a proper input file expression\n\n", line_info.title, line_info.filename);
                return(FALSE);
    	    }
  	    kstrcpy(buffer, filename);
	    kfree(filename);
	}
	else kstrcpy(buffer, line_info.filename);

	if (line_info.typeflag == KUIS_STDIN)
	    ksprintf(cmd_segment, "kcp -i %s -o - | ", buffer);
	else  ksprintf(cmd_segment, "| kcp -i - -o %s -force", buffer);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}


/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_int_cmd
|
|       Purpose: Constructs the appropriate command line segment
|                for an Integer selection according to its variable 
|                name and value.  For example, "-s 5".
|
|         Input: selection - selection for which to constr CLUI segment
|                expand    - TRUE if filenames are to be expanded, and
|                            expressions are to be evaluated; FALSE otherwise
|
|        Output: cmd_segment - stores the command line segment
|       Returns: TRUE if a correct command line segment is created,
|                FALSE otherwise
|          Date: March 24, 1993
|    Written By: Danielle Argiro & Mark Young
| Modifications: 
|
----------------------------------------------------------------*/
static int kvf_constr_int_cmd(
    kselection *selection,
    char       *cmd_segment,
    int        expand)
{
	Line_Info line_info;
	char      error[KLENGTH];
	char      leftover[KLENGTH];
	char      temp[KLENGTH];
	int       int_val;
	
	cmd_segment[0] = '\0';

	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	
	if (line_info.typeflag != KUIS_INTEGER)
	{
	    errno = KCALL;
	    kerror("kforms", "kvf_constr_int_cmd", 
		   "selection of incorrect type passed in");
	    return(FALSE);
	}

	if ((line_info.optional) && (!line_info.opt_sel)) 
	    return(FALSE);

	if (line_info.literal == NULL)
	{
	    ksprintf(temp, "%d", line_info.int_val);
	    line_info.literal = kstrdup(temp);
	}
	leftover[0] = '\0';
	if (expand) 
	{
	    if ((ksscanf(line_info.literal,"%d%s", &int_val, leftover) != 2) ||
	        (leftover[0] != '\0'))
	    {
	        if (!(kvf_call_expr_handler(selection->back_form,
	    				    line_info.literal, KINT, 
				            (kaddr) &int_val, error)))
                {
	    	    errno = KINVALID_INPUT;
                    kerror("kforms", "kvf_constr_int_cmd", "'%s' %s\n is not a proper integer expression\n\n %s", line_info.title, line_info.literal, error);
                    return(FALSE);
	        }
 	    }
	    ksprintf(cmd_segment, "-%s %d ", line_info.variable, int_val);
	}
	else ksprintf(cmd_segment, "-%s '%s' ", 
		      line_info.variable, line_info.literal);
	

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_flt_cmd
|
|       Purpose: Constructs the appropriate command line segment
|                for a Float selection according to its variable 
|                name and value.  For example, "-f 5.9".
|
|         Input: selection - selection for which to constr CLUI segment
|                expand    - TRUE if filenames are to be expanded, and
|                            expressions are to be evaluated; FALSE otherwise
|
|        Output: cmd_segment - stores the command line segment
|       Returns: TRUE if a correct command line segment is created,
|                FALSE otherwise
|          Date: March 24, 1993
|    Written By: Danielle Argiro & Mark Young
| Modifications: 
|
----------------------------------------------------------------*/
static int kvf_constr_flt_cmd(
    kselection *selection,
    char       *cmd_segment,
    int        expand)
{
	Line_Info line_info;
	char      error[KLENGTH];
	char      leftover[KLENGTH];
	char      temp[KLENGTH];
	float     float_val;
	
	cmd_segment[0] = '\0';

	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	
	if (line_info.typeflag != KUIS_FLOAT)
	{
	    errno = KCALL;
	    kerror("kforms", "kvf_constr_flt_cmd", 
		   "selection of incorrect type passed in");
	    return(FALSE);
	}

	if ((line_info.optional) && (!line_info.opt_sel)) 
	    return(FALSE);

	leftover[0] = '\0';
	if (line_info.literal == NULL)
        {
            ksprintf(temp, "%g", line_info.float_val);
            line_info.literal = kstrdup(temp);
        }
	if (expand)
	{
	    if (ksscanf(line_info.literal,"%f%s", &float_val, leftover) != 2 ||
	        (leftover[0] != '\0'))
	    {
	        if (!(kvf_call_expr_handler(selection->back_form,
					    line_info.literal, KFLOAT,
				            (kaddr) &float_val, error)))
                {
	    	    errno = KINVALID_INPUT;
                    kerror("kforms", "kvf_constr_flt_cmd", "'%s' %s\n is not a proper float expression\n\n %s", line_info.title, line_info.literal, error);
                    return(FALSE);
	        }
 	    }
	    ksprintf(cmd_segment, "-%s %g ", line_info.variable, float_val);
	}
	else ksprintf(cmd_segment, "-%s '%s' ", 
		      line_info.variable, line_info.literal);

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_dbl_cmd
|
|       Purpose: Constructs the appropriate command line segment
|                for a Double selection according to its variable 
|                name and value.  For example, "-h 0.0000000019".
|
|         Input: selection - selection for which to constr CLUI segment
|                expand    - TRUE if filenames are to be expanded, and
|                            expressions are to be evaluated; FALSE otherwise
|
|        Output: cmd_segment - stores the command line segment
|       Returns: TRUE if a correct command line segment is created,
|                FALSE otherwise
|          Date: April 30, 1993
|    Written By: Danielle Argiro
| Modifications: 
|
----------------------------------------------------------------*/
static int kvf_constr_dbl_cmd(
    kselection *selection,
    char       *cmd_segment,
    int        expand)
{
	Line_Info line_info;
	char      error[KLENGTH];
	char      leftover[KLENGTH];
	char      temp[KLENGTH];
	double    double_val;
	
	cmd_segment[0] = '\0';

	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	
	if (line_info.typeflag != KUIS_DOUBLE)
	{
	    errno = KCALL;
	    kerror("kforms", "kvf_constr_dbl_cmd", 
		   "selection of incorrect type passed in");
	    return(FALSE);
	}

	if ((line_info.optional) && (!line_info.opt_sel)) 
	    return(FALSE);

	if (line_info.literal == NULL)
        {
            ksprintf(temp, "%g", line_info.double_val);
            line_info.literal = kstrdup(temp);
        }
	leftover[0] = '\0';

	if (expand)
	{
	   if (ksscanf(line_info.literal,"%lg%s", &double_val, leftover) != 2 ||
	       (leftover[0] != '\0'))
	   {
	       if (!(kvf_call_expr_handler(selection->back_form,
	   				line_info.literal, KDOUBLE,
	   			        (kaddr) &double_val, error)))
               {
	   	   errno = KINVALID_INPUT;
                   kerror("kforms", "kvf_constr_dbl_cmd", "'%s' %s\n is not a proper double expression\n\n %s", line_info.title, line_info.literal, error);
                   return(FALSE);
	       }
 	   }
	   ksprintf(cmd_segment, "-%s %g ", line_info.variable, double_val);
	}
	else ksprintf(cmd_segment, "-%s '%s' ",
                      line_info.variable, line_info.literal);

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_misc_cmd
|
|       Purpose: Constructs the appropriate command line segment
|                for a Logical, Flag, List, DisplayList or Cycle 
|                selection according to its variable name and value.  
|                For example, "-type 1".
|
|         Input: selection - selection for which to constr CLUI segment
|                expand    - TRUE if filenames are to be expanded, and
|                            expressions are to be evaluated; FALSE otherwise
|
|        Output: cmd_segment - stores the command line segment
|       Returns: TRUE if a correct command line segment is created,
|                FALSE otherwise
|          Date: March 24, 1993
|    Written By: Danielle Argiro & Mark Young
| Modifications: 
|
----------------------------------------------------------------*/
static int kvf_constr_misc_cmd(
    kselection *selection,
    char       *cmd_segment,
    int        expand)
{
	Line_Info line_info;
	
	cmd_segment[0] = '\0';

	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	
	if ((line_info.typeflag != KUIS_LIST)        &&
	    (line_info.typeflag != KUIS_CYCLE)       && 
	    (line_info.typeflag != KUIS_LOGICAL)     && 
	    (line_info.typeflag != KUIS_DISPLAYLIST) && 
	    (line_info.typeflag != KUIS_FLAG))
	{
	    errno = KCALL;
	    kerror("kforms", "kvf_constr_list_cmd", 
		   "selection of incorrect type passed in");
	    return(FALSE);
	}

	if ((line_info.optional) && (!line_info.opt_sel)) 
	    return(FALSE);

	if (line_info.typeflag == KUIS_FLAG)
	{
	    if (kstrcmp(line_info.variable, "P") == 0)
		return(TRUE);
	    if (line_info.opt_sel)
          	ksprintf(cmd_segment, "-%s ", line_info.variable);
	    else return(FALSE);
	}
	else if (line_info.typeflag == KUIS_LOGICAL)
	{
	    if (line_info.logical_val)
		ksprintf(cmd_segment,  "-%s 1 ", line_info.variable);
	    else ksprintf(cmd_segment, "-%s 0 ", line_info.variable);
	}
	else
	{
            ksprintf(cmd_segment, "-%s %d ", 
		    line_info.variable, line_info.list_val);
	}

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}


/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_toggle_cmd
|
|       Purpose: Constructs the appropriate command line segment
|                for a Toggle selection according to its variable 
|                name and value.  For example, "-type byte".
|
|         Input: selection - selection for which to constr CLUI segment
|                expand    - TRUE if filenames are to be expanded, and
|                            expressions are to be evaluated; FALSE otherwise
|
|        Output: cmd_segment - stores the command line segment
|       Returns: TRUE if a correct command line segment is created,
|                FALSE otherwise
|          Date: March 24, 1993
|    Written By: Danielle Argiro & Mark Young
| Modifications: 
|
----------------------------------------------------------------*/
static int kvf_constr_toggle_cmd(
    kselection *selection,
    char       *cmd_segment,
    int        expand)
{
	int        num;
	char       toggle_name[KLENGTH];
	char       scale[KLENGTH];
	kselection *toggle;
	Line_Info  line_info;
	
	cmd_segment[0] = '\0';

	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	
	if (line_info.typeflag != KUIS_TOGGLE)
	{
	    errno = KCALL;
	    kerror("kforms", "kvf_constr_toggle_cmd", 
		   "selection of incorrect type passed in");
	    return(FALSE);
	}

	if ((line_info.optional) && (!line_info.opt_sel)) 
	    return(FALSE);

        if (line_info.toggle_val == 0)
            line_info.toggle_val = line_info.toggle_def;
  
	ksprintf(toggle_name, "%s", line_info.variable);
  
        num = line_info.toggle_val-1;
        toggle = selection->toggle_next;
        while (num > 0)
        {
            toggle = toggle->next;
            num = num - 1;
        }
        num = line_info.toggle_val;
	kvf_gen_parse(toggle->line, &line_info);
  
	if (line_info.typeflag == KUIS_INTEGER)
	    ksprintf(cmd_segment, "-%s %d ", toggle_name, line_info.int_def);
	       
	else if (line_info.typeflag == KUIS_FLOAT)
	{
	    if (line_info.precision == 0)
	        ksprintf(scale, "-%%s %%g ", line_info.precision);
	    else if (line_info.precision == -1)
	        ksprintf(scale, "-%%s %%f ", line_info.precision);
	    else ksprintf(scale, "-%%s %%.%df ", line_info.precision);
	    ksprintf(cmd_segment, scale, toggle_name, line_info.float_def);
	}
	       
	else if (line_info.typeflag == KUIS_DOUBLE)
	{
	    if (line_info.precision == 0)
	        ksprintf(scale, "-%%s %%g ", line_info.precision);
	    else if (line_info.precision == -1)
                ksprintf(scale, "-%%s %%f ", line_info.precision);
	    else ksprintf(scale, "-%%s %%.%df ", line_info.precision);
	    ksprintf(cmd_segment, scale, toggle_name, line_info.double_def);
	}
	       
	else if (line_info.typeflag == KUIS_INPUTFILE)
	    ksprintf(cmd_segment, "-%s %s ", toggle_name, line_info.file_def);
	       
	else if (line_info.typeflag == KUIS_OUTPUTFILE)
	    ksprintf(cmd_segment, "-%s %s ", toggle_name, line_info.file_def);
	       
	else if (line_info.typeflag == KUIS_STRING)
	    ksprintf(cmd_segment, "-%s '%s' ", toggle_name, 
		    line_info.string_def);
	     
        else ksprintf(cmd_segment, "-%s %d ", toggle_name, num);

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}


/*--------------------------------------------------------------
|
|  Routine Name: kvf_constr_routine_cmd
|
|       Purpose: Constructs the appropriate command line segment
|                for a Routine selection according to its routine 
|                name and machine.  For example, "vadd@borris.eece.unm.edu".
|
|         Input: selection - selection for which to construct CLUI segment
|                machine   - machine at which to execute command
|                expand    - TRUE if filenames are to be expanded, and
|                            expressions are to be evaluated; FALSE otherwise
|
|        Output: cmd_segment - stores the command line segment
|       Returns: TRUE if a correct command line segment is created,
|                FALSE otherwise
|          Date: March 24, 1993
|    Written By: Danielle Argiro & Mark Young
| Modifications: 
|
----------------------------------------------------------------*/

static int kvf_constr_routine_cmd(
    kselection *selection,
    char       *command,
    char       *machine,
    int        expand)
{
	Line_Info line_info;
	
	command[0] = '\0';

	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	
	if (line_info.typeflag != KUIS_ROUTINE)
	{
	    errno = KCALL;
	    kerror("kforms", "kvf_constr_routine_cmd", 
		   "selection of incorrect type passed in");
	    return(FALSE);
	}

	/* 
	 * I am assuming here that there is only one routine to be called 
         * from any one pane. This will need to be changed sometime.
         */
	if (machine == NULL)
	{
	    if ((!expand)||
		(kfullpath(line_info.routine, NULL, command) == NULL))
	        ksprintf(command,"%s ",line_info.routine);
	    else kstrcat(command, " ");
	}
	else
	{
	    if ((!expand) || 
		(kfullpath(line_info.routine, NULL, command) == NULL))
	        ksprintf(command,"%s@%s ", line_info.routine, machine);
	    else
	    {
	        kstrcat(command, "@");
	        kstrcat(command, machine);
	        kstrcat(command, " ");
	    }
	}
	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}



/************************************************************
*
*  Routine Name: kvf_modify_form_from_cmd - translate command line string
*                                           into form tree representation
*
*       Purpose: Gathers variables and values from the command line
*                string passed in, and modifies selections in the form tree 
*                according to those values. Note: this routine is used 
*                primarily by the kcodegen library, and by cantata.
*
*         Input: form     - pointer to the form tree
*                cmd      - the command line string
*                override - TRUE to over-ride recently-set values
*                                (selection->modified TRUE)
*                           FALSE to leave recently-set values alone
*                                (selection->modified FALSE)
*        Output: none
*       Returns: TRUE if form is correctly updated according to
*                command line, FALSE otherwise
*  Restrictions:
*    Written By: Danielle Argiro
*          Date:  March 24, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvf_modify_form_from_cmd(
    kform *form,
    char  *cmd,
    int   override)
{
	kselection *selection;

	if (!(kvf_check_for_single_pane(form, "kvf_modify_form_from_cmd")))
	    return(FALSE);

	selection = form->subform->guide->pane->sel_list;
	return(modify_selections_from_cmd(selection, cmd, override));
}

/*--------------------------------------------------------------
|
|  Routine Name: modify_selections_from_cmd
|
|       Purpose: Modifies the values stored in a selection list 
|                according to to the command string
|
|         Input: sel_list - pointer to the selection list
|                cmd      - the command line string
|                override - TRUE to over-ride recently-set values 
|                                (if modified TRUE)
|                           FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output: 
|       Returns: TRUE if selection list is correctly updated according to 
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
----------------------------------------------------------------*/
static int modify_selections_from_cmd(
    kselection *selection,
    char       *cmd,
    int        override)
{
	char  command[5*KLENGTH];

	if (cmd == NULL) return(TRUE);

	/* don't want "-infile" to register as "-i", thus the space */
	sprintf(command, "%s ", cmd);


	while (selection != NULL)
	{
	    switch (selection->type)
	    {
		case KUIS_INPUTFILE:
		case KUIS_STDIN:
		case KUIS_STDOUT:
		case KUIS_OUTPUTFILE:
		     if (!(modify_file_from_cmd(selection, command, 
					        override)))
			return(FALSE);
		     break;

		case KUIS_INTEGER:
		     if (!(modify_int_from_cmd(selection, command,
			                       override)))
			return(FALSE);
                     break;

		case KUIS_DOUBLE:
		     if (!(modify_double_from_cmd(selection, command, 
						  override)))
			return(FALSE);
		     break;

		case KUIS_FLOAT:
		     if (!(modify_float_from_cmd(selection, command, 
			                         override)))
			return(FALSE);
		     break;

		case KUIS_LOGICAL:
		     if (!(modify_logical_from_cmd(selection, command, 
				                   override)))
			return(FALSE);
		     break;

		case KUIS_FLAG:
		     if (!(modify_flag_from_cmd(selection, command, 
				                override)))
			return(FALSE);
		     break;

		case KUIS_STRING:
		case KUIS_STRINGLIST:
		     if (!(modify_string_from_cmd(selection, command, 
				                  override)))
			return(FALSE);
		     break;

		case KUIS_CYCLE:
		case KUIS_LIST:
		case KUIS_DISPLAYLIST:
		     if (!(modify_list_from_cmd(selection, command, 
				                override)))
			return(FALSE);
                     break;

		case KUIS_MUTEXCL:
		case KUIS_MUTINCL:
		case KUIS_GROUP:
		     if (!(modify_selections_from_cmd(selection->group_next,
						      cmd, override)))
		         return(FALSE);
		     break;

	        case KUIS_TOGGLE:
		     if (!(modify_toggle_from_cmd(selection, command, 
						  override)))
			return(FALSE);
                     break;

	    }
	    selection = selection->next;
	}
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: modify_file_from_cmd
|
|       Purpose: Modifies the values stored in an InputFile, OutputFile, 
|                Stdin, Stdout, AnswerInputFile, or AnswerOutputfile
|                selection according to to the command string
|
|         Input: selection - pointer to the selection
|                command   - the command line string
|                override  - TRUE to over-ride recently-set values
|                                (if modified TRUE)
|                            FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output:
|       Returns: TRUE if selection is correctly updated according to
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/
static int modify_file_from_cmd(
    kselection *selection,
    char       *command,
    int        override)
{
	char *variable, *cmd_seg;
	char temp[KLENGTH];
	char stringval[KLENGTH];
	char cmd_var[KLENGTH];
	int  optional, optsel;
	int  file_check = TRUE;

        kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTIONAL, &optional);
        kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTSEL, &optsel);
        variable = ktoken_to_string(selection->var_token);
        sprintf(temp, " -%s ", variable); 
	stringval[0] = '\0';

        /* 
	 * variable associated w/ selection does not appear on command line 
         */
        if ((cmd_seg = kstrstr(command, temp)) == NULL)
        {
            /* 
	     * if it's optional, and if we're either overriding
             * previous values, or if it hasn't been recently changed,
             * set optsel to false, indicating it is not selected 
             */
	    if ((optional) && (optsel != 2) &&
		(!selection->modified || override)) 
            {
		if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 0)))
                    return(FALSE);
            }
        }

	/*
	 * variable is on cmd line followed by *literal* value.
	 * just set the literal and return.
	 */
	else if (ksscanf(cmd_seg, " -%s '%[^']'", cmd_var, stringval) == 2)
	{

	    kvf_set_attribute(selection->back_kformstruct, KVF_LITERAL, 
			      stringval);
	    if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
            }
	    selection->modified = TRUE;
	}

	/* 
	 * variable is on cmd line, but not followed by value: 
	 * in general, this is an illegal construct.
	 * scan string after -var until you hit a space, or another '-'
         * (which would indicate the next clui option). 
	 *
	 * BUT, construct OK if ALL there is for the value is "-" 
	 * (this implies the use of stdin or stdout)
	 *
	 * It's ALSO OK if the variable is "a", "A", or "ap": these are special
	 * cases where if the filename doesn't appear, we know to substitute
	 * the value of $KHOROS_ANSFILE.
         */
        else if (((ksscanf(cmd_seg, " -%s %s", cmd_var, stringval) != 2) ||
		  ((stringval[0] == '-')  && 
		   (stringval[1] != '\0'))) && 
                  (kstrcasecmp(cmd_var, "A") != 0) &&
		  (kstrcmp(cmd_var, "ap") != 0))
        {
            kerror("kforms", "kvf_modify_form_from_cmd",
                   "Invalid or missing filename after \"-%s\" argument", 
		   variable);
            return(FALSE);
        }

	/* 
	 * false alarm: found only a subset of letters making up variable 
         */
	else if (kstrcmp(cmd_var, variable) != 0)
	    return(TRUE);

	/* 
	 * found proper construct: variable and it's value on command line 
 	 * only update selection values if we are overriding previous values,
         * or if it hasn't been recently changed.
         */
	else if ((!selection->modified) || (override))
        {
	    /* 
    	     * Here, we are about to set the filename.
	     *
	     * (1) If the variable is NOT "-A", "-a", or "-ap", set filename.  
	     *
	     * (2) If the variable IS "-A", "-a", or "-ap", but a real
	     *     filename (not another command line arg) does follow 
	     *     the variable: set the filename.
	     *
	     * (3) If the filename following variable is "-" by itself, 
	     *     set the filename: it's implying stdin or stdout.
	     */
            if ( ((kstrcasecmp(variable, "A") != 0) &&
		  (kstrcasecmp(variable, "ap") != 0))    ||

		 ((kstrcasecmp(variable, "A") == 0) && 
		  (kstrlen(stringval)!= 0)          &&
		  (stringval[0] != '-'))                 ||

		((kstrcmp(variable, "ap") == 0) && 
		 (kstrlen(stringval) != 0)      &&
		 (stringval[0] != '-'))                  ||

		(kstrcmp(stringval, "-") == 0) )
	    {
	        /* set the filename literal as specified */
                if (!(kvf_get_attribute(selection->back_kformstruct,
                                        KVF_FILE_CHECK, &file_check)))
                    return(FALSE);
		/* 
		 * if the input file does not exist (invalid input), DON'T 
		 * set filename value or the modified flag; kcodegen will print
		 * the error message if it's getting -i off cmd line, while
                 * cantata will go ahead and over-write the filename anyway
		 * returning TRUE avoids excess error messages.
	 	 */
		if ((selection->type == KUIS_INPUTFILE) && (file_check) &&
		    (kaccess(stringval, R_OK | F_OK) != 0))
		{
		   selection->modified = -1;
		   return(TRUE);
		}
/*
		else if ((selection->type == KUIS_OUTPUTFILE) && (file_check) &&
		    (kaccess(stringval, W_OK | F_OK) != 0))
		{
		   selection->modified = -1;
		   return(TRUE);
		}
*/

	        /* set the filename literal as specified */
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_LITERAL, stringval)))
                    return(FALSE);

		/*
	  	 *  valid input file (it exists) or output filename;
		 *  set the filename
	         */
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_FILE_NAME, stringval)))
		    return(FALSE);
	    }

	    /* if optional, indicate opt selected */
            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
	    }

	    /* now, it's been recently changed */
            selection->modified = TRUE;
	}
	return(TRUE);
}


/*--------------------------------------------------------------
|
|  Routine Name: modify_int_from_cmd
|
|       Purpose: Modifies the values stored in an Integer selection
|                according to to the command string
|
|         Input: selection - pointer to the selection
|                command   - the command line string
|                override  - TRUE to over-ride recently-set values
|                                (if modified TRUE)
|                            FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output:
|       Returns: TRUE if selection is correctly updated according to
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/

static int modify_int_from_cmd(
    kselection *selection,
    char       *command,
    int        override)
{
        char *variable, *cmd_seg;
        char temp[KLENGTH];
	char cmd_var[KLENGTH];
	char stringval[KLENGTH];
	int  intval;
	int  optional, optsel;

	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTIONAL, &optional);
	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTSEL, &optsel);
        variable = ktoken_to_string(selection->var_token);
        sprintf(temp, "-%s ", variable);

        /*
         * variable associated w/ selection does not appear on command line
         */
	if ((cmd_seg = kstrstr(command, temp)) == NULL)
        {
            /*
             * if it's optional, and if we're either overriding
             * previous values, or if it hasn't been recently changed,
             * set optsel to false, indicating it is not selected
             */
	    if ((optional) && (optsel != 2) &&
		(!selection->modified || override)) 
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 0)))
                    return(FALSE);
            }
        }

	/*
         * variable is on cmd line followed by *literal* value.
         * just set the literal and return.
         */
        else if (ksscanf(cmd_seg, " -%s '%[^']'", cmd_var, stringval) == 2)
        {

            kvf_set_attribute(selection->back_kformstruct, KVF_LITERAL,
                              stringval);
            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
            }
            selection->modified = TRUE;
        }

	/*
         * variable is on cmd line, but not followed by value: illegal construct
         */
        else if (ksscanf(cmd_seg, "-%s %d", cmd_var, &intval) != 2)
        {
            kerror("kforms", "kvf_modify_form_from_cmd",
                   "Invalid or missing integer value after \"-%s\" argument",
                   variable);
            return(FALSE);
        }

        /*
         * false alarm: found only a subset of letters making up variable
         */
	else if (kstrcmp(cmd_var, variable) != 0)
	    return(TRUE);

        /*
         * found proper construct: variable and it's value on command line
         * only update selection values if we are overriding previous values,
         * or if it hasn't been recently changed.
         */
	else if ((!selection->modified) || (override))
        {
	    /* set the string literal as specified */
	    ksprintf(stringval, "%d", intval);
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_LITERAL, stringval)))
                return(FALSE);

	    /* set the integer value as specified */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_INT_VAL, intval)))
                return(FALSE);

	    /* if optional, indicate opt selected */
            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
            }
	    /* now, it's been recently changed */
            selection->modified = TRUE;
        }
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: modify_float_from_cmd
|
|       Purpose: Modifies the values stored in an Float selection
|                according to to the command string
|
|         Input: selection - pointer to the selection
|                command   - the command line string
|                override  - TRUE to over-ride recently-set values
|                                (if modified TRUE)
|                            FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output:
|       Returns: TRUE if selection is correctly updated according to
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/

static int modify_float_from_cmd(
    kselection *selection,
    char       *command,
    int        override)
{
        char  *variable, *cmd_seg;
        char  temp[KLENGTH];
	char  cmd_var[KLENGTH];
	char  stringval[KLENGTH];
	float floatval;
	int   optional, optsel;

	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTIONAL, &optional);
	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTSEL, &optsel);
        variable = ktoken_to_string(selection->var_token);
        sprintf(temp, "-%s ", variable);

        /*
         * variable associated w/ selection does not appear on command line
         */
	if ((cmd_seg = kstrstr(command, temp)) == NULL)
        {
            /*
             * if it's optional, and if we're either overriding
             * previous values, or if it hasn't been recently changed,
             * set optsel to false, indicating it is not selected
             */
	    if ((optional) && (optsel != 2) &&
		(!selection->modified || override)) 
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 0)))
                    return(FALSE);
            }
        }

        /*
         * variable is on cmd line followed by *literal* value.
         * just set the literal and return.
         */
        else if (ksscanf(cmd_seg, " -%s '%[^']'", cmd_var, stringval) == 2)
        {

            kvf_set_attribute(selection->back_kformstruct, KVF_LITERAL,
                              stringval);
            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
            }
	    selection->modified = TRUE;
	}

	/*
         * variable is on cmd line, but not followed by value: illegal construct
         */
        else if (ksscanf(cmd_seg, "-%s %f", cmd_var, &floatval) != 2)
        {
            kerror("kforms", "kvf_modify_form_from_cmd",
                   "Invalid or missing float value after \"-%s\" argument",
                   variable);
            return(FALSE);
        }

        /*
         * false alarm: found only a subset of letters making up variable
         */
	else if (kstrcmp(cmd_var, variable) != 0)
	    return(TRUE);

        /*
         * found proper construct: variable and it's value on command line
         * only update selection values if we are overriding previous values,
         * or if it hasn't been recently changed.
         */
	else if ((!selection->modified) || (override))
        {
	    /* set the string literal as specified */
	    ksprintf(stringval, "%g", floatval);
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_LITERAL, stringval)))
                return(FALSE);

	    /* set the float value as specified */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_FLOAT_VAL, floatval)))
                return(FALSE);

	    /* if optional, indicate opt selected */
            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
            }
	    /* now, it's been recently changed */
            selection->modified = TRUE;
        }
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: modify_double_from_cmd
|
|       Purpose: Modifies the values stored in a Double selection
|                according to to the command string
|
|         Input: selection - pointer to the selection
|                command   - the command line string
|                override  - TRUE to over-ride recently-set values
|                                (if modified TRUE)
|                            FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output:
|       Returns: TRUE if selection is correctly updated according to
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/

static int modify_double_from_cmd(
    kselection *selection,
    char       *command,
    int        override)
{
        char   *variable, *cmd_seg;
        char   temp[KLENGTH];
	char   cmd_var[KLENGTH];
	char   stringval[KLENGTH];
	double doubleval;
	int    optional, optsel;

	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTIONAL, &optional);
	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTSEL, &optsel);
        variable = ktoken_to_string(selection->var_token);
        sprintf(temp, "-%s ", variable);

        /*
         * variable associated w/ selection does not appear on command line
         */
	if ((cmd_seg = kstrstr(command, temp)) == NULL)
        {
            /*
             * if it's optional, and if we're either overriding
             * previous values, or if it hasn't been recently changed,
             * set optsel to false, indicating it is not selected
             */
	    if ((optional) && (optsel != 2) &&
		(!selection->modified || override)) 
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 0)))
                    return(FALSE);
            }
        }

        /*
         * variable is on cmd line followed by *literal* value.
         * just set the literal and return.
         */
        else if (ksscanf(cmd_seg, " -%s '%[^']'", cmd_var, stringval) == 2)
        {

            kvf_set_attribute(selection->back_kformstruct, KVF_LITERAL,
                              stringval);
            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
            }
            selection->modified = TRUE;
        }

	/*
         * variable is on cmd line, but not followed by value: illegal construct
         */
        else if (ksscanf(cmd_seg, "-%s %lg", cmd_var, &doubleval) != 2)
        {
            kerror("kforms", "kvf_modify_form_from_cmd",
                   "Invalid or missing double value after \"-%s\" argument",
                   variable);
            return(FALSE);
        }

        /*
         * false alarm: found only a subset of letters making up variable
         */
	else if (kstrcmp(cmd_var, variable) != 0)
	    return(TRUE);

        /*
         * found proper construct: variable and it's value on command line
         * only update selection values if we are overriding previous values,
         * or if it hasn't been recently changed.
         */
	else if ((!selection->modified) || (override))
        {
	    /* set the double literal as specified */
	    ksprintf(stringval, "%g", doubleval);
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_LITERAL, stringval)))
                return(FALSE);

	    /* set the double value as specified */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_DOUBLE_VAL, doubleval)))
                return(FALSE);

	    /* if optional, indicate opt selected */
            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
            }
	    /* now, it's been recently changed */
            selection->modified = TRUE;
        }
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: modify_logical_from_cmd
|
|       Purpose: Modifies the values stored in a Logical selection
|                according to to the command string
|
|         Input: selection - pointer to the selection
|                command   - the command line string
|                override  - TRUE to over-ride recently-set values
|                                (if modified TRUE)
|                            FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output:
|       Returns: TRUE if selection is correctly updated according to
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/

static int modify_logical_from_cmd(
    kselection *selection,
    char       *command,
    int        override)
{
        char  *variable, *cmd_seg;
        char  temp[KLENGTH];
	char  cmd_var[KLENGTH];
	char  stringval[KLENGTH];
	int   logicval;
	int   optional, optsel;

	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTIONAL, &optional);
	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTSEL, &optsel);
        variable = ktoken_to_string(selection->var_token);
        sprintf(temp, "-%s ", variable);

        /*
         * variable associated w/ selection does not appear on command line
         */
	if ((cmd_seg = kstrstr(command, temp)) == NULL)
        {
            /*
             * if it's optional, and if we're either overriding
             * previous values, or if it hasn't been recently changed,
             * set optsel to false, indicating it is not selected
             */
	    if ((optional) && (optsel != 2) &&
		(!selection->modified || override)) 
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 0)))
                    return(FALSE);
            }
	    return(TRUE);
        }

	/*
	 *  variable is on cmd line followed by TRUE or FALSE representation
	 */
	stringval[0] = '\0';
	if (ksscanf(cmd_seg, "-%s %s", cmd_var, &stringval) == 2)
	{
	    if ((kstrcmp(stringval, "1") == 0)        ||
                (kstrcasecmp(stringval, "t") == 0)    ||
                (kstrcasecmp(stringval, "true") == 0) ||
                (kstrcasecmp(stringval, "y") == 0)    ||
                (kstrcasecmp(stringval, "yes") == 0))
	        logicval = TRUE; 

	    else if ((kstrcmp(stringval, "0") == 0)         ||
                     (kstrcasecmp(stringval, "f") == 0)     ||
                     (kstrcasecmp(stringval, "false") == 0) ||
                     (kstrcasecmp(stringval, "n") == 0)     ||
                     (kstrcasecmp(stringval, "no") == 0)) 
                logicval = FALSE; 

	    else
	    {
                kerror("kforms", "kvf_modify_form_from_cmd",
                       "Invalid logical value after \"-%s\" argument; allowed values include:\n1, 't', 'true', 'y', 'yes;\n0, 'f', 'false', 'n', and 'no'.",
                       variable);
                return(FALSE);
	    }
	}

	/*
         * variable is on cmd line, but not followed by value: illegal construct
         */
        else if (ksscanf(cmd_seg, "-%s %d", cmd_var, &logicval) != 2)
        {
            kerror("kforms", "kvf_modify_form_from_cmd",
                   "Invalid or missing logical value after \"-%s\" argument",
                   variable);
            return(FALSE);
        }

        /*
         * false alarm: found only a subset of letters making up variable
         */
	if (kstrcmp(cmd_var, variable) != 0)
	    return(TRUE);

        /*
         * found proper construct: variable and it's value on command line
         * only update selection values if we are overriding previous values,
         * or if it hasn't been recently changed.
         */
	if ((!selection->modified) || (override))
        {
	    /* set the logical value as specified */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_LOGIC_VAL, logicval)))
                return(FALSE);

	    /* if optional, indicate opt selected */
            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
            }
	    /* now, it's been recently changed */
            selection->modified = TRUE;
        }
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: modify_string_from_cmd
|
|       Purpose: Modifies the values stored in a String or StringList
|                selection according to to the command string
|
|         Input: selection - pointer to the selection
|                command   - the command line string
|                override  - TRUE to over-ride recently-set values
|                                (if modified TRUE)
|                            FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output:
|       Returns: TRUE if selection is correctly updated according to
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/
static int modify_string_from_cmd(
    kselection *selection,
    char       *command,
    int        override)
{
	char *variable, *cmd_seg;
	char temp[KLENGTH];
	char stringval[KLENGTH];
	char cmd_var[KLENGTH];
	int  optional, optsel;

        kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTIONAL, &optional);
        kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTSEL, &optsel);
        variable = ktoken_to_string(selection->var_token);
        sprintf(temp, "-%s ", variable); 
	stringval[0] = '\0';

        /* 
	 * variable associated w/ selection does not appear on command line 
         */
        if ((cmd_seg = kstrstr(command, temp)) == NULL)
        {
            /* 
	     * if it's optional, and if we're either overriding
             * previous values, or if it hasn't been recently changed,
             * set optsel to false, indicating it is not selected 
             */
	    if ((optional) && (optsel != 2) &&
		(!selection->modified || override)) 
            {
		if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 0)))
                    return(FALSE);
            }
        }

        /*
         * variable is on cmd line followed by *literal* value.
         * just set the literal and return.
         */
        else if (ksscanf(cmd_seg, " -%s '%[^']'", cmd_var, stringval) == 2)
        {

            kvf_set_attribute(selection->back_kformstruct, KVF_LITERAL,
                              stringval);

	    /* set the string as specified */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_STRING_VAL, stringval)))
		return(FALSE);

            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
            }
            selection->modified = TRUE;
        }

	/* 
	 * variable is on cmd line, but not followed by value: illegal construct
         */
        else if ((ksscanf(cmd_seg, "-%s \"%[^\"]", cmd_var, stringval) != 2) &&
		 (ksscanf(cmd_seg, "-%s '%[^\']", cmd_var, stringval) != 2) &&
		 (ksscanf(cmd_seg, "-%s %s", cmd_var, stringval) != 2))
        {
            kerror("kforms", "kvf_modify_form_from_cmd",
                   "Invalid or missing string after \"-%s\" argument", 
		   variable);
            return(FALSE);
        }

	/* 
	 * false alarm: found only a subset of letters making up variable 
         */
	else if (kstrcmp(cmd_var, variable) != 0)
	    return(TRUE);

	/* 
	 * found proper construct: variable and string value on command line 
	 * either surrounded in double quotes ("), single quotes ('), or plain.
 	 * only update selection values if we are overriding previous values,
         * or if it hasn't been recently changed.
         */
	else if ((!selection->modified) || (override))
        {
	    /* set the string literal as specified */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_LITERAL, stringval)))
                return(FALSE);

	    /* set the filename as specified */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_STRING_VAL, stringval)))
		return(FALSE);

	    /* if optional, indicate opt selected */
            if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
                    return(FALSE);
	    }

	    /* now, it's been recently changed */
            selection->modified = TRUE;
	}
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: modify_list_from_cmd
|
|       Purpose: Modifies the values stored in a List, Cycle, or DisplayList
|                selection according to to the command string
|
|         Input: selection - pointer to the selection
|                command   - the command line string
|                override  - TRUE to over-ride recently-set values
|                                (if modified TRUE)
|                            FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output:
|       Returns: TRUE if selection is correctly updated according to
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/
static int modify_list_from_cmd(
    kselection *selection,
    char       *command,
    int        override)
{

	char *variable, *cmd_seg;
        char temp[KLENGTH];
        char stringval[KLENGTH];
	char cmd_var[KLENGTH];
	int  intval;
	int  optional, optsel;

	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTIONAL, &optional);
	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTSEL, &optsel);
	variable = ktoken_to_string(selection->var_token);
        sprintf(temp, "-%s ", variable);

        /*
         * variable associated w/ selection does not appear on command line
         */
	if ((cmd_seg = kstrstr(command, temp)) == NULL)
        {
            /*
             * if it's optional, and if we're either overriding
             * previous values, or if it hasn't been recently changed,
             * set optsel to false, indicating it is not selected
             */
            if ((optional) && (optsel != 2) &&
		(!selection->modified || override))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 0)))
                    return(FALSE);
            }
	
        }

       /* 
        * found proper construct: variable and int value on command line
        * only update selection values if we are overriding previous values,
        * or if it hasn't been recently changed.
	*/
	else if ((ksscanf(cmd_seg, "-%s %d", cmd_var, &intval) == 2) &&
		 ((!selection->modified) || (override)))
        {
	    /* set the list value as specified by int value */
	    if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_LIST_VAL, intval)))
		return(FALSE);

	    /* if optional, indicate opt selected */
	    if (optional && (optsel != 2))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 1)))
		    return(FALSE);
            }
	    /* now, it's been recently changed */
            selection->modified = TRUE;
        }

       /* 
        * found another proper construct: variable and string value on command 
        * line; only update selection values if we are overriding previous 
        * values or if it hasn't been recently changed.
        */
        else if ((((ksscanf(cmd_seg, "-%s \"%[^\"]", cmd_var, stringval) == 2) ||
		  (ksscanf(cmd_seg, "-%s '%[^\']", cmd_var, stringval) == 2)  ||
		  (ksscanf(cmd_seg, "-%s %s", cmd_var, stringval) == 2))      &&
		  ((!selection->modified) || (override))))
	{
            /* set the list value as specified by string value */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                          KVF_LIST_ITEM, stringval)))
	    {
		errno = 0;
                kerror("kforms", "kvf_modify_form_from_cmd",
                       "Invalid list/cycle value after \"-%s\" argument",
                        variable);
                return(FALSE);
	    }

	    /* if optional, indicate opt selected */
            if (optional && (optsel != 2))
            {
	 	if (!(kvf_set_attribute(selection->back_kformstruct,
                                         KVF_OPTSEL, 1)))
		    return(FALSE);
            }
            /* now, it's been recently changed */
	    selection->modified = TRUE;
	}

        /*
         * false alarm: found only a subset of letters making up variable
         */
        else if (kstrcmp(cmd_var, variable) != 0)
	    return(TRUE);

        /*
         * variable is on cmd line, but not followed by value: illegal construct
         */
        else
        {
            kerror("kforms", "kvf_modify_form_from_cmd",
                   "Invalid or missing list/cycle value after \"-%s\" argument",
                   variable);
            return(FALSE);
        }
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: modify_flag_from_cmd
|
|       Purpose: Modifies the values stored in a Flag
|                selection according to to the command string
|
|         Input: selection - pointer to the selection
|                command   - the command line string
|                override  - TRUE to over-ride recently-set values
|                                (if modified TRUE)
|                            FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output:
|       Returns: TRUE if selection is correctly updated according to
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/
static int modify_flag_from_cmd(
    kselection *selection,
    char       *command,
    int        override)
{
	char *variable;
	char temp[KLENGTH];
	
        variable = ktoken_to_string(selection->var_token);
        sprintf(temp, "-%s ", variable);

        /* 
	 * flag does not appear on cmd line: mark as not present (false)
         * if we're either overriding previous values, or if it hasn't 
         * been recently changed
	 */
	if ( ((kstrstr(command, temp)) == NULL) &&
              (!selection->modified || override) )
	{
	    /* optsel false => flag false */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_OPTSEL, 0)))
		return(FALSE);
        }

        /* 
	 * flag appears on cmd line: mark as present (true)
         * if we're either overriding previous values, or if it hasn't 
         * been recently changed
	 */
	else if (!selection->modified || override)
	{
	    /* optsel true => flag true */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                           KVF_OPTSEL, 1)))
                return(FALSE);
	
	    /* now, it's been recently changed */
            selection->modified = TRUE;
        }

	return(TRUE);
}


/*--------------------------------------------------------------
|
|  Routine Name: modify_toggle_from_cmd
|
|       Purpose: Modifies the values stored in a toggle selection 
|                according to to the command string
|
|         Input: selection - pointer to the toggle selection 
|                cmd       - the command line string
|                override  - TRUE to over-ride recently-set values
|                                (if modified TRUE)
|                            FALSE to leave recently-set values alone
|                                (if modified FALSE)
|        Output:
|       Returns: TRUE if toggle selection is correctly updated according to
|                command line, FALSE otherwise
|          Date: March 24, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/

static int modify_toggle_from_cmd(
    kselection *selection,
    char       *command,
    int        override)
{
	char *variable, *cmd_seg;
        char temp[KLENGTH];
        char stringval[KLENGTH];
	char cmd_var[KLENGTH];
	int  optional, optsel;

	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTIONAL, &optional);
	kvf_get_attribute(selection->back_kformstruct,
                          KVF_OPTSEL, &optsel);
	variable = ktoken_to_string(selection->var_token);
        sprintf(temp, "-%s ", variable);

        /*
         * variable associated w/ selection does not appear on command line
         */
	if ((cmd_seg = kstrstr(command, temp)) == NULL)
        {
            /*
             * if it's optional, and if we're either overriding
             * previous values, or if it hasn't been recently changed,
             * set optsel to false, indicating it is not selected
             */
            if ((optional) && (optsel != 2) &&
		(!selection->modified || override))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_OPTSEL, 0)))
                    return(FALSE);
            }
        }

       /* 
        * found proper construct: variable and value on command 
        * line; only update selection values if we are overriding previous 
        * values or if it hasn't been recently changed.
        */
        else if (((ksscanf(cmd_seg, "-%s \"%[^\"]", cmd_var, stringval) == 2) ||
		 (ksscanf(cmd_seg, "-%s '%[^\']", cmd_var, stringval) == 2)  ||
		 (ksscanf(cmd_seg, "-%s %s", cmd_var, stringval) == 2)) &&
		 ((!selection->modified) || (override)))
	{
            /* set the list value as specified by string value */
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_TOGGLE_VAL, stringval)))
                            return(FALSE);

	    /* if optional, indicate opt selected */
            if (optional && (optsel != 2))
            {
	 	if (!(kvf_set_attribute(selection->back_kformstruct,
                                         KVF_OPTSEL, 1)))
		    return(FALSE);
            }
            /* now, it's been recently changed */
	    selection->modified = TRUE;
	}

        /*
         * false alarm: found only a subset of letters making up variable
         */
        else if (kstrcmp(cmd_var, variable) != 0)
	    return(TRUE);

        /*
         * variable is on cmd line, but not followed by value: illegal construct
         */
        else
        {
            kerror("kforms", "kvf_modify_form_from_cmd",
                   "Invalid or missing toggle value after \"-%s\" argument",
                   variable);
            return(FALSE);
        }
	return(TRUE);
}
