/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>       Utilities for Error Checking UIS                <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>               kvf_check_for_subform_buttons()         <<<<
   >>>>               kvf_check_for_guide_buttons()           <<<<
   >>>>               kvf_check_me_member_default()           <<<<
   >>>>               kvf_check_mi_member_default()           <<<<
   >>>>               kvf_check_me_group_default()            <<<<
   >>>>               kvf_check_double()                       <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*------------------------------------------------------------
|
|  Routine Name: kvf_check_for_subform_buttons
|
|       Purpose: A MasterForm [-S to -E] definition is not
|                allowed unless there are at least 2 subform
|                buttons;  otherwise, a single subform could
|                be substituted.  This routine checks to make
|                sure that there is at least 2 subform buttons
|                in the master definition.
|
|         Input: database  - internal representation of UIS 
|                index     - index into UIS of line after -S line
|
|        Output: Returns TRUE if MasterForm def has at least
|                two subform buttons, FALSE otherwise.
|          Date: Oct 06, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
int kvf_check_for_subform_buttons(
   char **database,
   int  index)
{
	int flag = 0;

	while ((flag != KUIS_END) && (database[index] != NULL))
	{
	    flag = kvf_get_line_type(database[index]);

	    /* skip over ME/MI groups */
	    if ((flag == KUIS_MUTEXCL) || 
	        (flag == KUIS_MUTINCL) ||
		(flag == KUIS_GROUP))
	    {
		while (flag != KUIS_END)
	           flag = kvf_get_line_type(database[index++]);
		flag = kvf_get_line_type(database[index]);
	    }

	    /* skip over toggles */
	    else if (flag == KUIS_TOGGLE)
	    {
                while (flag != KUIS_END)
                   flag = kvf_get_line_type(database[index++]);
                flag = kvf_get_line_type(database[index]);
            }   

	    /* include contents of submenus */
	    else if (flag == KUIS_STARTSUBMENU)
	    {
		index++;
		if (kvf_check_for_subform_buttons(database, index))
		    return(TRUE);
                while (flag != KUIS_END)
                   flag = kvf_get_line_type(database[index++]);
                flag = kvf_get_line_type(database[index]);
	    }
	    else index++;

	    if (flag == KUIS_SUBFORMBUTTON) 
		return(TRUE);
	}
	return(FALSE);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_check_for_guide_buttons
|
|       Purpose: A GuidePane [-G to -E] definition is not
|                allowed unless there are at least 2 guide
|                buttons;  otherwise, a single pane could
|                be substituted.  This routine checks to make
|                sure that there is at least 2 guide buttons
|                in the guidepane definition.
|
|         Input: database  - internal representation of UIS 
|                index     - index into UIS of line after -G line
|
|        Output: Returns TRUE if GuidePane def has at least
|                two guide buttons, FALSE otherwise.
|          Date: Oct 07, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
int kvf_check_for_guide_buttons(
   char **database,
   int  index)
{
	int flag = 0;

	while ((flag != KUIS_END) && (database[index] != NULL))
	{
	    flag = kvf_get_line_type(database[index]);

	    /* skip over ME/MI groups */
	    if ((flag == KUIS_MUTEXCL) || 
	        (flag == KUIS_MUTINCL) ||
		(flag == KUIS_GROUP))
	    {
		while (flag != KUIS_END)
	           flag = kvf_get_line_type(database[index++]);
		flag = kvf_get_line_type(database[index]);
	    }

	    /* skip over toggles */
	    else if (flag == KUIS_TOGGLE)
	    {
                while (flag != KUIS_END)
                   flag = kvf_get_line_type(database[index++]);
                flag = kvf_get_line_type(database[index]);
            }   

	    /* include contents of submenus */
	    else if (flag == KUIS_STARTSUBMENU)
	    {
		index++;
		if (kvf_check_for_guide_buttons(database, index))
		    return(TRUE);
                while (flag != KUIS_END)
                   flag = kvf_get_line_type(database[index++]);
                flag = kvf_get_line_type(database[index]);
	    }
	    else index++;

	    if (flag == KUIS_GUIDEBUTTON) 
		return(TRUE);
	}
	return(FALSE);
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_check_me_member_default
|
|       Purpose: Checks to be sure that only one member of a
|                ME group has its OptSel field set to 1.
|
|         Input: default_set     - indicates if a default has been set
|                opt_sel         - value of opt_sel field in UIS line
|                index           - index into UIS of current line
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: Returns FALSE if UIS line is in error, TRUE otherwise.
|      KUIS_COMMENTs: called by kvf_fill_in_group, kvf_fill_in_control.
|          Date: Jun 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
--------------------------------------------------------------*/

int kvf_check_me_member_default(
   int  *default_set,
   int  opt_sel,
   int  index,
   int  *filename_lookup,
   int  *linenum_lookup,
   char **filenames)
{
	if (opt_sel == 2)
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_create_form", "Error on line %d of %s; members of mutually exclusive groups may not have their 'opt_sel' field set to 2.", linenum_lookup[index], filenames[filename_lookup[index]]);
	    return(FALSE);
	}

	else if ((*default_set) && (opt_sel)) 
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_create_form", "Error on line %d of %s; only one member of the mutually exclusive group should have its 'opt_sel' field (4th field) set to 1 (TRUE). All other lines must have their 'opt_sel' fields set to 0 (FALSE).", linenum_lookup[index], filenames[filename_lookup[index]]);
	    return(FALSE);
	}
        else if (opt_sel)
        {
	    *default_set = TRUE;
	    return(TRUE);
	}
	else return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_check_mi_member_default
|
|       Purpose: Checks to be sure that all members of a
|                MI group has their OptSel fields set to the same value.
|
|         Input: default_set     - indicates if a default has been set
|                opt_sel         - value of OptSel from UIS line
|                index           - index into UIS of current line
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: Returns FALSE if UIS line is in error, TRUE otherwise.
|      KUIS_COMMENTs: called by kvf_fill_in_group, kvf_fill_in_control.
|          Date: Jun 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
--------------------------------------------------------------*/

int kvf_check_mi_member_default(
   int  *default_set,
   int  opt_sel,
   int  index,
   int  *filename_lookup,
   int  *linenum_lookup,
   char **filenames)
{
	if (opt_sel == 2)
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_create_form", "Error on line %d of %s; members of mutually inclusive groups may not have their 'opt_sel' field set to 2.", linenum_lookup[index], filenames[filename_lookup[index]]);
	    return(FALSE);
	}

	if (*default_set != opt_sel)
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_create_form", "Error on line %d of %s; all members of the mutually inclusive group must have their 'opt_sel' field (4th field) set to the same value; either all must be TRUE (1) or all must be FALSE (0). ", linenum_lookup[index], filenames[filename_lookup[index]]);
	    return(FALSE);
	}
        else return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_check_me_group_default
|
|       Purpose: Checks to be sure that a required ME group
|                has a default set.
|
|         Input: default_set     - indicates if a default has been set
|                index           - index into UIS of current line
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: Returns FALSE if UIS line is in error, TRUE otherwise.
|      KUIS_COMMENTs: called by kvf_fill_in_group, kvf_fill_in_control.
|          Date: Jun 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/

int kvf_check_me_group_default(
   int  *default_set,
   int  index,
   int  *filename_lookup,
   int  *linenum_lookup,
   char **filenames)
{
	if (*default_set == FALSE)
	{
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_create_form", "Error in mutually exclusive group beginning with [-C] specification on line %d of %s; one member of the mutually exclusive group should have its 'opt_sel' field (4th field) set to 1 (TRUE), indicating the default of the group.", linenum_lookup[index], filenames[filename_lookup[index]]);
	    return(FALSE);
	}
	else return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_check_double
|
|       Purpose: Checks a double value to see if it is in bounds.
|
|         Input: value    - the double value to be checked
|                upper    - the upper bound
|                lower    - the lower bound
|                variable - selection variable (for CLUI error message)
|                title    - selection title (for GUI error message)
|
|        Output: Returns TRUE if integer is ok, FALSE if not.
|          Date: April 30, 1993
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/

#define OUT_OF_BOUNDS_LT0  1
#define OUT_OF_BOUNDS_LE0  2
#define OUT_OF_BOUNDS_SPEC 3
#define OUT_OF_BOUNDS_GE0  4
#define OUT_OF_BOUNDS_GT0  5
#define BOUNDS_OK          6

int kvf_check_double(
   double value,
   double upper,
   double lower,
   char   *variable,
   char   *title)
{
        int  status, bounds;
	char mesg[KLENGTH], name[KLENGTH];

        bounds = kvf_check_bounds(value, upper, lower);

	/*
         * for CLUI, use variable; 
         * for GUI, use title if not NULL, else use variable
         */
        name[0] ='\0';
        if (kvf_context == KVF_CLUI)
            ksprintf(name, "%s", variable);
        else if (title != NULL)
            ksprintf(name, "%s", title);
        else ksprintf(name, "%s", variable);
           

	if (kvf_context == KVF_CLUI)
	    ksprintf(mesg, "Invalid double %g provided for CLUI argument '%s'",
		     value, name);
	else ksprintf(mesg, "Invalid double %g provided for GUI selection '%s'",
                     value, name);

        switch (bounds)
        {
            case OUT_OF_BOUNDS_LT0:
                 kerror("kforms", "kvf_check_double",
                        "%s; double value must be < 0", mesg);
		 status = FALSE;
		 break;

            case OUT_OF_BOUNDS_LE0:
                 kerror("kforms", "kvf_check_double",
                        "%s; double value must be <= 0", mesg);
		 status = FALSE;
		 break;

            case OUT_OF_BOUNDS_SPEC:
                 kerror("kforms", "kvf_check_double",
                        "%s; double value must be greater than %g, less than %g", mesg, lower, upper);
		 status = FALSE;
		 break;

            case BOUNDS_OK:
		 status = TRUE;
		 break;

            case OUT_OF_BOUNDS_GE0:
                 kerror("kforms", "kvf_check_double",
                        "%s; double value must be >= 0.0", mesg);
		 status = FALSE;
		 break;

            case OUT_OF_BOUNDS_GT0:
                 kerror("kforms", "kvf_check_double",
                        "%s; double value must be > 0.0", mesg);
		 status = FALSE;
		 break;

            default:
		 errno = KINTERNAL;
                 kerror("kforms", "kvf_check_double",
                        "Bad status returned from kvf_check_bounds");
		 status = FALSE;
		 break;
        }
        return(status);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_check_bounds
|
|       Purpose: Checks a value to see if it is in the bounds specified.
|                Used for integers, floats, and doubles: cast to double
|
|
|         Input: value - the double value to be checked
|                upper - the upper bound
|                lower - the lower bound
|        Output: 
|       Returns: If (upper != lower) no bounds checking is done.
|                If (upper == lower == -2.0)
|                   value should be < 0
|                If (upper == lower == -1.0)
|                   value should be <= 0
|                If (upper == lower == 1.0)
|                   value should be >= 0
|                If (upper == lower == 2.0)
|                   value should be > 0
|          Date: April 10, 1994
|    Written By: Danielle Argiro
| Modifications:
|
----------------------------------------------------------------*/
int kvf_check_bounds(
   double value,
   double upper,
   double lower)
{
	int caseval, status;

    	if ((value >= lower) && (value <= upper))
            return(BOUNDS_OK);

	if (lower == upper) 
	{
	    caseval = (int) upper;
            switch (caseval) 
	    {
		case -2 :
                      if (value < 0.0) status = BOUNDS_OK;
                      else status = OUT_OF_BOUNDS_LT0;
                      break;

                case -1 :
                      if (value <= 0.0) status = BOUNDS_OK;
                      else status = OUT_OF_BOUNDS_LE0;
                      break;

                 case 1 :
                      if (value >= 0.0) status = BOUNDS_OK;
                      else status = OUT_OF_BOUNDS_GE0;
                      break;

                 case 2 :
                      if (value > 0.0) status = BOUNDS_OK;
                      else status = OUT_OF_BOUNDS_GT0;
                      break;

                default :  /* suppress bounds checking */
                      status = BOUNDS_OK;
                      break;
            }
	}
	else status = OUT_OF_BOUNDS_SPEC;

	return(status);
}


