 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>      Various Private Print Routines For the kexpr Library
   >>>>
   >>>>  Private:
   >>>>		init_variable()
   >>>>		kexpr_add_variable()
   >>>>		kexpr_check_variable()
   >>>>		kexpr_get_variable()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"
#include "y.tab.h"


/*
 * Variable declaration list
 */
static struct _constant variables[] =
{
    { "x",		0.0 },
    { "y",		1.0 },
    { NULL,		0   },
};

/*-----------------------------------------------------------
|
|  Routine Name: init_variable - initialize the kexpr parser
|
|       Purpose: This routine initializes the kexpr parser.  This call
|		 used to initialize the constants, functions, and
|		 global variable lists.  It MUST be called prior to ANY
|		 other calls to the kexpr library.
|
|         Input:
|        Output:
|	Returns:
|
|  Restrictions: 
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
|      Verified:
|  Side Effects:
| Modifications:
|
*-----------------------------------------------------------*/

static void init_variable(void)
{
	static int initialized = FALSE;

	if (initialized == TRUE)
	   return;

	/*
	 * Install the varlist for Constants, Functions, and Global
	 * variables.
	 */
	initialized = TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_add_variable()
|
|       Purpose: kexpr_add_variable() is called when we want to add a
|		 variable, or constant to the symbol list.
|		 The variable symbol does not need to be copied, since
|		 we want to share variables within the system.  The
|		 current symbol list is passed in and the number is
|		 then tacked onto the end of the symbol list.  If the
|		 symbol list is currently NULL then we will
|		 create a new list and add the variable to it.
|
|        Input:  symlist  - the current symbol list
|		 variable - the symbol to be added to the list
|
|       Output:  returns the new symbol list.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

SymbolList *kexpr_add_variable(
   SymbolList *symlist,
   Symbol     *variable)
{
	char	   error[KLENGTH];
	SymbolList *list, *entry;


	/*
	 *  Allocate the entry for our new variable
	 */
	if ((entry = (SymbolList *) kmalloc(sizeof(SymbolList))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
reference list for variable '%s'", variable->name);
	   kexpr_error(error);
	}
	entry->symbol = variable;
	entry->next   = NULL;

	/*
	 *  If the current symlist is NULL then return entry as our symlist.
	 */
	if (symlist == NULL)
	{
	   return(entry);
	}

	/*
	 *  The current symbol list is not empty so add the entry to the
	 *  end of the current list.
	 */
	list = symlist;
	while (list->next != NULL)
	   list = list->next;

	list->next = entry;
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_check_variable()
|
|       Purpose: kexpr_check_variable() is called when we want to
|		 determine if a variable is an expression or
|		 a dependent variable expression.  When defining
|		 a variable the user may want that variable to
|		 actually be the value of that expression, but
|		 sometimes they may want it to be symbols that
|		 represent that variable.  For instance,
|
|		    x = 10     (EXPRESSION)
|
|	 	 this is an expression, since it is not dependent
|		 on any other variables.  But if we had used
|
|		    x = y + z  (VARIABLE)
|
|		 then we say it is a variable dependent expression,
|		 since if y or z were to change then we would like
|		 x to also change.
|		
|
|        Input:  variable   - the current variable symbol
|
|       Output:  None, but it will set the type of variable depending if
|		 the symbol is an expression or variable dependent
|		 expression.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

void kexpr_check_variable(
   Symbol *variable)
{
	int	   type;
	Symbol	   *symbol;
	SymbolList *symlist;


	/*
	 *  If the current variable is NULL then just return
	 */
	if (variable == NULL)
	   return;

	/*
	 *  Race thru the variable's symbol list and see if the variable
	 *  is an expression or dependent expression.
	 */
	type = EXPRESSION;
	symlist = variable->symlist;
	while (symlist != NULL)
	{
	   symbol = symlist->symbol;
	   if (symbol == variable)
	   {
	      variable->type = EXPRESSION;
	      return;
	   }
	   else if ((symbol->type == VARIABLE || symbol->type == EXPRESSION ||
		symbol->type == UNDEFINED) && variable != symbol)
	   {
	      type = VARIABLE;
	   }
	   symlist = symlist->next;
	}
	variable->type = type;
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_get_variable - get variable symbol
|
|       Purpose: kexpr_get_variable() - pulls variable name from
|		 the expression string.
|
|         Input: string - the string which contains the desired function
|
|        Output: None, except that the resulting variable is stored
|		 in the global kexpr_lval.symbol.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/

Symbol *kexpr_get_variable(
   char *string,
   int  parent)
{
	Symbol  *symbol = NULL;
	Varlist *varlist, *currlist;


	/*
	 * Check to see if it is a current variable list.
	 */
	init_variable();
	if ((currlist = kexpr_get_varlist(current_id)) != NULL)
	{
	   if ((symbol = kexpr_get_symbol(currlist, string, parent)) != NULL)
	   {
	      return(symbol);
	   }
	}

	/*
	 * Check to see if it is in the global variable list.
	 */
	if ((varlist = kexpr_get_varlist(GLOBAL_ID)) != NULL)
	{
	   if ((symbol = kexpr_get_symbol(varlist, string, parent)) == NULL)
	   {
	      symbol = kexpr_add_symbol(currlist, string, UNDEFINED, 0.0);
	      return(symbol);
	   }
	}
	return(symbol);
}
