 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>      Various Private Utility Routines For the kexpr Library
   >>>>
   >>>>  Private:
   >>>>             kexpr_get_varlist()
   >>>>             kexpr_delete_varlist()
   >>>>             kexpr_get_symbol()
   >>>>             kexpr_add_symbol()
   >>>>             kexpr_delete_symbol()
   >>>>             kexpr_free_symbol()
   >>>>             kexpr_free_symlist()
   >>>>             kexpr_eval_expression()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"
#include "y.tab.h"

static Varlist  *var_head = NULL;

int	current_id;
jmp_buf	jump_dest;
int    	error_found = FALSE;
int    	jump_dest_enabled = FALSE;
Symbol  FinalValue, *CompiledSymbol;
char	*sptr, *string_array, *kexpr_error_string;


/*-----------------------------------------------------------
|
|  Routine Name: kexpr_get_varlist - get the varlist
|
|       Purpose: kexpr_get_varlist() gets the varlist associated
|		 for a particular id.  If no varlist exists for the
|		 id then a list is created and added to the "var_head".
|
|         Input: id - the variable identifier
|
|        Output: RETURNS: returns the varlist on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/


Varlist *kexpr_get_varlist(
   long id)
{
	Varlist *varlist;


	varlist = var_head;
	while (varlist != NULL)
	{
	   if (varlist->id == id)
	      break;

	   varlist = varlist->next;
	}

	if (varlist == NULL)
	{
	   if ((varlist = (Varlist *) kmalloc(sizeof(Varlist))) == NULL)
	      return(NULL);

	   varlist->id	    = id;
	   varlist->symlist = NULL;
	   varlist->parent  = NULL;
	   varlist->next    = var_head;
	   var_head         = varlist;
	}
	return(varlist);
}



/*-----------------------------------------------------------
|
|  Routine Name: kexpr_delete_varlist - delete the varlist
|
|       Purpose: kexpr_delete_varlist() deletes the varlist associated
|		 with a particular id.
|
|         Input: id - the variable identifier
|
|        Output: RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/


int kexpr_delete_varlist(
   long id)
{
	Varlist *varlist, *list;


	list = varlist = var_head;
	while (list != NULL)
	{
	   if (list->id == id)
	      break;

	   varlist = list;
	   list = list->next;
	}

	/*
	 *  If we don't find the id in the current list then just return
	 */
	if (list == NULL)
	   return(FALSE);

	/*
	 *  If the variable list is at the head then advance 'var_head' to the
	 *  next variable list, otherwise we just link around the variable list.
	 */
	if (varlist == list)
	   var_head = list->next;
	else
	   varlist->next = list->next;

	/*
	 *  Make sure that none of the other variable list use the deleted
	 *  varlist as a parent.
	 */
	varlist = var_head;
	while (varlist != NULL)
	{
	   if (varlist->parent == list)
	      varlist->parent = NULL;

	   varlist = varlist->next;
	}
	kexpr_free_symlist(list->symlist);
	kfree(list);

	return(TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: kexpr_get_symbol - get a symbol
|
|       Purpose: kexpr_get_symbol() gets the symbol associated with
|		 a particular varlist and name identifier.  If this
|		 list does not contain the particular symbol and
|		 the varlist is a child then the kexpr_get_symbol()
|		 is recursed with the parent's varlist.
|
|         Input: varlist - the list of variable list from which
|			   we search the symbol
|		 name    - the name of the symbol we are looking
|			   for.
|		 parent  - whether we should be allowed to search
|			   the parent's varlist if one exists.
|
|        Output: RETURNS: returns the symbol on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/


Symbol *kexpr_get_symbol(
   Varlist *varlist,
   char    *name,
   int     parent)
{
	Symbol	   *symbol;
	SymbolList *symlist;


	/*
	 *  Race through the symbol list looking to see if we can find the
	 *  symbol.
	 */
	symlist = varlist->symlist;
	while (symlist != NULL)
	{
	   symbol = symlist->symbol;
	   if (kstrcmp(name, symbol->name) == 0)
	      return(symbol);

	   symlist = symlist->next;
	}

	/*
	 *  If the list doesn't have the symbol then try list's parent
	 *  if one exists.
	 */
	if (varlist->parent != NULL && parent == TRUE)
	   symbol = kexpr_get_symbol(varlist->parent, name, parent);
	else
	   symbol = NULL;

	return(symbol);
}



/*-----------------------------------------------------------
|
|  Routine Name: kexpr_add_symbol - add a symbol to a varlist
|
|       Purpose: kexpr_add_symbol() adds the symbol associated
|		 with a particular varlist and name identifier.
|
|         Input: varlist - the list of variable list to which
|			   we add the symbol
|		 name    - the name of the symbol we are adding
|                type    - the type of the symbol we are adding
|                value   - the current value of that symbol
|
|        Output: RETURNS: returns the symbol on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/


Symbol *kexpr_add_symbol(
   Varlist *varlist,
   char    *name,
   int     type,
   double  value)
{
	Symbol	    *symbol;
	SymbolList  *symlist;


	if ((symlist = (SymbolList *) kmalloc(sizeof(SymbolList))) == NULL)
	   return(NULL);

	if ((symbol = (Symbol *) kcalloc(1, sizeof(Symbol))) == NULL)
	   return(NULL);

	symbol->name =  kstring_copy(name, NULL);
	symbol->type =  type;
	symbol->eval =  FALSE;
	symbol->Value = value;

	symlist->symbol  = symbol;
	symlist->next    = varlist->symlist;
	varlist->symlist = symlist;
	return(symbol);
}



/*-----------------------------------------------------------
|
|  Routine Name: kexpr_delete_symbol - delete a symbol from a varlist
|
|       Purpose: kexpr_delete_symbol() deletes the symbol associated
|		 with a particular varlist and name identifier.  If
|		 this list does not contain the particular symbol then
|		 we return false, otherwise we delete it from the varlist
|		 and then the kexpr_get_symbol() is recursed with the
|		 parent's varlist.
|
|         Input: id     - the id from which the symbol came from
|		 symbol - the name of the symbol we are adding
|
|        Output: RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/


int kexpr_delete_symbol(
   int    id,
   Symbol *symbol)
{
	Varlist	   *varlist;
	SymbolList *symlist, *list;

	/*
	 * Check to see if it is in the id varlist list.
	 */
	if ((varlist = kexpr_get_varlist(id)) == NULL)
	   return(FALSE);

	list = symlist = varlist->symlist;;
	while (list != NULL)
	{
	   if (list->symbol == symbol)
	      break;

	   symlist = list;
	   list = list->next;
	}

	if (list == NULL)
	   return(FALSE);

	if (symlist == list)
	   varlist->symlist = list->next;
	else
	   symlist->next = list->next;

	list->next = NULL;
	kexpr_free_symlist(list);
	return(TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: kexpr_free_symbol - free a symbol
|
|       Purpose: kexpr_free_symbol() is called when we want to free
|		 the contents of a symbol, but not the symbol itself.
|		 We do this before re-using the symbol.
|
|         Input: symbol - the current symbol to be freed
|
|        Output: None
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/


void kexpr_free_symbol(
   Symbol *symbol)
{
	Symbol     temp;
	SymbolList list;


	/*
	 *  If the current symbol is NULL then just return
	 */
	if (symbol == NULL)
	   return;

	/*
	 *  Free all the resources that are allocated with the symbol,
	 *  ie). the symlist and the string (if the symbol is of type
	 *  string).
	 */
	if (symbol->type == STRING && symbol->String != NULL)
	   kfree(symbol->String);
	else if (symbol->type != UNDEFINED && symbol->symlist != NULL &&
	         (symbol->type == VARIABLE || symbol->type == EXPRESSION))
	{
	   
	   list.symbol = symbol;
	   list.next   = NULL;
	   kexpr_eval_symlist(&list, FALSE, &temp);
	   symbol->Value = temp.Value;
	}

	if (symbol->symlist != NULL)
	   kexpr_free_symlist(symbol->symlist);

	symbol->symlist = NULL;
}



/*-----------------------------------------------------------
|
|  Routine Name: kexpr_free_symlist - free a symbol list
|
|       Purpose: kexpr_free_symlist() is called when we want to free
|		 the contents of a symbol list.
|
|         Input: symbol - the current symbol to be freed
|
|        Output: None
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/


void kexpr_free_symlist(
   SymbolList *symlist)
{
	Symbol	   *symbol;
	SymbolList *temp;


	/*
	 *  Free all the resources that are allocated with the symlist,
	 *  ie). the symbol and the symlist itself
	 */
	while (symlist != NULL)
	{
	   symbol = symlist->symbol;
	   if (symbol->type == NUMBER || symbol->type == STRING ||
	       symbol->type == OPERATOR)
	   {
	      kexpr_free_symbol(symbol);
	      kfree(symbol);
	   }
	   temp = symlist->next;
	   kfree(symlist);
	   symlist = temp;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_eval_expression - evaluate an expression
|
|       Purpose: kexpr_eval_expression() evaluates the string
|		 and returns whether we succesfully evaluated
|		 that expression.
|
|         Input: id     - the variable identifier
|		 string - the string to be evaluated
|
|        Output: error  - if an error occurred the message is
|			  stored in the string array and False
|			  returned.  If this field is NULL then
|			  the error message is printed to kstderr.
|		 RETURNS: returns the varlist on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/

int kexpr_eval_expression(
   long id,
   char *string,
   char *error)
{
	error_found = FALSE;
	kexpr_error_string = error;
	if (setjmp(jump_dest) == 0)
	{
	   current_id = id;
	   jump_dest_enabled = TRUE;
	   string_array = sptr = string;
	   (void) kexpr_parse();
	}
	jump_dest_enabled = FALSE;
	return(error_found == FALSE);
}
