 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>      Various Private Print Routines For the kexpr Library
   >>>>
   >>>>  Private:
   >>>>		kexpr_add_function()
   >>>>		kexpr_check_function()
   >>>>		kexpr_get_function()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"
#include "y.tab.h"

/*
 * Function declaration list
 */
static struct _function functions[] =
{
    { "sin",	  ksin,	       1 },
    { "cos",	  kcos,	       1 },
    { "tan",	  ktan,	       1 },

    { "sinh",	  ksinh,       1 },
    { "cosh",	  kcosh,       1 },
    { "tanh",	  ktanh,       1 },

    { "asin",	  kasin,       1 },
    { "acos",	  kacos,       1 },
    { "atan",	  katan,       1 },
    { "atan2",	  katan2,      2 },
    { "sinc",	  ksinc,       1 },

    { "asinh",	  kasinh,      1 },
    { "acosh",	  kacosh,      1 },
    { "atanh",	  katanh,      1 },

    { "ln",	  klog,	       1 },
    { "log2",	  klog2,       1 },
    { "log10",	  klog10,      1 },
    { "log1p",	  klog1p,      1 },

    { "exp",	  kexp,	       1 },
    { "exp2",	  kexp2,       1 },
    { "exp10",	  kexp10,      1 },
    { "expm1",	  kexpm1,      1 },

    { "pow",	  kpow,        2 },
    { "sqrt",	  ksqrt,       1 },
    { "cbrt",	  kcbrt,       1 },

    { "abs",	  kfabs,       1 },
    { "trunc",	  ktrunc,      1 },
    { "fmod",	  kfmod,       2 },
    { "ldexp",	  kldexp,      2 },

    { "floor",	  kfloor,      1 },
    { "ceil",	  kceil,       1 },
    { "fact",	  kfact,       1 },

    { "impulse",  kimpulse,    1 },
    { "step",	  kstep,       1 },
    { "sign",	  ksign,       1 },
    { "erf",	  kerf,        1 },
    { "erfc",	  kerfc,       1 },
    { "gamma",	  kgamma,      1 },

    { "j0",	  kj0,	       1 },
    { "j1",	  kj1,	       1 },
    { "y0",	  ky0,         1 },
    { "y1",	  ky1,	       1 },

    { "hypot",	  khypot,      2 },
    { "frexp",	  kfrexp,      2 },
    { "urng",	  kurng,       0 },

    { NULL,	  0,	       0 },
};

/*-----------------------------------------------------------
|
 | Routine Name: init_function - initialize the kexpr parser
|
|       Purpose: This routine initializes the kexpr parser.  This call
|		 used to initialize the constants, functions, and
|		 global variable lists.  It MUST be called prior to ANY
|		 other calls to the kexpr library.
|
|         Input:
|        Output:
|	Returns:
|
|  Restrictions: 
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
|      Verified:
|  Side Effects:
| Modifications:
|
*-----------------------------------------------------------*/

static void init_function(void)
{
	int	i;
	Symbol  *symbol;
	Varlist *varlist;
	static int initialized = FALSE;


	if (initialized == TRUE)
	   return;

	/*
	 * Install the varlist for Constants, Functions, and Global
	 * variables.
	 */
	initialized = TRUE;
	if ((varlist = kexpr_get_varlist(FUNCTION_ID)) != NULL)
	{
	   for (i = 0; functions[i].name != NULL; i++)
	   {
	       symbol = kexpr_add_symbol(varlist, functions[i].name, FUNCTION,
				0.0);
	       symbol->num     = functions[i].num_args;
	       symbol->Function = functions[i].routine;
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_add_function()
|
|       Purpose: kexpr_add_function() is called when we want to add a
|		 function to the symbol list.  The function symbol
|		 does not need to be copied, since we want to share
|		 functions within the system.  The current symbol list
|		 is passed in and the number is then tacked onto the
|		 end of the symbol list.  If the symbol list is currently
|		 NULL then we will create a new list and add the variable
|		 to it.
|
|        Input:  symlist    - the current symbol list
|		 function   - the symbol to be added to the list
|		 num_args   - the number of arguments specified with
|			      the function.
|
|       Output:  returns the new symbol list.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

SymbolList *kexpr_add_function(
   SymbolList *symlist,
   Symbol     *function,
   int	      num_args)
{
	char	   error[KLENGTH];
	SymbolList *list, *entry;


	/*
	 *  Allocate the entry for our new function
	 */
	if (function->num != num_args && function->type != UNDEFINED)
	{
	   (void) sprintf(error, "Error! Incorrect number of arguments \
specified.  Function '%s' takes %d arguments, but you specified %d.",
			function->name, (int) function->num, num_args);
	   kexpr_error(error);
	}

	/*
	 *  Allocate the entry for our new function
	 */
	if ((entry = (SymbolList *) kmalloc(sizeof(SymbolList))) == NULL)
	{
	   (void) sprintf(error, "Error! Ran out of memory trying to allocate \
reference list for function '%s'", function->name);
	   kexpr_error(error);
	}
	entry->symbol = function;
	entry->next   = NULL;

	/*
	 *  If the current symlist is NULL then return entry as our symlist.
	 */
	if (symlist == NULL)
	{
	   return(entry);
	}

	/*
	 *  The current symbol list is not empty so add the entry to the
	 *  end of the current list.
	 */
	list = symlist;
	while (list->next != NULL)
	   list = list->next;

	list->next = entry;
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_check_function()
|
|       Purpose: kexpr_check_function() is called when we want to
|		 create a user defined function.  The main
|		 purpose of this routine is make the function's
|		 argument list be local to only the function and
|		 not global to the rest of the system.
|
|        Input:  symbol   - the current variable symbol
|
|       Output:  None.  Make the functions symlist have only local
|		 variables and then update the function expression
|		 to use these local variables.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
|----------------------------------------------------------*/

void kexpr_check_function(
    Symbol     *function,
    SymbolList *arglist,
    SymbolList *symlist)
{
	SymbolList *list;
	Symbol	   *symbol, *temp;
	int	   num_args = 0;
	char	   error[KLENGTH];


	/*
	 *  If the current function is NULL then just return
	 */
	if (function == NULL)
	   return;

	/*
	 *  Change all function arguments from global to local variables.
	 *  At the same time change the expression to use these local
	 *  variables.
	 */
	while (arglist != NULL)
	{
	   symbol = arglist->symbol;
	   if (symbol->type == VARIABLE || symbol->type == EXPRESSION ||
	       symbol->type == UNDEFINED)
	   {
	      /*
	       *  Allocate the symbol to store our local variable
	       */
	      if ((temp = (Symbol *) kcalloc(1,sizeof(Symbol))) == NULL)
	      {
	         (void) sprintf(error, "Error! Ran out of memory trying to \
allocate local argument '%s' for function '%s' ", symbol->name, function->name);
		 kexpr_error(error);
	      }
	      temp->Value = 0.0;
	      temp->type  = EXPRESSION;
	      temp->name  = kstring_copy(symbol->name, NULL);
	      arglist->symbol = temp;

	      list = symlist;
	      while (list != NULL)
	      {
		 if (list->symbol == symbol)
		    list->symbol = temp;

	         list = list->next;
	      }
	   }
	   arglist = arglist->next; num_args++;
	}
	function->num = num_args;
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_get_function - get function symbol
|
|       Purpose: kexpr_get_function() - pulls function name from
|		 the expression string.
|
|         Input: string - the string which contains the desired function
|
|        Output: None, except that the resulting function is stored
|		 in the global kexpr_lval.symbol.
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/

Symbol *kexpr_get_function(
  char	*string)
{
	Varlist *varlist;
	Symbol  *symbol = NULL;


	/*
	 * Check check to see if it is a defined function
	 */
	init_function();
	if ((varlist = kexpr_get_varlist(FUNCTION_ID)) != NULL)
	{
	   if ((symbol = kexpr_get_symbol(varlist, string, TRUE)) != NULL)
	   {
	      return(symbol);
	   }
	}

	/*
	 * Check to see if it is in the current varlist list.
	 */
	if ((varlist = kexpr_get_varlist(current_id)) != NULL)
	{
	   if ((symbol = kexpr_get_symbol(varlist, string, TRUE)) == NULL)
	   {
	      symbol = kexpr_add_symbol(varlist, string, UNDEFINED, 0.0);
	      return(symbol);
	   }
	}
	return(symbol);
}
