/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>              Routines to Generate *.3 file            <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kgen_clui_generate_man3file()          <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*---------------------------------------------------------------
|
|  Routine Name: kgen_clui_generate_man3file
|
|       Purpose: Creates the man3 file for the library as a whole
|
|         Input: library   -  library object
|                prog_spec -  pointer to the internal PS struct
|                opath     -  path to the library object
|                tb_name   -  name of toolbox
|		 name      -  name of library object being generated
|
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro and Steve Jorgensen
|          Date: Nov 19, 1993 
| Modifications: 
|
--------------------------------------------------------------*/

int  kgen_clui_generate_man3file(
   kobject library,
   char    **prog_spec,
   char    *opath,
   char    *tb_name,
   char    *oname)
{
        kstring      routine     = "kgen_clui_generate_man3file()";
        kobject      toolbox     = NULL;
	kobject      file_object = NULL;
	int          force_flag  = FALSE;
	time_t       clck;
	struct tm   *tmval;
	kfile       *file;
        kstring      filepath;
	kstring      copyright   = NULL;
	char         temp[KLENGTH], date[KLENGTH], directory[KLENGTH];
	header_info *header;
        klist       *headerlist;
	int          isnewfile   = FALSE;
	kstring      short_description = NULL;

        /*
         * get file object representing man3 page
         */
	if (!kcms_get_attributes(library,
				 KCMS_CMOBJ_GEN_MAN3,   &file_object,
				 KCMS_CMOBJ_SHORT_DESC, &short_description,
				 KCMS_END)
	    || !kcms_query_bit(library, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE,
			       &force_flag))
	    return(FALSE);

        /* 
         * 1st time man3 page has been created - create file object, so that 
         * kcms registers the man3 page in the kcms database
         */
	if (file_object == NULL)
        {
            ksprintf(temp, "%s/man/%s.3", opath, oname);
            file_object = kcms_create_fileobj(library, temp, NULL,
                                              KCMS_FOBJ_TYPE_MAN,
                                              KCMS_FOBJ_SUBTYPE_MAN3,
                                              KCMS_FOBJ_GEN_MAN3,
                                              KCMS_FOBJ_ACCESS_RDWR);
            if (file_object == NULL)
                return(FALSE);
            isnewfile = TRUE;
            filepath = temp;
	}

	/*
         * the man3 file has already been created & added to kcms database;
         * simply get the path to the man3 page.
         */
	else if (!kcms_get_attribute(file_object, KCMS_PATH, &filepath))
            return(FALSE);

	/*
         *  if somehow the database GHOST_MAN3 key got deleted from the kcms
         *  database, but the man1 file really is there, don't want to
         *  clobber it!
         */
        if ((isnewfile) && (kaccess(filepath, R_OK) == 0))
        {
            kerror(KCODEGEN, "kgen_clui_generate",
                   "WARNING!  CMS database appears to have been corrupted;  cms dbm key GHOST_MAN3 is missing, but the %s man3 does exist.  Regeneration of man3 file will NOT take place. Adding GHOST_MAN3 cms key to database; next run of ghostwriter should be OK.", oname);
	    return(FALSE);
        }

        /*
         * get the directory to the man3 file.  if the directory does
         * not exist, it will get created by kmake_dir();  if kmake_dir()
         * cannot create it, it will return the ENOENT error.
         */
        (void) kdirname(filepath, directory);
        if (!kmake_dir(directory, 0777) && errno != ENOENT)
        {
             kerror(KCODEGEN, "kgen_clui_generate_man3file",
                    "Can't create directory %s in which to deposit man3 file",
                    directory);
             return(FALSE);
        }

	/*
	 *  see if man3 file already exists.  If so, prompt to over-write,
	 *  return TRUE if the answer is NO
	 */
	if (!force_flag &&
	    koverwrite(kget_notify(), filepath) == FALSE)
         return(TRUE);

	/*
	 *  create & open man3 file  
	 */
        if (kaccess(filepath, R_OK) != 0)
	   isnewfile = TRUE;
	if ((file = kfopen(filepath, "w"))== NULL)
	{
	     kerror(KCODEGEN, "kgen_clui_generate_man3file",
	            "Could not create file '%s'", filepath);
	     return(FALSE);
	}
	
	/*
	 * print out first line of man3 page with time & date
	 */
	clck = time(NULL);
	tmval = localtime(&clck);
	if (tmval != NULL)
	{
	     strftime( date, KLENGTH, "%h %d, %Y", tmval);
	}
	
	kfprintf(file, ".TH \"%s\" \"%s\" \"COMMANDS\" \"\" \"%s\"\n",
	         oname, kstring_upper(tb_name, NULL), date);
	
	/* 
	 * print out name & description
	 */
	kfprintf(file, ".SH NAME\n");
	if (short_description == NULL)
	   short_description = "(no short description specified)";
	kfprintf(file, "%s \\- %s\n", oname, short_description);
	
	/*
  	 * print out description
	 */
	kfprintf(file, ".SH DESCRIPTION\n");
	if (kstrlen(prog_spec[MAN3_LONGDESC]) > 0) 
	    kfprintf(file, "%s\n", prog_spec[MAN3_LONGDESC]);
	
	/* 
	 * print out the list of library functions
	 */
	kfprintf(file, ".SH \"LIST OF LIBRARY FUNCTIONS\"\n");
	kcms_get_attribute(library, KCMS_CMOBJ_HEADERLIST, &headerlist);
	while (headerlist!= NULL)
        {
	    header = (header_info *) headerlist->client_data;
	    if (header->name != NULL)
		kfprintf(file, ".whatis %s %s\n", oname, header->name);
	    headerlist = klist_next(headerlist);
        }

	
	/* 
	 * print out additional information
	 */
	kfprintf(file, ".SH \"ADDITIONAL INFORMATION\"\n");
	if (kstrlen(prog_spec[MAN3_ADDINFO]) > 0) 
	    kfprintf(file, "%s\n", prog_spec[MAN3_ADDINFO]);
	
	/* 
	 * print out files
	 */
	kfprintf(file, ".SH \"LOCATION OF SOURCE FILES:\"\n");
	kfprintf(file, "$%s/objects/library/%s/src\n", 
		 kstring_upper(tb_name, NULL), oname);

	kfprintf(file, ".SH \"LOCATION OF PUBLIC INCLUDE FILE:\"\n");
	kfprintf(file, "$%s/include/%s/%s.h\n", 
		 kstring_upper(tb_name, NULL), oname, oname);

	kfprintf(file, ".SH \"YOU MUST INCLUDE:\"\n");
	kfprintf(file, "#include <%s.h>\n", kstring_lower(tb_name, NULL));
	
	/* 
	 * print out "see also"
	 */
	kfprintf(file, ".SH \"SEE ALSO\"\n");
	if (kstrlen(prog_spec[MAN3_SEEALSO]) > 0) 
	   kfprintf(file, "%s\n", prog_spec[MAN3_SEEALSO]);
	
	/* 
	 * print out restrictions
	 */
	kfprintf(file, ".SH \"SEE MANUAL\" \n");
	if (kstrlen(prog_spec[MAN3_MANUAL]) > 0) 
	   kfprintf(file, "%s\n", prog_spec[MAN3_MANUAL]);

	kfprintf (file, ".SH COPYRIGHT\n");
	

	/*
	 * include the copyright
	 */
	if ((toolbox = kcms_open_toolbox(tb_name)) == NULL)
	{
	   if (tb_name == NULL)
	      kerror(KCODEGEN, "kgen_clui_generate_man3file",
	          "Cannot get copyright for man 3 file, NULL input toolbox.  "
		  "Copyright will be left blank in the man page");
	   else
	      kerror(KCODEGEN, "kgen_clui_generate_man3file",
	          "Cannot open toolbox '%s' to get copyright for man 3 file.  "
		  "Copyright will be left blank in the man page",
		  tb_name);
	}
	else
	{
	   copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_SHORT,
				          KCMS_LANG_NONE);
	   kcms_close(toolbox);
	}
	if (copyright != NULL)
	{
	    kfprintf(file, "%s\n", copyright);
	    kfree(copyright);
	}

	kfclose(file);
	kannounce(KCODEGEN, routine, "done generating %s man page", oname);
	
	
	/*
	 *  Add the manpage to the toolbox manpage database
	 */
	if (filepath != NULL)
	{
	   kdbm   *dbm;
	   kdatum key, data;
	
	   ksprintf(temp, "$%s/repos/db/manpage", tb_name);
	   if ((dbm = kdbm_open(temp, O_WRONLY | O_CREAT, 0666)) != NULL)
	   {
	      key.dptr   = oname;
  	      key.dsize  = kstrlen(key.dptr) + 1;
	      data.dptr  = filepath;
	      data.dsize = kstrlen(data.dptr) + 1;
	      kdbm_store(dbm, key, data, KDBM_REPLACE);
	      kdbm_close(dbm);
	   }
	}
	return(TRUE);
}


