/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>             kcms_env_getpath()
   >>>>             kcms_env_addtb()
   >>>>             kcms_env_deletetb()
   >>>>   Public:
   >>>>             kcms_query_toolboxes()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

#define DEFAULT_TOOLBOX	     "BOOTSTRAP"
#define DEFAULT_TOOLBOX_FILE "$BOOTSTRAP/repos/Toolboxes"


/*-----------------------------------------------------------
| Routine Name:	kcms_env_getpath - get a path entry from a specific toolbox
|
| Purpose:	This subroutine returns the path assoicated with
|		a specific toolbox.  The path is part of the global
|		structure used internally by the khoros routines
|		and should be treated as a read-only resource.
|
| Input:	variable - the name of the toolbox
|
| Output:	none
| Returns:	The cooresponding toolbox path, or NULL if
|		not found or on error
|
| Written By:	Mark Young
| Date:		Jul 03, 1992
------------------------------------------------------------*/
char *
kcms_env_getpath(
   char  *variable)
{
   char  cleanup[KLENGTH];


   ktoolbox_file_initialize();
   if (!kstring_cleanup(variable, cleanup))
      return (NULL);

   kstring_upper(cleanup, cleanup);
   return kstrdup(kgetenv(cleanup));
}

/*-----------------------------------------------------------
| Routine Name:	kcms_env_addtb - add a new toolbox to the list.
|
| Purpose:	This subroutine adds a new toolbox to the global
|		toolbox list.  This routine takes all the information
|		required to store a Khoros Toolbox entry.
|
| Input:	variable    - the name of the toolbox
|		path        - the path for the top directory of the toolbox
|
| Output:	none
| Returns:	return TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Mark Young
| Date:		Jul 03, 1992
------------------------------------------------------------*/
int 
kcms_env_addtb(
   char  *variable,
   char  *path)
{
   kstring  routine           = "kcms_env_addtb()";
   kfile    *file;
   int      i;
   int      num;
   int      status;
   int      written            = FALSE;
   char     *filenames;
   char     *tmp;
   char     **list;
   char     tbname[KLENGTH];
   char     tbvar[KLENGTH];
   char     tbpath[KLENGTH];
   char     temp[KLENGTH];
   char     filename[KLENGTH];


   if ((filenames = (char *)kgetenv("KHOROS_TOOLBOX")) == NULL)
   {
      kerror(KCMS, routine, "Environment variable KHOROS_TOOLBOX not set.\n"
             "This is a required environment variable.  Setting it to %s\n",
             DEFAULT_TOOLBOX_FILE);
      filenames = DEFAULT_TOOLBOX_FILE;
   }

   kstring_cleanup(path, tbpath);
   kstring_cleanup(variable, tbvar);
   kstring_upper(tbvar, tbvar);
   while (filenames != NULL)
   {
      if ((tmp = kstrchr(filenames, ':')) != NULL)
      {
	 kstring_ncopy(filenames, tmp - filenames, filename);
	 tmp++;
      }
      else
	 kstring_copy(filenames, filename);


      /* KHOROS_TOOLBOX variable cannot be opened */
      filenames = tmp;
      if ((file = kfinput(filename)) != NULL)
      {
	 list = kparse_file_scan_delimit(file, KPARSE_BOF, KPARSE_EOF,
				 KLITERAL, "\n", NULL, NULL, &num, &status);
	 kfclose(file);
      }
      else
      {
	 num = 0;
	 list = NULL;
      }

      /* KHOROS_TOOLBOX variable cannot be opened */
      if ((file = kfoutput(filename)) == NULL)
      {
	 if (!written)
	    kwarn(KCMS, routine, "Unable to open toolbox file '%s' "
		  "for writing, so could not add reference for toolbox `%s'.",
		  filename, tbvar);
	 continue;
      }

      for (i = 0; i < num; i++)
      {
	 if (ksscanf(list[i], "%[^#:]:", tbname) == 1 &&
	     kstrcasecmp(tbname, tbvar) == 0)
	 {
	    written = TRUE;
	    kfprintf(file, "%s:%s\n", tbvar, tbpath);
	 }
	 else
	    kfprintf(file, "%s\n", list[i]);
      }

      if (!written)
      {
	 written = TRUE;
	 kfprintf(file, "%s:%s\n", tbvar, tbpath);
      }
      kfclose(file);
      karray_free(list, num, NULL);
   }
   ksprintf(temp, "%s=%s", tbvar, tbpath);
   kputenv(temp);
   return written;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_env_deletetb - delete a toolbox entry from the list
|
| Purpose:	This routine removes a toolbox entry from the
|		global toolbox list.  The toolbox name is taken as
|		an input parameter.
|
| Input:	variable - the name of the toolbox to delete
|
| Output:	none
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Mark Young
| Date:		Jul 03, 1992
------------------------------------------------------------*/
int
kcms_env_deletetb(
   char     *variable)
{
   kstring  routine  = "kcms_env_deletetb()";
   kfile    *file;
   int      i;
   int      num;
   int      status;
   char     *filenames;
   char     *tmp;
   char     **list;
   char     tbname[KLENGTH];
   char     tbvar[KLENGTH];
   char     filename[KLENGTH];
   int      mentioned;


   if ((filenames = (char *)kgetenv("KHOROS_TOOLBOX")) == NULL)
   {
      kwarn(KCMS, routine, "Environment variable KHOROS_TOOLBOX not set.\n"
            "This is a required environment variable.\n"
            "Setting it to %s\n", DEFAULT_TOOLBOX_FILE);
      filenames = DEFAULT_TOOLBOX_FILE;
   }

   kstring_cleanup(variable, tbvar);
   kstring_upper(tbvar, tbvar);
   while (filenames != NULL)
   {
      if ((tmp = kstrchr(filenames, ':')) != NULL)
      {
	 kstring_ncopy(filenames, tmp - filenames, filename);
	 tmp++;
      }
      else
	 kstring_copy(filenames, filename);


      /* KHOROS_TOOLBOX variable cannot be opened */
      filenames = tmp;
      if ((file = kfinput(filename)) == NULL)
      {
	 kerror(KCMS, routine, "Unable to open toolbox file '%s' "
                "for reading.", filename);
	 continue;
      }

      list = kparse_file_scan_delimit(file, KPARSE_BOF,
		                      KPARSE_EOF, KLITERAL, "\n", NULL, NULL,
                                      &num, &status);
      if (list == NULL)
      {
	 kerror(KCMS, routine, "Cannot read contents of toolbox file '%s'",
                filename);
	 continue;
      }
      kfclose(file);

      /*-- is the toolbox mentioned? ----------------------------------*/
      mentioned = FALSE;
      for (i = 0; i < num; i++)
      {
	 if (ksscanf(list[i], "%[^#:]:", tbname) != 1 ||
             kstrcasecmp(tbname, tbvar) == 0)
	 {
	    mentioned = TRUE;
	 }
      }

      if (!mentioned)
	 continue;

      /* KHOROS_TOOLBOX variable cannot be opened */
      if ((file = kfoutput(filename)) == NULL)
      {
	 kwarn(KCMS, routine,
	       "Unable to write to toolbox file `%s' -- reference to "
	       "toolbox `%s' will remain.", filename, tbvar);
	 continue;
      }

      for (i = 0; i < num; i++)
      {
	 if (ksscanf(list[i], "%[^#:]:", tbname) != 1 ||
             kstrcasecmp(tbname, tbvar) != 0)
	 {
	    kfprintf(file, "%s\n", list[i]);
	 }
      }
      kfclose(file);
      karray_free(list, num, NULL);
   }
   kremenv(tbvar);

   return TRUE;
}


/************************************************************
* Routine Name:	kcms_query_toolboxes - determine available toolboxes
*
* Purpose:	This routine determines the list of available toolboxes,
*		and returns an array of string pointers containing the names
*		of the toolboxes.
*
*		The use of this function is illustrated by the following code:
*		!\f(CW\s-1
*		!   kstring  *toolbox_names;
*		!   int       toolbox_count;
*		!   int       i;
*		!
*		!   toolbox_names = kcms_query_toolboxes(&toolbox_count);
*		!   for (i = 0; i < toolbox_count; i++)
*		!      kprintf("Toolbox %d is %s\n", i, toolbox_names[i]);
*		!\s+1\fP
*
* Output:	count - A pointer to an integer.  The number of toolbox
*			names returned will be stored in the integer
*			referenced.
* Returns:	An array of string pointers, or NULL on failure.
*
* Written By:	Steven Jorgensen, Neil Bowers
* Date:		30-aug-94
*************************************************************/
kstring *
kcms_query_toolboxes(
   int  *count)
{
   return ktoolbox_file_list2array(count);
}
