#include "bootstrap.h"

static void set_and_notify(
   int   notify,
   char *message,
   kvalist)
  {
    kva_list args;

    kinfo(KSTANDARD,"Set variable here, and follow the set with a notify to the user with kvinfo.  This allows the programmer to put the variable argument list where it is most appropriate to make coding more efficient\n");
    kva_start(args, message);
    kvinfo(notify, message, args);
    kva_end(args);
  }

static void clean_and_notify(
   char *lib,
   char *routine,
   int  category,
   char *message,
   kvalist)
  {
    kva_list args;

    kinfo(KSTANDARD,"Clean state information here, and follow the cleanup with a notify to the user with kverror.  This allows the programmer to put the variable argument list where it is most appropriate to make coding more efficient\n");
    kva_start(args, message);
    errno = category;
    kverror(lib, routine, message, args);
    kva_end(args);
  }

main(
   int argc,
   char *argv[],
   char *envp[])
  {
    int val = 20;
    khoros_initialize(argc, argv, envp, "BOOTSTRAP");
    kprintf("This routine shows examples of how to call kverror, kerror,\n");
    kprintf("kvinfo, and kinfo\n\n");

    kinfo(KSTANDARD, "This call to kinfo automatically formats the string to 60 characters or less per line, inserting '\\n's as appropriate.  The three notify modes are KFORCE, KSTANDARD, and KVERBOSE.  These modes will always print the information, print the information if KHOROS_NOTIFY is not QUIET, or print the information only when KHOROS_NOTIFY is set to VERBOSE.  You can also use the format parameter as a printf format.  So, you can put a '%%d' and follow the message parameter with an integer, or '%%f' will put floats in, and so on.\n");

    kinfo(KSTANDARD, "For example, suppose you have a program that wants to warn the user that a value is out of range, the programmer, can than use kinfo to say, \n\"myprog: the value for X is '%d', but a valid range is 1-10.\".\nNext an example of kvinfo\n", val);

    kinfo(KSTANDARD, "The routine kvinfo is similar to kinfo, but instead of the variable argument list, it takes a kva_list variable.  This kva_list variable was initialized by a kva_start call in a routine that takes a variable argument list.  For example, suppose you want to set a variable and notify the user about the effects of setting that variable.  The next message will be generated from a call to a routine called set_and_notify defined in example.c\n");

    set_and_notify(KSTANDARD, "This is my notify message in set_and_notify.  Perhaps I just want it to print, \"Variable %s changed from %d to %d\".\n", "blah", 3, 10);

    kinfo(KSTANDARD, "Now, suppose you have an error in your program.  (maybe a malloc error).  Anyway, this error can be printed in a standard format so that users will know won't confuse an error with a warning.  It also give an indication of where in the source the error occured so the programmer can track bug reports.  The following error is an example of a possible error message that may be printed on a memory allocation failure.  Again, like kinfo, kerror allows printf formatting in the message string.\n");

    kerror(NULL, "notify_example", "Could not allocate '%d' bytes.", sizeof(double));
    kinfo(KSTANDARD, "Note, the category is one of a number of standard error categories defined in $BOOTSTRAP/include/kutils/knotify.h\n");

    kinfo(KSTANDARD, "Next, suppose you have an error, and you want to clean up some state variables before printing the error message and exiting.  You could put the clean up and the error reporting call into one routine if you use kverror.  The routine clean_and_notify() defined in example.c does this.  Suppose the error is that an open error occurred on an input file.\n");
    clean_and_notify(NULL, "main", KCANT_OPEN_INFILE, "The file '%s' cannot be opened.", "myfile.c");
  }
