/*
Copyright 1994 Silicon Graphics, Inc. -- All Rights Reserved

If the Software is acquired by or on behalf of an entity of government
of  the  United States of America, the following provision applies: U.
S.  GOVERNMENT  RESTRICTED  RIGHTS  LEGEND:    Use,   duplication   or
disclosure of Software by the Government is subject to restrictions as
set forth in FAR 52.227-19(c)(2) or  subparagraph  (c)(1)(ii)  of  the
Rights  in  Technical  Data  and  Computer  Software  clause  at DFARS
252.227-7013 and/or in similar or successor clauses in the FAR, or the
DOD  or  NASA  FAR Supplement. Unpub-lished- rights reserved under the
Copyright  Laws  of  the  United  States.  Contractor/manufacturer  is
SILICON  GRAPHICS,  INC.,  2011  N. Shoreline Blvd., Mountain View, CA
94039- 7311.

Silicon Graphics, Inc. hereby grants  to  you  a  non-exclusive,  non-
transferable,  personal, paid-up license to use, modify and distribute
the Software solely with SGI computer products.  You must include,  in
all  copies  of  the  Software  and  any associated documentation, the
copyright notice and restricted rights legend set forth above.

THE SOFTWARE IS PROVIDED  TO  YOU  "AS-IS"  AND  WITHOUT  ANY  SUPPORT
OBLIGATION  OR  WARRANTY  OF  ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
INCLUDING WITHOUT  LIMITATION,  ANY  WARRANTY  OF  MERCHANTABILITY  OR
FITNESS  FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SGI BE LIABLE FOR
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF  LIABILITY,
ARISING  OUT  OF  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

You agree that you will not export or re-export the Software, directly
or  indirectly,  unless  (a)  the  Export  Administration of the U. S.
Department of Commerce explicitly permits the export or  re-export  of
the  Software  or  (b)  the  Office  of  Export Licensing of the U. S.
Department of Commerce has granted au-thorization to  you  in  writing
for the  export or re- export the Software.

If you fail to fulfill any  of  the  foregoing  obligations,  SGI  may
pursue  all  available  legal  remedies  to  enforce  these  terms and
conditions, and SGI may,  at  any  time  after  your  default  hereof,
terminate  the  license  and  rights  granted  to  you hereunder.  You
further agree that, if SGI terminates this license for  your  default,
you  will, within ten (10) days after any such termination, deliver to
SGI or  render  unusable  all  Software  originally  provided  to  you
hereunder and any copies thereof embodied in any medium.
*/


#include "cppArgs.h"

#include <stdlib.h>
#include <stdio.h>
#include <alloca.h>

#include "genSetup.H"
#include "genLoadData.h"
#include "genLoadSlice.h"
#include "genByteCompile.H"
#include "misMatrix.h"
#include "misImage.H"
#include "misComputeNormals.H"
#include "volInit.h"
#include "hciUpdateScreen.h"


#ifdef VR_IRISGL

/* Texture ID counters */
int nextId = 1;
int nextTexId = 1;
int nextTevId = 1;

float BoxFilter[3*2]      = { 0.333, 0.333, 0.333, 0.333, 0.333, 0.333 };
float TriangleFilter[3*2] = { 0.100, 0.800, 0.100, 0.100, 0.800, 0.100 };

#endif

#ifdef VR_OPENGL

GLfloat BoxFilter[3]      = { 0.333, 0.333, 0.333 };
GLfloat TriangleFilter[3] = { 0.100, 0.800, 0.100 };

#endif


/*
 * This routine computes the texture scale and offset for the Ith
 * brick in a dimension with BRICKS bricks.
 */
static void SetBrickScaleAndOffset(int i, int bricks, int res, float *scl, float *off)
{
  *scl = 1.0;
  *off = 0.0;

#ifdef VR_OPENGL
  *scl *= (res - 1.0) / res;
  *off += 0.5 / res;
#endif

  /* Only one brick in this dimension */
  if (bricks == 1)
    return;

  if (i == 0) {

    /* First brick in this dimension */
    *off += *scl * (0.0);
    *scl *= 1.0 - (0.5 / (res - 1.0));

  } else if (i == (bricks-1)) {

    /* Last brick in this dimension */
    *off += *scl * (0.5 / (res - 1.0));
    *scl *= 1.0 - (0.5 / (res - 1.0));

  } else {

    /* Interior brick in this dimension */
    *off += *scl * (0.5 / (res - 1.0));
    *scl *= 1.0 - (1.0 / (res - 1.0));
  }
}  


static void SetBrickClipPlanes(int i, int j, float plane[6][4], int *nPlanes)
{
  int np=0;

  /*
   * WARNING!
   *
   * This code only knows how to handle the case
   * where only 2 clip planes are required due to the
   * fact that the volume is divided by exactly 2 in each
   * dimension.  This is not general purpose!
   *
   */

  if (i == 0) {

    plane[np][0] = -1.0;
    plane[np][1] =  0.0;
    plane[np][2] =  0.0;
    plane[np][3] =  0.0;

  } else {

    plane[np][0] =  1.0;
    plane[np][1] =  0.0;
    plane[np][2] =  0.0;
    plane[np][3] =  0.0;
  }

  np++;

  if (j == 0) {

    plane[np][0] =  0.0;
    plane[np][1] = -1.0;
    plane[np][2] =  0.0;
    plane[np][3] =  0.0;

  } else {

    plane[np][0] =  0.0;
    plane[np][1] =  1.0;
    plane[np][2] =  0.0;
    plane[np][3] =  0.0;
  }

  np++;
  *nPlanes = np;
}


static void SetupVolumeData(VRState *state, VRVolumeData *volumeData[], int volNumber, 
			     int xOff, int yOff, int zOff)
{
  VRWorld *world = state->world;
  VRVolumeData *vd = volumeData[volNumber];

#if 0
  if (world->multiResMode) {

    if (!world->readCompiled) {
      fprintf(stderr, "Multi-resolution mode is only supported on compiled data files!\n");
      exit( EXIT_FAILURE );
    }

    if ((xOff < 0) || (yOff < 0) || (zOff < 0)) {

      if (world->byteCompiling) {

	/* Load the entire volume */
	vd->subvolume = FALSE;
	vd->xOff = 0;
	vd->yOff = 0;
	vd->zOff = 0;

	vd->xRes = world->xeRes[volNumber];
	vd->yRes = world->yeRes[volNumber];
	vd->zRes = world->zeRes[volNumber];

	vd->xScl = world->xScl[volNumber];
	vd->yScl = world->yScl[volNumber];
	vd->zScl = world->zScl[volNumber];

      } else {

	/* Load the entire, downsampled volume */
	vd->subvolume = FALSE;
	vd->xOff = 0;
	vd->yOff = 0;
	vd->zOff = 0;

	vd->xRes = world->xsRes[volNumber];
	vd->yRes = world->ysRes[volNumber];
	vd->zRes = world->zsRes[volNumber];

	vd->xScl = world->xScl[volNumber];
	vd->yScl = world->yScl[volNumber];
	vd->zScl = world->zScl[volNumber];
      }

    } else {

      /* Load a subvolume */
      vd->subvolume = TRUE;
      vd->xOff = xOff;
      vd->yOff = yOff;
      vd->zOff = zOff;

      vd->xRes = world->xsRes[volNumber];
      vd->yRes = world->ysRes[volNumber];
      vd->zRes = world->zsRes[volNumber];

      vd->xScl = world->xScl[volNumber] * world->xsRes[volNumber] /
	         world->xeRes[volNumber];
      vd->yScl = world->yScl[volNumber] * world->ysRes[volNumber] /
	         world->yeRes[volNumber];
      vd->zScl = world->zScl[volNumber] * world->zsRes[volNumber] /
	         world->zeRes[volNumber];

    }

  } else {
#endif

    /* We are not running in multi-resolution mode */
    vd->subvolume = FALSE;
    vd->xOff = 0;
    vd->yOff = 0;
    vd->zOff = 0;

    vd->xRes = world->xeRes[volNumber] / world->xStride[volNumber];
    vd->yRes = world->yeRes[volNumber] / world->yStride[volNumber];
    vd->zRes = world->zeRes[volNumber] / world->zStride[volNumber];
    vd->tRes = world->teRes[volNumber];

    vd->xScl = world->xScl[volNumber];
    vd->yScl = world->yScl[volNumber];
    vd->zScl = world->zScl[volNumber];

#if 0
  }
#endif
}


static void AllocateBricks(VRState *state, VRVolumeData *volumeData[], int volNumber)
{
  VRVolumeData *vd = volumeData[volNumber];
  long textureMem;
  int nbricks;
  int i, j, k, t;
  int bnum;
  Brick *b;

  textureMem = state->world->textureMem;

#ifdef VR_KONA
  /* Return size divided by two since texels are minimum 16-bit */
  textureMem = textureMem / 2;
#endif

#ifdef VR_VENICE
  /* Return size divided by two since texels are minimum 16-bit */
  textureMem = textureMem / 2;
#endif

#ifdef VR_IMPACT
  /* Return size divided by four since we store RGBA textures */
  textureMem = textureMem / 4;
#endif

  state->world->usableTexMem = textureMem;

  if ((state->world->brickResGiven[volNumber]) && !state->world->downSample) {

    vd->nxBricks = vd->xRes / state->world->xbRes[volNumber];
    vd->nyBricks = vd->yRes / state->world->ybRes[volNumber];
    vd->nzBricks = vd->zRes / state->world->zbRes[volNumber];

  } else if ((state->world->slabs > 0) && !state->world->downSample) {

    vd->nxBricks = 1;
    vd->nyBricks = 1;
    vd->nzBricks = state->world->slabs;

  } else {

    state->world->doTexturedSurfs = TRUE;
    state->world->nPlanes         = 4;

    nbricks = (vd->xRes*vd->yRes*vd->zRes) / textureMem;

    if (nbricks < 4) {

      vd->nxBricks = 2;
      vd->nyBricks = 2;
      vd->nzBricks = 1;

    } else {

      vd->nxBricks = 2;
      vd->nyBricks = 2;
      vd->nzBricks = nbricks/4; 
    }
  }

  vd->brick[BRICK8BITS] = (Brick***) malloc(sizeof(Brick**) * vd->tRes);
  if (state->world->define4BitBricks)
    vd->brick[BRICK4BITS] = (Brick***) malloc(sizeof(Brick**) * vd->tRes);

  vd->nBricks = vd->nxBricks*vd->nyBricks*vd->nzBricks;

  vd->sbrick[BRICK8BITS] = (Brick**) malloc(sizeof(Brick*) * vd->nBricks);
  if (state->world->define4BitBricks)
    vd->sbrick[BRICK4BITS] = (Brick**) malloc(sizeof(Brick*) * vd->nBricks);

  for (t=0; t<vd->tRes; t++) {

    vd->brick[BRICK8BITS][t] = (Brick**) malloc(sizeof(Brick*) * vd->nBricks);
    if (state->world->define4BitBricks)
      vd->brick[BRICK4BITS][t] = (Brick**) malloc(sizeof(Brick*) * vd->nBricks);

    bnum = 0;

    for (k=0; k<vd->nzBricks; k++)
      for (j=0; j<vd->nyBricks; j++)
	for (i=0; i<vd->nxBricks; i++) {

	  b = (Brick*) malloc(sizeof(Brick));
	  vd->brick[BRICK8BITS][t][bnum] = b;
	  if (b == NULL) {
	    fprintf(stderr, "Unable to allocate brick memory\n");
	    exit( EXIT_FAILURE );
	  }

	  if (state->world->define4BitBricks) {
	    b = (Brick*) malloc(sizeof(Brick));
	    vd->brick[BRICK4BITS][t][bnum] = b;
	    if (b == NULL) {
	      fprintf(stderr, "Unable to allocate brick memory\n");
	      exit( EXIT_FAILURE );
	    }
	  }

	  bnum++;
	}
  }
}


static void AllocateBrickData(VRState *state, VRVolumeData *volumeData[], int volNumber)
{
  VRVolumeData  *vd = volumeData[volNumber];
  Brick         *b;
  int            i, j, k, t, mode;
  int            maxMode = state->world->define4BitBricks ? 2 : 1;
  int            bnum, bricksize, xRes, yRes, zRes;

#ifdef VR_INTERLEAVE
  unsigned char *idata;
#endif

  xRes = vd->xRes / vd->nxBricks;
  yRes = vd->yRes / vd->nyBricks;
  zRes = vd->zRes / vd->nzBricks;

  bricksize = xRes * yRes * zRes;

  for (mode=0; mode<maxMode; mode++) {
    for (t=0; t<vd->tRes; t++) {

      bnum = 0;

      for (k=0; k<vd->nzBricks; k++)
	for (j=0; j<vd->nyBricks; j++)
	  for (i=0; i<vd->nxBricks; i++) {

	    b = vd->brick[mode][t][bnum];

	    b->xRes = vd->xRes / vd->nxBricks;
	    b->yRes = vd->yRes / vd->nyBricks;
	    b->zRes = vd->zRes / vd->nzBricks;

	    if (mode == BRICK8BITS) {

	      if (!state->world->doTextures || state->world->readCompiled)
		b->data = NULL;
	      else {
		b->data = (unsigned char*) malloc(bricksize*sizeof(char));
		if (b->data == NULL) {
		  fprintf(stderr, "Unable to allocate texture memory\n");
		  exit( EXIT_FAILURE );
		}
	      }

	    } else {

	      b->data = vd->brick[BRICK8BITS][t][bnum]->data;
	    }

	    bnum++;
	  }
    }
  }

#ifdef VR_INTERLEAVE
  if (state->world->doTextures) {
    for (t=0; t<vd->tRes; t++) {

      for (i=0; i<vd->nBricks/2; i++) {

	idata = (unsigned char*) malloc(bricksize*2);
	for (j=0; j<2; j++) {

	  /* Record the interleaved data information */
	  b = vd->brick[BRICK8BITS][t][i*2 + j];
	  b->idata = idata;
	}
      }

      if (state->world->define4BitBricks) {
	for (i=0; i<vd->nBricks/4; i++) {
	  
	  idata = (unsigned char*) malloc(bricksize*4);
	  for (j=0; j<4; j++) {

	    /* Record the interleaved data information */
	    b = vd->brick[BRICK4BITS][t][i*4 + j];
	    b->idata = idata;
	  }
	}
      }
    }
  }
#endif
}


static void InitBricks(VRState *state, VRVolumeData *volumeData[], int volNumber)
{
  VRVolumeData *vd = volumeData[volNumber];
  int        i, j, k, t, mode;
  Brick     *b;
  int        bnum;
  int        maxMode = state->world->define4BitBricks ? 2 : 1;

#ifdef VR_OPENGL
#ifdef VR_TEXOBJS
  GLuint *texIds;
  int     texIdx;
  GLsizei texNum;
#endif
#endif

  for (mode=0; mode<maxMode; mode++) {

    for (t=0; t<vd->tRes; t++) {

      bnum = 0;

      for (k=0; k<vd->nzBricks; k++)
	for (j=0; j<vd->nyBricks; j++)
	  for (i=0; i<vd->nxBricks; i++) {

	    b = vd->brick[mode][t][bnum];

	    b->xOff = (float) i;
	    b->yOff = (float) j;
	    b->zOff = (float) k;

	    b->xRes = vd->xRes / vd->nxBricks;
	    b->yRes = vd->yRes / vd->nyBricks;
	    b->zRes = vd->zRes / vd->nzBricks;

	    b->xScl = 1.0;
	    b->yScl = 1.0;
	    b->zScl = 1.0;

	    b->voxelBits = (mode == BRICK8BITS) ? 8 : 4;
	    b->own = (mode == BRICK8BITS);

	    if (state->world->downSample) {
	      SetBrickScaleAndOffset(i, vd->nxBricks, b->xRes, &(b->txScl), &(b->txOff));
	      SetBrickScaleAndOffset(j, vd->nyBricks, b->yRes, &(b->tyScl), &(b->tyOff));
	      SetBrickScaleAndOffset(0, 1, b->zRes, &(b->tzScl), &(b->tzOff));
	    } else {
	      b->txOff = b->tyOff = b->tzOff = 0.0;
	      b->txScl = b->tyScl = b->tzScl = 1.0;
	    }

	    if (state->world->doTexturedSurfs)
	      SetBrickClipPlanes(i, j, b->plane, &(b->nPlanes));
	    else
	      b->nPlanes = 0;

	    bnum++;
	  }
    }
  }

  if (state->world->doTextures) {
    for (t=0; t<vd->tRes; t++) {

#ifdef VR_OPENGL
#ifdef VR_TEXOBJS
      texIdx = 0;
      texNum = 2 * vd->nBricks;
      texIds = (GLuint*) alloca(texNum * sizeof(GLuint));
      glGenTexturesEXT(texNum, texIds);
#endif
#endif

      for (i=0; i<vd->nBricks/2; i++) {
	for (j=0; j<2; j++) {

	  b = vd->brick[BRICK8BITS][t][i*2 + j];

#ifdef VR_IRISGL
#ifdef VR_INTERLEAVE
	  /* Setup the texture IDs */
	  b->texId = nextTexId;
	  b->tevId = nextTevId++;

	  /* Record the interleaved data information */
	  b->iown  = j == 0;
#else
	  /* Setup the texture ID */
	  b->texId = nextTexId++;
#endif
#endif

#ifdef VR_OPENGL
#ifdef VR_TEXOBJS
	  /* Setup the texture ID */
	  b->texId = texIds[texIdx++];
#else
#ifdef VR_INTERLEAVE
	  NO_SUPPORT_WITH(VR_FEAT_INTERLEAVE, VR_MODE_OPENGL);
#else
	  /* Setup the texture ID */
	  b->texId = glGenLists(1);
#endif
#endif
#endif
	}

#ifdef VR_IRISGL
#ifdef VR_INTERLEAVE
	nextTexId++;
#endif
#endif
      }

      if (state->world->define4BitBricks) {
	for (i=0; i<vd->nBricks/4; i++) {
	  for (j=0; j<4; j++) {

	    b = vd->brick[BRICK4BITS][t][i*4 + j];

#ifdef VR_IRISGL
#ifdef VR_INTERLEAVE
	    /* Setup the texture IDs */
	    b->texId = nextTexId;
	    b->tevId = nextTevId++;

	    /* Record the interleaved data information */
	    b->iown  = j == 0;
#else
	    /* Setup the texture ID */
	    b->texId = nextTexId++;
#endif
#endif
	
#ifdef VR_OPENGL
#ifdef VR_TEXOBJS
	    /* Setup the texture ID */
	    b->texId = texIds[texIdx++];
#else
#ifdef VR_INTERLEAVE
	    NO_SUPPORT_WITH(VR_FEAT_INTERLEAVE, VR_MODE_OPENGL);
#else
	    /* Setup the texture ID */
	    b->texId = glGenLists(1);
#endif
#endif
#endif
	  }

#ifdef VR_IRISGL
#ifdef VR_INTERLEAVE
	  nextTexId++;
#endif
#endif
	}
      }
    }
  }
}

#ifdef VR_INTERLEAVE
static int OpenBrickFile(VRState *state, char *basename, int writing,
			  int x, int y, int z, int b, int sv)
{
  char buf[256];
  int fd, flags;

  /*
   * Generate the file name.
   *
   * Normal names:       basename-x-y-z-b.cvol
   * Subvolume names:    basename-sv-x-y-z-b.cvol
   */

  sprintf(buf, "%s%s%d-%d-%d-%d.cvol", basename, sv ? "-sv-" : "-", x, y, z, b);

  if (state->view->debug > 4)
    printf("%sing file \"%s\"...\n", writing ? "Writ" : "Read", buf);

  /* Setup the flags */
  if (writing)
    flags = O_RDWR|O_CREAT;
  else
    flags = O_RDONLY;

  /* Open the file */
  fd = open(buf, flags, 0666);
  if (fd == -1) {
    fprintf(stderr, "Couldn't open file \"%s\"!\n", buf);
    fprintf(stderr, "volren: %s\n", strerror(oserror()));
    exit( EXIT_FAILURE );
  }

  return( fd );
}
#endif


/*
 * InterleaveTexture repacks texture memory to take advantage of the
 * bank interleaving of texture memory.  This is done as follows:
 *
 * For 8-bit Textures: 
 *    t00 t10 t01 t11 t02 t12 ... t0n-1 t1n-1
 *
 * For 4-bit Textures:
 *    t00 t10 t20 t30 t01 t11 t21 t31 ... t0n-1 t1n-1 t2n-1 t3n-1
 *
 *     where: txy is the y-th voxel from the x-th texture brick
 *
 * The implementation here is a bit dense, but is designed to run fast
 * so it is done with a single loop and pointers to the source and
 * destination memory locations.  (These are stored in
 * registers). Offset is the the offset into the word (for 8-bit
 * textures) or long word (for 4-bit textures). Stride is the distance
 * or stride between voxels coming from the same brick (measured in
 * bytes).
 *
 * This is just "classic" memory bank interleaving, see Hennesey and
 * Patterson, Computer Architecture - A Quantitative Approach, pages
 * 429-431.
 */
#ifdef VR_INTERLEAVE
static void InterleaveTexture(unsigned char *t, unsigned char *src,
			       int x, int y, int z, int stride, int offset)
{
  register unsigned char *d, *s, *e;

  d = &t[offset];
  e = &src[x*y*z];

  for (s=src; s<e; s++,d+=stride)
    *d = *s;
}
#endif


static void ReadSlices(VRState *state, VRVolumeData *volumeData[], int volNumber)
{
  VRVolumeData *vd = volumeData[volNumber];
  char msg[200];
  int i, j, t, volOffset, bLayer, bOffset, slice;
  int nx, ny, nz, rx, ry, rz;
  int sx, sy, sz, ix, iy, iz;
  float zoomx, zoomy;
  unsigned long *imgbuf;
  Brick *b8;

#ifdef VR_OPENGL
  int doConv, doHist;
  float tbuf[256];
  float hbuf[256];

  char *extensions = (char*) glGetString(GL_EXTENSIONS);

  if (strstr(extensions, "EXT_convolution"))
    doConv = TRUE;
  else
    doConv = FALSE;

  if (strstr(extensions, "EXT_histogram"))
    doHist = TRUE;
  else
    doHist = FALSE;
#endif

  FocusView(state);

  /*
   * Set things up to use an auxilary buffer to do the convolution
   * with the hardware and readback from the window in a safe manner
   */
#ifdef VR_AUXBUFFER
#ifdef VR_IRISGL
  if (ilbuffer(1) != 1) {
    fprintf(stderr, "Could not allocate aux buffer space\n");
    exit( EXIT_FAILURE );
  }
#endif

#ifdef VR_OPENGL
  NO_SUPPORT_WITH(VR_FEAT_AUXBUFFER, VR_MODE_OPENGL);
#endif
#endif

  nx = vd->nxBricks;
  ny = vd->nyBricks;
  nz = vd->nzBricks;
  rx = vd->xRes;
  ry = vd->yRes;
  rz = vd->zRes;
  sx = state->world->xStride[volNumber];
  sy = state->world->yStride[volNumber];
  sz = state->world->zStride[volNumber];
  ix = rx*sx;
  iy = sy*ry;
  iz = sz*rz;
  zoomx = 1.0 / sx;
  zoomy = 1.0 / sy;

  /* Setup the slice reading code */
  LoadSliceSetup(state, ix, iy, iz, volNumber);

  /* Fast allocate space for the image on the stack! */
  imgbuf = alloca(ix*iy*sizeof(unsigned long));

  /* Prepare the histogram table */
  if (doHist) {
#ifdef VR_OPENGL
    glHistogramEXT(GL_HISTOGRAM_EXT, 256, GL_LUMINANCE, GL_FALSE);
    glEnable(GL_HISTOGRAM_EXT);
#endif
    for (i=0; i<256; i++)
      hbuf[i] = 0.0;
  }

  /* Loop over all the time data */
  for (t=0; t<vd->tRes; t++) {

    /* Compute the offset for the t'th volume */
    volOffset = t * iz;

    /* Read the slices into the bricks */
    for (slice=0; slice<rz; slice++) {

      int diskSlice, bSlice, bLayer, bOffset;
      int cx1, cy1;			/* Offset to center the image */
      int cx2, cy2;			/* Top right of centered image */

      cx1 = (state->view->width  - rx) / 2;  cx1 = (cx1 >= 0) ? cx1 : 0;
      cy1 = (state->view->height - ry) / 2;  cy1 = (cy1 >= 0) ? cy1 : 0;
      cx2 = cx1 + ix - 1;
      cy2 = cy1 + iy - 1;

      /* Notify the user of where we're at */
      sprintf(msg, "Reading volume [%d/%d] time [%d/%d] slice [%d/%d]",
	      volNumber+1, state->world->nVols, t+1, vd->tRes, slice+1, rz);
      NotifyInfo(msg);

      if (state->world->handedness[volNumber] == RIGHT)
	bSlice = slice;
      else
	bSlice = (rz - slice - 1);
      bLayer = bSlice / (rz/nz);
      bOffset = bSlice % (rz/nz);

      /* Disk file slice to load */
      diskSlice = slice * sz + volOffset;

      /* Load this slice */
      LoadSlice(state, ix, iy, diskSlice, volNumber, imgbuf);

      if (doHist)
#ifdef VR_OPENGL
	glEnable(GL_HISTOGRAM_EXT);
#endif

      /*
       * If we have auxbuffer, draw the image to the screen for the
       * user's benefit and then draw it, with possible convolution to
       * the auxbuffer.  All reads will be done from the safer auxbuffer.
       */
#ifdef VR_AUXBUFFER
#ifdef VR_IRISGL
      /* Draw the image to the window */
      pixmode(PM_SIZE, 8);
      pixmode(PM_INPUT_FORMAT, PM_LUMINANCE);
      pixmode(PM_INPUT_TYPE,   PM_UNSIGNED_BYTE);
      pixmode(PM_TTOB, 0);
      
      viewport(0, state->view->width-1, 0, state->view->height-1);
      cpack(state->world->backColor);
      clear();
      rectzoom(zoomx, zoomy);
      lrectwrite(cx1, cy1, cx2, cy2, imgbuf);
      rectzoom(1.0, 1.0);
      
      /* Draw the image into the IL buffer */
      ildraw(1);
      readsource(SRC_ILBUFFER_1);
      
      pixmode(PM_SIZE, 8);
      pixmode(PM_INPUT_FORMAT, PM_LUMINANCE);
      pixmode(PM_INPUT_TYPE,   PM_UNSIGNED_BYTE);
      pixmode(PM_TTOB, 0);
      viewport(0, state->view->width-1, 0, state->view->height-1);
      
      rectzoom(zoomx, zoomy);
      if (state->world->downSample) {
	convolve(CV_SEPARABLE, CV_REDUCE, 3, 3, state->world->resampleFilter, 0.0);
	lrectwrite(cx1, cy1, cx2, cy2, imgbuf);
	convolve(CV_OFF, NULL, NULL, NULL, NULL, NULL);
      } else
	lrectwrite(cx1, cy1, cx2, cy2, imgbuf);
      rectzoom(1.0, 1.0);
#endif
      
#ifdef VR_OPENGL
      NO_SUPPORT_WITH(VR_FEAT_AUXBUFFER, VR_OPENGL);
#endif
#else
#ifdef VR_IRISGL
      pixmode(PM_SIZE, 8);
      pixmode(PM_INPUT_FORMAT, PM_LUMINANCE);
      pixmode(PM_INPUT_TYPE,   PM_UNSIGNED_BYTE);
      pixmode(PM_TTOB, 0);
      
      viewport(0, state->view->width-1, 0, state->view->height-1);
      cpack(state->world->backColor);
      clear();
      
      rectzoom(zoomx, zoomy);
      if (state->world->downSample) {
	convolve(CV_SEPARABLE, CV_REDUCE, 3, 3, state->world->resampleFilter, 0.0);
	lrectwrite(cx1, cy1, cx2, cy2, imgbuf);
	convolve(CV_OFF, NULL, NULL, NULL, NULL, NULL);
      } else
	lrectwrite(cx1, cy1, cx2, cy2, imgbuf);
      rectzoom(1.0, 1.0);
#endif
      
#ifdef VR_OPENGL
      /* Draw the image to the window */
      glViewport(0, 0, state->view->width-1, state->view->height-1);
      glClearColor(COLOR_TO_OGL(state->world->backColor));
      glClear(GL_COLOR_BUFFER_BIT);
      
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity();
      glOrtho(0.0, state->view->width-1.0, 0.0, state->view->height-1.0, -1.0, 1.0);
      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity();

      glPixelZoom(zoomx, zoomy);
      glRasterPos2i(cx1, cy1);
      if (state->world->downSample) {
	if (doConv) {
	  glConvolutionParameteriEXT(GL_SEPARABLE_2D_EXT, GL_CONVOLUTION_BORDER_MODE_EXT,
				     GL_REDUCE_EXT);
	  glSeparableFilter2DEXT(GL_SEPARABLE_2D_EXT, GL_LUMINANCE,
				 3, 3, GL_LUMINANCE, GL_FLOAT,
				 state->world->resampleFilter, state->world->resampleFilter);
	  glEnable(GL_SEPARABLE_2D_EXT);
	}
	glDrawPixels(ix, iy, GL_LUMINANCE, GL_UNSIGNED_BYTE, imgbuf);
	if (doConv)
	  glDisable(GL_SEPARABLE_2D_EXT);
      } else {
	glDrawPixels(ix, iy, GL_LUMINANCE, GL_UNSIGNED_BYTE, imgbuf);
      }
      glPixelZoom(1.0, 1.0);
#endif
#endif
      
      /* Add in the histogram values for this slice */
      if (doHist) {
#ifdef VR_OPENGL
	glGetHistogramEXT(GL_HISTOGRAM_EXT, GL_TRUE, GL_LUMINANCE, GL_FLOAT, tbuf);
	glDisable(GL_HISTOGRAM_EXT);
#endif
	for (i=0; i<256; i++)
	  hbuf[i] += tbuf[i];
      }
      
      /* Read the screen back into the appropriate bricks */
      for (j=0; j<ny; j++) {
	for (i=0; i<nx; i++) {

	  int x, y;

	  x = cx1 + i * rx / nx;
	  y = cy1 + j * ry / ny;

	  b8 = vd->brick[BRICK8BITS][t][bLayer*nx*ny + j*nx + i];

	  if (state->world->downSample) {
	    if (i > 0) x -= 2;
	    if (j > 0) y -= 2;
	  }

#ifdef VR_IRISGL
	  lrectread(x, y, x + b8->xRes-1, y + b8->yRes-1,
		    (unsigned long *) &b8->data[b8->xRes * b8->yRes * bOffset]);
#endif

#ifdef VR_OPENGL
	  /*
	   * Since this is luminance data, r,g,b channels should be
	   * identical.  So to read it back, we need only read the red
	   * channel.  This seems to work better than reading
	   * GL_LUMINANCE, which is either broken or does a funky
	   * conversion which changes the data.
	   */
	  glReadPixels(x, y, b8->xRes, b8->yRes, GL_RED, GL_UNSIGNED_BYTE,
		       (unsigned long *) &b8->data[b8->xRes * b8->yRes * bOffset]);
#endif
	}
      }
      
#ifdef VR_AUXBUFFER
#ifdef VR_IRISGL
      ildraw(0);
      readsource(SRC_AUTO);
#endif
#endif
      
#ifdef VR_IRISGL
      swapbuffers();
#endif
      
#ifdef VR_OPENGL
      glXSwapBuffers(state->view->disp, state->view->wind);
#endif
    }
  }

  /* Normalize the histogram values and save them */
  if (doHist) {
    float max = 0.0;
#ifdef VR_OPENGL
    glDisable(GL_HISTOGRAM_EXT);
#endif
    for (i=0; i<256; i++)
      max = (hbuf[i] > max) ? hbuf[i] : max;
    for (i=0; i<256; i++) {
      vd->hist[i] = hbuf[i] / max;
      /* Possibly print out the histogram table */
      if (state->view->debug == 8) {
	printf("%4.2f (%5.0f) ", vd->hist[i], hbuf[i]);
	if ((i % 8) == 7)
	  printf("\n");
      }
    }
    vd->histValid = TRUE;
  }

#ifdef VR_AUXBUFFER
#ifdef VR_IRISGL
  ilbuffer(0);
#endif
#endif

#ifdef VR_IRISGL
  viewport(0, state->view->width-1, 0, state->view->height-1);
#endif

#ifdef VR_OPENGL
  glViewport(0, 0, state->view->width-1, state->view->height-1);
#endif

  /* Write the images back to the screen */
  if ((state->view->debug == 2) || (state->view->debug == 3)) {
    for (t=0; t<vd->tRes; t++)
      for (bLayer=0; bLayer<vd->nzBricks; bLayer++)
	for (bOffset=0; bOffset<vd->zRes/vd->nzBricks; bOffset++) {

#ifdef VR_IRISGL
	  cpack(state->world->backColor);
	  clear();
	  swapbuffers();
	  clear();
	  swapbuffers();
#endif

#ifdef VR_OPENGL
	  glClearColor(COLOR_TO_OGL(state->world->backColor));
	  glClear(GL_COLOR_BUFFER_BIT);
	  glXSwapBuffers(state->view->disp, state->view->wind);
	  glClear(GL_COLOR_BUFFER_BIT);
	  glXSwapBuffers(state->view->disp, state->view->wind);
#endif

	  for (j=0; j<ny; j++) {
	    for (i=0; i<nx; i++) {

	      int x, y;

	      x = i * rx / nx;
	      y = j * ry / ny;

	      b8 = vd->brick[BRICK8BITS][t][bLayer*nx*ny + j*nx + i];

#ifdef VR_IRISGL
	      lrectwrite(x, y, x + b8->xRes-1, y + b8->yRes-1,
			 (unsigned long*) &b8->data[b8->xRes * b8->yRes * bOffset]);
#endif

#ifdef VR_OPENGL
	      glMatrixMode(GL_PROJECTION);
	      glLoadIdentity();
	      glOrtho(0.0, state->view->width-1.0, 0.0, state->view->height-1.0, -1.0, 1.0);
	      glMatrixMode(GL_MODELVIEW);
	      glLoadIdentity();

	      glRasterPos2i(x, y);
	      glDrawPixels(b8->xRes, b8->yRes, GL_LUMINANCE, GL_UNSIGNED_BYTE,
			   (unsigned long*) &b8->data[b8->xRes * b8->yRes * bOffset]);
#endif

	      if (state->view->debug == 2) {
		char buf [1024];

#ifdef VR_IRISGL
		swapbuffers();
#endif

#ifdef VR_OPENGL
		glXSwapBuffers(state->view->disp, state->view->wind);
#endif
	      
		printf("Showing slice #%03d of brick [%02d,%02d,%02d] ", 
		       bOffset, i, j, bLayer);
		gets(buf);

#ifdef VR_IRISGL
		swapbuffers();
#endif

#ifdef VR_OPENGL
		glXSwapBuffers(state->view->disp, state->view->wind);
#endif
	      }
	    }
	  }

#ifdef VR_IRISGL
	  swapbuffers();
#endif

#ifdef VR_OPENGL
	  glXSwapBuffers(state->view->disp, state->view->wind);
#endif
	}
  }
}


#ifdef VR_INTERLEAVE
static void InterLeaveTextures(VRState *state, VRVolumeData *volumeData[], int volNumber)
{
  VRVolumeData *vd = volumeData[volNumber];
  int i, j, t;
  Brick *b;
  char msg[200];
  int x, y, z;

  b = vd->brick[BRICK8BITS][0][0];
  x = b->xRes;
  y = b->yRes;
  z = b->zRes;

  for (t=0; t<vd->tRes; t++) {

    /* Repack 8-bit textures */
    for (i=0; i<vd->nBricks/2; i++) {
      for (j=0; j<2; j++) {

	sprintf(msg, "Defining 8-bit texture brick #%d of %d\n [ size = (%dx%dx%d) ]\n",
		i*2 + j + 1, vd->nBricks, x, y, z);
	NotifyInfo(msg);

	b = vd->brick[BRICK8BITS][t][i*2 + j];

	/* Interleave the 8-bit data into the interleaved buffer */
	InterleaveTexture((unsigned char*) b->idata, b->data, x, y, z, 2, j);
      }
    }

    if (state->world->define4BitBricks) {

      /* Repack 4-bit textures */
      for (i=0; i<vd->nBricks/4; i++) {
	for (j=0; j<4; j++) {

	  sprintf(msg, "Defining 4-bit texture brick #%d of %d\n [ size = (%dx%dx%d) ]\n",
		  i*4 + j + 1, vd->nBricks, x, y, z);
	  NotifyInfo(msg);

	  b = vd->brick[BRICK4BITS][t][i*4 + j];

	  /* Interleave the 4-bit data into the interleaved buffer */
	  InterleaveTexture((unsigned char*) b->idata, b->data, x, y, z, 4, j);
	}
      }
    }

    for (i=0; i<vd->nBricks; i++) {

      b = vd->brick[BRICK8BITS][t][i];
      if (b->own && b->data)
	free(b->data);
      b->data = NULL;

      if (state->world->define4BitBricks) {
	b = vd->brick[BRICK4BITS][t][i];
	if (b->own && b->data)
	  free(b->data);
	b->data = NULL;
      }
    }
  }
}
#endif


#ifdef VR_INTERLEAVE
static void ReadBricks(VRState *state, VRVolumeData *volumeData[], int volNumber)
{
  VRVolumeData *vd = volumeData[volNumber];
  int i, j, t, fd;
  Brick *b, *ba[4];
  char msg[200];
  int x, y, z;
  int xo, yo, zo;
  unsigned char *idata;

  b = vd->brick[BRICK8BITS][0][0];
  x = b->xRes;
  y = b->yRes;
  z = b->zRes;

  for (t=0; t<vd->tRes; t++) {

    /* Read 8-bit textures */
    for (i=0; i<vd->nBricks/2; i++) {

      sprintf(msg, "Reading 8-bit interleaved compiled texture chunk #%d of %d\n"
	      " [ size = (%dx%dx%d) ]\n", i+1, vd->nBricks/2, x, y, z);
      NotifyInfo(msg);

      b = vd->brick[BRICK8BITS][t][i*2];

      xo = b->xOff + vd->xOff;
      yo = b->yOff + vd->yOff;
      zo = b->zOff + vd->zOff;

      for (j=0; j<2; j++)
	ba[j] = vd->brick[BRICK8BITS][t][i*2+j];

      /* Open the brick file */
      fd = OpenBrickFile(state, state->world->basename[volNumber], 0,
			 xo, yo, zo, 8, vd->subvolume);

      /* Read in the structures and data */
      idata = b->idata;
      ReadCompiledBricks(state, fd, (unsigned long*) idata, x, y, z, 2, ba, 2);
      close(fd);

      /* Reset the idata pointers for these bricks */
      for (j=0; j<2; j++) {
	ba[j]->data  = NULL;
	ba[j]->idata = idata;
      }
    }

    if (state->world->define4BitBricks) {

      /* Read 4-bit textures */
      for (i=0; i<vd->nBricks/4; i++) {

	sprintf(msg, "Reading 4-bit interleaved compiled texture chunk #%d of %d\n"
		" [ size = (%dx%dx%d) ]\n", i+1, vd->nBricks/4, x, y, z);
	NotifyInfo(msg);

	b = vd->brick[BRICK4BITS][t][i*4];

	xo = b->xOff + vd->xOff;
	yo = b->yOff + vd->yOff;
	zo = b->zOff + vd->zOff;

	for (j=0; j<4; j++)
	  ba[j] = vd->brick[BRICK4BITS][t][i*4+j];

	/* Open the brick file */
	fd = OpenBrickFile(state, state->world->basename[volNumber], 0,
			   xo, yo, zo, 4, vd->subvolume);

	/* Read in the structures and data */
	idata = b->idata;
	ReadCompiledBricks(state, fd, (unsigned long*) b->idata, x, y, z, 4, ba, 4);
	close(fd);

	/* Reset the idata pointers for these bricks */
	for (j=0; j<4; j++) {
	  ba[j]->data  = NULL;
	  ba[j]->idata = idata;
	}
      }
    }
  }
}
#endif


#ifdef VR_INTERLEAVE
static void WriteBricks(VRState *state, VRVolumeData *volumeData[], int volNumber)
{
  VRVolumeData *vd = volumeData[volNumber];
  int i, t, fd;
  Brick *b;
  char msg[200];
  int x, y, z;

  b = vd->brick[BRICK8BITS][0][0];
  x = b->xRes;
  y = b->yRes;
  z = b->zRes;

  for (t=0; t<vd->tRes; t++) {

    /* Write 8-bit textures */
    for (i=0; i<vd->nBricks/2; i++) {

      sprintf(msg, "Writing 8-bit texture brick #%d of %d\n [ size = (%dx%dx%d) ]\n",
	      i+1, vd->nBricks/2, x, y, z);
      NotifyInfo(msg);

      b = vd->brick[BRICK8BITS][t][i*2];

      fd = OpenBrickFile(state, state->world->outname, 1,
			 b->xOff, b->yOff, b->zOff, 8, vd->subvolume);
      WriteCompiledBricks(state, fd, (unsigned long*) b->idata, x, y, z, 2,
			  &(vd->brick[BRICK8BITS][t][i*2]), 2);
      close(fd);
    }

    if (state->world->define4BitBricks) {

      /* Write 4-bit textures */
      for (i=0; i<vd->nBricks/4; i++) {

	sprintf(msg, "Writing 4-bit texture brick #%d of %d\n [ size = (%dx%dx%d) ]\n",
		i+1, vd->nBricks/4, x, y, z);
	NotifyInfo(msg);

	b = vd->brick[BRICK4BITS][t][i*4];

	fd = OpenBrickFile(state, state->world->outname, 1,
			   b->xOff, b->yOff, b->zOff, 4, vd->subvolume);
	WriteCompiledBricks(state, fd, (unsigned long*) b->idata, x, y, z, 4,
			    &(vd->brick[BRICK4BITS][t][i*4]), 4);
	close(fd);
      }
    }
  }
}
#endif


void InstallTextures(VRVolumeData *volumeData[], int volNumber)
{
#ifdef VR_IRISGL
static float tex_options[2][11] = {
  { TX_MINFILTER, TX_TRILINEAR, TX_MAGFILTER, TX_TRILINEAR, TX_WRAP, TX_CLAMP,
    TX_INTERNAL_FORMAT,   TX_IA_8, TX_EXTERNAL_FORMAT, TX_PACK_8, TX_NULL },
  { TX_MINFILTER, TX_TRILINEAR, TX_MAGFILTER, TX_TRILINEAR, TX_WRAP, TX_SELECT,
    TX_INTERNAL_FORMAT, TX_RGBA_4, TX_EXTERNAL_FORMAT, TX_PACK_8, TX_NULL } };

#ifdef VR_INTERLEAVE
static float tev_4_options[4][4] = {
  { TV_MODULATE, TV_COMPONENT_SELECT, TV_I_GETS_A, TV_NULL },
  { TV_MODULATE, TV_COMPONENT_SELECT, TV_I_GETS_B, TV_NULL },
  { TV_MODULATE, TV_COMPONENT_SELECT, TV_I_GETS_G, TV_NULL },
  { TV_MODULATE, TV_COMPONENT_SELECT, TV_I_GETS_R, TV_NULL } };

static float tev_8_options[2][4] = {
  { TV_MODULATE, TV_COMPONENT_SELECT, TV_I_GETS_A, TV_NULL },
  { TV_MODULATE, TV_COMPONENT_SELECT, TV_I_GETS_I, TV_NULL } };
#else
static float tev_options[2] = { TV_MODULATE, TV_NULL };
#endif
#endif

  VRVolumeData *vd = volumeData[volNumber];
  int i, t, count;
  char msg[200];
  Brick *b;
#ifdef VR_IRISGL
#ifdef VR_INTERLEAVE
  int j;
#endif
  unsigned char *data;
#endif

  for (t=0; t<vd->tRes; t++) {

#ifdef VR_INTERLEAVE
    count = vd->nBricks / 2;
#else
    count = vd->nBricks;
#endif

    /* Install 8-bit textures */
    for (i=0; i<count; i++) {

#ifdef VR_INTERLEAVE
      b = vd->brick[BRICK8BITS][t][i*2];
#ifdef VR_IRISGL
      data = b->idata;
#endif
      sprintf(msg, "Installing 8-bit interleaved compiled texture chunk #%d of %d\n",
	      i+1, vd->nBricks/2);
#else
      b = vd->brick[BRICK8BITS][t][i];
#ifdef VR_IRISGL
      data = b->data;
#endif
      sprintf(msg, "Installing 8-bit compiled texture chunk #%d of %d\n",
	      i+1, vd->nBricks);
#endif

      NotifyInfo(msg);

#ifdef VR_IRISGL
#ifdef VR_INTERLEAVE
      texdef3d(b->texId, 2, b->xRes, b->yRes, b->zRes,
	       (unsigned long *) data, 0, tex_options[BRICK8BITS]);

      for (j=0; j<2; j++) {
	b = vd->brick[BRICK8BITS][t][i*2 + j];
	tevdef(b->tevId, 0, tev_8_options[j]);
      }
#else
      texdef3d(b->texId, 2, b->xRes, b->yRes, b->zRes,
	       (unsigned long *) data, 0, tex_options[BRICK8BITS]);
      tevdef(b->tevId, 0, tev_8_options);
#endif
#endif

#ifdef VR_OPENGL
#ifdef VR_INTERLEAVE
      NO_SUPPORT_WITH(VR_FEAT_INTERLEAVE, VR_MODE_OPENGL);
#else
#ifdef VR_TEXOBJS
      glBindTextureEXT(GL_TEXTURE_3D_EXT, b->texId);
#endif
      glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_WRAP_R_EXT, GL_CLAMP);
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_WRAP_S, GL_CLAMP);
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_WRAP_T, GL_CLAMP);

#ifdef VR_KONA
      glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_INTENSITY8_EXT,
		      b->xRes, b->yRes, b->zRes, 0, GL_LUMINANCE,
		      GL_UNSIGNED_BYTE, b->data);
#endif

#ifdef VR_VENICE
      glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_INTENSITY8_EXT,
		      b->xRes, b->yRes, b->zRes, 0, GL_LUMINANCE,
		      GL_UNSIGNED_BYTE, b->data);
#endif

#ifdef VR_IMPACT
      glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_RGBA8_EXT,
		      b->xRes, b->yRes, b->zRes, 0, GL_COLOR_INDEX,
		      GL_UNSIGNED_BYTE, NULL);
#endif

      b->forceDownload = TRUE;
      b->forceRedef = FALSE;
#endif
#endif
    }

    if (state->world->define4BitBricks) {

#ifdef VR_INTERLEAVE
      count = vd->nBricks / 4;
#else
      count = vd->nBricks;
#endif

      /* Install 4-bit textures */
      for (i=0; i<count; i++) {

#ifdef VR_INTERLEAVE
	b = vd->brick[BRICK4BITS][t][i*4];
#ifdef VR_IRISGL
	data = b->idata;
#endif
	sprintf(msg, "Installing 4-bit interleaved compiled texture chunk #%d of %d\n",
		i+1, vd->nBricks/4);
#else
	b = vd->brick[BRICK4BITS][t][i];
#ifdef VR_IRISGL
	data = b->data;
#endif
	sprintf(msg, "Installing 4-bit compiled texture chunk #%d of %d\n",
		i+1, vd->nBricks);
#endif

	NotifyInfo(msg);

#ifdef VR_IRISGL
#ifdef VR_INTERLEAVE
	texdef3d(b->texId, 4, b->xRes, b->yRes, b->zRes,
		 (unsigned long *) data, 0, tex_options[BRICK4BITS]);

	for (j=0; j<4; j++) {
	  b = vd->brick[BRICK4BITS][t][i*4 + j];
	  tevdef(b->tevId, 0, tev_4_options[j]);
	}
#else
	texdef3d(b->texId, 4, b->xRes, b->yRes, b->zRes,
		 (unsigned long *) data, 0, tex_options[BRICK4BITS]);
	tevdef(b->tevId, 0, tev_4_options);
#endif
#endif

#ifdef VR_OPENGL
#ifdef VR_INTERLEAVE
	NO_SUPPORT_WITH(VR_FEAT_INTERLEAVE, VR_MODE_OPENGL);
#else
#ifdef VR_TEXOBJS
	glBindTextureEXT(GL_TEXTURE_3D_EXT, b->texId);
#endif
	glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_WRAP_R_EXT, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_WRAP_T, GL_CLAMP);

#ifdef VR_KONA
	glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_INTENSITY4_EXT,
			b->xRes, b->yRes, b->zRes, 0, GL_LUMINANCE,
			GL_UNSIGNED_BYTE, b->data);
#endif

#ifdef VR_VENICE
	glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_INTENSITY4_EXT,
			b->xRes, b->yRes, b->zRes, 0, GL_LUMINANCE,
			GL_UNSIGNED_BYTE, b->data);
#endif

#ifdef VR_IMPACT
	glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_RGBA4_EXT,
			b->xRes, b->yRes, b->zRes, 0, GL_COLOR_INDEX,
			GL_UNSIGNED_BYTE, NULL);
#endif

	b->forceDownload = TRUE;
	b->forceRedef = FALSE;
#endif
#endif
      }
    }
  }
}


void LoadVRPolygonData(VRState *state, VRPolygonData *polygonData)
{
  FILE *file;
  int lineno=0, o, t, p, n;
  char line[1024];

  file = fopen(state->world->polygonFile, "r");
  if (file == NULL) {
    fprintf(stderr, "Unable to open the specified polygon file!\n");
    exit( EXIT_FAILURE );
  }

  do {
    fgets(line, sizeof(line), file);
    lineno++;
  } while ((line[0] == '#') || (line[0] == '\n'));

  polygonData->nObjects = atoi(line);
  if (polygonData->nObjects <= 0) {
    fprintf(stderr, "Illegal object count in polygon file, must be greater than zero!\n");
    fprintf(stderr, "Error occured at line #%d.\n", lineno);
    exit( EXIT_FAILURE );
  }

  polygonData->object = (PolyObject*) malloc(polygonData->nObjects *
					     sizeof(PolyObject));
  if (polygonData->object == NULL) {
    fprintf(stderr, "Unable to allocate space for polygon objects!\n");
    exit( EXIT_FAILURE );
  }

  for (o=0; o<polygonData->nObjects; o++) {

    char *ptr;

    PolyObject *obj = &polygonData->object[o];

    do {
      fgets(line, sizeof(line), file);
      lineno++;
    } while ((line[0] == '#') || (line[0] == '\n'));
      
    /* Remove the newline */
    ptr = line;
    while (*ptr != '\n') ptr++;
    *ptr = NULL;

    for (t=0; t<MAX_POTYPES; t++)
      if (!strcasecmp(line, polyObjectName[t])) {
	obj->type = t;
	break;
      }

    if (t == MAX_POTYPES) {
      fprintf(stderr, "Unknown object type: %s", line);
      fprintf(stderr, "Error occured at line #%d.\n", lineno);
      exit( EXIT_FAILURE );
    }

    do {
      fgets(line, sizeof(line), file);
      lineno++;
    } while ((line[0] == '#') || (line[0] == '\n'));

    obj->nPoints = atoi(line);
    if (obj->nPoints <= 0) {
      fprintf(stderr, "Illegal part count in object in polygon file, "
	      "must be greater than zero!\n");
      fprintf(stderr, "Error occured at line #%d.\n", lineno);
      exit( EXIT_FAILURE );
    }

    obj->point = (coord*) malloc(obj->nPoints * pointsPerPart[obj->type] * sizeof(coord));
    if (obj->point == NULL) {
      fprintf(stderr, "Unable to allocate space for polygon objects!\n");
      exit( EXIT_FAILURE );
    }

    for (p=0; p<obj->nPoints; p++) {

      int o = p * pointsPerPart[obj->type];

      for (n=0; n<pointsPerPart[obj->type]; n++) {
	do {
	  fgets(line, sizeof(line), file);
	  lineno++;
	} while ((line[0] == '#') || (line[0] == '\n'));

	if (sscanf(line, "%f %f %f", &(obj->point[o+n].x), &(obj->point[o+n].y), &(obj->point[o+n].z)) != 3) {
	  fprintf(stderr, "Unable to read an object point!\n");
	  fprintf(stderr, "Error occured at line #%d.\n", lineno);
	  exit( EXIT_FAILURE );
	}
      }
    }

    obj->normal = (coord*) malloc(obj->nPoints * pointsPerPart[obj->type] * sizeof(coord));
    if (obj->normal == NULL) {
      fprintf(stderr, "Unable to allocate space for polygon objects!\n");
      exit( EXIT_FAILURE );
    }

    switch (obj->type) {

    case DISJOINT_TRIANGLES:
      for (p=0; p<obj->nPoints; p++) {

	coord v1, v2, v3, n1;
	int o = p * pointsPerPart[obj->type];

	coord_sub(obj->point[o+0], obj->point[o+1], v1);
	coord_sub(obj->point[o+2], obj->point[o+1], v2);
	coord_cross_product(v2, v1, v3);
	coord_normalize(v3, n1);

	for (n=0; n<pointsPerPart[obj->type]; n++)
	  coord_assign(n1, obj->normal[o+n]);
      }
      break;

    case QUADRILATERAL_STRIP:
      GenerateSurfaceNormals(obj->point, NORMAL_IS_Z, obj->normal, 2, obj->nPoints);
      break;
    }

    obj->nPoints *= pointsPerPart[obj->type];
  }
}


static float ReadFloat(FILE *fp, int j, int m)
{
  float val;

  if (fscanf(fp, "%f", &val) == 1) {

    return( val );

  } else {

    fprintf(stderr, "Unable to parse table \"%s\"!\n",
	    state->world->mapName[m]);
    fprintf(stderr, "Failed reading initial %s table modifier!\n",
	    ((j<2)?((j==0)?"red":"green"):((j==2)?"blue":"alpha")));
    exit( EXIT_FAILURE );
  }
  return( 0.0 );
}


static int ReadDecimal(FILE *fp, int i, int j, int m)
{
  int val;

  if (fscanf(fp, "%d", &val) == 1) {

    return( val );

  } else {

    fprintf(stderr, "Unable to parse table \"%s\"!\n",
	    state->world->mapName[m]);
    fprintf(stderr, "Failed reading index %d %s element!\n",
	    i, ((j<2)?((j==0)?"red":"green"):((j==2)?"blue":"alpha")));
    exit( EXIT_FAILURE );
  }
  return( 0.0 );
}


static char* FileToMap(char *file)
{
  char *name, *t;

  /* Remove the leading path */
  t = strrchr(file, '/');
  if (t == NULL) {
    t = file;
  } else {
    t++;
  }

  /* Copy the filename */
  name = malloc(strlen(t)+1);
  strcpy(name, t);

  /* Remove any trailing extension */
  t = strrchr(name, '.');
  if (t != NULL)
    *t = NULL;

  return( name );
}


void LoadVRTableData(VRState *state, VRTableData *tableData)
{
  FILE *fp;
  char msg[200];
  float base[4], *color, *alpha;
  int i, m, c=0, a=0;

  tableData->nColormaps = 0;
  tableData->nAlphamaps = 0;

  for (m=0; m<state->world->nMaps; m++) {

    switch (state->world->mapType[m]) {

    case TABLEMAP:

      /* Read in a table map */
      fp = fopen(state->world->mapName[m], "r");
      if (fp) {

	sprintf(msg, "Reading in tablemap #%d: %s", m+1,
		state->world->mapName[m]);
	NotifyInfo(msg);

	for (i=0; i<4; i++)
	  base[i] = ReadFloat(fp, i, m);

	color = tableData->cmap[c] = (float*) malloc(256*3*sizeof(float));
	alpha = tableData->amap[a] = (float*) malloc(256*1*sizeof(float));

	for (i=0; i<256; i++) {
	  *color++ = ReadDecimal(fp, i, 0, m) / 255.0 * base[0];
	  *color++ = ReadDecimal(fp, i, 1, m) / 255.0 * base[1];
	  *color++ = ReadDecimal(fp, i, 2, m) / 255.0 * base[2];
	  *alpha++ = ReadDecimal(fp, i, 3, m) / 255.0 * base[3];
	}

      } else {
	
	perror(state->world->mapName[m]);
	exit( EXIT_FAILURE ); 
      }
      
      tableData->colormapName[c++] =
      tableData->alphamapName[a++] = FileToMap(state->world->mapName[m]);

      break;

    case COLORMAP:

      /* Read in a colormap */
      fp = fopen(state->world->mapName[m], "r");
      if (fp) {

	sprintf(msg, "Reading in colormap #%d: %s", m+1,
		state->world->mapName[m]);
	NotifyInfo(msg);

	for (i=0; i<3; i++)
	  base[i] = ReadFloat(fp, i, m);

	color = tableData->cmap[c] = (float*) malloc(256*3*sizeof(float));

	for (i=0; i<256; i++) {
	  *color++ = ReadDecimal(fp, i, 0, m) / 255.0 * base[0];
	  *color++ = ReadDecimal(fp, i, 1, m) / 255.0 * base[1];
	  *color++ = ReadDecimal(fp, i, 2, m) / 255.0 * base[2];
	}

      } else {
	
	perror(state->world->mapName[m]);
	exit( EXIT_FAILURE ); 
      }
      
      tableData->colormapName[c++] = FileToMap(state->world->mapName[m]);

      break;

    case ALPHAMAP:

      /* Read in an alphamap */
      fp = fopen(state->world->mapName[m], "r");
      if (fp) {

	sprintf(msg, "Reading in alphamap #%d: %s", m+1,
		state->world->mapName[m]);
	NotifyInfo(msg);

	for (i=3; i<4; i++)
	  base[i] = ReadFloat(fp, i, m);

	alpha = tableData->amap[a] = (float*) malloc(256*1*sizeof(float));

	for (i=0; i<256; i++)
	  *alpha++ = ReadDecimal(fp, i, 3, m) / 255.0 * base[3];

      } else {
	
	perror(state->world->mapName[m]);
	exit( EXIT_FAILURE ); 
      }
      
      tableData->alphamapName[a++] = FileToMap(state->world->mapName[m]);

      break;
    }
  }

  tableData->nColormaps = c;
  tableData->nAlphamaps = a;
}


void LoadVRElevData(VRState *state, VRElevData *elevData[], int i)
{
  char msg[128];

#ifdef VR_IRISGL
  static float tex_options[11] = {
    TX_MINFILTER, TX_BILINEAR, TX_MAGFILTER, TX_BILINEAR, TX_WRAP, TX_CLAMP,
    TX_INTERNAL_FORMAT, TX_RGB_5, TX_EXTERNAL_FORMAT, TX_PACK_8, TX_NULL };

  static float tev_options[2] = { TV_MODULATE, TV_NULL };
#endif

  VRElevData *ed = elevData[i];

  sprintf(msg, "Defining elevation [%d/%d]", i+1, state->world->nElevs);
  NotifyInfo(msg);

  if (state->world->elevColorImg[i]) {

    int cx, cy;
    int rx, ry;
    unsigned char *buf, *color=NULL;

    sprintf(msg, "Defining elevation [%d/%d] color texture data", i+1, state->world->nElevs);
    NotifyInfo(msg);

    ed->haveTexture = TRUE;

    /* Load the texture data */
    color = ReadImage(state->world->elevColorImg[i], TRUE, TRUE, -1, -1, &rx, &ry, NULL);

#ifdef VR_OPENGL
    FocusView(state);

    /* Draw the image to the window */
    glViewport(0, 0, state->view->width-1, state->view->height-1);
    glClearColor(COLOR_TO_OGL(state->world->backColor));
    glClear(GL_COLOR_BUFFER_BIT);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, state->view->width-1.0, 0.0, state->view->height-1.0, -1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glRasterPos2i(0, 0);
    glDrawPixels(rx, ry, GL_RGB, GL_UNSIGNED_BYTE, color);
    glXSwapBuffers(state->view->disp, state->view->wind);
#endif

    /* Now get it into a buffer that is a power of two on a side. */
    cx = NextPowerOfTwo(rx);
    cy = NextPowerOfTwo(ry);

    buf = (unsigned char*) malloc(cx*cy*3*sizeof(unsigned char));
    if (buf == NULL) {
      (void) fprintf(stderr, "Unable to allocate space for elevation texture!\n");
      exit( EXIT_FAILURE );
    }

    memset(buf, 0x80, cx*cy*3*sizeof(unsigned char));
    ExtractImage(buf, cx, cy, color, rx, ry, 0, 0, 0, 0, rx, ry, 3);
    free(color);

    /* Record the scale in the elevData */
    ed->txScl = ((float) rx) / cx;
    ed->tyScl = ((float) ry) / cy;

#ifdef VR_IRISGL
    ed->texId = nextTexId++;
    ed->tevId = nextTevId++;

    texdef2d(ed->texId, 3, cx, cy, (unsigned long*) buf, 0, tex_options);
    tevdef(ed->tevId, 0, tev_options);
#endif

#ifdef VR_OPENGL
#ifdef VR_TEXOBJS
    glGenTexturesEXT(1, &(ed->texId));
    glBindTextureEXT(GL_TEXTURE_2D, ed->texId);
#endif

    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB8_EXT, cx, cy, 0, GL_RGB,
                 GL_UNSIGNED_BYTE, (const GLvoid*) buf);

#endif

    free(buf);
  }

  if (state->world->elevElevImg[i]) {

    int ex, ey, x, y;

    unsigned short bits = 8;
    unsigned long maxValue = 1<<bits;
    unsigned char *ptr, *elev=NULL;

    sprintf(msg, "Defining elevation [%d/%d] height data", i+1, state->world->nElevs);
    NotifyInfo(msg);

    /* Copy the data into the polys array */
    elev = ReadImage(state->world->elevElevImg[i], FALSE, TRUE, -1, -1, &ex, &ey, NULL);

    ed->polyWidth  = ex;
    ed->polyHeight = ey;

    if (ed->polys)
      free(ed->polys);
    ed->polys = (coord*) malloc(ed->polyWidth * ed->polyHeight * sizeof(coord));
    if (ed->polys == NULL) {
      (void) fprintf(stderr, "Unable to allocate space for elevation points!\n");
      exit( EXIT_FAILURE );
    }

    ptr = elev;
    for (y=0; y<ey; y++)
      for (x=0; x<ex; x++) {
	ed->polys[y*ex+x].x = (2.0 * x / (ex-1) - 1.0);
	ed->polys[y*ex+x].y = (2.0 * y / (ey-1) - 1.0);
	ed->polys[y*ex+x].z = ((float) (*ptr++)) / maxValue;
      }
    free(elev);

    /* Generate the elevation's normals */
    if (ed->normals)
      free(ed->normals);
    ed->normals = (coord*) malloc(ed->polyWidth * ed->polyHeight * sizeof(coord));
    if (ed->normals == NULL) {
      (void) fprintf(stderr, "Unable to allocate space for elevation normals!\n");
      exit( EXIT_FAILURE );
    }

    GenerateSurfaceNormals(ed->polys, NORMAL_IS_Z, ed->normals,
			   ed->polyWidth, ed->polyHeight);
  }
}


void LoadVRVolumeData(VRState *state, VRVolumeData *volumeData[], int volNumber,
		       int xOff, int yOff, int zOff)
{
  /* Setup the volume data basic values */
  SetupVolumeData(state, volumeData, volNumber, xOff, yOff, zOff);

  /* Compute the brick counts in each dimension and allocate the bricks */
  AllocateBricks(state, volumeData, volNumber);

  /* Allocate the brick data */
  AllocateBrickData(state, volumeData, volNumber);

  InitBricks(state, volumeData, volNumber);

  if (state->world->doTextures) {
    if (state->world->readCompiled) {

#ifdef VR_INTERLEAVE
      ReadBricks(state, volumeData, volNumber);
      InstallTextures(volumeData, volNumber);
#else
      NO_SUPPORT_WITHOUT(VR_FEAT_INTERLEAVE, VR_MODE_COMPILING);
#endif

    } else {

      ReadSlices(state, volumeData, volNumber);
#ifdef VR_INTERLEAVE
      InterLeaveTextures(state, volumeData, volNumber);
#endif
      InstallTextures(volumeData, volNumber);
    }
  }
}


void ReloadVRVolumeData(VRState *state, VRVolumeData *volumeData[], int volNumber)
{
#ifdef VR_INTERLEAVE
  state->world->initing = TRUE;
  ReadBricks(state, volumeData, volNumber);
  InstallTextures(volumeData, volNumber);
  state->world->initing = FALSE;
#else
  NO_SUPPORT_WITHOUT(VR_FEAT_INTERLEAVE, VR_MODE_COMPILING);
#endif
}


void CompileVRVolumeData(VRState *state, VRVolumeData *volumeData[], int volNumber)
{
#ifdef VR_INTERLEAVE
  if (state->world->readCompiled) {
    fprintf(stderr, "Why compile already compiled data?\n");
    exit( EXIT_FAILURE );
  }

  if (state->world->multiResMode) {

    SetupVolumeData(state, volumeData, volNumber, -1, -1, -1);
    AllocateBricks(state, volumeData, volNumber);
    AllocateBrickData(state, volumeData, volNumber);

    InitBricks(state, volumeData, volNumber);
    ReadSlices(state, volumeData, volNumber);
    InterLeaveTextures(state, volumeData, volNumber);

  } else {

    SetupVolumeData(state, volumeData, volNumber, -1, -1, -1);
    AllocateBricks(state, volumeData, volNumber);
    AllocateBrickData(state, volumeData, volNumber);

    InitBricks(state, volumeData, volNumber);
    ReadSlices(state, volumeData, volNumber);
    InterLeaveTextures(state, volumeData, volNumber);

    /* Write the bricks back to disk interleaved */
    WriteBricks(state, volumeData, volNumber);

  }
#else
  NO_SUPPORT_WITHOUT(VR_FEAT_INTERLEAVE, VR_MODE_COMPILING);
#endif
}
