/*
Copyright 1994 Silicon Graphics, Inc. -- All Rights Reserved

If the Software is acquired by or on behalf of an entity of government
of  the  United States of America, the following provision applies: U.
S.  GOVERNMENT  RESTRICTED  RIGHTS  LEGEND:    Use,   duplication   or
disclosure of Software by the Government is subject to restrictions as
set forth in FAR 52.227-19(c)(2) or  subparagraph  (c)(1)(ii)  of  the
Rights  in  Technical  Data  and  Computer  Software  clause  at DFARS
252.227-7013 and/or in similar or successor clauses in the FAR, or the
DOD  or  NASA  FAR Supplement. Unpub-lished- rights reserved under the
Copyright  Laws  of  the  United  States.  Contractor/manufacturer  is
SILICON  GRAPHICS,  INC.,  2011  N. Shoreline Blvd., Mountain View, CA
94039- 7311.

Silicon Graphics, Inc. hereby grants  to  you  a  non-exclusive,  non-
transferable,  personal, paid-up license to use, modify and distribute
the Software solely with SGI computer products.  You must include,  in
all  copies  of  the  Software  and  any associated documentation, the
copyright notice and restricted rights legend set forth above.

THE SOFTWARE IS PROVIDED  TO  YOU  "AS-IS"  AND  WITHOUT  ANY  SUPPORT
OBLIGATION  OR  WARRANTY  OF  ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
INCLUDING WITHOUT  LIMITATION,  ANY  WARRANTY  OF  MERCHANTABILITY  OR
FITNESS  FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SGI BE LIABLE FOR
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF  LIABILITY,
ARISING  OUT  OF  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

You agree that you will not export or re-export the Software, directly
or  indirectly,  unless  (a)  the  Export  Administration of the U. S.
Department of Commerce explicitly permits the export or  re-export  of
the  Software  or  (b)  the  Office  of  Export Licensing of the U. S.
Department of Commerce has granted au-thorization to  you  in  writing
for the  export or re- export the Software.

If you fail to fulfill any  of  the  foregoing  obligations,  SGI  may
pursue  all  available  legal  remedies  to  enforce  these  terms and
conditions, and SGI may,  at  any  time  after  your  default  hereof,
terminate  the  license  and  rights  granted  to  you hereunder.  You
further agree that, if SGI terminates this license for  your  default,
you  will, within ten (10) days after any such termination, deliver to
SGI or  render  unusable  all  Software  originally  provided  to  you
hereunder and any copies thereof embodied in any medium.
*/


#include "cppArgs.h"

#include <stdlib.h>
#include <ctype.h>
#include <Vk/VkApp.h>

#include "genParseArgs.H"


static char defExtension[] = ".tiff";

static char *usage_str =
"usage: %s [intfopts] -- [volopts] -- object [-- object [...]]\n"
"(intfopts)            Interface Options:\n"
"\n"
"  -C colormap         - Colormap file name (may be specified multiple times)\n"
"  -A alphamap         - Alphamap file name (may be specified multiple times)\n"
"  -T tablemap         - Tablemap (color & alpha) file name\n"
"                        (may be specified multiple times)\n"
#if 0
"  -M display_name     - Display (host:display.screen) of additional monitors\n"
"                        (may be specified multiple times)\n"
#endif
"  -P file             - Polygon data file to load\n"
"  -S count            - The number of slices to use when rendering in z\n"
"  -r width height     - Rendering window width and height (default: 512x512)\n"
"  -p count            - Set the number of clipping planes to use (default: 6)\n"
"  -e                  - Allow stereo mode (default: allocate only mono buffers)\n"
"  -a                  - Don't modulate alphamap by the transfer function\n"
"  -m                  - Don't modulate colormap by the transfer function\n"
"  -c obj color        - Set color of object obj\n"
"                        Valid objects are:\n"
"                            back         The background color\n"
"                            frame        The volume's frame\n"
"                            cframe       The current volume's frame\n"
"                            plane        The clip planes' frame\n"
"                            cplane       The current clip plane's frame\n"
"                            point        The point color\n"
"                            cpoint       The current point's color\n"
"                            xvec         The X axis vector's color\n"
"                            yvec         The Y axis vector's color\n"
"                            zvec         The Z axis vector's color\n"
"                            upvec        The up vector's color\n"
"                            northvec     The north vector's color\n"
"                            frontpoly    The front of the polygons' color\n"
"                            backpoly     The back of the polygons' color\n"
"                            frontelev    The front of the elevation's color\n"
"                            backelev     The back of the elevation's color\n"
"                            xysurf       The XY surfaces' color\n"
"                            zsurf        The Z surfaces' color\n"
"                            surf#        The #th surface's color\n"
"  -F file             - Filename for snapshot image (default: \"snapfile.tif\")\n"
"  -f width height     - Size of snapshot file (default: 1000x1000)\n"
"  -Q                  - Supported for backwards compatibility only\n"
"  -O                  - Display compilation options then exit\n"
"  -V                  - Verbose progress\n"
"  -D level            - Print out debugging information\n"
"\n"
"(volopts)             Volume Options:\n"
"\n"
"  -m count            - Break the volume up into count slabs in z\n"
"  -r resample         - Turn on resampling to remove seams based upon resample\n"
"                        Any of the following values will turn resampling off\n"
"                        (default: on; overrides -m):\n"
"                            false  f      no     n      off   -\n"
"  -f filter           - Select a resampling filter (default: box)\n"
"                        Valid filters are:\n"
"                            box    triangle\n"
"  -4                  - Create 4-bit bricks (default: only define 8-bit bricks)\n"
"  -U dir              - Choose an up vector\n"
"                        Valid choices are: {x,y,z}\n"
"  -N angle            - Choose a north vector\n"
"                        (specified as an angle degrees around the up vector)\n"
#if 0
"  -C cdata            - Just byte compile the data then exit\n"
"                        (put the files in \"cdata\"+suffix)\n"
"  -R                  - Read byte compiled data instead of slices\n"
"                        (read the files from \"data\"+suffix)\n"
"  -A                  - Run in interactive data acquisition mode\n"
"                        (reread the data when a SIGUSR1 is receieved)\n"
#endif
"  -V                  - Verbose progress\n"
"  -D level            - Print out debugging information\n"
"\n"
"(object)              Per Object Information:\n"
"\n"
"  Format: objtype [objopts] objinfo\n"
"  objtype             - The type of object about to be defined.\n"
"                        Valid object types are:\n"
"                            volume    elevation\n"
"  objopts             - The options for this object.  Each object has its\n"
"                        own list of options.  See below.\n"
"  objinfo             - The required information for this object.  Each\n"
"                        object has its own required information.  See below.\n"
"\n"
"(volume options)      Per Volume Options:\n"
"\n"
"  -h                  - Toggle handedness (default: left, images stack down)\n"
"  -X scale            - Relative voxel spacing in X (default: 1.0)\n"
"  -Y scale            - Relative voxel spacing in Y (default: 1.0)\n"
"  -Z scale            - Relative voxel spacing in Z (default: 1.0)\n"
"  -T number           - Resolution of the volume in time (default: 1)\n"
"  -S millisecs        - Time interval before switching to next volume\n"
"                        in time sequence (default: 3000)\n"
"  -e extension        - Slice file name extension (default: \".tiff\")\n"
"  -o z_off            - The number of the first slice in Z\n"
"  -O x_off y_off      - The X/Y dimensional offsets\n"
"  --                  - Must be specified for subsequent volumes if no\n"
"                        other volume options are given; marks the beginning\n"
"                        of subsequent volumes\n"
"\n"
"(volume info)         Per Volume Information:\n"
"\n"
"  Format: x y z data\n"
"  [ All resolutions must be a power of two! ]\n"
"  x                   - Resolution of the volume in X\n"
"  y                   - Resolution of the volume in Y\n"
"  z                   - Resolution of the volume in Z\n"
"  data                - Path to data files to read\n"
"                        (read the files from \"data\"###\"suffix\")\n"
"\n"
"(elevation options)   Per Elevation Options:\n"
"\n"
"  -X scale            - Relative pixel spacing in X (default: 1.0)\n"
"  -Y scale            - Relative pixel spacing in Y (default: 1.0)\n"
"  -Z scale            - Relative elevation size (default: 1.0)\n"
"  -L                  - Disable lighting of elevation data.\n"
"  -R angle            - Image rotation, counter-clockwise, in X-Y plane.\n"
"  -T xt yt zt         - Translation of image center in X, Y, and Z.\n"
"\n"
"(elevation info)      Per Elevation Information:\n"
"\n"
"  Format: [colorimg] elevimg\n"
"  [ The images need not have the same size! ]\n"
"  colorimg            - An RGB image specifying the elevation color data.\n"
"  elevimg             - A greyscale image specifying the elevation data.\n"
"\n"
;


char *progname = "volren";


static void usage( void )
{
  fprintf( stderr, usage_str, progname );
  exit( EXIT_FAILURE );
}


static long DecodeAndPackColor( Colormap cmap, char *colorString, long defColor )
{
  XColor cell;
  int s;

  s = XParseColor(theApplication->display(), cmap, colorString, &cell);
  if (!s) {
    fprintf(stderr, "Unable to parse color name string (\"%s\") into a valid color!\n",
            colorString );
    return( defColor );
  }

  return( ((0xff            ) << 24) |
          ((cell.blue   >> 8) << 16) |
          ((cell.green  >> 8) <<  8) |
          ((cell.red    >> 8) <<  0) );
}


static void DecodeUserColors( VRWorld *world )
{
  Colormap cmap;
  int i;

  XtVaGetValues(theApplication->baseWidget(), XtNcolormap, &cmap, NULL);

  if (world->backColorString)
    world->backColor =
      DecodeAndPackColor(cmap, world->backColorString, world->backColor);

  if (world->frameColorString)
    world->frameColor =
      DecodeAndPackColor(cmap, world->frameColorString, world->frameColor);

  if (world->curFrameColorString)
    world->curFrameColor =
      DecodeAndPackColor(cmap, world->curFrameColorString, world->curFrameColor);

  if (world->planeColorString)
    world->planeColor =
      DecodeAndPackColor(cmap, world->planeColorString, world->planeColor);

  if (world->curPlaneColorString)
    world->curPlaneColor =
      DecodeAndPackColor(cmap, world->curPlaneColorString, world->curPlaneColor);

  if (world->pointColorString)
    world->pointColor =
      DecodeAndPackColor(cmap, world->pointColorString, world->pointColor);

  if (world->curPointColorString)
    world->curPointColor =
      DecodeAndPackColor(cmap, world->curPointColorString, world->curPointColor);

  if (world->xVectorColorString)
    world->xVectorColor =
      DecodeAndPackColor(cmap, world->xVectorColorString, world->xVectorColor);

  if (world->yVectorColorString)
    world->yVectorColor =
      DecodeAndPackColor(cmap, world->yVectorColorString, world->yVectorColor);

  if (world->zVectorColorString)
    world->zVectorColor =
      DecodeAndPackColor(cmap, world->zVectorColorString, world->zVectorColor);

  if (world->upVectorColorString)
    world->upVectorColor =
      DecodeAndPackColor(cmap, world->upVectorColorString, world->upVectorColor);

  if (world->northVectorColorString)
    world->northVectorColor =
      DecodeAndPackColor(cmap, world->northVectorColorString, world->northVectorColor);

  if (world->frontPolyColorString)
    world->frontPolyColor =
      DecodeAndPackColor(cmap, world->frontPolyColorString, world->frontPolyColor);

  if (world->backPolyColorString)
    world->backPolyColor =
      DecodeAndPackColor(cmap, world->backPolyColorString, world->backPolyColor);

  if (world->frontElevColorString)
    world->frontElevColor =
      DecodeAndPackColor(cmap, world->frontElevColorString, world->frontElevColor);

  if (world->backElevColorString)
    world->backElevColor =
      DecodeAndPackColor(cmap, world->backElevColorString, world->backElevColor);

  if (world->xySurfColorString)
    world->xySurfColor =
      DecodeAndPackColor(cmap, world->xySurfColorString, world->xySurfColor);

  if (world->zSurfColorString)
    world->zSurfColor =
      DecodeAndPackColor(cmap, world->zSurfColorString, world->zSurfColor);

  for (i=0; i<MAX_SURFS; i++)
    if (world->enumSurfColorString[i])
      world->enumSurfColor[i] =
	DecodeAndPackColor(cmap, world->enumSurfColorString[i], world->enumSurfColor[i]);
}


static int ReadEntireRes( VRWorld *world, int volNumber, char **argv, int optind )
{
  world->xeRes[volNumber] = atoi(argv[optind++]);
  world->yeRes[volNumber] = atoi(argv[optind++]);
  world->zeRes[volNumber] = atoi(argv[optind++]);

  if ( !PowerOfTwo(world->xeRes[volNumber]) || !PowerOfTwo(world->yeRes[volNumber]) ||
       !PowerOfTwo(world->zeRes[volNumber]) ) {

    fprintf( stderr,
	     "The volume resolution (\"%4d,%4d,%4d\") must be a power of two on each side!\n",
	     world->xeRes[volNumber], world->yeRes[volNumber], world->zeRes[volNumber] );
    usage();
  }

  world->xScl[volNumber] *= 1;
  world->yScl[volNumber] *= (float)world->yeRes[volNumber] / (float)world->xeRes[volNumber];
  world->zScl[volNumber] *= (float)world->zeRes[volNumber] / (float)world->xeRes[volNumber];

  return( optind );
}


static void MakeEmptyVolume( VRWorld *world, int volNumber )
{
  char *argv[] = { "4", "4", "4" };

  ReadEntireRes(world, volNumber, argv, 0);
  world->basename[volNumber] = NULL;
}


void LoadVRWorld( VRWorld *world, int argc, char **argv )
{
  int compilationOptions=FALSE;
  int ch;

  progname = strrchr(argv[0], '/');

  if (progname != NULL) progname++;
  else                  progname = argv[0];

  if (argc >= 2)
    if (!strcmp(argv[1], "-magic")) {
      optind++;
      world->magicExtensions = TRUE;
    }

  /*
   * Set up the default values
   */
  world->disp               = theApplication->display();
  world->scrn               = DefaultScreen(world->disp);

  while ((ch = getopt(argc, argv, "C:A:T:M:P:F:S:p:er:z:f:amc:QOD:V")) != EOF) {

    switch ((char) ch) {

    default:
    case '?':
      fprintf( stderr, "%s: unknown interface option -%c\n",
	       progname, ch );
      usage();
      break;

    case 'C':
      if (world->nCMaps < MAX_MAPS) {
	world->mapName[world->nMaps] = optarg;
	world->mapType[world->nMaps] = COLORMAP;
	world->nMaps++;
	world->nCMaps++;
      } else
	fprintf(stderr,
		"Warning: Unable to load map \"%s\", only %d colormaps are allowed!\n",
		optarg, MAX_MAPS);
      break;

    case 'A':
      if (world->nAMaps < MAX_MAPS) {
	world->mapName[world->nMaps] = optarg;
	world->mapType[world->nMaps] = ALPHAMAP;
	world->nMaps++;
	world->nAMaps++;
      } else
	fprintf(stderr,
		"Warning: Unable to load map \"%s\", only %d alphamaps are allowed!\n",
		optarg, MAX_MAPS);
      break;

    case 'T':
      if ((world->nCMaps < MAX_MAPS) && (world->nAMaps < MAX_MAPS)) {
	world->mapName[world->nMaps] = optarg;
	world->mapType[world->nMaps] = TABLEMAP;
	world->nMaps++;
	world->nCMaps++;
	world->nAMaps++;
      } else
	fprintf(stderr,
		"Warning: Unable to load map \"%s\", only %d color/alphamaps are allowed!\n",
		optarg, MAX_MAPS);
      break;

    case 'M':
      // Add the default DISPLAY environment variable monitor
      if ( world->nMonitors == 0 ) {
	world->monitorName[0] = getenv("DISPLAY");
	world->nMonitors++;
      }

      if ( world->nMonitors < MAX_MONITORS ) {
	world->monitorName[world->nMonitors] = optarg;
	world->nMonitors++;
      }
      break;

    case 'P':
      world->polygonFile = optarg;
      break;

    case 'F':
      world->snapFile = optarg;
      break;

    case 'S':
      world->slices = atoi(optarg);
      if (world->slices < world->sliceMin ) {
	fprintf( stderr,
		 "The slice count must be at least %d, setting it to %d\n",
		 world->sliceMin, world->sliceMin );
	world->slices = world->sliceMin;
      }
      if (world->slices > world->sliceMax ) {
	fprintf( stderr,
		 "The slice count must be no greater than %d, setting it to %d\n",
		 world->sliceMax, world->sliceMax );
	world->slices = world->sliceMax;
      }
      break;

    case 'p':
      world->nPlanes = atoi(optarg);
      if (world->nPlanes < 1)
      {
	fprintf( stderr, "You must have at least one slice plane!\n" );
	world->nPlanes = 1;
      }
      if (world->nPlanes > MAX_PLANES)
      {
	fprintf( stderr, "No more than %d clipping planes are supported!\n",
		 MAX_PLANES );
	world->nPlanes = MAX_PLANES;
      }
      break;

    case 'e':
#ifdef VR_STEREO
      world->allowStereoMode = TRUE;
#else
      fprintf(stderr, "Warning: Program not compiled with stereo support!\n");
      fprintf(stderr, "Warning: Stereo will be disabled!\n");
#endif
      break;

    case 'r':
      world->width = atoi(optarg);
      world->height = atoi(argv[optind++]);
      if (world->width > 1024)  world->width = 1024;
      if (world->width < 10)    world->width = 10;
      if (world->height > 1024) world->height = 1024;
      if (world->height < 10)   world->height = 10;
      break;

    case 'f':
      world->snapWidth = atoi(optarg);
      world->snapHeight = atoi(argv[optind++]);
      if (world->snapWidth > 10000)  world->snapWidth = 10000;
      if (world->snapWidth < 10)     world->snapWidth = 10;
      if (world->snapHeight > 10000) world->snapHeight = 10000;
      if (world->snapHeight < 10)    world->snapHeight = 10;
      break;

    case 'a':
      world->modAmap = !world->modAmap;
      break;

    case 'm':
      world->modCmap = !world->modCmap;
      break;

    case 'c':
      if      (!strcasecmp(optarg, "frame"))
	world->frameColorString      = argv[optind++];
      else if (!strcasecmp(optarg, "cframe"))
	world->curFrameColorString   = argv[optind++];
      else if (!strcasecmp(optarg, "plane"))
	world->planeColorString      = argv[optind++];
      else if (!strcasecmp(optarg, "cplane"))
	world->curPlaneColorString    = argv[optind++];
      else if (!strcasecmp(optarg, "point"))
	world->pointColorString       = argv[optind++];
      else if (!strcasecmp(optarg, "cpoint"))
	world->curPointColorString    = argv[optind++];
      else if (!strcasecmp(optarg, "xvec"))
	world->xVectorColorString     = argv[optind++];
      else if (!strcasecmp(optarg, "yvec"))
	world->yVectorColorString     = argv[optind++];
      else if (!strcasecmp(optarg, "zvec"))
	world->zVectorColorString     = argv[optind++];
      else if (!strcasecmp(optarg, "upvec"))
	world->upVectorColorString    = argv[optind++];
      else if (!strcasecmp(optarg, "northvec"))
	world->northVectorColorString = argv[optind++];
      else if (!strcasecmp(optarg, "frontpoly"))
	world->frontPolyColorString   = argv[optind++];
      else if (!strcasecmp(optarg, "backpoly"))
	world->backPolyColorString    = argv[optind++];
      else if (!strcasecmp(optarg, "frontelev"))
	world->frontElevColorString   = argv[optind++];
      else if (!strcasecmp(optarg, "backelev"))
	world->backElevColorString    = argv[optind++];
      else if (!strcasecmp(optarg, "back"))
	world->backColorString        = argv[optind++];
      else if (!strcasecmp(optarg, "xysurf"))
	world->xySurfColorString   = argv[optind++];
      else if (!strcasecmp(optarg, "zsurf"))
	world->zSurfColorString   = argv[optind++];
      else if (!strncasecmp(optarg, "surf", 4)) {
	int i = atoi(optarg+4) - 1;
	if ((i >= 0) && (i < (MAX_SURFS-1)))
	  world->enumSurfColorString[i] = argv[optind++];
      } else
      {
	fprintf( stderr, "%s: illegal object name given to -c (\"%s\").\n",
		 progname, optarg );
	usage();
      }
      break;

    case 'Q':
      // No longer supported, now we always display in the status bar
      break;

    case 'O':
      compilationOptions = TRUE;
      break;

    case 'V':
      world->verbose = TRUE;
      break;

    case 'D':
      world->debug = atoi(optarg);
      break;
    }
  }

  if (compilationOptions) {

    printf("\n");
    printf("Compilation information for %s version %s (for %s):\n",
#ifdef VR_IRISGL
	   VR_NAME_IRISGL,
#endif
#ifdef VR_OPENGL
           VR_NAME_OPENGL,
#endif
	   VR_VERSION,
	   VR_NAME_MACHINE);
#ifdef VR_INTERLEAVE
    printf("  %s\n", VR_NAME_INTERLEAVE);
#endif
#ifdef VR_TEXOBJS
    printf("  %s\n", VR_NAME_TEXOBJS);
#endif
#ifdef VR_DISPLIST
    printf("  %s\n", VR_NAME_DISPLIST);
#endif
#ifdef VR_AUXBUFFER
    printf("  %s\n", VR_NAME_AUXBUFFER);
#endif
#ifdef VR_TLUTS
    printf("  %s\n", VR_NAME_TLUTS);
#endif
#ifdef VR_SUBTEX
    printf("  %s\n", VR_NAME_SUBTEX);
#endif
#ifdef VR_STEREO
    printf("  %s\n", VR_NAME_STEREO);
#endif
#ifdef VR_MERGING
    printf("  %s\n", VR_NAME_MERGING);
#endif
#ifdef VR_MATTING
    printf("  %s\n", VR_NAME_MATTING);
#endif
#ifdef VR_SHADING
    printf("  %s\n", VR_NAME_SHADING);
#endif
    printf("\n");
    exit( EXIT_SUCCESS );
  }

  while ((ch = getopt(argc, argv, "m:r:f:4U:N:C:RAVD:")) != EOF) {

    switch ((char)ch) {

    default:
    case '?':
      fprintf( stderr, "%s: unknown brick option -%c\n",
	       progname, ch );
      usage();
      break;

    case 'm':
      world->slabs = atoi(optarg);
      world->downSample = FALSE;
      if (world->slabs < 3) {
	fprintf( stderr,
		 "The slab count must be at least 4, setting it to 4\n");
	world->slabs = 4;
      }
      break;

    case 'r':
      if (!strcasecmp(optarg, "no") || !strcasecmp(optarg, "n") ||
	  !strcasecmp(optarg, "false") || !strcasecmp(optarg, "f") ||
	  !strcasecmp(optarg, "off") || !strcasecmp(optarg, "-"))
	world->downSample = FALSE;
      else {
	fprintf( stderr, "%s: illegal boolean given to -r (\'%s\').\n",
		 progname, optarg );
	usage();
      }
      break;

    case 'f':
      if      (!strcasecmp(optarg, "box"))
	world->resampleFilter = BoxFilter;
      else if (!strcasecmp(optarg, "triangle"))
	world->resampleFilter = TriangleFilter;
      else {
	fprintf( stderr, "%s: illegal filter given to -f (\'%s\').\n",
		 progname, optarg[0] );
	usage();
      }
      break;

    case '4':
      world->define4BitBricks = TRUE;
      break;

    case 'U':
      if        (tolower(optarg[0]) == 'x') {
	world->surfaceNormal = NORMAL_IS_X;
      } else if (tolower(optarg[0]) == 'y') {
	world->surfaceNormal = NORMAL_IS_Y;
      } else if (tolower(optarg[0]) == 'z') {
	world->surfaceNormal = NORMAL_IS_Z;
      } else {
	fprintf( stderr, "%s: illegal direction given to -U (\'%c\').\n",
		 progname, optarg[0] );
	usage();
      }
      break;

    case 'N':
      world->drawAxes = FALSE;
      world->northAngle = atof(optarg);
      while (world->northAngle < -180.0)
	world->northAngle += 360.0;
      while (world->northAngle >  180.0)
	world->northAngle -= 360.0;
      world->northAngle = world->northAngle/180.0*M_PI;
      break;

    case 'C':
      world->byteCompiling = TRUE;
      world->outname = optarg;
      break;

    case 'R':
      world->readCompiled = TRUE;
      break;

    case 'A':
      world->allowDataReread = TRUE;
      break;

    case 'V':
      world->verbose = TRUE;
      break;

    case 'D':
      world->debug = atoi(optarg);
      break;
    }
  }

  int volNumber = 0;
  int elevNumber = 0;

  while (optind < argc) {

    int objType;

    if (!strcmp(argv[optind], "--"))
      optind++;

    if (!strcmp(argv[optind], "volume")) {
      objType = OBJ_VOLUME;
      optind++;
    } else if (!strcmp(argv[optind], "elevation")) {
      objType = OBJ_ELEVATION;
      optind++;
    } else if (argv[optind][0] == '-')
      // Assume that the script is old format and a volume follows
      objType = OBJ_VOLUME;
    else {
      fprintf(stderr, "%s: unknown object type \"%s\"\n",
	      progname, argv[optind]);
      usage();
    }

    switch (objType) {
    case OBJ_VOLUME:

      world->extension[volNumber] = defExtension;

      while ((ch = getopt(argc, argv, "hX:Y:Z:T:S:e:o:O:N")) != EOF) {
	switch ((char)ch) {
	default:
	case '?':
	  fprintf( stderr, "%s: unknown volume option -%c\n",
		  progname, ch );
	  usage();
	  break;

	case 'h':
	  world->handedness[volNumber] = RIGHT;
	  break;

	case 'X':
	  world->xScl[volNumber] = atof(optarg);
	  break;

	case 'Y':
	  world->yScl[volNumber] = atof(optarg);
	  break;

	case 'Z':
	  world->zScl[volNumber] = atof(optarg);
	  break;

	case 'T':
	  world->teRes[volNumber] = atoi(optarg);
	  world->timeVariantVolumes = TRUE;
	  break;

	case 'S':
	  world->timeInterval[volNumber] = atoi(optarg);
	  break;

	case 'e':
	  world->extension[volNumber] = optarg;
	  break;

	case 'o':
	  world->sOff[volNumber] = atoi(optarg);
	  break;

	case 'O':
	  world->xOff[volNumber] = atoi(optarg);
	  world->yOff[volNumber] = atoi(argv[optind++]);
	  break;
	}
      }

      int argsLeft=optind;
      while ((argsLeft < argc) && (argv[argsLeft][0] != '-'))
	argsLeft++;
      argsLeft -= optind;
 
      if (argsLeft == 4) {
	optind = ReadEntireRes(world,    volNumber, argv, optind);
      } else {
	fprintf(stderr, "Invalid number of parameters (four required)!\n");
	fprintf(stderr, "Expected format (for each volume): x y z data\n");
	fprintf(stderr, "Provided %s %d parameter%s.\n", (argsLeft==1) ? "was" : "were",
		argsLeft, (argsLeft==1) ? "" : "s");
	usage();
      }

      world->basename[volNumber] = argv[optind];
      optind++;

      volNumber++;
      world->nVols++;
      break;

    case OBJ_ELEVATION:

      while ((ch = getopt(argc, argv, "X:Y:Z:LR:T:")) != EOF) {
	switch ((char)ch) {
	default:
	case '?':
	  fprintf( stderr, "%s: unknown elevation option -%c\n",
		  progname, ch );
	  usage();
	  break;

	case 'X':
	  world->elevXScl[elevNumber] = atof(optarg);
	  break;

	case 'Y':
	  world->elevYScl[elevNumber] = atof(optarg);
	  break;

	case 'Z':
	  world->elevZScl[elevNumber] = atof(optarg);
	  break;

	case 'L':
	  world->elevLight[elevNumber] = FALSE;
	  break;

	case 'R':
	  world->elevRot[elevNumber] = atof(optarg);
	  break;

	case 'T':
	  world->elevXTrn[elevNumber] = atof(optarg++);
	  world->elevYTrn[elevNumber] = atof(argv[optind++]);
	  world->elevZTrn[elevNumber] = atof(argv[optind++]);
	  break;
	}
      }

      argsLeft=optind;
      while ((argsLeft < argc) && (argv[argsLeft][0] != '-'))
	argsLeft++;
      argsLeft -= optind;
 
      if (argsLeft == 1) {

	world->elevElevImg[elevNumber]  = argv[optind++];

      } else if (argsLeft == 2) {

	if (world->frontElevColorString == NULL)
	  world->frontElevColor = OPAQUE_WHITE;
	if (world->backElevColorString == NULL)
	  world->backElevColor = OPAQUE_WHITE;

	world->elevColorImg[elevNumber] = argv[optind++];
	world->elevElevImg[elevNumber]  = argv[optind++];

      } else {
	fprintf(stderr, "Invalid number of parameters (one or two required)!\n");
	fprintf(stderr, "Expected format (for each elevation): "
		"[colorimg] elevimg\n");
	fprintf(stderr, "Provided %s %d parameter%s.\n", (argsLeft==1) ? "was" : "were",
		argsLeft, (argsLeft==1) ? "" : "s");
	usage();
      }

      elevNumber++;
      world->nElevs++;
      break;
    }
  }

  if (world->multiResMode && (world->nVols > 1)) {
    fprintf(stderr, "Multiple resolution mode is not supported for "
	    "multiple volumes!\n");
    exit( EXIT_FAILURE );
  }

  if (world->nVols == 0) {
    world->doTextures = FALSE;
    MakeEmptyVolume(world, world->nVols++);
  }

#ifndef VR_MERGING
  world->allowRenderMode[MERGE_RENDER] = FALSE;
#endif
#ifndef VR_MATTING
  world->allowRenderMode[MATTE_RENDER] = FALSE;
#endif
#ifndef VR_SHADING
  world->allowRenderMode[SHADE_RENDER] = FALSE;
#endif

  if (world->verbose) {

    if (world->downSample) {
      if (world->brickResGiven[0])
	printf("Volren will ignore brick resolution provided "
	       "as down sampling overrides it.\n");

      if (world->slabs > 0)
	printf("Volren will ignore slab count (-m) provided "
	       "as down sampling overrides it.\n");

      printf("Volren will down sample the data as it is read in.\n");

    } else {

      if (world->brickResGiven[0]) {
	if (world->slabs != 0)
	  printf("Volren will ignore slab count (-m) provided "
		 "as provided brick resolution overrides it.\n");

	printf("Volren will create %d by %d by %d bricks as requested.\n",
	       world->xbRes, world->ybRes, world->zbRes);

      } else if (world->slabs != 0) {

	printf("Volren will create %d slabs of volume bricks.\n",
	       1, 1, world->slabs);
      }
    }

    printf("Volren is running in %s resolution mode.\n",
	   world->multiResMode ? "multiple" : "single");

    if (world->multiResMode)
      printf("Volren will display %d by %d by %d subvolumes.\n",
	     world->xsRes, world->ysRes, world->zsRes);

    if (world->basename[0])
      printf("Volren will read %s data, getting it from \"%s###%s\".\n",
	     world->readCompiled ? "byte compiled" : "slice", world->basename[0],
	     world->extension[0]);

    if (world->byteCompiling)
      if (world->readCompiled)
	printf("Volren will ignore byte compile flag (-C) as the data is already compiled.\n");
      else
	printf("Volren will byte compile the slice data, placing it in \"%s\"+suffix.\n",
	       world->outname);
  }

  DecodeUserColors( world );
}
