 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for msvd
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lmsvd
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lmsvd - compute singular value decomposition (SVD)
* 
*       Purpose: lmsvd computes the singular value decomposition of a
*                general matrix, A=USVt, where U and V are orthogonal/unitary
*                and S is diagonal, real, and non-negative. Here, Vt means
*                transpose(V).
*
*                For MxN matrix A, U is MxM, S is MxN, and V is NxN where
*                the columns of U and V contain the left and right singular
*                vectors respectively and the diagonal elements of S are the
*                singular values in descending sorted order.
*
*                lmsvd uses the DGESVD and ZGESVD routines from LAPACK to
*                obtain the decomposition. 
*
*         Input: *                kobject in_obj - input matrix object
*
*        Output: *                kobject u_obj - matrix object to receive the U matrix
*                kobject s_obj - matrix object to receive the S matrix
*                kobject v_obj - matrix object to receive the V matrix
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lmsvd(
kobject in_obj,
kobject u_obj,
kobject s_obj,
kobject v_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kmatrix", *rtn = "lmsvd";
        double *A=NULL,*U=NULL,*S=NULL,*V=NULL,*WORK=NULL,*SVEC=NULL;
        double *RWORK=NULL;
        kdcomplex *AZ=NULL,*UZ=NULL,*SZ=NULL,*VZ=NULL,*WORKZ=NULL;
        int i,j,m,n,num,nplanes,lda,*ipiv,info;
        int type, otype;
        int wc,hc,dc,tc,ec;
        int lwork_len,rwork_len,s_len;
        klist *objlist=NULL;
        char jobu,jobvt;
        double tmp;
        kdcomplex tmpz;

        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input object");
            return(FALSE);
          }
        if (u_obj == KOBJECT_INVALID && s_obj == KOBJECT_INVALID &&
            v_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Must have at least one output object");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");

        /* See if the data type of the input object is real or complex */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_DATA_TYPE,&type));
        if (type == KCOMPLEX || type == KDCOMPLEX) otype = KDCOMPLEX;
        else otype = KDOUBLE;

        /* Constrain the input data to double real or double complex for read */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_DATA_TYPE, otype));

        /* See how big the data is */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec));
        /* Set up to process input by planes */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,wc,hc,1,1,1));
        nplanes = dc*tc*ec;
        s_len = kmin(wc,hc);

        /* Set the output object sizes and data types */
        if (u_obj != KOBJECT_INVALID)
          {
            if (!kpds_query_value(u_obj)) KCALL(kpds_create_value(u_obj));
            KCALL(kpds_set_attributes(u_obj,
                                 KPDS_VALUE_SIZE,hc,hc,dc,tc,ec,
                                 KPDS_VALUE_DATA_TYPE,otype,NULL));
            KCALL((u_obj = kpds_reference_object(u_obj)));
            objlist = klist_add(objlist,u_obj,"KOBJECT");
            KCALL(kpds_set_attribute(u_obj,KPDS_VALUE_REGION_SIZE,
                                            hc,hc,1,1,1));
          }
        if (s_obj != KOBJECT_INVALID)
          {
            if (!kpds_query_value(s_obj)) KCALL(kpds_create_value(s_obj));
            KCALL(kpds_set_attributes(s_obj,
                                 KPDS_VALUE_SIZE,wc,hc,dc,tc,ec,
                                 KPDS_VALUE_DATA_TYPE,otype,NULL));
            KCALL((s_obj = kpds_reference_object(s_obj)));
            objlist = klist_add(objlist,s_obj,"KOBJECT");
            KCALL(kpds_set_attribute(s_obj,KPDS_VALUE_REGION_SIZE,
                                            wc,hc,1,1,1));
          }
        if (v_obj != KOBJECT_INVALID)
          {
            if (!kpds_query_value(v_obj)) KCALL(kpds_create_value(v_obj));
            KCALL(kpds_set_attributes(v_obj,
                                 KPDS_VALUE_SIZE,wc,wc,dc,tc,ec,
                                 KPDS_VALUE_DATA_TYPE,otype,NULL));
            KCALL((v_obj = kpds_reference_object(v_obj)));
            objlist = klist_add(objlist,v_obj,"KOBJECT");
            KCALL(kpds_set_attribute(v_obj,KPDS_VALUE_REGION_SIZE,
                                            wc,wc,1,1,1));
          }

        /* Allocate plane buffers */
        if (otype == KDOUBLE)
          {
            KCALL(!((U=(double *)kmalloc(hc*hc*sizeof(double))) == NULL));
            objlist = klist_add(objlist,U,"KMALLOC");
            KCALL(!((S=(double *)kmalloc(wc*hc*sizeof(double))) == NULL));
            objlist = klist_add(objlist,S,"KMALLOC");
            KCALL(!((V=(double *)kmalloc(wc*wc*sizeof(double))) == NULL));
            objlist = klist_add(objlist,V,"KMALLOC");
            KCALL(!((A=(double *)kmalloc(wc*hc*sizeof(double))) == NULL));
            objlist = klist_add(objlist,A,"KMALLOC");
          }
        else
          {
            KCALL(!((UZ=(kdcomplex *)kmalloc(hc*hc*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,UZ,"KMALLOC");
            KCALL(!((SZ=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,SZ,"KMALLOC");
            KCALL(!((VZ=(kdcomplex *)kmalloc(wc*wc*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,VZ,"KMALLOC");
            KCALL(!((AZ=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,AZ,"KMALLOC");
          }

        /* Allocate SVEC array */
        KCALL(!((SVEC=(double *)kmalloc(s_len*sizeof(double))) == NULL));
        objlist = klist_add(objlist,SVEC,"KMALLOC");

        /* Allocate WORK array */
        if (otype == KDOUBLE)
          {
            lwork_len = 2*kmax(3*kmin(wc,hc)+kmax(wc,hc),5*kmin(wc,hc-4));
            KCALL(!((WORK=(double *)kmalloc(lwork_len*sizeof(double))) == NULL));
            objlist = klist_add(objlist,WORK,"KMALLOC");
          }
        else
          {
            lwork_len = 2*(2*kmin(wc,hc)+kmax(wc,hc));
            KCALL(!((WORKZ=(kdcomplex *)kmalloc(lwork_len*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,WORKZ,"KMALLOC");
            rwork_len = 5*kmax(wc,hc);
            KCALL(!((RWORK=(double *)kmalloc(rwork_len*sizeof(double)))== NULL));
            objlist = klist_add(objlist,RWORK,"KMALLOC");

          }

        jobu = 'A';
        if (u_obj == KOBJECT_INVALID) jobu = 'N';
        jobvt = 'A';
        if (v_obj == KOBJECT_INVALID) jobvt = 'N';

        for (num=0; num<nplanes; num++)
          {
            /* Get a plane from the input object */
            if (otype == KDOUBLE)
              A = (double *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)A);
            else
              AZ=(kdcomplex *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)AZ);

             /* Transpose so that F77 likes it */
            if (otype == KDOUBLE)
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    S[j*hc+i] = A[i*wc+j];
              }
            else
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    SZ[j*hc+i] = AZ[i*wc+j];
              }

            /* Obtain SVD */
            if (otype == KDOUBLE)
              DGESVD(&jobu,&jobvt,&hc,&wc,S,&hc,
                     SVEC,U,&hc,V,&wc,WORK,&lwork_len,&info);
            else ZGESVD(&jobu,&jobvt,&hc,&wc,SZ,&hc,
                     SVEC,UZ,&hc,VZ,&wc,WORKZ,&lwork_len,RWORK,&info);

            if (info != 0)
              {
                kinfo(KSTANDARD,"lmsvd: LAPACK {dz}gesvd() returned with INFO=%d\n\
where INFO < 0 -> for INFO = -i, the i'th argument had an illegal value and\n\
      INFO > 0 -> for INFO = i, SVD failed to converge, i off-diagonal\n\
                                elements did not converge to zero.\n",info);
              }

            /* Untranspose U so that C likes it */
            if (otype == KDOUBLE)
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<i; j++)
                    {
                      tmp = U[j*hc+i];
                      U[j*hc+i] = U[i*hc+j];
                      U[i*hc+j] = tmp;
                    }
              }
            else
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<i; j++)
                    {
                      tmpz = UZ[j*hc+i];
                      UZ[j*hc+i] = UZ[i*hc+j];
                      UZ[i*hc+j] = tmpz;
                    }
              }

            /* V is supplied by the SVD as V', so it is already the way
               we want it */

            /* Read the values out of SVEC and plant'em in S */
            if (otype == KDOUBLE)
              {
                kbzero(S,wc*hc*sizeof(double));
                for (i=0; i<kmin(wc,hc); i++) S[i*wc+i] = SVEC[i];
              }
            else
              {
                kbzero(SZ,wc*hc*sizeof(kdcomplex));
                for (i=0; i<kmin(wc,hc); i++) SZ[i*wc+i].r = SVEC[i];
              }

            /* Write the data to the output objects */
            if (otype == KDOUBLE)
              {
                if (u_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(u_obj,KPDS_VALUE_REGION,(kaddr)U));
                if (s_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(s_obj,KPDS_VALUE_REGION,(kaddr)S));
                if (v_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(v_obj,KPDS_VALUE_REGION,(kaddr)V));
              }
            else
              {
                if (u_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(u_obj,KPDS_VALUE_REGION,(kaddr)UZ));
                if (s_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(s_obj,KPDS_VALUE_REGION,(kaddr)SZ));
                if (v_obj != KOBJECT_INVALID)
                  KCALL(kpds_put_data(v_obj,KPDS_VALUE_REGION,(kaddr)VZ));
              }
          }

        (void)klist_free(objlist, (kfunc_void)lkcall_free);

	return TRUE;
}
/* -library_code_end */
