 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for mcovar
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lmcovar
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* Around 10000 vectors seems to be a reasonable comprimise for most
architectures and operating systems. Your timings may vary and might
be improved by playing with this number. Look out for interactions with
the DMS buffer size though. */
#define MAX_VECTORS 10000

/* Special mapping macro for 5D access to the data region where
   (wi,hi,di,ti,ei) is the element we want access and
   (w,h,d,t,e) is the dimension of the region. Note that
   the use of this gimmick does not make for the most
   efficient code, but it is at least correct, which is
   the main goal. */
#define PIXEL(wi,hi,di,ti,ei) \
          ((ei)*w*h*d*t+(ti)*w*h*d+(di)*w*h+(hi)*w+(wi))

/* Special code compression games. Also not the most efficient,
   but will get the correct answer without fail. The order of
   loops is set up in an attempt to let the memory system
   do the best it can with the cache. If you flip the loop
   order it will cost about 10% more execution time. */
#define SCAN_OVER_REGION \
               for (ei=0; ei<e; ei++) \
                 { \
                   for (ti=0; ti<t; ti++) \
                     { \
                       for (di=0; di<d; di++) \
                         { \
                           for (hi=0; hi<h; hi++) \
                             { \
                               for (wi=0; wi<w; wi++) \
                                 {

#define SCAN_OVER_REGION_CLOSE \
                                 } \
                             } \
                         } \
                     } \
                 }
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lmcovar - compute the covariance matrix for a given set of data vectors
* 
*       Purpose: lmcovar is used to compute the covariance matrix for the set
*                of data vectors supplied in the input data object. The
*                covariance matrix is computed for all w*h*d*t data vectors,
*                each of length e. The output will be of type DOUBLE or DCOMPLEX
*                depending on the input data object.
*
*         Input: in_obj - input data object
*
*        Output: out_obj - output covariance matrix object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lmcovar(
kobject in_obj,
kobject out_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kmatrix", *rtn = "lmcovar";
        klist *objlist=NULL;
        int nvecs,nrgns;
        int n,k,offset;
        int type,ttype;
        int w,h,d,t,e;
        int wi,hi,di,ti,ei;
        int wc,hc,dc,tc,ec;
        double *A=NULL,*m=NULL,*vec=NULL;
        kdcomplex *AZ=NULL,*mZ=NULL,*vecZ=NULL;
        
        /* Make sure we have a valid input and output objects */
        if (in_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input object");
            return(FALSE);
          }
        if (out_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus output object");
            return(FALSE);
          }
 
        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be automatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");
 
        /* See if the data type of the input object is real or complex */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_DATA_TYPE,&type));
        if (type == KCOMPLEX || type == KDCOMPLEX) ttype = KDCOMPLEX;
        else ttype = KDOUBLE;
 
        /* Constrain the input data to double or double complex for read */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_DATA_TYPE, ttype));
 
        /* See how big the data is */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec));
 
        /* Come up with some kind of chunk size that makes sense */
        w=wc; h=1; d=1; t=1; e=ec;
        if (wc*hc < MAX_VECTORS) h=hc;
        if (wc*hc*dc < MAX_VECTORS) d=dc;
        if (wc*hc*dc*tc < MAX_VECTORS) t=tc;
        nrgns = wc*hc*dc*tc/(w*h*d*t);
        nvecs = wc*hc*dc*tc;

        /* Set up to process input by vectors */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,w,h,d,t,e));

        /* Set the output object sizes and data types */
        if (!kpds_query_value(out_obj)) KCALL(kpds_create_value(out_obj));
        KCALL(kpds_set_attributes(out_obj,
                             KPDS_VALUE_SIZE,ec,ec,1,1,1,
                             KPDS_VALUE_DATA_TYPE,ttype,NULL));
        KCALL((out_obj = kpds_reference_object(out_obj)));
        objlist = klist_add(objlist,out_obj,"KOBJECT");
	
        /* Set the presentation data type to match that used internally */
        KCALL(kpds_set_attributes(out_obj,KPDS_VALUE_DATA_TYPE,ttype,NULL));

        /* Set up to write the output as the covariance matrix */
        KCALL(kpds_set_attribute(out_obj,KPDS_VALUE_REGION_SIZE,ec,ec,1,1,1));

        /* Allocate buffers */
        if (ttype == KDCOMPLEX)
          {
            KCALL(!((AZ=(kdcomplex *)kmalloc(ec*ec*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,AZ,"KMALLOC");
            kbzero(AZ,ec*ec*sizeof(kdcomplex));
            KCALL(!((mZ=(kdcomplex *)kmalloc(ec*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,mZ,"KMALLOC");
            kbzero(mZ,ec*sizeof(kdcomplex));
            KCALL(!((vecZ=(kdcomplex *)kmalloc(w*h*d*t*e*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,vecZ,"KMALLOC");
          }
        else
          {
            KCALL(!((A=(double *)kmalloc(ec*ec*sizeof(double)))== NULL));
            objlist = klist_add(objlist,A,"KMALLOC");
            kbzero(A,ec*ec*sizeof(double));
            KCALL(!((m=(double *)kmalloc(ec*sizeof(double)))== NULL));
            objlist = klist_add(objlist,m,"KMALLOC");
            kbzero(m,ec*sizeof(double));
            KCALL(!((vec=(double *)kmalloc(w*h*d*t*e*sizeof(double)))== NULL));
            objlist = klist_add(objlist,vec,"KMALLOC");
          }

        /* First compute the mean vector */
        for (n=0; n<nrgns; n++)
          {
            if (ttype == KDCOMPLEX)
             {
               vecZ=(kdcomplex *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)vecZ);
               SCAN_OVER_REGION
                 mZ[ei] = kdcadd(mZ[ei],vecZ[PIXEL(wi,hi,di,ti,ei)]);
               SCAN_OVER_REGION_CLOSE
             }
            else
             {
               vec=(double *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)vec);
               SCAN_OVER_REGION
                 m[ei] += vec[PIXEL(wi,hi,di,ti,ei)];
               SCAN_OVER_REGION_CLOSE
             }
          }
        if (ttype == KDCOMPLEX)
          {
            for (ei=0; ei<ec; ei++)
              {
                mZ[ei].r /= nvecs;
                mZ[ei].i /= nvecs;
              }
          }
        else
          {
            for (ei=0; ei<ec; ei++) m[ei] /= nvecs;
          }

        /* Reset the position back to 0,0,0,0,0 and then compute 
           the covariance matrix */
        kpds_set_attribute(out_obj,KPDS_VALUE_POSITION,0,0,0,0,0);
        for (n=0; n<nrgns; n++)
          {
            if (ttype == KDCOMPLEX)
             {
               vecZ=(kdcomplex *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)vecZ);
               /* Subtract the mean vector from each vector in the region */
               SCAN_OVER_REGION
                 offset = PIXEL(wi,hi,di,ti,ei);
                 vecZ[offset] = kdcsub(vecZ[offset],mZ[ei]);
               SCAN_OVER_REGION_CLOSE
               SCAN_OVER_REGION
                 for (k=0; k<ec; k++) 
                   AZ[ei*ec+k] = kdcadd(AZ[ei*ec+k],
                                 kdcmult(vecZ[PIXEL(wi,hi,di,ti,ei)],
                                         vecZ[PIXEL(wi,hi,di,ti,k)]));
               SCAN_OVER_REGION_CLOSE
              }
            else
              {
               vec=(double *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)vec);
               SCAN_OVER_REGION
                 offset = PIXEL(wi,hi,di,ti,ei);
                 vec[offset] = vec[offset]-m[ei];
               SCAN_OVER_REGION_CLOSE
               SCAN_OVER_REGION
                 for (k=0; k<ec; k++) 
                   A[ei*ec+k] += 
                     vec[PIXEL(wi,hi,di,ti,ei)]*vec[PIXEL(wi,hi,di,ti,k)];
               SCAN_OVER_REGION_CLOSE
              }
          }
        /* Normalize the covariance matrix estimnate */
        if (ttype == KDCOMPLEX)
          {
            for (k=0; k<ec*ec; k++)
              {
                AZ[k].r /= nvecs;
                AZ[k].i /= nvecs;
              }
          }
        else
          {
            for (k=0; k<ec*ec; k++) A[k] /= nvecs;
          }

        /* Write out the covariance matrix */
        if (ttype == KDCOMPLEX)
          {
            KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)AZ));
          }
        else
          {
            KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)A));
          }
        
        (void)klist_free(objlist, (kfunc_void)lkcall_free);
	return TRUE;
}
/* -library_code_end */
