 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for igradient
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	ligradient
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#define KERNEL_DEPTH   1         /* depth size of the kernel */
#define KERNEL_TIME    1         /* time size of the kernel */
#define KERNEL_HEIGHT  3         /* kernel height (rows) */
#define KERNEL_BANDS   1         /* number of bands in kernel */
#define REGION_COL_INC        0         /* region col increment factor */
#define REGION_ROW_INC        1         /* region row increment factor */
#define REGION_BAND_INC       1         /* region band increment factor */
#define REGION_ROW_OFFSET    -1         /* region row offset */
#define REGION_COL_OFFSET    -1         /* region column offset */
#define REGION_BAND_OFFSET    0         /* region band offset */
#define REGION_DEPTH_OFFSET   0         /* region depth offset */
#define REGION_TIME_OFFSET    0         /* region time offset */
#define SQROOT2		1.414213562
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: ligradient - perform image differentiation using gradient operator
* 
*       Purpose: takes an image and performs the differentiation (high pass 
*		 spatial filtering or edge enhancement) operation by using 
*		 standard gradient operators.  The standard gradient operators 
*		 supported by ligradient are: Robert's, Sobel, Prewitt and 
*		 Isotropic).  The Robert's operators use a 2x2 kernel, whereas 
*		 the other three use a 3x3 kernel.  Each operator uses two 
*		 templates to enhance edges in the horizontal and vertical 
*		 directions only.  For edge enhancements in other directions 
*		 other operators should be used. 
*
*		 There are two approximations to the true gradient.  The user 
*		 can select between calculating the magnitude of the gradient 
*		 by using squares and square roots (Euclidean distance 
*		 evaluation), or by using absolute differences.  The latter 
*		 is computational faster.
*
*		 Mask data is ignored.  
*
*         Input: src - the source data object to be processed
*		 gradient_type - type of gradient calculation
*                operation_type - absolute values (default) or squares and
*                                 square root operations
*        Output: dest - the destination data object
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Ramiro Jordan, Jeremy Worley
*          Date: Apr 12, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int ligradient ( 
		 kobject src,
                 int gradient_type,
                 int operation_type,
                 kobject dest )
/* -library_def_end */

/* -library_code */
{
   kobject    src_ref, dest_ref; /* source and destination reference objects */
   int width;                   /* image width (num cols) */
   int height;                  /* image height (num rows) */
   int depth;                   /* depth and time give number of images */
   int tsize;                    
   int num_bands;               /* number of bands per image */
   int num_regions;             /* number of regions per image */
   int region_width;            /* region width */
   int data_type;                  /* object data type */
   int pdata_type;                 /* data type for processing */
   long   *idata=NULL, *ival=NULL;
   long   ixval=0, iyval=0;
   double *ddata=NULL, *dval=NULL;
   double dxval=0.0, dyval=0.0;
   int    k, h, i, j, col;

  /* Check to see if the source and destination objects passed into lroutine
   * (ligradient) are valid data objects. With these checks pushed into the
   * lroutine the driver routine need not necessarily check the status on the
   * kpds_input and kpds_output calls, though it is a good idea to do it.
   * These checks are provided here to make the lroutine complete in its
   * implementation so that it can be called by other programs.
   * (Reusable code..).
   */
 
   if ( !src ) {
      kerror("kim age_proc", "ligradient", "Source object is not a valid data object.");
       return(FALSE);
   }
 
   if ( !dest ) {
      kerror("kimage_proc", "ligradient", "Destination object is not a valid data object.");
       return (FALSE);
   }
 
  /*
   * Check the source object to make sure that it can be processed.
   * This algorithm requires that the source object contain value data.
   * It will not process masked data.
   */
   if (! kpds_query_value(src) ) {
      kerror("kimage_proc", "ligradient", "Source object does not contain value data");
      return(FALSE);
   }
   if ( kpds_query_mask(src) ) {
      kerror("kimage_proc", "ligradient", "Cannot process masked data, use krmseg to remove the mask segment.");
      return(TRUE);
   }
 
  /*
   *  Set up a reference to the source object so that the library does
   *  not have any side effects on the source object's attributes.
   */
   if ((src_ref = kpds_reference_object(src)) == KOBJECT_INVALID) {
       kerror("kimage_proc", "ligradient", "Failed to reference source object.");
       return(FALSE);
   }
 
  /*
   *  Set up a reference to the destination object to facilitate processing.
   */
   if ((dest_ref = kpds_reference_object(dest)) == KOBJECT_INVALID) {
       kerror("kimage_proc", "ligradient", "Failed to reference destination object.");
       return(FALSE);
   }
 
  /*
   * Now set the presentation attributes on the source reference object.
   * These will not be applied to the destination object.
   *
   * The source object may contain a collection of images, thus we get
   * the KPDS_VALUE_SIZE attribute which provides the following information:
   *    - width    = number of columns in the image
   *    - height   = number of rows in the image
   *    - elements = number of bands (elements) per image
   *    - depth and tsize = number of images in the object
   *
   */
   if (! kpds_get_attribute(src_ref, KPDS_VALUE_SIZE,
                  &width, &height, &depth, &tsize, &num_bands) ) {
      kerror("kimage_proc", "ligradient", "kpds_get_attribute failed for KPDS_VALUE_SIZE.");
      return(FALSE);
   }
 
  /*
   * We want to specify the size of the region to process.  Want to
   * select a set of rows from the source image as the region to
   * process.
   *
   * Set the region size to:
   *      - region_width = width of source image + 2 columns
   *           (one extra column at each side of the region)
   *      - KERNEL_HEIGHT = heigth of region (3)
   *      - KERNEL_BANDS = number of bands in the region (1)
   *      - KERNEL_DEPTH = depth size of the region (1)
   *      - KERNEL_TIME  = time size of the region (1)
   * Specifying the region this way allows the program to get a strip
   * of data at a time.
   */
   region_width = width;
   region_width += 2;
   if (! kpds_set_attribute(src_ref, KPDS_VALUE_REGION_SIZE, region_width,
          KERNEL_HEIGHT, KERNEL_DEPTH, KERNEL_TIME, KERNEL_BANDS)) {
      kerror("kimage_proc", "ligradient", "kpds_set_attribute failed for KPDS_REGION_SIZE");
      return(FALSE);
   }
 
  /*
   * Set the source object KPDS_VALUE_OFFSET attribute.
   * This attribute specifies the offset into the data position for processing.
   *
   * Offset position set to:
   *     - REGION_COL_OFFSET = column offset position (-1)
   *     - REGION_ROW_OFFSET = row offset position (-1)
   *     - REGION_BAND_OFFSET = band (element) offset number (0)
   *     - REGION_DEPTH_OFFSET = depth offset number (0)
   *     - REGION_TIME_OFFSET  = time offset number (0)
   *
   *
   * you are here (outside of the original data set)
   *    X  __________________
   *      | | | | | | | | .........   row 0
   *       ------------------
   *      | | | | | | | | .........   row 1
   *       ------------------
   *      | | | | | | | | .........   row 2
   *       ------------------
   *       .   .
   *       .   .
   *       .   .
   *                    each cell represents a pixel
   *       c   c
   *       o   o
   *       l   l
   *
   *       0   2
   */
   if (! kpds_set_attribute(src_ref, KPDS_VALUE_OFFSET, 
			    REGION_COL_OFFSET,
			    REGION_ROW_OFFSET, 
			    REGION_DEPTH_OFFSET, 
			    REGION_TIME_OFFSET, 
			    REGION_BAND_OFFSET)) {
      kerror("kimage_proc", "ligradient", "kpds_set_attribute failed for KPDS_VALUE_OFFSET");
      return(FALSE);
   }
 
  /*
   * Set KPDS_VALUE_INTERPOLATE attribute on source object so that the
   * region elements beyond the input image boundary are zero padded (default).
   * This is an attribute of how the data is to be presented.
   *
   * Interpolate option is set to:
   *     - KPAD = resize image by zero padding
   *
   * you are here (outside of the original data set)
   *    X  0 0 0 0 0 0 0 0 0
   *       __________________
   *    0 | | | | | | | | .........   row 0
   *       ------------------
   *    0 | | | | | | | | .........   row 1
   *       ------------------
   *    0 | | | | | | | | .........   row 2
   *       ------------------
   *       .   .
   *       .   .
   *       .   .
   *                    each cell represents a pixel
   *       c   c
   *       o   o
   *       l   l
   *
   *       0   2
   *
   * Enables a procedure to perform operations with templates (windows)
   * that are not multiple integers of the overall object size.
   */
   if (! kpds_set_attribute(src_ref, KPDS_VALUE_INTERPOLATE, KPAD) ) {
      kerror("kimage_proc", "ligradient", "kpds_set_attribute failed for KPDS_VALUE_INTERPOLATE");
      return(FALSE);
   }
 
  /*
   * Obtain data type of object to be processed.
   *    - data_type = object data type
   */
   if (! kpds_get_attribute(src_ref, KPDS_VALUE_DATA_TYPE, &data_type) ) {
      kerror("kimage_proc", "ligradient", "kpds_get_attribute failed for KPDS_VALUE_DATA_TYPE");
      return(FALSE);
   }
 
  /*
   * Select the appropriate common data type for processing the data.
   * Want to minimize any corruption on the data integrity.
   * Recommended data types are:
   *   - KLONG = signed long integer
   *   - KULONG = unsigned long integer
   *   - KDOUBLE = double precision floating point
   *   - KDCOMPLEX = double precision complex
   */
   pdata_type = kdatatype_cast_process(data_type,data_type,KANYTYPE);
 
  /*
   * *********************************************************************
   *                  Major loop for integer processing.                 *
   * *********************************************************************
   * If the source data is integer representable, process as long int
   * (KLONG).
   *
   * In the destination object the data type will be automatically casted
   * to the appropriate data type.
   */
   if ( (pdata_type == KLONG)  || (pdata_type == KULONG) ) {
      if (! kpds_set_attribute(src_ref, KPDS_VALUE_DATA_TYPE, KLONG)) {
         kerror("kimage_proc", "ligradient", "kpds_set_attribute failed for KPDS_VALUE_DATA_TYPE");
         return(FALSE);
      }
      if (! kpds_set_attribute(dest_ref, KPDS_VALUE_DATA_TYPE, pdata_type)) {
         kerror("kimage_proc", "ligradient", "kpds_set_attribute failed for KPDS_VALUE_DATA_TYPE");
         return(FALSE);
      }
 
     /*
      * Get a row in which to temporarily store the results of the gradient.
      */
      if ( (ival = (long *)kpds_get_data(dest_ref, KPDS_VALUE_LINE, ival)) == NULL) {
               kerror("kimage_proc", "ligradient", "kpds_get_data failed for ival") ;
               return(FALSE);
      }
 
     /*
      * Outer loops: variable h and i are the number of images in object.
      */
      for (h = 0; h < tsize; h++) {
	 for (i = 0; i < depth; i++) {
	    /*
	     * 1st Middle loop: variable is the number of bands (elements)
	     * to process.
	     *    - num_bands = number of bands (elements) per image
	     */
	    for ( k = 0; k < num_bands; k++) {
	       /*
		* 2nd Middle loop: variable is the number of regions to process
		*    - num_regions = number of rows (height)
		*/
	       num_regions = height;
	       for ( j = 0; j < num_regions; j++) {
		  /*
		   * Get a region of data and calculate gradient across 
		   * that region.
		   */
		  kpds_set_attribute(src_ref, KPDS_VALUE_POSITION, 
				     0, j, i, h, k);
		  idata = (long *)(kpds_get_data(src_ref, KPDS_VALUE_REGION, 
						 idata));
		  if (idata == NULL) {
		     kerror("kimage_proc", "ligradient", "kpds_get_data failed for idata.");
		     return(FALSE);
		  }

		  /*
		   * Inner loop: variable is the number of columns.
		   *   - region_width = width of source image + 2 columns
		   *     (one extra column at each side of the region)
		   */
		  for (col=1; col<region_width-1; col++) {
		     /*
		      * Robert's operator.
		      */
		     if (gradient_type == 0) {
			ixval =  idata[region_width + col + 1] -
			   idata[2*region_width + col]; 
			iyval =  idata[region_width + col] -
			   idata[2*region_width + col + 1]; 
		     }
		     /*
		      * Sobel operator.
		      */
		     if (gradient_type == 1) {
			ixval =  idata[col+1] + 2*idata[region_width + col+1] +
			   idata[2*region_width + col+1] -
			   (idata[col-1] + 2*idata[region_width + col-1] +
                            idata[2*region_width + col-1]);
			iyval =  idata[col-1] + 2*idata[col] + idata[col+1] -
			   (idata[2*region_width + col-1] +
                            2*idata[2*region_width + col] +
                            idata[2 * region_width + col+1]);
		     }
		     /*	
		      * Prewitt operator.
		      */
		     if (gradient_type == 2) {
			ixval =  idata[col+1] + idata[region_width + col+1] +
			   idata[2*region_width + col+1] -
			   (idata[col-1] + idata[region_width + col-1] +
                            idata[2*region_width + col-1]);
			iyval =  idata[col-1] + idata[col] + idata[col+1] -
			   (idata[2*region_width + col-1] +
                            idata[2*region_width + col] +
                            idata[2 * region_width + col+1]);
		     }
		     /*	
		      * Isotropic operator.
		      */
		     if (gradient_type == 3) {
			ixval =  idata[col+1] + SQROOT2*idata[region_width + col+1] +
			   idata[2*region_width + col+1] -
			   (idata[col-1] + SQROOT2*idata[region_width + col-1]
                            + idata[2*region_width + col-1]);
			iyval =  idata[col-1] + SQROOT2*idata[col] + idata[col+1] -
			   (idata[2*region_width + col-1] +
                            SQROOT2*idata[2*region_width + col] +
                            idata[2 * region_width + col+1]);
		     }
		     if(operation_type)
			ival[col-1] = kabs(ixval) + kabs(iyval);
		     else
			ival[col-1] = sqrt((double)((ixval*ixval) + (iyval*iyval))
			   );
		  }
		  if (! kpds_put_data(dest_ref, KPDS_VALUE_LINE, ival) ) {
		     kerror("kimage_proc", "ligradient", "kpds_put_data failed for destination object.");
		     return(FALSE);
		  }
	       }
	    }
	 }
      }
   } /* end of loop for integer processing */
 
  /*
   * If the data is not integer representable, process as double.
   */
   else {
     /*
      * If data type is COMPLEX, warn user and exit.
      */
      if ( (data_type == KCOMPLEX) || (data_type == KDCOMPLEX) ) {
            kerror("kimage_proc", "ligradient", "data type is COMPLEX, cannot process.");
            return(FALSE);
         }
     /*
      * *********************************************************************
      *                  Major loop for float processing.                   *
      * *********************************************************************
      * Set the data type of the source and destination references to KDOUBLE.
      */
      if (! kpds_set_attribute(src_ref, KPDS_VALUE_DATA_TYPE, KDOUBLE)) {
         kerror("kimage_proc", "ligradient", "kpds_set_attribute failed for KPDS_VALUE_TYPE");
         return(FALSE);
      }
      if (! kpds_set_attribute(dest_ref, KPDS_VALUE_DATA_TYPE, KDOUBLE)) {
         kerror("kimage_proc", "ligradient", "kpds_set_attribute failed for KPDS_VALUE_TYPE");
         return(FALSE);
      }
 
     /*
      * Get a row in which to temporarily store the results of the gradient.
      */
      if ( (dval = (double *)kpds_get_data(dest_ref, KPDS_VALUE_LINE, dval)) == NULL) {
            kerror("kimage_proc", "ligradient", "kpds_get_data failed for dval.");
            return(FALSE);
      }
 
     /*
      * Outer loops: variable h and i are the number of images in object.
      */
      for ( h = 0; h < tsize; h++ ) {
	 for ( i = 0; i < depth; i++) {
	    /*
	     * 1st Middle loop: variable is the number of bands (elements)
	     * to process.
	     *    - num_bands = number of bands (elements) per image
	     */
	    for ( k = 0; k < num_bands; k++) {
	       /*
		* 2nd Middle loop: variable is the number of regions to process
		*    - num_regions = number of rows (height)
		*/
	       num_regions = height;
	       for ( j = 0; j < num_regions; j++) {
		  /*
		   * Get a region of data and calculate gradient across that 
		   * region.
		   */
		  kpds_set_attribute(src_ref, KPDS_VALUE_POSITION, 
				     0, j, i, h, k);
		  ddata = (double *)(kpds_get_data(src_ref, 
						   KPDS_VALUE_REGION, ddata));
		  if (ddata == NULL) {
		     kerror("kimage_proc", "ligradient", "kpds_get_data failed for ddata.");
		     return(FALSE);
		  }
		  /*
		   * Inner loop: variable is the number of columns.
		   *   - region_width = width of source image + 2 columns
		   *     (one extra column at each side of the region)
		   */
		  for (col=1; col<region_width-1; col++) {
		     /*	
		      * Robert's operator.
		      */
		     if (gradient_type == 0) {
			dxval =  ddata[region_width + col + 1] -
			   ddata[2*region_width + col]; 
			dyval =  ddata[region_width + col] -
			   ddata[2*region_width + col + 1]; 
		     }
		     /*	
		      * Sobel operator.
		      */
		     if (gradient_type == 1) {
			dxval =  ddata[col+1] + 2*ddata[region_width + col+1] +
			   ddata[2*region_width + col+1] -
			   (ddata[col-1] + 2*ddata[region_width + col-1] +
                            ddata[2*region_width + col-1]);
			dyval =  ddata[col-1] + 2*ddata[col] + ddata[col+1] -
			   (ddata[2*region_width + col-1] +
                            2*ddata[2*region_width + col] +
                            ddata[2 * region_width + col+1]);
		     }
		     /*	
		      * Prewitt operator.
		      */
		     if (gradient_type == 2) {
			dxval =  ddata[col+1] + ddata[region_width + col+1] +
			   ddata[2*region_width + col+1] -
			   (ddata[col-1] + ddata[region_width + col-1] +
                            ddata[2*region_width + col-1]);
			dyval =  ddata[col-1] + ddata[col] + ddata[col+1] -
			   (ddata[2*region_width + col-1] +
                            ddata[2*region_width + col] +
                            ddata[2 * region_width + col+1]);
		     }
		     /*
		      * Isotropic operator.
		      */
		     if (gradient_type == 3) {
			dxval =  ddata[col+1] + SQROOT2*ddata[region_width + col+1] +
			   ddata[2*region_width + col+1] -
			   (ddata[col-1] + SQROOT2*ddata[region_width + col-1]
                            + ddata[2*region_width + col-1]);
			dyval =  ddata[col-1] + SQROOT2*ddata[col] + ddata[col+1] -
			   (ddata[2*region_width + col-1] +
                            SQROOT2*ddata[2*region_width + col] +
                            ddata[2 * region_width + col+1]);
		     }
		     if(operation_type)
			dval[col-1] = kabs(dxval) + kabs(dyval);
		     else
			dval[col-1] = sqrt((double)((dxval*dxval)
						    + (dyval*dyval)));
		  }
		  if (! kpds_put_data(dest_ref, KPDS_VALUE_LINE, dval) ) {
		     kerror("kimage_proc", "ligradient", "kpds_put_data failed for destination object.");
		     return(FALSE);
		  }
	       }
	    }
	 }
      }
   }

   return(TRUE);
}
/* -library_code_end */
