 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for ifilt_design
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lifilt_design
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
static double pix_dist PROTO((double, double, int, int, double, double, int));
static double ideal_lpf PROTO((double, double, double, int));
static double but_lpf   PROTO((double, double, double, int));
static double exp_lpf   PROTO((double, double, double, int));
static double ch1_lpf   PROTO((double, double, double, int));
static double ch2_lpf   PROTO((double, double, double, int));
static double lpf       PROTO((double, double, double, int, int));
static double hpf       PROTO((double, double, double, int, int));
static double bpf       PROTO((double, double, double, int, int));
static double bsf       PROTO((double, double, double, int, int));

kfunc_double btbl[5] = { ideal_lpf, but_lpf, exp_lpf, ch1_lpf, ch2_lpf };
kfunc_double ttbl[4] = { lpf, hpf, bpf, bsf };
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lifilt_design - generate 2-dimensional frequency filter
* 
*       Purpose: This routine generates a spatial frequency image of real values
*		 that represents the response of the filter type and brand
*		 that was specified.  The types it supports are Low Pass,
*		 High Pass, Band Pass, and Band Stop filters.   The generator
*		 function brands are Ideal, Butterworth, Exponential,
*		 Chebyshev I, and Chebyshev II.  All output is radially
*		 semmetric about the origin of the image, and the origin can
*		 be either the center of the image or the top left corner
*		 of the image.  Cutoff frequencies are normalized between
*		 0 and 0.5 where 0 is DC, and 0.5 is the maximum circle/elipse
*		 that will cover the image.  Image height and width do not
*		 have to be the same, but pixel distance to the origin will be
*		 normalized by image height and width dimensions.  For example,
*		 if you have an image of size 256x128 with the origin at the
*		 center of the image, then the pixel at 0,128 and 64,0 will
*		 both be normalized to 0.5.  This will produce filters that
*		 look like an elipse with the major axis in the direction
*		 of the largest image dimension, and the minor axis in the
*		 smallest dimension.
*
*         Input: type          - type of filter.
*		 brand         - brand of filter.
*		 order         - filter order.
*		 pixel_height  - vertical size of a pixel
*		 pixel_width   - horizontal size of a pixel
*		 image_height  - height of the image.
*		 image_width   - width of the image.
*		 lower_cutoff  - lower cutoff for band pass or band stop,
*				 and cutoff for high pass filters.
*		 upper_cutoff  - upper cutoff for band pass or band stop,
*				 and cutoff for low pass filters.
*		 centered      - if TRUE, it puts low frequencies in the center
*				 of the image.  Otherwise it puts them in the
*				 corners.
*        Output: freq_image    - An opened kobject to store the frequency image
*				 in
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: The parameters pixel_height and pixel_width are not yet 
*		 supported by lifilt_design.  Currently, only a value of 1 is 
*		 accepted for these parameters.
*    Written By: Steven Jorgensen
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lifilt_design (
	int	type,
	int	brand,
	int	order,
	double	pixel_height,
	double	pixel_width,
	int	image_height,
	int	image_width,
	double	lower_cutoff,
	double	upper_cutoff,
	int	centered,
	kobject freq_image)
/* -library_def_end */

/* -library_code */
{
	kobject fi_ref = NULL;
	double *line;
	int i, j;
	double d;

	/*
	 * error check the parameters passed in by the user.
	 * type is checked by the switch statement below.
	 */
	if (brand < KFILT_IDEAL || brand > KFILT_CHEBYSHEV_II)
	{
		kerror("kimage_proc", "lifilt_design", "Invalid Filter Brand, KFILT_IDEAL, KFILT_BUTTERWORTH, KFILT_EXPONENTIAL, KFILT_CHEBYSHEV_I, and KFILT_CHEBYSHEV_II are the only valid brands.");
		return(FALSE);
	}
	if (pixel_height != 1.0)
	{
		kerror("kimage_proc", "lifilt_design","Pixel height must be 1");
		return(FALSE);
	}
	if (pixel_width != 1.0)
	{
		kerror("kimage_proc", "lifilt_design", "Pixel width must be 1");
		return(FALSE);
	}
	if (image_height <= 0)
	{
		kerror("kimage_proc", "lifilt_design", "Image height must be a positive value.");
		return(FALSE);
	}
	if (image_width <= 0)
	{
		kerror("kimage_proc", "lifilt_design", "Image width must be a positive value.");
		return(FALSE);
	}
	if (!freq_image)
	{
		kerror("kimage_proc", "lifilt_design", "Frequency Image object is not a valid data object.");
		return(FALSE);
	}

	if (type < KFILT_LP || type > KFILT_BS)
	{
		kerror("kimage_proc", "lifilt_design", "Invalid filter type, valid types are KFILT_LP, KFILT_HP, KFILT_BP, or KFILT_BS");
		return(FALSE);
	}
	if ((type == KFILT_BP || type == KFILT_BS) &&
	    ((upper_cutoff - lower_cutoff) <= 0))
	{
		kerror("kimage_proc", "lifilt_design", "Upper cutoff value must be greater than the Lower cutoff value for this type of filter.");
		return(FALSE);
	}

	if ((fi_ref = kpds_reference_object(freq_image)) == KOBJECT_INVALID)
	{
		kerror("kimage_proc", "lifilt_design", "Failed to reference frequency image object.");
		return(FALSE);
	}

	if (! kpds_set_attributes(fi_ref, KPDS_VALUE_DATA_TYPE, KDOUBLE,
				  KPDS_VALUE_SIZE, image_width, image_height,
				  1, 1, 1, NULL))
	{
		kerror("kimage_proc", "lifilt_design", "Could not set reference data type to double complex or reference size.");
		return(FALSE);
	}

	/*
	 * alloc a line of data
	 */
	if ((line = (double *) kcalloc((unsigned)image_width, sizeof(double))) 
		 == NULL)
	{
		kerror("kimage_proc", "lifilt_design", "Cannot allocate '%d' bytes for a line of double image data.", sizeof(double));
		return(FALSE);
	}

	/*
	 * for each line in image do
	 */
	for (i = 0; i < image_height; i++)
	{
		/*
		 * for each element in the line do
	         */
		for (j = 0; j < image_width; j++)
		{
			/* compute distance from origin */
			d = pix_dist((double) j, (double) i,
				     image_width, image_height,
				     pixel_width, pixel_height,
				     centered);

			/* compute pixel scale value based on filter type */
			line[j] = (ttbl[type])(d, upper_cutoff, lower_cutoff,
					       order, brand);
		/* end element loop */
		}
		/* put the line */
		if (! kpds_put_data(fi_ref, KPDS_VALUE_LINE, line))
		{
			kerror("kimage_proc", "lifilt_design", "kpds_put_data failed for frequency image object.");
			free(line);
			return(FALSE);
		}
	/* end of line loop */
	}
   kfree(line);
   return(TRUE);
}

static double pix_dist(
	double x,
	double y,
	int ix,
	int iy,
	double pw,
	double ph,
	int centered)
{
	double dx, dy, d;
	/*
	 * centered implies that the low freqencies are at the center of
	 * the image.
	 */
	if (centered)
	{
		if (x < ((double) ix)/2.0)
			dx = ((double) ix)/2.0 - x;
		else
			dx = x - ((double) ix)/2.0;
		if (y < ((double) iy)/2.0)
			dy = ((double) iy)/2.0 - y;
		else
			dy = y - ((double) iy)/2.0;
	}
	/*
	 * Implies that the low freqencies are at the corners of
	 * the image.
	 */
	else
	{
		if (x < ((double) ix)/2.0)
			dx = x;
		else
			dx = ((double) ix) - x;
		if (y < ((double) iy)/2.0)
			dy =  y;
		else
			dy = ((double) iy) - y;
	}
	/*
	 * pixel size considerations.  Probably incorrect, but then no
	 * one has figured this junk out yet.
	 */
	dx *= pw;
	dy *= ph;

	/*
	 * When we devide by ix^2 and iy^2, we get an extra factor of 4
	 * in the denominator, but we want to normalize the value to
	 * be between 0 and 0.5 on the maximum elipse that covers the
	 * image, so that would require us to divide by 2, so we'll just
	 * multiply by two to bring the divide by 4 to a divide by 2.
	 */
	dx = dx * dx;
	dy = dy * dy;
	dx = dx / ((double) ix * ix);
	dy = dy / ((double) iy * iy);
	d = ksqrt(dx + dy);
	return(d);
}

static double cheb_ord(
	double x,
	int order)
{
	if (order < 0)
		return(1.0);
	if (order == 1)
		return(x);
	return(2.0 * x * cheb_ord(x, order -1) - cheb_ord(x, order -2));
}

static double lpf(
	double x,
	double uc,
	double lc,
	int order,
	int brand)
{
	return((btbl[brand])(x, uc, lc, order));
}

static double hpf(
	double x,
	double uc,
	double lc,
	int order,
	int brand)
{
	if (x == 0.0)
		return(0.0);
	x = lc * lc / x;
	return((btbl[brand])(x, lc, uc, order));
}

static double bpf(
	double x,
	double uc,
	double lc,
	int order,
	int brand)
{
	double val1;
	val1 = uc -lc;
	if (x == 0.0)
		return(0.0);
	x = kabs((uc * lc - x * x) / x);
	return((btbl[brand])(x, val1, lc, order));
}

static double bsf(
	double x,
	double uc,
	double lc,
	int order,
	int brand)
{
	double val1, val2;
	val1 = uc - lc;
	val2 = x * x - uc * lc;
	if (val2 == 0.0)
		return(0.0);
	x = kabs(x * val1 * val1 / (uc * lc - x * x));
	return((btbl[brand])(x, val1, lc, order));
}

static double ideal_lpf (
	double x,
	double uc,
	double lc,
	int order)
{
	return((x <= uc) ? 1.0 : 0.0);
}

static double but_lpf (
	double x,
	double uc,
	double lc,
	int order)
{
	return (1.0 / (1.0 + kpow(x / uc, 2.0 * (double) order)));
}

static double exp_lpf (
	double x,
	double uc,
	double lc,
	int order)
{
	double val;
	val = x / uc;
	val = kpow( val, (double) order);
	val = kexp((-0.347)*val);
	return(val);
}

static double ch1_lpf (
	double x,
	double uc,
	double lc,
	int order)
{
	double val;
	val = cheb_ord((x / uc), order);
	val = 1.0 / (1.0 + (val * val));
	return(val);
}

static double ch2_lpf (
	double x,
	double uc,
	double lc,
	int order)
{
	double val;
	if (x == 0.0)
		return(1.0);
	val = cheb_ord(uc / x, order);
	val = 1.0 / (1.0 + (1.0 / (val * val)));
	return(val);
}
/* -library_code_end */
