
/*********************************************************************

Copyright (C) 1993, 1994, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

*********************************************************************/

#include <design.h>

#include <geometry.h>
#include "rmonster.h"
#include "rmatrix.h"
#include "rmonster_objs.h"
#include "shader.h"
#include "volrender.h"
#include "zbuffer.h"
#include "sort_trans.h"

/**
  * voxel shader function declarations.
**/

static void voxel_dot();
static void voxel_splat_homogenous();
static void voxel_splat_homogenous_gauss();

static void (*voxelshaderfuncs[])() =
{
    voxel_dot,
    voxel_splat_homogenous,
    voxel_splat_homogenous_gauss
};

/**
  * functions to get or compute locations from uniform, rectilinear
  * and curvilinear volumes.  the init function is called first to
  * set up some prelim stuff.
**/
static void o_loc_init PROTO((rmonster_prim *p, vertex_3d *));
static void o_curv PROTO((int, int, int, vertex_3d **));
static void o_rect PROTO((int, int, int, vertex_3d **));
static void o_unif PROTO((int, int, int, vertex_3d **));

static void c_init PROTO((rmonster_prim *,surface_colors_rgba *));
static void c_ptr PROTO((int, int, int, surface_colors_rgba **));

static void form_voxel PROTO((voxel *,int,int,int,surface_colors_rgba *));
static int voxel_on_screen PROTO((voxel *,int,int));
static void colors_rgba_avg PROTO((surface_colors_rgba **,int,surface_colors_rgba *d));

/**
  * the volrender routine is the entry point into the direct volume
  * renderer.  it is assumed that the location data is in screen space,
  * that the values in the colors array are RGBA tuples.  these RGBA tuples
  * may be the result of some type of shading calculation.
  *
  * in this routine, we will step through the volume (at present, we simply
  * go from 0..size-1 in each of U,V,W blindly, although we could use some
  * btf or ftb mechanism...this isn't explicitly necessary due to the fact
  * that pixel compositing happens later, in the rendering back-end) and
  * render each voxel.
  *
  * a voxel is defined as the hexahedra formed by 8 grid nodes.  the colors
  * on each corner may possibly be different, so depending upon the color
  * interpolation scheme used in the different voxel rendering routines, we
  * may get different results.
**/

void
volrender(rmonsterobj *r,
	  rmonster_prim *p,
	  vertex_3d *locs,
	  surface_colors_rgba *color,
	  int width, int height)
{
    int iu,iv,iw;
    void (*shaderfunc)();
    int in_use_shader;
    voxel V;

    in_use_shader = r->shader - VOLUME_SHADER_BASE;
    if (in_use_shader < 0)  /* trying to use a surface shader, eh? */
	in_use_shader = 0;
    shaderfunc = voxelshaderfuncs[in_use_shader];

    o_loc_init(p,locs);
    c_init(p,color);
    
    for (iw=0;iw<p->mesh_dims[2]-1;iw++)
    {
	for (iv=0;iv<p->mesh_dims[1]-1;iv++)
	{
	    for (iu=0;iu<p->mesh_dims[0]-1;iu++)
	    {
		form_voxel(&V,iu,iv,iw,color);

		if (voxel_on_screen(&V,width,height))
		    (*shaderfunc)(&V); 
		    iu = iu;
	    }
	}
    }
}

static int
voxel_on_screen(voxel *v,
		int w,
		int h)
{
    /**
      * NOTE: we have not yet implemented front/back trivial
      * accept/reject.
    **/
    
    /* check to see if box min coords > 0 */
    if ((v->vmin.v[0] < 0.) || (v->vmin.v[1] < 0))
	return(0);

    /* check to see if box max coords < (w,h) */
    if ((v->vmax.v[0] > w) || (v->vmax.v[1] > h))
	return(0);

    /* f/b clipping here? */
    
    return(1);
}

static void
form_voxel(voxel *v,
	   int iu,
	   int iv,
	   int iw,
	   surface_colors_rgba *c)
{
    /* call routine to load coords */
    o_curv(iu,iv,iw,&(v->l[0]));
    o_curv(iu+1,iv,iw,&(v->l[1]));
    o_curv(iu,iv+1,iw,&(v->l[2]));
    o_curv(iu+1,iv+1,iw,&(v->l[3]));
	   
    o_curv(iu,iv,iw+1,&(v->l[4]));
    o_curv(iu+1,iv,iw+1,&(v->l[5]));
    o_curv(iu,iv+1,iw+1,&(v->l[6]));
    o_curv(iu+1,iv+1,iw+1,&(v->l[7]));

    /* call routine to load colors. */
    c_ptr(iu,iv,iw,&(v->d[0]));
    c_ptr(iu+1,iv,iw,&(v->d[1]));
    c_ptr(iu,iv+1,iw,&(v->d[2]));
    c_ptr(iu+1,iv+1,iw,&(v->d[3]));
    
    c_ptr(iu,iv,iw+1,&(v->d[4]));
    c_ptr(iu+1,iv,iw+1,&(v->d[5]));
    c_ptr(iu,iv+1,iw+1,&(v->d[6]));
    c_ptr(iu+1,iv+1,iw+1,&(v->d[7]));

    lcompute_bbox_ptrs(v->l,8,&(v->vmin),&(v->vmax));
}

static surface_colors_rgba *cbase;
static int c_usize,c_uvsize,c_xfsize;
static void c_ptr PROTO((int, int, int, surface_colors_rgba **));

static void c_init(rmonster_prim *p,
		   surface_colors_rgba *c)
{
    cbase = c;
    c_usize = p->mesh_dims[0];
    c_uvsize = p->mesh_dims[1] * c_usize;
}

static void
c_ptr(int iu,
      int iv,
      int iw,
      surface_colors_rgba **c)
{
    int offset;
    offset = iu + iv*c_usize + iw * c_uvsize;
    *c = cbase + offset;
}


static vertex_3d *lbase;
static int l_usize,l_uvsize,l_xfsize;

static void
o_loc_init (rmonster_prim *p,
	    vertex_3d *locs)
{
    lbase = locs;
    /* the following is valid only for curvilinear data */
    l_usize = p->mesh_dims[0];
    l_uvsize = p->mesh_dims[1] * l_usize;
/*    l_xfsize = sizeof(vertex_3d); */

    /* for rectilinear data, we would keep pointers to separate arrays
       for each of x,y,z, and for uniform data, we would compute the
       vertex as we need it. */
}


static void
o_curv(int iu,
       int iv,
       int iw,
       vertex_3d **coord)
{
    long offset;
    offset = iu + iv*l_usize + iw*l_uvsize;
/*    kmemcpy(coord,lbase+offset,lxfsize); */
    *coord = lbase+offset;
}


static void
o_rect(int iu,
       int iv,
       int iw,
       vertex_3d **coord)
{
}


static void
o_unif(int iu,
       int iv,
       int iw,
       vertex_3d **coord)
{
}


/**
  * voxel shader functions.
**/

static void
voxel_dot(voxel *v)
{
    /**
      * here, we'll compute the weighted average of the voxels
      * color and opacity, and write a single dot at the center
      * of the voxel.
    **/
    surface_colors_rgba c_avg;
    vertex_3d l_avg;
    int i;
    float zb;
    int x,y;

    for (i=0;i<3;i++)
	l_avg.v[i] = v->vmin.v[i] + (v->vmax.v[i] - v->vmin.v[i]) * 0.5;

    x = (int)l_avg.v[0];
    y= (int)l_avg.v[1];
    
    zbuf_read(x,y,&zb);
    
    if (l_avg.v[2] < zb)
    {
	/**
	 * first, check to see if the deepest Z point on the voxel
	 * is in front of whatever is in the opaque Z buffer.
	 **/
    
	colors_rgba_avg(v->d,8,&c_avg);
	add_zbucket_entry(x,y,l_avg.v[2],c_avg.r,c_avg.g,c_avg.b,c_avg.a);
    }
}

static void
voxel_splat_homogenous(voxel *v)
{
    /**
      * here, we'll compute the weighted average of the voxels
      * color and opacity.  homogenous refers to the fact that the
      * splat will be basically a square (polygon) of constant
      * shade and opacity.
      *
    **/
    surface_colors_rgba c_avg;
    vertex_3d l_avg;
    int ix,iy,i;
    float zb;
    int x,y;

    colors_rgba_avg(v->d,8,&c_avg);
    for (i=0;i<3;i++)
	l_avg.v[i] = v->vmin.v[i] + (v->vmax.v[i] - v->vmin.v[i]) * 0.5;
    
    for (iy = (int)v->vmin.v[1];iy < (int)v->vmax.v[1];iy++)
    {
	for (ix = (int)v->vmin.v[0]; ix < (int)v->vmax.v[0];ix++)
	{
	    zbuf_read(ix,iy,&zb);
	    if (l_avg.v[2] < zb)
	    {
		add_zbucket_entry(ix,iy,l_avg.v[2],c_avg.r,c_avg.g,
				  c_avg.b,c_avg.a);

	    }
	}
    }

}

/**
  * the following two parameters control how two things in volume rendering.
  * the VOXEL_COVERAGE parameter controls the size of the splat with respect
  * to the two dimensional image plane projection of the voxel.  a value
  * of 1.0 will cause the splat size to be exactly the min/max extents
  * of the voxel footprint.  a value of 2.0 causes the splat size to
  * be twice the size of the min/max extents of the voxel footprint.  in
  * addition to affecting the runtime of the volume renderer, increasing
  * this value will have the effect of "blurring" the rendering.
  *
  * the STD_DEV parameter controls how the voxel opacity is concentrated.
  * as per westover's 1990 siggraph paper, a pretty decent value for this
  * is 2.5.   this value is used in the following equation which is a
  * 2d gaussian filter used to spread the opacity over the footprint of
  * the voxel:
  *
  * 1/(2*pi) * 1/STD_DEV**2 *
  *       e ** (-(x-xmean)**2/(2*STD_DEV**2)-((y-ymean)**2/(2*STD_DEV**2)))
  *
  * at some point, this should be put into a table for efficiency.  it
  * can be put into a table for uniform volumes rendered using parallel
  * projection (all voxels have the same footprint).
  *
  * the way the table will be built is as follows: there will be a twoD
  * array of pointers.  the index of each axis indicates the width of
  * the footprint, so the table at entry (1,2) corresponds to a footprint
  * that is 1 pixel wide by 2 pixels high.  each entry of the table
  * contains the scalar computed below.  probably the most general way
  * (not the absolute most efficient) to do this is to initialize the
  * table to nil.  as voxels are processed, if a table entry is needed
  * and it is nil, it is computed and stored.  in the case of uniform
  * volumes rendered with parallel projection, the table entry will be
  * computed only once.
**/

/*#define STD_DEV 2.5 */

#define STD_DEV 3.0
/*#define VOXEL_COVERAGE 1.6 */
#define VOXEL_COVERAGE 2.0

static void
voxel_splat_homogenous_gauss(voxel *v)
{
    register int ix,iy;
    int ixmin,iymin,ixmax,iymax,xd,yd;
    float z_avg,opaque_z,t1;
    float xmean,ymean;
    surface_colors_rgba c_avg;
    vertex_3d l_avg;
    vertex_3d *min,*max;
    int i;

#if 0
    double scale1 = 0.025464791; /* 1/(2*pi)^2 * 1/(STD_DEV)^2 */
    double scale2 = -0.08; /* -1/(STD_DEV)^2 */
    
/*    extern double kexp(); */
#endif
    double d1,d2;

    double scale1,scale2; 

    scale1= 0.15915494 * (1./(STD_DEV*STD_DEV));
    scale2 = -0.5/(STD_DEV*STD_DEV);

    colors_rgba_avg(v->d,8,&c_avg);
    for (i=0;i<3;i++)
	l_avg.v[i] = v->vmin.v[i] + (v->vmax.v[i] - v->vmin.v[i]) * 0.5;

    min = &(v->vmin);
    max = &(v->vmax);

    t1 = (max->v[0] - min->v[0])*VOXEL_COVERAGE;  /* as per westover's paper */
    xmean = (max->v[0] - min->v[0])*0.5 + min->v[0];
    ixmin = xmean - t1*0.5;
    ixmax = xmean + t1*0.5;
    
    t1 = (max->v[1] - min->v[1])*VOXEL_COVERAGE;  /* as per westover's paper */
    ymean = (max->v[1] - min->v[1])*0.5 + min->v[1];
    iymin = ymean - t1*0.5;
    iymax = ymean + t1*0.5;
#if 0
    ixmin = min->v[0];
    ixmax = max->v[0];
    iymin = min->v[1];
    iymax = max->v[1];
#endif
#if 0
    r = rgba[0];
    g = rgba[1];
    b = rgba[2];
    a = rgba[3];
#endif
/*    z_avg = (min->v[2] + max->v[2])*0.5; */
    z_avg = min->v[2];

    for (iy=iymin;iy<=iymax;iy++)
    {
	for (ix=ixmin;ix<=ixmax;ix++)
	{
	    zbuf_read(ix,iy,&opaque_z);
	    if (z_avg < opaque_z)
	    {

		d1 = (ix - xmean);
		d1 = d1*d1 * scale2;
/*		d1 = (ix - xmean)*(ix - xmean) * scale2; */
		d2 = (iy - ymean);
		d2 = d2*d2 * scale2;
/*		d2 = (iy - ymean)*(iy - ymean) * scale2; */
		d1 = d1 + d2;
		d2 = exp(d1);
/*		d2 *= scale1; */  /* keep normalized 0..1 */
		t1 = c_avg.a * d2;
		add_zbucket_entry(ix,iy,z_avg,c_avg.r,c_avg.g,c_avg.b,t1);
#if 0
		add_zbucket_entry(ix,iy,z_avg,r,g,b,a); 
#endif
	    }
	}
    }
}


static void
colors_rgba_avg(surface_colors_rgba **c,
		int n,
		surface_colors_rgba *d)
{
    double inv;
    double dr,dg,db,da;
    int i;
    surface_colors_rgba *rp;
    
    inv = 1./(double)n;

    dr = dg = db = da = 0.;
    
    for (i=0;i<n;i++)
    {
	rp = c[i];
	dr += rp->r;
	dg += rp->g;
	db += rp->b;
	da += rp->a;
    }

    d->r = dr * inv;
    d->g = dg * inv;
    d->b = db * inv;
    d->a = da * inv;
}
