
/*********************************************************************

Copyright (C) 1994, 1995, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus it is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

Sun Jan  1 17:50:35 PST 1995 (wes) - changes to show_box() routine so that
    the line clipping is done to the XOR box presented to the user.  

*********************************************************************/


/*
 * Copyright (C) 1993, 1994, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

#include <design.h>
#include <geometry.h>

#include "rmonster.h"
#include "imaging.h"
#include "zbuffer.h"
#include "sort_trans.h"
#include "rmonster_objs.h"
#include "rmonster_mouse.h"
#include "rmonster_cursors.h"
#include "rmatrix.h"
#include "camera.h"
#include "imaging.h"

#include "xvinclude.h"
#include "xvisual/xvisual.h"

#include <X11/cursorfont.h>

static xvobject rmonster_xvobject;

/**
  * the following should be ____
**/
static double DELTA_SCALE = 50.;
/*#define DELTA_SCALE 50.0 */


static Display *rmonster_display;
static Window rmonster_window;
static Screen *rmonster_screen;
static GC rmonster_gc_equiv;
static Cursor zencursor,translate_cursor,xy_rotate_cursor,
    z_rotate_cursor,scale_cursor;
static int current_rotate_mode;
static double z_rotate_portal_radius;
#define Z_ROTATE 0
#define XY_ROTATE 1

static int cursor_x= -1, cursor_y = -1;
static int virgin;

/**
  * routines for building a temporary object used to represent, in
  * a minimal wireframe representation, objects which will be transformed
  * interactively by the user.
  *
  * the bounding box consists of 8 vertices.  they are ordered thus:

                              max
          7-------------------6
	 /|                  /|
	3-------------------2 |
	| |                 | |
	| 4-----------------|-5
	|/                  |/
	0------------------ 1
      min
 *       
**/

void get_parent_xfrm PROTO((matrix4x4 *));
void get_parent_xfrm_inverse PROTO((matrix4x4 *));
void sync_root_box PROTO((void));

static rmonsterobj *ref_obj;
static vertex_3d box_verts3d[8];
static vertex_3d box_center,trans_vect,trans_vect_orig;
static vertex_3d bv[8];
static unsigned int ix[8],iy[8];
static matrix4x4 view,model,comp,comp_inverse,I,view_rotations_inverse;
static matrix4x4 parent_xfrm,parent_xfrm_inverse,view_rotations;
static matrix4x4 m_ry,m_rx,m_rz;
static matrix4x4 m_scale;
static double dscale;
static int imagew,imageh;
KGEOM_VERTEX_TYPE dx,dy,dz,xscale,yscale,xd,yd,rx,ry,rz;
static int projection;
static int zbox_x_min,zbox_x_max,zbox_y_min,zbox_y_max,zbox_width,zbox_height,
    zbox_x_center,zbox_y_center;

/* the following paramter controls the size of the rotate circle
   that appears on the screen when you rotate the object.  the
   value 0.1666 is 1/6.  this is the distance from the left and
   right side of the screen (or top and bottom, depending upon which
   is the smallest dimension) which contains the circle.  in other
   words, if this value is 1/6, then the radius of the rotate circle
   is 1/3 the size of the smallest image dimension.  if you changed it
   to 0.25, the circle would be smaller, if you changed it to 0.10, the
   circle would be larger.  to me, 1/6 seems aesthetically pleasing. */

#define UL_ZROTATE_CIRCLE 0.1666

static void
build_outline_object()
{
    vertex_3d *mn,*mx;
    
    ref_obj = get_current_object();

    mn = &(ref_obj->bmin);
    mx = &(ref_obj->bmax);
    
    VCOPY(box_verts3d,mn);  /* do the min vertex. */
    VCOPY(box_verts3d+6,mx); /* do the max vertex. */

    VCOPY(box_verts3d+1,mn);
    box_verts3d[1].v[0] = mx->v[0];

    VCOPY(box_verts3d+2,mx);
    box_verts3d[2].v[2] = mn->v[2];

    VCOPY(box_verts3d+3,mn);
    box_verts3d[3].v[1] = mx->v[1];

    VCOPY(box_verts3d+4,mn);
    box_verts3d[4].v[2] = mx->v[2];

    VCOPY(box_verts3d+5,mx);
    box_verts3d[5].v[1] = mn->v[1];

    VCOPY(box_verts3d+7,mx);
    box_verts3d[7].v[0] = mn->v[0];

    matrix_4x4copy(&model,&(ref_obj->xfrm)); 

    get_parent_xfrm(&parent_xfrm);
    get_parent_xfrm_inverse(&parent_xfrm_inverse);

    VCOPY(&trans_vect,&(ref_obj->translate_vect));
    VCOPY(&trans_vect_orig,&trans_vect);
    VCOPY(&box_center,&(ref_obj->center));
}

static void
set_up_xfrms()
{
    vertex_3d eye,at,flength_vec;
    double flength,dfov,t;
    float fov;
    int i,min_dim;
    
    identity_4x4(&I);
    
    get_view_xform(&view);
    get_image_dims(&imagew,&imageh);
    mmul_4x4(&model,&view,&comp);
    matrix_inverse(&comp,&comp_inverse);
    get_view_rotation_matrix(&view_rotations);
    matrix_inverse(&view_rotations,&view_rotations_inverse);

    /**
      * compute a scale factor which is applied to mouse motions and
      * used in computing translational components.  the goal here is to
      * enable a mapping such that a 1 pixel change in the mouse position
      * results in a one pixel change in the object if it lies in the
      * projection plane.  objects closer to the viewer than the projection
      * plane will move more than this, objects further away will move
      * less than this amount.
    **/
    get_eye(&eye);
    get_at(&at);
    get_fov(&fov);

    for (i=0;i<3;i++)
	flength_vec.v[i] = eye.v[i] - at.v[i];

    flength = 0.;
    for (i=0;i<3;i++)
	flength += flength_vec.v[i]*flength_vec.v[i];
    flength = ksqrt(flength);

    dfov = fov * 0.5;
    dfov = DEGREES_TO_RADIANS(dfov);
    t = ktan(dfov);

    DELTA_SCALE = t * 0.5 * MAX(imagew,imageh);
    DELTA_SCALE = flength/DELTA_SCALE;

    dx = DELTA_SCALE;
    dy = DELTA_SCALE;

    /**
      * for the Z component, we will scale mouse motions so that the
      * user can move the object either +/-Z in an amount equal to the
      * half the focal length of the camera.
    **/

    dz = flength*0.25;

    /**
      * for x/y rotations, we want to have 2pi radians worth of
      * rotation across the body of the screen.
    **/
    ry = 2.*PI/(double)imagew;
    rx = 2.*PI/(double)imageh;

    xd = xscale = ((KGEOM_VERTEX_TYPE)imagew - 0.001)/2.0;
    yd = yscale = ((KGEOM_VERTEX_TYPE)imageh - 0.001)/2.0;

    /**
      * for scaling, we will permit the user to double the size
      * of the object when moving the mouse halfway across the
      * screen in X.
    **/
    dscale = 2./(double)imagew;

    min_dim = MIN(imagew,imageh);
    
    zbox_x_center = imagew >> 1;
    zbox_y_center = imageh >> 1;
    
    dscale = 2./(double)min_dim;
    
    zbox_x_min = zbox_x_center - (0.5 - UL_ZROTATE_CIRCLE) * min_dim;
    zbox_x_max = zbox_x_center + (0.5 - UL_ZROTATE_CIRCLE) * min_dim;
    
    zbox_y_min = zbox_y_center - (0.5 - UL_ZROTATE_CIRCLE) * min_dim;
    
    zbox_y_max = zbox_y_center + (0.5 - UL_ZROTATE_CIRCLE) * min_dim;


    zbox_width = zbox_x_max - zbox_x_min;
    zbox_height = zbox_y_max - zbox_y_min;

    z_rotate_portal_radius = (zbox_width > zbox_height) ? zbox_height : zbox_width;
    z_rotate_portal_radius = z_rotate_portal_radius * 0.5;

    get_projection(&projection);
}

static void
iso_scale_box(int idel)
{
    matrix4x4 t,t2;
    double acc;

    acc = (double)idel * dscale;

    identity_4x4(&t);
    t.m[0][0] += acc;
    t.m[1][1] += acc;
    t.m[2][2] += acc;

    mmul_4x4(&t,&model,&t2);
    matrix_4x4copy(&model,&t2);
}

static int
get_rotate_mode(int x,int y)
{
    /* figure out if we're inside or outside of the "circle" formed
       by the Z rotate portal.*/
    double d;
    int dx,dy;

    dx = x - zbox_x_center;
    dy = y - zbox_y_center;

    d = (double)(dx * dx) + (double)(dy * dy);
    d = ksqrt(d);
    
    if (d > z_rotate_portal_radius)
	return(Z_ROTATE);
    else
	return(XY_ROTATE);
}


static void
rotate_box_y(int delx)
{
    double st,ct;
    double a;
    matrix4x4 t,t2;
    
    identity_4x4(&t);
    
    a = -1. * (double)delx * ry;
    st = sin(a);
    ct = cos(a);

    t.m[0][0] = t.m[2][2] = ct;
    t.m[0][2] = st;
    t.m[2][0] = -1. * st;

    mmul_4x4(&view_rotations,&t,&t); 
    mmul_4x4(&t,&view_rotations_inverse,&t); 

    mmul_4x4(&model,&t,&t2); 
    matrix_4x4copy(&m_ry,&t2);
    matrix_4x4copy(&model,&t2);
    mmul_4x4(&t2,&view,&comp);
}

static void
rotate_box_z(int delx,int dely,int newx, int newy)
{
    double st,ct;
    double a1,a2;
    matrix4x4 t,t2;
    int sign,ox,oy;

    /* figure out if positive or negative z axis rotation.
       the scheme we'll use is to look at the z component of a
       vector cross product, to determine if the center of the
       screen is on the "right" or "left" of the current mouse trajectory.
     **/

    ox = zbox_x_center - newx;
    oy = zbox_y_center - newy;
    sign = delx * oy -	dely * ox;
    if (sign > 0)
	sign = -1;
    else if (sign < 0)
	sign = 1;
    else  /* huh? */
	sign = 0;
    
    identity_4x4(&t);

    /* compute theta of rotation - this is dependent upon the
       number of pixels the cursor has moved since last time, as well
       as the distance from the center of the screen.
    */

    a1 = ksqrt((double)(delx*delx) + (double)(dely*dely));
    a2 = ksqrt((double)(ox * ox) + (double)(oy * oy));
    a1 = katan(a1/a2);
    a1 = FABS(a1);
    a1 *= (double)sign;

    st = ksin(a1);
    ct = kcos(a1);

    t.m[0][0] = t.m[1][1] = ct;
    t.m[0][1] = st;
    t.m[1][0] = -1. * st;

    mmul_4x4(&view_rotations,&t,&t); 
    mmul_4x4(&t,&view_rotations_inverse,&t);
    mmul_4x4(&model,&t,&t2);

    matrix_4x4copy(&m_rz,&t2);
    matrix_4x4copy(&model,&t2);
    mmul_4x4(&t2,&view,&comp);
}

static void
rotate_box_x(int delx)
{
    double st,ct;
    double a;
    matrix4x4 t,t2;
    
    identity_4x4(&t);
    
    a = (double)delx * rx;
    st = sin(a);
    ct = cos(a);

    t.m[1][1] = t.m[2][2] = ct;
    t.m[1][2] = st * -1.;
    t.m[2][1] = st;

    
    mmul_4x4(&view_rotations,&t,&t);
    mmul_4x4(&t,&view_rotations_inverse,&t);
    mmul_4x4(&model,&t,&t2);
    matrix_4x4copy(&m_rx,&t2);
    matrix_4x4copy(&model,&t2);
    mmul_4x4(&t2,&view,&comp);
}

static void
translate_box_xy(int delx,
		 int dely)
{
    double ddx,ddy,ddz;
    matrix4x4 *m;

    m = &view_rotations_inverse;

    ddx = delx * m->m[0][0] + dely * m->m[1][0];
    ddy = delx * m->m[0][1] + dely * m->m[1][1];
    ddz = delx * m->m[0][2] + dely * m->m[1][2];

    trans_vect.v[0] += dx * ddx;
    trans_vect.v[1] += dy * ddy;
    trans_vect.v[2] += dx * ddz;
}

static void
translate_box_z(int delx,
		int dely)
{
    int delta;

    delta = delx + dely;
    trans_vect.v[2] += dx * delta;
}

static void
show_z_rotate_box()
{
    XDrawArc(rmonster_display,rmonster_window,rmonster_gc_equiv,
	     zbox_x_min,zbox_y_min,
	     (zbox_x_max - zbox_x_min),(zbox_y_max-zbox_y_min),
	     0,360*64);
}

static void
show_box()
{
    int i;
    vertex_3d work;
    float zmin; /* we should move this to a static variable for performance,
		   cuz we don't need to obtain this value on each entry into
		   this routine. typically, zmin isn't going to change while
		   an interactive transformation is happening.  1/1/95 wes*/
    int clipped[12];  /* boolean visible/not visible flags */
    vertex_3d larray[12][2];
    int ixarray[12][2],iyarray[12][2];

    get_zmin(&zmin);

    geom_xform_points(&model,&trans_vect,&box_center,box_verts3d,bv,
		      &parent_xfrm,NULL,8);
		      
    npoint_xfrm(&(bv[0].v[0]),&view,&(bv[0].v[0]),8);

    /* set up for clipping.  here, we have to construct an array of
       line segments */

    for (i=0;i<4;i++)
    {
        VCOPY(&(larray[i][0]),&(bv[i]));
	VCOPY(&(larray[i][1]),&(bv[(i+1)%4]));
    }
    for (i=0;i<4;i++)
    {
        VCOPY(&(larray[i+4][0]),&(bv[i+4]));
	if (i != 3)
	    VCOPY(&(larray[i+4][1]),&(bv[i+5]));
        else
	    VCOPY(&(larray[i+4][1]),&(bv[4]));
    }
    for (i=8;i<12;i++)
    {
        VCOPY(&(larray[i][0]),&(bv[i%8]));
	VCOPY(&(larray[i][1]),&(bv[i%8+4]));
    }

    /* now clip the array of segments */
    for (i=0;i<12;i++)
    {
        if (projection == KPROJECTION_PERSPECTIVE)
        {
	    clipped[i] = clip_line_persp(&(larray[i][0].v[0]),
					 &(larray[i][0].v[1]),
					 &(larray[i][0].v[2]),
					 &(larray[i][1].v[0]),
					 &(larray[i][1].v[1]),
					 &(larray[i][1].v[2]),
					 &zmin,1,0);
        }
	else
	{
	    clipped[i] = clip_line_parallel(&(larray[i][0].v[0]),
					    &(larray[i][0].v[1]),
					    &(larray[i][0].v[2]),
					    &(larray[i][1].v[0]),
					    &(larray[i][1].v[1]),
					    &(larray[i][1].v[2]),
					    &zmin,1,0);
	}
    }

    /* do the perspective divide and map to screen space */
    for (i=0;i<12;i++)
    {
        if (clipped[i] == CHILL)
	{
	    if (projection == KPROJECTION_PERSPECTIVE)
	    {
	      larray[i][0].v[0] = larray[i][0].v[0]/larray[i][0].v[2];
	      larray[i][0].v[1] = larray[i][0].v[1]/larray[i][0].v[2];

	      larray[i][1].v[0] = larray[i][1].v[0]/larray[i][1].v[2];
	      larray[i][1].v[1] = larray[i][1].v[1]/larray[i][1].v[2];
	    }

	    larray[i][0].v[0] = larray[i][0].v[0] * xscale + xd;
	    larray[i][0].v[1] = larray[i][0].v[1] * yscale + yd;
	    larray[i][1].v[0] = larray[i][1].v[0] * xscale + xd;
	    larray[i][1].v[1] = larray[i][1].v[1] * yscale + yd;
	}
    }

    for (i=0;i<12;i++)
    {
        if (clipped[i] == CHILL)
	{
	    ixarray[i][0] = (int)larray[i][0].v[0];
	    ixarray[i][1] = (int)larray[i][1].v[0];

	    iyarray[i][0] = imageh - (int)(larray[i][0].v[1]) - 1;
	    iyarray[i][1] = imageh - (int)(larray[i][1].v[1]) - 1;
	}
    }


    /* draw a bunch of lines */
    for (i=0;i<12;i++)
    {
        if (clipped[i] == CHILL)
	    XDrawLine(rmonster_display, rmonster_window, rmonster_gc_equiv, 
		      ixarray[i][0], iyarray[i][0],ixarray[i][1], iyarray[i][1]);
    }
}


/***
  * routines which setup things for interactive rendering, ie,
  * setting up callbacks for button presses, etc.
**/

void
rmonster_resize(xvobject obj, kaddr cliendData,
		XEvent *event, Boolean *dispatch)
{
    unsigned int newx,newy;
    int oldw,oldh;

    /** using the debugger, it seems that the event type ConfigureNotify
      is used to signal a size change of the image widget, so we'll key on
      that event type.  we'll also check existing image dimensions to
      see if there is any change, given that i'm anticipating that
      ConfigureNotify could be used to signal things other than just a
      size change request.  1/19/95, wes. **/
    
    if (event->type == ConfigureNotify)
    {
	newx = event->xconfigure.width;
	newy = event->xconfigure.height;

	get_image_dims(&oldw,&oldh);

	if ((oldw != newx) || (oldh != newy))
	{
	    oldw = newx;
	    oldh = newy;  /* use int instead of unsigned int */
	    change_image_dims(oldw,oldh);
	    /* -- sad if geometry takes long to render, but better
	       then befuddled users staring at the screen waiting -- */
	    renderer_invoke_func();
	}
    }
    /* repaint imaging pane GUI to reflect new dims. */
    repaint_imaging_pane();

}

/*
 *   Event handler for initializing gc_equiv
 * 	this will be called on expose events
 *  	(There's probably a more elegant way to do this - Mark will know)
 */
void
rmonster_create_gc(xvobject obj, kaddr clientData, 
		   XEvent *event, Boolean  *dispatch) 
{
    Pixmap zenmap,zenmask;
    XColor fg,bg;
    XGCValues	  values;	
    unsigned long  mask;
    int status,xhot,yhot;
    unsigned int w,h;

    rmonster_display = xvw_display(obj);
    rmonster_window  = xvw_window(obj);
    rmonster_screen = xvw_screen(obj);
    rmonster_xvobject = obj;

    mask = GCLineWidth | GCFunction | GCSubwindowMode ;
    values.line_width = 0;
    values.function = GXinvert;  /* (NOT src) XOR dst */
    
    values.foreground = (unsigned long)(-1);
    values.background = (unsigned long)(0);

    values.subwindow_mode = IncludeInferiors;
    rmonster_gc_equiv = XCreateGC(rmonster_display, rmonster_window,
				  mask, &values);

    /* Okay, we've initialzed gc_equiv - we don't have to do this again */
     xvw_remove_event(obj, ExposureMask, rmonster_create_gc, NULL); 
   
    /**
     * build cursor definitions
     **/
    translate_cursor = XCreateFontCursor(rmonster_display,XC_diamond_cross);
    xy_rotate_cursor = XCreateFontCursor(rmonster_display,XC_fleur);
    z_rotate_cursor = XCreateFontCursor(rmonster_display,XC_exchange);
    scale_cursor = XCreateFontCursor(rmonster_display,XC_sizing);
    zenmap = XCreateBitmapFromData(rmonster_display,rmonster_window,
				   (char *)zen_bits,17,17);
    zenmask = XCreateBitmapFromData(rmonster_display,rmonster_window,
				    (char *)zen_mask,17,17);
	
    fg.pixel = XWhitePixel(rmonster_display,XDefaultScreen(rmonster_display));
    fg.flags = bg.flags = DoRed | DoGreen | DoBlue;
    fg.red = fg.green = fg.blue = -1;
    bg.red = bg.green = bg.blue = 0;
    bg.pixel = XBlackPixel(rmonster_display,XDefaultScreen(rmonster_display));
    
    zencursor = XCreatePixmapCursor(rmonster_display,zenmap,zenmask,&fg,&bg,1,1);

}


/**
  * action handlers for rmonster.
**/

void
btn3_down_func(xvobject o,
	       kaddr client_data,
	       XEvent *event)
{
    int x,y;

    XDefineCursor(rmonster_display,rmonster_window,translate_cursor);

    xvw_get_attribute(o, XVW_IMAGE_XPOSITION, &cursor_x);
    xvw_get_attribute(o, XVW_IMAGE_YPOSITION, &cursor_y);

    virgin = 1;
    build_outline_object();
    set_up_xfrms();
    show_box();

}


void
btn3_motion_func(xvobject o,
	       kaddr client_data,
	       XEvent *event)
{
    int newx,newy;
    int delx,dely;

    show_box();
    
    xvw_get_attribute(o, XVW_IMAGE_XPOSITION, &newx);
    xvw_get_attribute(o, XVW_IMAGE_YPOSITION, &newy);

    virgin = 0;
    
    delx = newx - cursor_x;
    dely = cursor_y - newy;

    translate_box_xy(delx,dely);

    cursor_x = newx;
    cursor_y = newy;

    show_box();
}
void
btn3_up_func(xvobject o,
	       kaddr client_data,
	       XEvent *event)
{
    int i;
    matrix4x4 m2;
    vertex_3d t2;
    
    show_box();
    XUndefineCursor(rmonster_display,rmonster_window);

    if (virgin == 0)
    {
	for (i=0;i<3;i++)
	    trans_vect.v[i] = trans_vect.v[i] - trans_vect_orig.v[i];
	for (i=0;i<3;i++)
	    ref_obj->translate_vect.v[i] += trans_vect.v[i];

	set_current_object(ref_obj);
	
	sync_root_box();
	
/*	software_render_image(); */
	renderer_invoke_func();
    }
}

void
shift_btn3_down_func(xvobject o,
		     kaddr client_data,
		     XEvent *event)
{
    int x,y;

    XDefineCursor(rmonster_display,rmonster_window,translate_cursor);

    xvw_get_attribute(o, XVW_IMAGE_XPOSITION, &cursor_x);
    xvw_get_attribute(o, XVW_IMAGE_YPOSITION, &cursor_y);

    virgin = 1;

    build_outline_object();
    set_up_xfrms();
    show_box();
}


void
shift_btn3_motion_func(xvobject o,
		       kaddr client_data,
		       XEvent *event)
{
    int newx,newy;
    int delx,dely;

    show_box();
    
    xvw_get_attribute(o, XVW_IMAGE_XPOSITION, &newx);
    xvw_get_attribute(o, XVW_IMAGE_YPOSITION, &newy);

    delx = newx - cursor_x;
    dely = cursor_y - newy;

    virgin = 0;

    translate_box_z(delx,dely);

    cursor_x = newx;
    cursor_y = newy;

    show_box();
}

void
shift_btn2_down_func(xvobject o,
		     kaddr client_data,
		     XEvent *event)
{
    int x,y;

    XDefineCursor(rmonster_display,rmonster_window,scale_cursor);

    xvw_get_attribute(o, XVW_IMAGE_XPOSITION, &cursor_x);
    xvw_get_attribute(o, XVW_IMAGE_YPOSITION, &cursor_y);

    virgin = 1;

    build_outline_object();
    set_up_xfrms();
    show_box();
}


void
shift_btn2_motion_func(xvobject o,
		       kaddr client_data,
		       XEvent *event)
{
    int newx,newy;
    int delx,dely;

    show_box();
    
    xvw_get_attribute(o, XVW_IMAGE_XPOSITION, &newx);
    xvw_get_attribute(o, XVW_IMAGE_YPOSITION, &newy);

    delx = newx - cursor_x;

    virgin = 0;

    iso_scale_box(delx);

    cursor_x = newx;
    cursor_y = newy;

    show_box();
}


void
btn2_down_func(xvobject o,
	       kaddr client_data,
	       XEvent *event)
{
    int x,y;

    xvw_get_attribute(o, XVW_IMAGE_XPOSITION, &cursor_x);
    xvw_get_attribute(o, XVW_IMAGE_YPOSITION, &cursor_y);

    virgin = 1;

    build_outline_object();
    set_up_xfrms();
    show_z_rotate_box();
    show_box();

    current_rotate_mode = get_rotate_mode(cursor_x,cursor_y);

    if (current_rotate_mode == Z_ROTATE)
	XDefineCursor(rmonster_display,rmonster_window,z_rotate_cursor);
    else
	XDefineCursor(rmonster_display,rmonster_window,xy_rotate_cursor);
	
}


void
btn2_motion_func(xvobject o,
	       kaddr client_data,
	       XEvent *event)
{
    int delx,dely,newx,newy;
    int rmode;

    show_box();
    
    xvw_get_attribute(o, XVW_IMAGE_XPOSITION, &newx);
    xvw_get_attribute(o, XVW_IMAGE_YPOSITION, &newy);

    delx = newx - cursor_x;
    dely = cursor_y - newy;

    virgin = 0;

    rmode = get_rotate_mode(newx,newy);

    if (rmode != current_rotate_mode) /* cursor change */
    {
	XUndefineCursor(rmonster_display,rmonster_window);
	if (rmode == Z_ROTATE)
	    XDefineCursor(rmonster_display,rmonster_window,z_rotate_cursor);
	else
	    XDefineCursor(rmonster_display,rmonster_window,xy_rotate_cursor);
	current_rotate_mode = rmode;
    }

    /* here, we're going to figure out if we're inside or outside the
       Z-rotate portal.  if inside, we'll rotate about x/y axes, otherwise,
       we'll rotate about the Z axis.  we might have to adjust the cursor
       to reflect the change. */

    if (rmode == Z_ROTATE)
	rotate_box_z(delx,-dely,newx,newy);
    else
    {
	rotate_box_y(delx);
	rotate_box_x(dely);
    }

    cursor_x = newx;
    cursor_y = newy;

    show_box();
}
void
btn2_up_func(xvobject o,
	       kaddr client_data,
	       XEvent *event)
{
    matrix4x4 t;
    
    show_box();
    show_z_rotate_box();
    
    XUndefineCursor(rmonster_display,rmonster_window);


    /* now, premultiply all the accumulated rotations and stuff
       back into the object transformation matrix. */

    if (virgin == 0)
    {
	matrix_4x4copy(&(ref_obj->xfrm),&model);

	set_current_object(ref_obj);
	sync_root_box();
    
/*	software_render_image(); */
	renderer_invoke_func();
    }
}


void
shift_btn2_up_func(xvobject o,
		   kaddr client_data,
		   XEvent *event)
{
    matrix4x4 t;
    
    show_box();
    XUndefineCursor(rmonster_display,rmonster_window);


    /* now, premultiply all the accumulated rotations and stuff
       back into the object transformation matrix. */

    if (virgin == 0)
    {
	matrix_4x4copy(&(ref_obj->xfrm),&model);

	set_current_object(ref_obj);
	sync_root_box();
    
/*	software_render_image(); */
        renderer_invoke_func();
    }
}

void
left_key_func(xvobject o,
	      kaddr client_data,
	      XEvent *event)
{
    double st,ct;
    double a;
    matrix4x4 t,t2;
    float f;
#ifdef DEBUG    
    fprintf(stderr,"left key pressed\n");
#endif
    ref_obj = get_current_object();
    matrix_4x4copy(&model,&(ref_obj->xfrm)); 
    get_parent_xfrm(&parent_xfrm);
    get_parent_xfrm_inverse(&parent_xfrm_inverse);

    set_up_xfrms();

    get_yrot_relative(&f);
    
    identity_4x4(&t);
    
    a = DEGREES_TO_RADIANS(f);
    st = ksin(a);
    ct = kcos(a);

    t.m[0][0] = t.m[2][2] = ct;
    t.m[0][2] = st;
    t.m[2][0] = -1. * st;

    mmul_4x4(&view_rotations,&t,&t);
    mmul_4x4(&t,&view_rotations_inverse,&t);
    mmul_4x4(&model,&t,&t2);
    matrix_4x4copy(&m_rx,&t2);
    matrix_4x4copy(&model,&t2);
    mmul_4x4(&t2,&view,&comp);
    
    matrix_4x4copy(&(ref_obj->xfrm),&model);
    set_current_object(ref_obj);
    sync_root_box();
    
    renderer_invoke_func();
}

void
right_key_func(xvobject o,
	       kaddr client_data,
	       XEvent *event)
{
    double st,ct;
    double a;
    matrix4x4 t,t2;
    float f;
#ifdef DEBUG    
    fprintf(stderr,"right key pressed\n");
#endif
    ref_obj = get_current_object();
    matrix_4x4copy(&model,&(ref_obj->xfrm)); 
    get_parent_xfrm(&parent_xfrm);
    get_parent_xfrm_inverse(&parent_xfrm_inverse);

    set_up_xfrms();

    get_yrot_relative(&f);
    f *= -1.;
    
    identity_4x4(&t);
    
    a = DEGREES_TO_RADIANS(f);
    st = ksin(a);
    ct = kcos(a);

    t.m[0][0] = t.m[2][2] = ct;
    t.m[0][2] = st;
    t.m[2][0] = -1. * st;

    mmul_4x4(&view_rotations,&t,&t);
    mmul_4x4(&t,&view_rotations_inverse,&t);
    mmul_4x4(&model,&t,&t2);
    matrix_4x4copy(&m_rx,&t2);
    matrix_4x4copy(&model,&t2);
    mmul_4x4(&t2,&view,&comp);
    
    matrix_4x4copy(&(ref_obj->xfrm),&model);
    set_current_object(ref_obj);
    sync_root_box();
    
    renderer_invoke_func();
}

void
up_key_func(xvobject o,
	    kaddr client_data,
	    XEvent *event)
{
    /* effect a positive rotation about the X-axis */
    double st,ct;
    double a;
    matrix4x4 t,t2;
    float f;
#ifdef DEBUG    
    fprintf(stderr,"up key pressed\n");
#endif    
    ref_obj = get_current_object();
    matrix_4x4copy(&model,&(ref_obj->xfrm)); 
    get_parent_xfrm(&parent_xfrm);
    get_parent_xfrm_inverse(&parent_xfrm_inverse);

    set_up_xfrms();

    get_xrot_relative(&f);
    
    identity_4x4(&t);
    
    a = DEGREES_TO_RADIANS(f);
    st = sin(a);
    ct = cos(a);

    t.m[1][1] = t.m[2][2] = ct;
    t.m[1][2] = st * -1.;
    t.m[2][1] = st;

    
    mmul_4x4(&view_rotations,&t,&t);
    mmul_4x4(&t,&view_rotations_inverse,&t);
    mmul_4x4(&model,&t,&t2);
    matrix_4x4copy(&m_rx,&t2);
    matrix_4x4copy(&model,&t2);
    mmul_4x4(&t2,&view,&comp);
    
    matrix_4x4copy(&(ref_obj->xfrm),&model);
    set_current_object(ref_obj);
    sync_root_box();
    
    renderer_invoke_func();
}

void
down_key_func(xvobject o,
	      kaddr client_data,
	      XEvent *event)
{
    /* effect a negative rotation about the X axis */
    double st,ct;
    double a;
    matrix4x4 t,t2;
    float f;
#ifdef DEBUG    
    fprintf(stderr,"down key pressed\n");
#endif    
    ref_obj = get_current_object();
    matrix_4x4copy(&model,&(ref_obj->xfrm)); 
    get_parent_xfrm(&parent_xfrm);
    get_parent_xfrm_inverse(&parent_xfrm_inverse);

    set_up_xfrms();

    get_xrot_relative(&f);
    f *= -1.;
    
    identity_4x4(&t);
    
    a = DEGREES_TO_RADIANS(f);
    st = sin(a);
    ct = cos(a);

    t.m[1][1] = t.m[2][2] = ct;
    t.m[1][2] = st * -1.;
    t.m[2][1] = st;

    
    mmul_4x4(&view_rotations,&t,&t);
    mmul_4x4(&t,&view_rotations_inverse,&t);
    mmul_4x4(&model,&t,&t2);
    matrix_4x4copy(&m_rx,&t2);
    matrix_4x4copy(&model,&t2);
    mmul_4x4(&t2,&view,&comp);
    
    matrix_4x4copy(&(ref_obj->xfrm),&model);
    set_current_object(ref_obj);
    sync_root_box();
    
    renderer_invoke_func();
}

void
page_up_key_func(xvobject o,
		 kaddr client_data,
		 XEvent *event)
{
    /* effect a positive rotation about the Z-axis */
    double st,ct;
    double a;
    matrix4x4 t,t2;
    float f;
#ifdef DEBUG    
    fprintf(stderr,"page up key pressed\n");
#endif    

    ref_obj = get_current_object();
    matrix_4x4copy(&model,&(ref_obj->xfrm)); 
    get_parent_xfrm(&parent_xfrm);
    get_parent_xfrm_inverse(&parent_xfrm_inverse);

    set_up_xfrms();

    get_zrot_relative(&f);
    
    identity_4x4(&t);
    
    a = DEGREES_TO_RADIANS(f);
    st = sin(a);
    ct = cos(a);

    st = ksin(a);
    ct = kcos(a);

    t.m[0][0] = t.m[1][1] = ct;
    t.m[0][1] = st;
    t.m[1][0] = -1. * st;

    mmul_4x4(&view_rotations,&t,&t);
    mmul_4x4(&t,&view_rotations_inverse,&t);
    mmul_4x4(&model,&t,&t2);
    matrix_4x4copy(&m_rx,&t2);
    matrix_4x4copy(&model,&t2);
    mmul_4x4(&t2,&view,&comp);
    
    matrix_4x4copy(&(ref_obj->xfrm),&model);
    set_current_object(ref_obj);
    sync_root_box();
    
    renderer_invoke_func();

}

void
page_down_key_func(xvobject o,
		   kaddr client_data,
		   XEvent *event)
{
    /* effect a positive rotation about the Z-axis */
    double st,ct;
    double a;
    matrix4x4 t,t2;
    float f;
#ifdef DEBUG    
    fprintf(stderr,"page down key pressed\n");
#endif    

    ref_obj = get_current_object();
    matrix_4x4copy(&model,&(ref_obj->xfrm)); 
    get_parent_xfrm(&parent_xfrm);
    get_parent_xfrm_inverse(&parent_xfrm_inverse);

    set_up_xfrms();

    get_zrot_relative(&f);
    f *= -1.;
    
    identity_4x4(&t);
    
    a = DEGREES_TO_RADIANS(f);
    st = sin(a);
    ct = cos(a);

    st = ksin(a);
    ct = kcos(a);

    t.m[0][0] = t.m[1][1] = ct;
    t.m[0][1] = st;
    t.m[1][0] = -1. * st;

    mmul_4x4(&view_rotations,&t,&t);
    mmul_4x4(&t,&view_rotations_inverse,&t);
    mmul_4x4(&model,&t,&t2);
    matrix_4x4copy(&m_rx,&t2);
    matrix_4x4copy(&model,&t2);
    mmul_4x4(&t2,&view,&comp);
    
    matrix_4x4copy(&(ref_obj->xfrm),&model);
    set_current_object(ref_obj);
    sync_root_box();
    
    renderer_invoke_func();
}


void
set_up_rmonster_mouse_button_definitions(xvobject xvimage,
					 kobject output_kimage)
{
    /**
      * here, we assign routines to various mouse buttons.
      *
      * we will use the right button for translations, the middle
      * button for rotations, and the left button for picks.
      *
      * action handlers are established for:
      *     right button down, right button motion, right button up -
      *          produce translations in the image plane
      *     <shift>+right button - produces translations perpindicular
      *          to the image plane
      *
      *     middle button - rotations.  which axis depends upon
      *          where the cursor is on the screen and which way
      *          the mouse moves.
      *
      *     shift+middle = isometric scaling
      *   
    **/

    xvw_add_event(xvimage, ExposureMask, rmonster_create_gc, NULL);
    xvw_add_event(xvimage, StructureNotifyMask, rmonster_resize,NULL);

    /* right button definitions: down, motion, up */
    xvw_add_action(xvimage,"<Btn3Down>",btn3_down_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Btn3Motion>",btn3_motion_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Btn3Up>",btn3_up_func,NULL,TRUE);
    
    /* shift_right button definitions: down, motion, up.  note that
     the same routines for init and completion are used here as in
     plain ole btn3_down.  only the shift_btn3_motion func is different */
    
    xvw_add_action(xvimage,"Shift<Btn3Down>",btn3_down_func,NULL,TRUE); 
    xvw_add_action(xvimage,"Shift<Btn3Motion>",shift_btn3_motion_func,NULL,TRUE);
    xvw_add_action(xvimage,"Shift<Btn3Up>",btn3_up_func,NULL,TRUE); 
    
    /* middle button definitions: down, motion, up - rotations*/
    xvw_add_action(xvimage,"<Btn2Down>",btn2_down_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Btn2Motion>",btn2_motion_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Btn2Up>",btn2_up_func,NULL,TRUE);

    /* keyboard actions */

    xvw_add_action(xvimage,"<Key>Left",left_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>Right",right_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>Up",up_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>Down",down_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>Page_Up",page_up_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>Page_Down",page_down_key_func,NULL,TRUE);

    /* -- silly ole X, the above "standard" keycodes only work on sun -- */

    /* -- define some other key sequences  -- */

    /* -- (x,X) for X  (y,Y) for Y  (z,Z) for Z -- */
    /* -- chosen cuz they are ez to recall -- */
    xvw_add_action(xvimage,"<Key>x",left_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>X",right_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>y",up_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>Y",down_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>z",page_up_key_func,NULL,TRUE);
    xvw_add_action(xvimage,"<Key>Z",page_down_key_func,NULL,TRUE);

    /* shift-middle button definitions: scaling */

    /* first, delete old actions.. */
#if 0
    xvw_remove_action(xvimage,"Shift<Btn2Down>",btn2_down_func,NULL);
    xvw_remove_action(xvimage,"Shift<Btn2Motion>",btn2_motion_func,NULL);
    xvw_remove_action(xvimage,"Shift<Btn2Up>",btn2_up_func,NULL);
#endif
    
    xvw_add_action(xvimage,"Shift<Btn2Down>",shift_btn2_down_func,NULL,TRUE);
    xvw_add_action(xvimage,"Shift<Btn2Motion>",shift_btn2_motion_func,NULL,TRUE);
    xvw_add_action(xvimage,"Shift<Btn2Up>",shift_btn2_up_func,NULL,TRUE);

}


void
show_render_cursor()
{
   if (rmonster_display == NULL)
      return;

    xvw_busy(rmonster_xvobject,FALSE);
    XDefineCursor(rmonster_display,rmonster_window,zencursor);
    XFlush(rmonster_display);

}

void
clear_render_cursor()
{
   if (rmonster_display == NULL)
      return;

    XUndefineCursor(rmonster_display,rmonster_window);
    XFlush(rmonster_display);

}

static float x_rot_relative,y_rot_relative,z_rot_relative;

void
set_xrot_relative(float f)
{
    x_rot_relative = f;
}
void
set_yrot_relative(float f)
{
    y_rot_relative = f;
}
void
set_zrot_relative(float f)
{
    z_rot_relative = f;
}
void
get_xrot_relative(float *f)
{
    *f = x_rot_relative;
}
void
get_yrot_relative(float *f)
{
    *f = y_rot_relative;
}
void
get_zrot_relative(float *f)
{
    *f = z_rot_relative;
}
