
/*********************************************************************

Copyright (C) 1993, 1994, 1995 Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

1/1/95, wes. changed the way that arrays returned by kgeom_get_attribute()
    are handled.  previously, this call returned an array.  presently, it
    returns a pointer to an area of memory under control of the system.
    hence, it can't be free'd, nor otherwise used.  it is read-only and
    volatile.  the changes made were to cull out free's and to do 
    kmemcpy's where previously these arrays were copied directly into 
    rmonster_obj structs.

10 jan 95, wes.    cleaned up file detect code.
30 jan 95, wes.    connected triangle strips without normals will no
    longer be converted to disjoint triangles.  rather, the normals
    will be computed on a per-face basis, resulting in images in which
    the objects appear faceted.  users now bear the responsiblility
    of generating appropriate normals if this behavior is not desireable.

*********************************************************************/



#include <design.h>

#include "rmonster.h"

#include "rmonster_objs.h"
#include "shader.h"
#include "clip.h"

#include "rmatrix.h"

int reread_input_object PROTO((xvobject, char *, kaddr));
int i_zero=0,i_one=1,i_two=2,i_three=3;

static rmonsterobj root;
static int num_objects_in_tree=0;
static rmonsterobj *current_object = &root;
static matrix4x4 current_parent_xfrm,current_parent_xfrm_inverse;

static void free_rmonsterobj PROTO((rmonsterobj *r));
static void free_rmonster_prim PROTO((rmonster_prim *p));

static float fdetect_wait_time=2.0;  /* time in seconds between select() calls
					on input file descriptors. */

static rmonsterobj *input_objects[NUMBER_RMONSTER_INPUT_OBJECT_PORTS]=
{
    NULL,
    NULL,
    NULL,
    NULL
};


/**
  * the following utility routines are responsible for
  * converting the various khoros specific primitives into
  * rmonster "native" format.
  *
  * in the future, there will be a "low memory" version of
  * this code which won't make a copy of all the vertex, etc.
  * information.
  *
  * NOTE: the order of the following routines is the same as in
  * $DATASERV/include/kappserv/kgeometry.h!!!!
**/



static void rm_nullprim PROTO((rmonster_prim *, kobject, int, int));
static void rm_kpolyline_dis PROTO((rmonster_prim *, kobject, int, int));
static void rm_kpolyline_con PROTO((rmonster_prim *, kobject, int, int));
static void rm_ktriangles_dis PROTO((rmonster_prim *, kobject, int, int));
static void rm_ktriangles_con PROTO((rmonster_prim *, kobject, int, int));
static void rm_kspheres PROTO((rmonster_prim *, kobject, int, int));
static void rm_kcylinders PROTO((rmonster_prim *, kobject, int, int));
static void rm_kquadmesh PROTO((rmonster_prim *, kobject, int, int));
static void rm_koctmesh PROTO((rmonster_prim *, kobject, int, int));
static void rm_kdirpoints PROTO((rmonster_prim *, kobject, int, int));
static void rm_kpolyhedra_face PROTO((rmonster_prim *, kobject, int, int));
static void rm_kpolyhedra_verts PROTO((rmonster_prim *, kobject,int, int));
static void rm_ktexture2d PROTO((rmonster_prim *, kobject, int, int));
static void rm_ktexture3d PROTO((rmonster_prim *, kobject, int, int));

static int primitives[] = 
{
	KNULL_PRIMITIVE,
	KGEOM_POLYLINE_DISJOINT,
	KGEOM_POLYLINE_CONNECTED,
	KGEOM_TRIANGLES_DISJOINT,
	KGEOM_TRIANGLES_CONNECTED,
	KGEOM_SPHERES, 
	KGEOM_CYLINDERS, 
	KGEOM_DIRECTED_POINTS, 
	KGEOM_QUADMESH, /* a more complex set of primitives is now available */ 

	KGEOM_OCTMESH,  /* a more complex set of primitives is now available */
#if 0
	KGEOM_TEXTURE2D, /* a more complex set of primitives is now available */
	KGEOM_TEXTURE3D, /* a more complex set of primitives is now available */
#endif
};

static void (*convert_prim_funcs[])() =
{
    rm_nullprim,
    rm_kpolyline_dis,
    rm_kpolyline_con,
    rm_ktriangles_dis,
    rm_ktriangles_con,
    rm_kspheres,
    rm_kcylinders,
    rm_kdirpoints,
    rm_kquadmesh,
    rm_koctmesh,
#if 0
    rm_ktexture2d,
    rm_ktexture3d
#endif
};

/* determine the number of primitive types from the above table */
static int num_prim_types = sizeof(primitives) / sizeof(primitives[0]);

#define NUM_PRIM_TYPES num_prim_types


static void copy_kattribs_to_rmonsterobj PROTO((kobject,rmonsterobj *,int));
static void compute_rmonster_bbox PROTO((rmonsterobj *));
static void determine_transparency PROTO((rmonsterobj *));
static void determine_default_shader PROTO((rmonsterobj *));
static int search_for_named_object PROTO((char *, rmonsterobj **, rmonsterobj *));
static void union_bounding_boxes PROTO((rmonsterobj *a,rmonsterobj *b,vertex_3d *bmin, vertex_3d *bmax));

int load_child_names PROTO((char **, int *, rmonsterobj *, int));

/**
  * this file contains routines which are repsonsible for
  * converting between khoros geom objects and rmonster
  * objects.  this routine knows about the clui_info_struct
  * as well as the rmonster-specific objects.
**/


int
init_object_list(clui_info_struct *rmc)
{
    int status;
    rmonsterobj *t;
    int root_num_kids = 0;
    

    /* initialize the root object */
    init_rmonster_obj(&root,ROOT_OBJECT_NAME);
    current_object = &root;
    num_objects_in_tree = 1;

    /* now, read in each of the objects.it would be nice if
     the i1_files in the command structure were in an array....
     also, this is where the scene services interface will go
     to grab kobjects whenever that finally gets built */

    if (rmc->i1_flag)
    {
	t = create_rmonsterobj();
	status = convert_kobj_to_rmonsterobj(rmc->i1_file,t);
	if (status != CHILL)
/* KERROR CAT: "init_object_list" */
	    kerror("GEOMETRY","rmonster","failed to convert kobject (-i) to internal format");
	else
	{
	    add_child(&root,t);
	    num_objects_in_tree++;
/*	    replace_input_object(0,t); */
	    input_objects[0] = t;
	    xvw_add_detectfile(NULL,rmc->i1_file,fdetect_wait_time,
			       reread_input_object,&i_zero);
	}
    }

    /* check the additional flags.  this is where having an
       array would be nice.  c'est la vie. */

    if (rmc->i2_flag)
    {
	t = create_rmonsterobj();
	status = convert_kobj_to_rmonsterobj(rmc->i2_file,t);
	if (status != CHILL)
/* KERROR CAT: "init_object_list" */
	    kerror("GEOMETRY","rmonster","failed to convert kobject (-i2) to internal format");
	else
	{
	    add_child(&root,t);
	    num_objects_in_tree++;
/*	    replace_input_object(1,t); */
	    input_objects[1] = t;
	    xvw_add_detectfile(NULL,rmc->i2_file,fdetect_wait_time,
			       reread_input_object,&i_one);
	}
    }
    
    if (rmc->i3_flag)
    {
	t = create_rmonsterobj();
	status = convert_kobj_to_rmonsterobj(rmc->i3_file,t);
	if (status != CHILL)
/* KERROR CAT: "init_object_list" */
	    kerror("GEOMETRY","rmonster","failed to convert kobject (-i3) to internal format");
	else
	{
	    add_child(&root,t);
	    num_objects_in_tree++;
/*	    replace_input_object(2,t); */
	    input_objects[2] = t;
	    xvw_add_detectfile(NULL,rmc->i3_file,fdetect_wait_time,
			       reread_input_object,&i_two);
	}
    }
    
    if (rmc->i4_flag)
    {
	t = create_rmonsterobj();
	status = convert_kobj_to_rmonsterobj(rmc->i4_file,t);
	if (status != CHILL)
/* KERROR CAT: "init_object_list" */
	    kerror("GEOMETRY","rmonster","failed toc onvert kobject (-i4) to internal format");
	else
	{
	    add_child(&root,t);
	    num_objects_in_tree++;
/*	    replace_input_object(3,t); */
	    input_objects[3] = t;
	    xvw_add_detectfile(NULL,rmc->i4_file,fdetect_wait_time,
			       reread_input_object,&i_three);
	}
    }

    set_current_object(&root);
    
    return(CHILL);
}

void
init_rmonster_obj(rmonsterobj *r,char *name)
{
    memset((char *)r,0,sizeof(rmonsterobj));
    kstrncpy(r->name,name,KGEOM_MAX_NAME_LENGTH);

    identity_4x4(&(r->xfrm));

    /* default surface attributes */
    r->attribs.amb = KGEOM_AMBIENT_DEFAULT;
    r->attribs.diff = KGEOM_DIFFUSE_DEFAULT;
    r->attribs.spec = KGEOM_SPECULAR_COEFFICIENT_DEFAULT;
    r->attribs.spec_exp = KGEOM_SPECULAR_EXPONENT_DEFAULT;
    r->attribs.spec_metal = KGEOM_METAL_DEFAULT;

    /* set center to origin */
    r->center.v[0] = r->center.v[1] = r->center.v[2];

    /* default color */
    r->colors.r = r->colors.g = r->colors.b = r->colors.a = 1.;

    /* default display status */
    r->do_display = TRUE;

    /* default shader */
    r->shader = KSHADER_INHERIT;
}


rmonsterobj *
create_rmonsterobj()
{
    rmonsterobj *t;
    t = (rmonsterobj *)kmalloc(sizeof(rmonsterobj));
    init_rmonster_obj(t,NULL);
    return(t);
}

int
add_child(rmonsterobj *parent, rmonsterobj *child)
{
    int i;
    vertex_3d union_box_min,union_box_max;
    
    if ((parent->nchildren == MAXOFFSPRING) ||
	(child->nparents == MAXOFFSPRING))
	return(WHACKED);
    
    parent->children[parent->nchildren++] = child;
    child->parents[child->nparents++] = parent;

    /**
      * update parent's bounding box to reflect extents of the
      * child's bounding box.  this is union type operation.
    **/

    union_bounding_boxes(parent,child,&union_box_min,&union_box_max);

    VCOPY(&(parent->bmin),&union_box_min);
    VCOPY(&(parent->bmax),&union_box_max);

    for (i=0;i<3;i++)
	parent->center.v[i] = parent->bmin.v[i] + 0.5 * (parent->bmax.v[i] - parent->bmin.v[i]);

    return(CHILL);
}

int
convert_kobj_to_rmonsterobj(char *file,
			    rmonsterobj *r)
{
    kobject k;
    char *name=NULL;
    int nprims,i, j;
    rmonster_prim *p;
    int *primlist=NULL;
    
    k = kgeom_open_input_object(file);
    if (k == KOBJECT_INVALID)
    {
	kerror("GEOMETRY","convert_kobj_to_rmonsterobj","The input object <%s> is problematic, and can't be opened.",file);
	return(WHACKED);
    }

    /* get the name */
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_NAME,&name);
    /* assign name if there is one   -- added 7/11/93  // Steve K */
    if (name)
      strncpy((char *)r->name,(char *)name,KGEOM_MAX_NAME_LENGTH);

    if (kgeom_query_primitive_list(k))
    {
       /* get the number of primitives in this object */
       kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,&nprims);
       r->nprims = nprims;

       /* grab some memory for the primitives. */
       r->prims = (rmonster_prim *)kmalloc(sizeof(rmonster_prim)*nprims);
       memset((char *)(r->prims),0,sizeof(rmonster_prim)*nprims);

       kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_PRIMITIVE_LIST,&primlist);

       kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_LAYOUT,&(r->layout));
       kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_HAS_ALPHA,&(r->has_alpha));

       /* -- set some friendly presntation attributes // SK -- */
       kgeom_set_attributes(k,KGEOM_OBJECT,
	                    KGEOM_LOCATION_DATA_TYPE, KFLOAT,
	                    KGEOM_COLOR_DATA_TYPE,    KFLOAT,
	                    KGEOM_NORMAL_DATA_TYPE,   KFLOAT,
	                    KGEOM_LOCATION_SIZE,      3,
	                    NULL);
    
       for (i=0;i<nprims;i++)
       {
   	   /* TEMPORARY KLUDGE */
	   kgeom_set_attribute(k,KGEOM_OBJECT,KGEOM_PRIMITIVE_POSITION,i);
	   /* END KLUDGE */
	
	   p = r->prims+i;
	   p->type = primlist[i];

           for (j=0;j<NUM_PRIM_TYPES;j++)
              if (p->type == primitives[j])
              {
	         (*convert_prim_funcs[j])(p,k,r->layout,r->has_alpha);
                 break;
              }
       }
       copy_kattribs_to_rmonsterobj(k,r,r->has_alpha);
       determine_default_shader(r);
    }
    
/*    kfree(name);  1/1/95 don't free stuff returned by get_attrib(). wes*/
    kgeom_close_object(k);
    return(CHILL);
}

int
return_object_names(char ***list,int *n)
{
    char *t;
    int i;
    
    t = kmalloc(sizeof(char)*KGEOM_MAX_NAME_LENGTH);
    *list = (char **)kmalloc(sizeof(char *)*num_objects_in_tree);
    
    kstrncpy((char *)t,(char *)(root.name),KGEOM_MAX_NAME_LENGTH);

    *list[0] = t;  /* set entry 0 to the root object. */
    *n = 1;

    for (i=0;i<root.nchildren;i++)
	load_child_names(*list,n,root.children[i],1);

    return(CHILL);
}

int
load_child_names(char **list,int *n,rmonsterobj *r,int level)
{
    int i;
    char *t,*t2;

    t = t2 = kmalloc(sizeof(char)*KGEOM_MAX_NAME_LENGTH);
    
    for (i=0;i<level;i++)
	*t2++ = ' ';
    
    kstrncpy((char *)t2,(char *)(r->name),KGEOM_MAX_NAME_LENGTH-level);
    list[*n] = t;
    *n += 1;

    if (r->nchildren > 0)
    {
	/* recursively descend into tree */
	for (i=0;i<r->nchildren;i++)
	    load_child_names(list,n,r->children[i],level+1);
    }

    return(CHILL);
}

static void rm_nullprim (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
}

static void rm_kpolyline_dis (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
    int nverts;
    float *verts=NULL,*colors=NULL;
    
    kgeom_get_attribute(k,KGEOM_POLYLINE_DISJOINT,KGEOM_NUMBER_VERTICES,&nverts);
#ifdef DEBUG
    kprintf("\tthere are %d verts in this kpolyline_dis prim.\n",nverts);
#endif
    
    kgeom_get_data(k,KGEOM_POLYLINE_DISJOINT,&verts,&colors);

    if (verts == NULL) /* strange object */
	p->nverts = 0;
    else
	p->nverts = nverts;
    p->verts = (vertex_3d *)verts;

    if (colors == NULL)
	p->ncolors = 0;
    else if (layout == KPER_VERTEX)
	p->ncolors = nverts;
    else /*  (layout == KPER_VECTOR) */
	p->ncolors = nverts/2;

    if (has_alpha)
	p->rgba_color_list = (surface_colors_rgba *)colors;
    else
	p->rgb_color_list = (surface_colors_rgb *)colors;
    
    p->type = KGEOM_POLYLINE_DISJOINT;

    p->clipfunc = clip_polyline_dis;

}

static void rm_kpolyline_con (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
    int nverts;
    float *verts=NULL,*colors=NULL;
    
    kgeom_get_attribute(k,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,&nverts);
#ifdef DEBUG
    kprintf("\tthere are %d verts in this kpolyline_dis prim.\n",nverts);
#endif

    kgeom_get_data(k,KGEOM_POLYLINE_CONNECTED,&verts,&colors);

    if (verts == NULL) /* strange object */
	p->nverts = 0;
    else
	p->nverts = nverts;
    p->verts = (vertex_3d *)verts;

    if (colors == NULL)
	p->ncolors = 0;
    else if (layout == KPER_VERTEX)
	p->ncolors = nverts;
    else /*  (layout == KPER_VECTOR) */
	p->ncolors = nverts/2;

    if (has_alpha)
	p->rgba_color_list = (surface_colors_rgba *)colors;
    else
	p->rgb_color_list = (surface_colors_rgb *)colors;
    
    p->type = KGEOM_POLYLINE_CONNECTED;

    p->clipfunc = clip_polyline_con;
}

static void rm_ktriangles_dis (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
    int nverts;
    vertex_3d p1,p2;
    vertex_3d *vnormals,*vverts;
    int nnormals,i;
    int index,j;
    double dmag;
    float *verts=NULL,*colors=NULL,*normals=NULL,*tcoords=NULL;
    
    kgeom_get_attribute(k,KGEOM_TRIANGLES_DISJOINT,KGEOM_NUMBER_VERTICES,&nverts);
#ifdef DEBUG
    kprintf("\tthere are %d verts in this ktriangles_dis prim.\n",nverts);
#endif
    
    kgeom_get_data(k,KGEOM_TRIANGLES_DISJOINT,&verts,&colors,&normals,&tcoords);

    if (verts == NULL) /* strange object */
	p->nverts = 0;
    else
	p->nverts = nverts;
    p->verts = (vertex_3d *)verts;

    if (colors == NULL)
	p->ncolors = 0;
    else if (layout == KPER_VERTEX)
	p->ncolors = nverts;
    else /*  (layout == KPER_FACE) */
	p->ncolors = nverts/3;

    if (has_alpha)
	p->rgba_color_list = (surface_colors_rgba *)colors;
    else
	p->rgb_color_list = (surface_colors_rgb *)colors;

    if (normals == NULL)
    {
	/**
	  * this section of code has not been tested.  8/17/94
	**/
	/**
	 * if there were no normals present in the object, compute them
	 * here.
	 **/
	if (layout == KPER_VERTEX)
	    nnormals = nverts;
	else
	    nnormals = nverts/3;
	vnormals = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nnormals);
	vverts = (vertex_3d *)verts;
	
	for (i=0;i<nnormals;i++)
	{
	    /**
	      * compute index so that we grab the correct vertices for
	      * each triangle.  
	    **/

	    if (layout == KPER_VERTEX)
		index = i/3;
	    else
		index = i*3;
	    
	    for (j=0;j<3;j++)
	    {
		p1.v[j] = vverts[index+1].v[j] - vverts[index].v[j];
		p2.v[j] = vverts[index+2].v[j] - vverts[index+1].v[j];
	    }
	    
	    vertex_cross(&p1,&p2,vnormals+i);
	    vertex_unit(vnormals+i,&dmag);
	    
	    if (i & 1) /* flip every other normal */
	    {
		for (j=0;j<3;j++)
		    vnormals[i].v[j] *= -1.;
	    }
	}
	p->nthings = nnormals;
	normals = (float *)vnormals;
    }
    else if (layout == KPER_VERTEX)
	p->nthings = nverts;
    else /* layout == KPER_FACE */
	p->nthings = nverts/3;
    p->prim_specifics.normals = (vertex_3d *)normals;

    p->type = KGEOM_TRIANGLES_DISJOINT;

    p->clipfunc = clip_polygon_cheap;

    /* blow off texture coordinates until things get straightened out */
}


static void rm_ktriangles_con (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
    int nverts;
    vertex_3d p1,p2;
    vertex_3d *vnormals,*vverts;
    int nnormals,i;
    int index,j;
    double dmag;
    float *verts=NULL,*colors=NULL,*normals=NULL,*tcoords=NULL;

    /**
      * 1/30/95
      * tstrips which don't have normals will have them computed here
      * in this routine.  we have two choices, either to compute the
      * normals per-face (cross product of triangle edges), or to compute
      * the normals so as to approximate some kind of smooth surface.
      * we choose to compute the normals in the former way.  the images
      * resulting from computing normals in this way will appear faceted.
      * if the user wishes to have the t-strip approximate a smooth
      * surface, they must compute the normals themselves outside of
      * rmonster.
    **/
    
    kgeom_get_attribute(k,KGEOM_TRIANGLES_CONNECTED,KGEOM_NUMBER_VERTICES,&nverts);

#ifdef DEBUG
    kprintf("\tthere are %d verts in this ktriangles_con prim.\n",nverts);
#endif

    kgeom_get_data(k,KGEOM_TRIANGLES_CONNECTED,&verts,&colors,&normals,&tcoords);

    /**
      * check for whether or not we're going to build a disjoint or
      * connected primitive.
    **/

    /* the layout attribute in the kgeom object refers to BOTH colors
       and normals */
    
    if (layout == KPER_FACE) 
    {
	/**
	  * build a disjoint triangle primitive.
	**/
	vertex_3d *work_verts,*work_normals,*work_tcoords,*v;
	surface_colors_rgba *wc_a;
	surface_colors_rgb  *wc;
	int work_nverts,do_tc,do_colors,offset;

	work_nverts = (nverts-2)*3;

	do_colors = (colors == NULL) ? 0 : 1;
	do_tc = (tcoords == NULL) ? 0 : 1;
	work_verts = (vertex_3d *)kmalloc(sizeof(vertex_3d)*work_nverts);
	work_normals = (vertex_3d *)kmalloc(sizeof(vertex_3d)*work_nverts);

	v = (vertex_3d *)verts;

	/**
	  * copy over the vertices from connected format to disconnected
	  * format.
	**/
	offset=0;
	for (i=0;i<(nverts-2);i++)  /* loop on triangles */
	{
	    if (i & 1)
	    {
		VCOPY(work_verts+offset,v+i+1);
		offset++;
		VCOPY(work_verts+offset,v+i);
		offset++;
		VCOPY(work_verts+offset,v+i+2);
		offset++;
	    }
	    else
	    {
		VCOPY(work_verts+offset,v+i);
		offset++;
		VCOPY(work_verts+offset,v+i+1);
		offset++;
		VCOPY(work_verts+offset,v+i+2);
		offset++;
	    }
	}
	p->nverts = work_nverts;
	p->verts = (vertex_3d *)work_verts;
	kfree(verts);

	/**
	  * compute normals, if necessary, using the new vertex list.
	**/
	if (normals == NULL)
	{
	    vverts = work_verts;
	    offset=0;
	    for (i=0;i<(nverts-2);i++)  /* loop on triangles */
	    {
		index = i*3;
		for (j=0;j<3;j++)
		{
		    p1.v[j] = vverts[index+1].v[j] - vverts[index].v[j];
		    p2.v[j] = vverts[index+2].v[j] - vverts[index+1].v[j];
		}
	    
		vertex_cross(&p1,&p2,work_normals+offset);
		vertex_unit(work_normals+offset,&dmag);

		/**
		  * flip normal.
		**/
		for (j=0;j<3;j++)
		    work_normals[offset].v[j] *= -1.;

		/* replicate that normal over the entire triangle */
		VCOPY(work_normals+offset+1,work_normals+offset);
		offset++;
		VCOPY(work_normals+offset+1,work_normals+offset);
		offset+=2;

	    }
	}
	else  /* have normals already, need to redistribute */
        {
	    if (layout == KPER_VERTEX)
	    {
		v = (vertex_3d *)normals;
		offset=0;
		for (i=0;i<(nverts-2);i++)  /* loop on triangles */
		{
		    if (i & 1)
		    {
			VCOPY(work_normals+offset,v+i+1);
			offset++;
			VCOPY(work_normals+offset,v+i);
			offset++;
			VCOPY(work_normals+offset,v+i+2);
			offset++;
		    }
		    else
		    {
			VCOPY(work_normals+offset,v+i);
			offset++;
			VCOPY(work_normals+offset,v+i+1);
			offset++;
			VCOPY(work_normals+offset,v+i+2);
			offset++;
		    }
		}
	    }
	    else /* layout == KPER_FACE */
	    {
		v = (vertex_3d *)normals;
		offset=0;
		for (i=0;i<(nverts-2);i++)  /* loop on triangles */
		{
		    VCOPY(work_normals+offset,v+i);
		    offset++;
		    VCOPY(work_normals+offset,v+i);
		    offset++;
		    VCOPY(work_normals+offset,v+i);
		    offset++;
		}
	    }
	} /* end dealing with normals */
	p->nthings = work_nverts;
	p->prim_specifics.normals = (vertex_3d *)work_normals;
	
	/**
	  * deal with color information, if any
	**/

	if (colors)
	{
	    float *f;
	    int stride;
	    
	    f = colors;

	    if (has_alpha)
	    {
		stride = 4;
		wc_a = (surface_colors_rgba *)
		   kmalloc(sizeof(surface_colors_rgba)*work_nverts);
	    }
	    else
	    {
		stride = 3;
		wc = (surface_colors_rgb *)
		   kmalloc(sizeof(surface_colors_rgb)*work_nverts);
	    }

	    /*
	     *   previously did not consider per-face with no alpha
	     *   (alpha values of 1. were inserted)  this case can
	     *   be tested with a color legend generated from gcmaplegend.
	     *
	     *   sort out the colors differently depending on 
	     *   whether or not there is alpha information.
	     *
	     *   I suspect I did this in the most difficult way
	     *   possible :(  // SK
	     */
	    if (has_alpha)
	    {
	       if (layout == KPER_VERTEX)
	       {
		  offset=0;
		  for (i=0;i<(nverts-2);i++)  /* loop on triangles */
		  {
		     if (i & 1)
		     {
			VCOPY((vertex_3d *)(wc_a+offset),
			      (vertex_3d *)(f+(i+1)*stride));
			wc_a[offset].a = *(f+(i+1)*stride+3);
			offset++;
			
			VCOPY((vertex_3d *)(wc_a+offset),
			      (vertex_3d *)(f+(i)*stride));
			wc_a[offset].a = *(f+(i)*stride+3);
			offset++;
			
			VCOPY((vertex_3d *)(wc_a+offset),
			      (vertex_3d *)(f+(i+2)*stride));
			wc_a[offset].a = *(f+(i+2)*stride+3);
			offset++;
		     }
		     else
		     {
			VCOPY((vertex_3d *)(wc_a+offset),
			      (vertex_3d *)(f+(i)*stride));
			wc_a[offset].a = *(f+(i)*stride+3);
			offset++;
			
			VCOPY((vertex_3d *)(wc_a+offset),
			      (vertex_3d *)(f+(i+1)*stride));
			wc_a[offset].a = *(f+(i+1)*stride+3);
			offset++;
			
			VCOPY((vertex_3d *)(wc_a+offset),
			      (vertex_3d *)(f+(i+2)*stride));
			wc_a[offset].a = *(f+(i+2)*stride+3);
			offset++;
		     }
		  }
	       }
	       else /* layout == KPER_FACE */
	       {
		  f = colors;
		  offset=0;
		  for (i=0;i<(nverts-2);i++)  /* loop on triangles */
		  {
		     VCOPY((vertex_3d *)(wc_a+offset),
			   (vertex_3d *)(f+(i)*stride));
		     wc_a[offset].a = *(f+(i)*stride+3);
		     offset++;
		     VCOPY((vertex_3d *)(wc_a+offset),
			   (vertex_3d *)(f+(i)*stride));
		     wc_a[offset].a = *(f+(i)*stride+3);
		     offset++;
		     VCOPY((vertex_3d *)(wc_a+offset),
			   (vertex_3d *)(f+(i)*stride));
		     wc_a[offset].a = *(f+(i)*stride+3);
		     offset++;
		  }
	       } /* end dealing with colors */
	       p->rgb_color_list = NULL;
	       p->rgba_color_list = wc_a;
	       p->ncolors = work_nverts;
	       kfree(colors);
	    }
	    else /* has no alpha */
	    {
	       if (layout == KPER_VERTEX)
	       {
		  offset=0;
		  for (i=0;i<(nverts-2);i++)  /* loop on triangles */
		  {
		     if (i & 1)
		     {
			VCOPY((vertex_3d *)(wc+offset),
			      (vertex_3d *)(f+(i+1)*stride));
			offset++;
			
			VCOPY((vertex_3d *)(wc+offset),
			      (vertex_3d *)(f+(i)*stride));
			offset++;
			
			VCOPY((vertex_3d *)(wc+offset),
			      (vertex_3d *)(f+(i+2)*stride));
			offset++;
		     }
		     else
		     {
			VCOPY((vertex_3d *)(wc+offset),
			      (vertex_3d *)(f+(i)*stride));
			offset++;
			
			VCOPY((vertex_3d *)(wc+offset),
			      (vertex_3d *)(f+(i+1)*stride));
			offset++;
			
			VCOPY((vertex_3d *)(wc+offset),
			      (vertex_3d *)(f+(i+2)*stride));
			offset++;
		     }
		  }
	       }
	       else /* layout == KPER_FACE */
	       {
		  f = colors;
		  offset=0;
		  for (i=0;i<(nverts-2);i++)  /* loop on triangles */
		  {
		     VCOPY((vertex_3d *)(wc+offset),
			   (vertex_3d *)(f+(i)*stride));
		     offset++;
		     VCOPY((vertex_3d *)(wc+offset),
			   (vertex_3d *)(f+(i)*stride));
		     offset++;
		     VCOPY((vertex_3d *)(wc+offset),
			   (vertex_3d *)(f+(i)*stride));
		     offset++;
		  }
	       } /* end dealing with colors */
	       p->rgb_color_list = wc;
	       p->rgba_color_list = NULL;
	       p->ncolors = work_nverts;
	       kfree(colors);
	    }
	}
	else
	{
	   p->ncolors = 0;
	   p->rgba_color_list = NULL;
	   p->rgb_color_list = NULL;
	}

	p->type = KGEOM_TRIANGLES_DISJOINT;
    }
    else			/* layout is per-vertex */
    {
	/**
	  * build a connected triangle primitive.  we assume per-vertex
	  * everything.
	**/
	p->nverts = nverts;
	p->verts = (vertex_3d *)verts;

	if (colors)
	    p->ncolors = nverts;
	else
	    p->ncolors = 0;
	if (has_alpha)
	    p->rgba_color_list = (surface_colors_rgba *)colors;
	else
	    p->rgb_color_list = (surface_colors_rgb *)colors;

	if (normals == NULL)	/* no normals.  compute them per-face */
	{
	    vertex_3d *w_norms;
	    vertex_3d *work,*lverts;
	    int n,i,j;

	    n = nverts-2;
	    lverts = (vertex_3d *)verts;
	    
	    p->nthings = n; /* per face normals will be computed */
	    w_norms = (vertex_3d *)kmalloc(sizeof(vertex_3d)*n);
	    work = (vertex_3d *)kmalloc(sizeof(vertex_3d)*(n+1));

	    /* compute c[i] = a[i]-b[i] */
	    for (i=0;i<n+1;i++)
	    {
		for (j=0;j<3;j++)
		    work[i].v[j] = lverts[i+1].v[j] - lverts[i].v[j];
	    }

	    for (i=0;i<n;i++)
		vertex_cross(work+i,work+i+1,w_norms+i);

	    nnormal_unit(w_norms,w_norms,n);

	    for (i=0;i<n;i++)
	    {

		if (!(i&1))
		{ 
		    for (j=0;j<3;j++)
			w_norms[i].v[j] *= -1.;
		}

	    }
	    p->prim_specifics.normals = w_norms;

	    kfree(work);

	}
	else
	{
	    p->nthings = nverts;
	    p->prim_specifics.normals = (vertex_3d *)normals;
	}
	
	p->type = KGEOM_TRIANGLES_CONNECTED;

    }

    p->clipfunc = clip_polygon_cheap;


    /* blow off texture coordinates until things get straightened out */
}

static void rm_kspheres (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
    int nspheres;
    float *v=NULL,*c=NULL,*r=NULL;

    kgeom_set_attribute(k,KGEOM_SPHERES,KGEOM_LOCATION_DATA_TYPE,KFLOAT);
    kgeom_set_attribute(k,KGEOM_SPHERES,KGEOM_RADIUS_DATA_TYPE,KFLOAT);
    kgeom_set_attribute(k,KGEOM_SPHERES,KGEOM_COLOR_DATA_TYPE,KFLOAT);

    kgeom_get_attribute(k,KGEOM_SPHERES,KGEOM_NUMBER_VERTICES,&nspheres);
    kgeom_get_data(k,KGEOM_SPHERES,&v,&c,&r);

    
    if (v != NULL)
    {
	p->nverts = nspheres;
	p->verts = (vertex_3d *)v;
    }
    else
    {
	p->nverts = 0;
	p->verts = NULL;
    }

    if (r != NULL)
    {
	p->prim_specifics.radii = r;
	p->nthings = nspheres;
    }
    else
    {
	p->prim_specifics.radii = NULL;
	p->nthings = 0;
    }

    if (c != NULL)
    {
	p->ncolors = nspheres;
	if (has_alpha)
	{
	    p->rgba_color_list = (surface_colors_rgba *)c;
	    p->rgb_color_list = NULL;
	}
	else
	{
	    p->rgba_color_list = NULL;
	    p->rgb_color_list = (surface_colors_rgb *)c;
	}
    }
    else
    {
	p->ncolors = 0;
    }

    p->clipfunc = clip_polygon_cheap;

    
}

static void rm_kcylinders (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
}

static void rm_kquadmesh (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
    int usize,vsize,nverts,has_loc,has_color,has_normals;
    float *verts=NULL,*normals=NULL,*colors=NULL,*texture=NULL;
    vertex_3d *tn=NULL;
    
    kgeom_get_attribute(k,KGEOM_QUADMESH,KGEOM_QUADMESH_SIZE,&usize,&vsize);
    nverts = usize*vsize;

    has_loc =  kgeom_get_data(k,KGEOM_QUADMESH_LOCATION_ALL,&verts);
    has_color = kgeom_get_data(k,KGEOM_QUADMESH_COLOR_ALL,&colors);
    has_normals = kgeom_get_data(k,KGEOM_QUADMESH_NORMAL_ALL,&normals);

    p->type = KGEOM_QUADMESH;
    if (verts == NULL) /* strange object */
    {
	p->nverts = 0;
	p->mesh_dims[0] = 0;
	p->mesh_dims[1] = 0;
	p->mesh_dims[2] = 0;
    }
    else
    {
	p->nverts = usize*vsize;
	p->mesh_dims[0] = usize;
	p->mesh_dims[1] = vsize;
	p->mesh_dims[2] = 1; /* for sanity reasons */
	p->nverts = nverts;
    }

    if (!has_normals)  /* then compute them */
    {
	/**
	  * make a call to compute a bunch of per-vertex
	  * normals for the vertices.  we'll compute them once
	  * here and keep them around rather than recomputing
	  * them each time we rerender.
        **/
/*	lcompute_normals(verts,&normals,usize,vsize); */
	lcompute_quadmesh_vertex_normals((vertex_3d *)verts,&tn,
					 usize,vsize);
	normals = (float *)tn;
    }
    
    p->verts = (vertex_3d *)verts;

    if (colors == NULL)
	p->ncolors = 0;
    else if (layout == KPER_VERTEX)
	p->ncolors = nverts;
    else /*  (layout == KPER_FACE) */
	p->ncolors = (usize-1)*(vsize-1);

    if (has_alpha)
	p->rgba_color_list = (surface_colors_rgba *)colors;
    else
	p->rgb_color_list = (surface_colors_rgb *)colors;

    if (normals == NULL)
	p->nthings = 0;
    else if (layout == KPER_VERTEX)
	p->nthings = nverts;
    else /* layout == KPER_FACE */
	p->nthings = (usize-1)*(vsize-1);
    p->prim_specifics.normals = (vertex_3d *)normals;

    p->clipfunc = clip_polygon_cheap;
    /* blow off texture coordinates until things get straightened out */

}

static void rm_koctmesh (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
    /**
      * right now, we are handling ONLY curvilinear meshes.  this needs
      * to be changed to handle UNIFORM and RECTILINEAR meshes.
    **/
    
    int usize,vsize,wsize,nverts,has_loc,has_color,has_normals;
    float *verts=NULL,*normals=NULL,*colors=NULL,*texture=NULL;
    
    kgeom_get_attribute(k,KGEOM_OCTMESH,KGEOM_OCTMESH_SIZE,&usize,&vsize,&wsize);
    nverts = usize*vsize*wsize;

    has_loc =  kgeom_get_data(k,KGEOM_OCTMESH_LOCATION_ALL,&verts);
    has_color = kgeom_get_data(k,KGEOM_OCTMESH_COLOR_ALL,&colors);
    has_normals = kgeom_get_data(k,KGEOM_OCTMESH_NORMAL_ALL,&normals);

    p->type = KGEOM_OCTMESH;
    if (verts == NULL) /* strange object */
    {
	p->nverts = 0;
	p->mesh_dims[0] = 0;
	p->mesh_dims[1] = 0;
	p->mesh_dims[2] = 0;
    }
    else
    {
	p->nverts = usize*vsize*wsize;
	p->mesh_dims[0] = usize;
	p->mesh_dims[1] = vsize;
	p->mesh_dims[2] = wsize;
	p->nverts = nverts;
    }

    if (!has_normals)  /* then compute them */
    {
#if 0
	/**
	  * make a call to compute a bunch of per-vertex
	  * normals for the vertices.  we'll compute them once
	  * here and keep them around rather than recomputing
	  * them each time we rerender.
        **/
/*	lcompute_normals(verts,&normals,usize,vsize); */
	lcompute_quadmesh_vertex_normals((vertex_3d *)verts,&normals,
					 usize,vsize);
#endif
    }
    
    p->verts = (vertex_3d *)verts;

    if (colors == NULL)
	p->ncolors = 0;
    else if (layout == KPER_VERTEX)
	p->ncolors = nverts;
    else /*  (layout == KPER_FACE) */
	p->ncolors = (usize-1)*(vsize-1)*(wsize-1);

    if (has_alpha)
	p->rgba_color_list = (surface_colors_rgba *)colors;
    else
	p->rgb_color_list = (surface_colors_rgb *)colors;

    if (normals == NULL)
	p->nthings = 0;
    else if (layout == KPER_VERTEX)
	p->nthings = nverts;
    else /* layout == KPER_FACE */
	p->nthings = (usize-1)*(vsize-1);
    p->prim_specifics.normals = (vertex_3d *)normals;

    p->clipfunc = clip_nullfunc;
    /* blow off texture coordinates until things get straightened out */
}

static void rm_kdirpoints (rmonster_prim *p, kobject k,int layout, int has_alpha)
{
    kfprintf(kstderr," directed point primitives not yet supported.\n");
}


static void rm_kpolyhedra_face (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
}

static void rm_kpolyhedra_verts (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
}


static void rm_ktexture2d (rmonster_prim *p, kobject k, int layout, int has_alpha)
{
    kfprintf(kstderr," texture 2d primitives not yet supported.\n");
}

static void rm_ktexture3d (rmonster_prim *p, kobject k, int layout, int has_alpa)
{
    kfprintf(kstderr," texture 3d primitives not yet supported.\n");
}

static void
copy_kattribs_to_rmonsterobj(kobject k,rmonsterobj *p, int has_alpha)
{
    double d;
    int status,i,j,offset;
    float *f,work[4];
    float *bmin,*bmax,*tmp;

    f = work;
    /**
      * for any object attribute which isn't defined, we use a
      * default or otherwise try and do something sensible.
    **/

    /* first, do the surface attributes */
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_AMBIENT,&d);
    if (!status)
	p->attribs.amb = KGEOM_AMBIENT_DEFAULT;
    else
	p->attribs.amb = (float)d;

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_DIFFUSE,&d);
    if (!status)
	p->attribs.diff = KGEOM_DIFFUSE_DEFAULT;
    else
	p->attribs.diff = (float)d;

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_SPECULAR_COEFFICIENT,&d);
    if (!status)
	p->attribs.spec = KGEOM_SPECULAR_COEFFICIENT_DEFAULT;
    else
	p->attribs.spec = (float)d;

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_SPECULAR_EXPONENT,&d);
    if (!status)
	p->attribs.spec_exp = KGEOM_SPECULAR_EXPONENT_DEFAULT;
    else
	p->attribs.spec_exp = (float)d;

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_METAL,&d);
    if (!status)
	p->attribs.spec_metal = KGEOM_METAL_DEFAULT;
    else
	p->attribs.spec_metal = (float)d;

    /* now, do the surface color */

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_COLOR,&f);
    if (status)
    {
	p->colors.r = f[0];
	p->colors.g = f[1];
	p->colors.b = f[2];
	if (has_alpha)
	    p->colors.a = f[3];
	else
	    p->colors.a = 1.0;
    }

    /* do the bounding box */
    bmin = bmax = NULL;
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_BOUNDING_BOX,&bmin,&bmax);
    if (!status)
    {
	/* no box defined, compute one. */
	compute_rmonster_bbox(p);
    }
    else
    {
	for (i=0;i<3;i++)
	{
	    p->bmin.v[i] = (float)bmin[i];
	    p->bmax.v[i] = (float)bmax[i];
	}
/*	kfree(bmin); don't free stuff returned by get_attr().  wes 1/1/95*/
/*	kfree(bmax); ditto. */
    }

    /* do the center point */
    bmin = NULL;
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_CENTER,&bmin);
    if (!status)
    {
	/* no center point, compute one. */
	for (i=0;i<3;i++)
	    p->center.v[i] = (p->bmax.v[i] + p->bmin.v[i]) * 0.5;
    }
    else
    {
	for (i=0;i<3;i++)
	    p->center.v[i] = (float)bmin[i];
/*	kfree(bmin); don't free stuff returned by get_attr(). 1/1/95 wes */
    }


    /* get the transformation matrix. */
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_MATRIX,&bmin);
    if (!status)
	identity_4x4(&(p->xfrm));
    else
    {
	offset=0;
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
		p->xfrm.m[i][j] = (KGEOM_VERTEX_TYPE)bmin[offset++];
	
/*	kfree(bmin); */
    }

    /** 1/1/95 change how the following is handled to make copies of
      stuff returned by get_attribute().  1/1/95 wes. **/
#if 0
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_ROTATE,&(p->rotate_vect.v[0]));
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_SCALE,&(p->scale_vect.v[0]));
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_TRANSLATE,&(p->translate_vect.v[0]));
#endif
#if 0
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_ROTATE,&tmp);
    kmemcpy(&(p->rotate_vect.v[0]),tmp,sizeof(vertex_3d));

    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_SCALE,&tmp);
    kmemcpy(&(p->scale_vect.v[0]),tmp,sizeof(vertex_3d));

    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_TRANSLATE,&tmp);
    kmemcpy(&(p->translate_vect.v[0]),tmp,sizeof(vertex_3d));
#endif
    /* get the display status */
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_VISIBLE,&i);
    if (!status)
	p->do_display = KGEOM_VISIBLE_DEFAULT;
    else
	p->do_display = i;

    /* determine if any part of this object is transparent */
    if (has_alpha)
    {
	/* need to scan through the list of colors */
	determine_transparency(p);
    }
    else
	p->is_transparent = FALSE;

    
       
}

static void
compute_rmonster_bbox(rmonsterobj *p)
{
    float *mins,*maxs;
    float *f;
    int i;

    mins = (float *)kmalloc(sizeof(float)*p->nprims*3);
    maxs = (float *)kmalloc(sizeof(float)*p->nprims*3);

    for (i=0;i<p->nprims;i++)
    {
	f = (float *)&(p->prims[i].verts[0]);
	vect_min(f,p->prims[i].nverts*3,0,3,mins+i);
	vect_min(f+1,p->prims[i].nverts*3,0,3,mins+i+p->nprims);
	vect_min(f+2,p->prims[i].nverts*3,0,3,mins+i+(p->nprims*2));
	
	vect_max(f,p->prims[i].nverts*3,0,3,maxs+i);
	vect_max(f+1,p->prims[i].nverts*3,0,3,maxs+i+p->nprims);
	vect_max(f+2,p->prims[i].nverts*3,0,3,maxs+i+(p->nprims*2));
    }

    for (i=0;i<3;i++)
	vect_min(mins,p->nprims,i*p->nprims,1,&(p->bmin.v[i]));
    
    for (i=0;i<3;i++)
	vect_max(maxs,p->nprims,i*p->nprims,1,&(p->bmax.v[i]));
}

static void
determine_transparency(rmonsterobj *p)
{
    int has_prim_color,i,j;
    int has_some_transparency;
    surface_colors_rgba *c;
    
    /* if there is no color information at the primitive level,
       then just look at the global object color vector. */
    for (i=0;i<p->nprims;i++)
    {
	if (p->prims[i].ncolors != 0)
	    break;
    }
    if (i < p->nprims)
	has_prim_color = TRUE;
    else
	has_prim_color = FALSE;

    if (has_prim_color == FALSE)
    {
	if (p->colors.a != 1.)
	    p->is_transparent = TRUE;
	else
	    p->is_transparent = FALSE;
    }
    else
    {
	has_some_transparency = FALSE;
	for (i=0;i<p->nprims;i++)
	{
	    c = p->prims[i].rgba_color_list;
	    if (c != NULL)
	    {
		for (j=0;j<p->prims[i].ncolors;j++,c++)
		    if (c->a != 1.)
		    {
			has_some_transparency = TRUE;
			break;
		    }
	    }
	    if (has_some_transparency == TRUE)
		break;
	}
	if (has_some_transparency == TRUE)
	    p->is_transparent = TRUE;
	else
	    p->is_transparent = FALSE;
    }
}

static void
determine_default_shader(rmonsterobj *r)
{
    /* taken from rmonster_objs.h: */
    /**
      * since a geom object is comprised of several primitives,
      * the notion of an object level shader applying to heterogeneous
      * primitives is slippery.  for now, we are assuming that
      * all the primitive types are homogeneous (ie, all volumetric,
      * all surface based, or all wireframe) and that all of the
      * primitives will use the object-level shader.
    **/

    if (r->nprims == 0)
	r->shader = KSHADER_INHERIT;
    else
    {
	/* just look at the first primitive - assume they're
	   all homogeneous */
	
	switch (r->prims[0].type)
	{
        case KGEOM_POLYLINE_DISJOINT:
        case KGEOM_POLYLINE_CONNECTED:
            r->shader = KSHADER_NO_LIGHT;
            break;
            
        case KGEOM_TRIANGLES_DISJOINT:
        case KGEOM_TRIANGLES_CONNECTED:
        case KGEOM_CYLINDERS:
        case KGEOM_QUADMESH:
        case KGEOM_DIRECTED_POINTS:
            r->shader = KSHADER_FLAT;
            break;

        case KGEOM_SPHERES:
            r->shader = KSHADER_FLAT;
            /*  r->shader = KSHADER_GOUROUD;  slow.. :(  // Steve K */
            break;

        case KGEOM_OCTMESH:
/*          r->shader = KSHADER_SPLAT_HOMOGENEOUS; */
            r->shader = KSHADER_VOXEL_DOT;
            break;

#if 0
        case KGEOM_TEXTURE2D:
        case KGEOM_TEXTURE3D:
#endif
        default:
            r->shader = KSHADER_INHERIT;
            break;
	}
    }
}

int
find_rmonsterobj(char *search_for_name, rmonsterobj **r)
{
    int status;

    status = search_for_named_object(search_for_name,r,&root);
    return(status);
}

static int
search_for_named_object(char *name, rmonsterobj **rval, rmonsterobj *r)
{
    int i,status;

    if (kstrcmp(name,r->name) == 0)
    {
	*rval = r;
	return(CHILL);
    }
    else
    {
	for (i=0;i<r->nchildren;i++)
	{
	    status = search_for_named_object(name,rval,r->children[i]);
	    if (status == CHILL)
		return(CHILL);
	}
    }
    return(WHACKED);
}

rmonsterobj *get_current_object()
{
    return(current_object);
}

void
get_parent_xfrm(matrix4x4 *m)
{
    matrix_4x4copy(m,&current_parent_xfrm);
}

void
get_parent_xfrm_inverse(matrix4x4 *m)
{
    matrix_4x4copy(m,&current_parent_xfrm_inverse);
}


static void
build_parent_xfrm(rmonsterobj *r,
		  matrix4x4 *m)
{
    rmonsterobj *p;
    matrix4x4 t1,comp;
    
    if (r->nparents > 1)  /* deep shit */
    {
	kerror("GEOMETRY","rmonster","this object has more than one parent. you cannot transform this object. ");
	return;
    }
    else if (r->nparents == 0) /* top of tree */
    {
	return;
    }
    else
    {
	p = r->parents[0];
	/* validity check on p ?? */

	/* construct forward transformation for p. this consists of
	   the following matrix concatenations:
	       -Center * Scale * Rotate * Center * Translate
	   this composite matrix will be postmultiplied to the one
	   passed in to this routine.
	 */
	identity_4x4(&t1);
	identity_4x4(&comp);

	t1.m[3][0] = -1.*p->center.v[0];
	t1.m[3][1] = -1.*p->center.v[1];
	t1.m[3][2] = -1.*p->center.v[2];

	/* for now, scaling and rotation are collapsed into a single
	   matrix in the object.  */
	
	mmul_4x4(&t1,&(p->xfrm),&comp);

	t1.m[3][0] = p->center.v[0] + p->translate_vect.v[0];
	t1.m[3][1] = p->center.v[1] + p->translate_vect.v[1];
	t1.m[3][2] = p->center.v[2] + p->translate_vect.v[2];

	mmul_4x4(&comp,&t1,&comp);

	
	build_parent_xfrm(p,&comp);
	mmul_4x4(m,&comp,m);
    }
}

int
set_current_object(rmonsterobj *r)
{
    vertex_3d t;
    matrix4x4 m;
    
    /* check to make sure it exists?? */
    current_object = r;

    /**
      * also, set the current transformation matrix and center point.
      * this is the spot where we'll catch impossible conditions, such
      * as attempting to set the cumulative transformation (bottom-up)
      * for objects with more than one parent.
      *
      * we store, along with a pointer to the current object, a single
      * transformation matrix which represents that transformation
      * of the parent.  in other words, to compute the position in view
      * space, we need the transformation of the current object, plus
      * the transformation of the parent.  since the reference object may
      * be at an arbitrary place in the object tree, we need to backwards-
      * compute a transformation, representing the transformation for
      * the parent.  the viewspace vector would computed by transforming
      * and object's vertex value through its local transformation matrix,
      * then through that of it's parent (postmultiplication).
      *
      * NOTE: this backwards transformation should not be confused with
      * the inverse of the parent's transform.
      *
      * the static variable current_parent_xfrm holds the above described
      * transformation matrix, and is computed by the routine
      * build_parent_xfrm().
    **/
       
    identity_4x4(&m);

    build_parent_xfrm(r,&m);
    matrix_4x4copy(&current_parent_xfrm,&m);
    matrix_inverse(&current_parent_xfrm,&current_parent_xfrm_inverse);

    /* update the transformation UI */
    update_transformation_pane(r);
    update_object_pane(r);

    return(CHILL);
}

int
set_object_shader (rmonsterobj *r, int shader)
{
    /* range check ?? */
    r->shader = shader;
    return(CHILL);
}

int
get_object_shader (rmonsterobj *r, int *shader)
{
    *shader = r->shader;
    return(CHILL);
}

int
set_object_display_status (rmonsterobj *r, int value)
{
    /** range check ?? */
    r->do_display = value;
    return(CHILL);
}

int
get_object_display_status (rmonsterobj *r, int *value)
{
    *value = r->do_display;
    return(CHILL);
}

int
get_object_rgba (rmonsterobj *o, float *r, float *g, float *b, float *a)
{
    *r = o->colors.r;
    *g = o->colors.g;
    *b = o->colors.b;
    *a = o->colors.a;
    return(CHILL);
}

int
set_object_rgba (rmonsterobj *o, float r, float g, float b, float a)
{
    /* range check on input ?? */
    o->colors.r = r;
    o->colors.g = g;
    o->colors.b = b;
    o->colors.a = a;
    determine_transparency(o);
    return(CHILL);
}

int
set_object_amb (rmonsterobj *r, float f)
{
    /* range check input?? */
    r->attribs.amb = f;
    return(CHILL);
}

int
get_object_amb (rmonsterobj *r, float *f)
{
    *f = r->attribs.amb;
    return(CHILL);
}

int
set_object_diff (rmonsterobj *r, float f)
{
    /* check input ?? */
    r->attribs.diff = f;
    return(CHILL);
}

int
get_object_diff (rmonsterobj *r, float *f)
{
    *f = r->attribs.diff;
    return(CHILL);
}

int
set_object_spec (rmonsterobj *r, float f)
{
    /* check input ?? */
    r->attribs.spec = f;
    return(CHILL);
}

int
get_object_spec (rmonsterobj *r , float *f)
{
    /* check input ?? */
    *f = r->attribs.spec;
    return(CHILL);
}

int
set_object_spec_exp (rmonsterobj *r, float f)
{
    /* check input ?? */
    r->attribs.spec_exp = f;
    return(CHILL);
}

int
get_object_spec_exp (rmonsterobj *r, float *f)
{
    *f = r->attribs.spec_exp;
    return(CHILL);
}

int
set_object_metal (rmonsterobj *r, float f)
{
    /* check input ? */
    r->attribs.spec_metal = f;
    return(CHILL);
}

int
get_object_metal (rmonsterobj *r, float *f)
{
    *f = r->attribs.spec_metal;
    return(CHILL);
}

rmonsterobj *
get_root_object()
{
    return(&root);
}

static void
union_bounding_boxes(rmonsterobj *a,
		     rmonsterobj *b,
		     vertex_3d *bmin,
		     vertex_3d *bmax)
{
    vertex_3d l_amin,l_amax,l_bmin,l_bmax;
    
    /**
      * if the boxes of either object are {0.,0.,0},{0.,0.,0.},
      * we will assume that the box hasn't been defined.
      *
      * the box in object "a" will not be transformed (matrix or translate
      * vector).  the box in object "b" will be transformed.
    **/

    int a_undefined=0,b_undefined=0,i;
    
    VCOPY(&l_amin,&(a->bmin));
    VCOPY(&l_amax,&(a->bmax));
    VCOPY(&l_bmin,&(b->bmin));
    VCOPY(&l_bmax,&(b->bmax));

    if ((l_amin.v[0] == 0.) &&
	(l_amin.v[1] == 0.) &&
	(l_amin.v[2] == 0.) &&
	(l_amax.v[0] == 0.) &&
	(l_amax.v[1] == 0.) &&
	(l_amax.v[2] == 0.))
	a_undefined = 1;

    if ((l_bmin.v[0] == 0.) &&
	(l_bmin.v[1] == 0.) &&
	(l_bmin.v[2] == 0.) &&
	(l_bmax.v[0] == 0.) &&
	(l_bmax.v[1] == 0.) &&
	(l_bmax.v[2] == 0.))
	b_undefined = 1;
    
    /* transform "b" points back to the center of the object. */

    npoint_diff((KGEOM_VERTEX_TYPE *)&(l_bmin.v[0]),
		(KGEOM_VERTEX_TYPE *)&(b->center.v[0]),
		(KGEOM_VERTEX_TYPE *)&(l_bmin.v[0]),1);

    npoint_diff((KGEOM_VERTEX_TYPE *)&(l_bmax.v[0]),
		(KGEOM_VERTEX_TYPE *)&(b->center.v[0]),
		(KGEOM_VERTEX_TYPE *)&(l_bmax.v[0]),1);


    /* apply these points through the transformation matrix */
    npoint_xfrm(&(l_bmin.v[0]),&(b->xfrm),&(l_bmin.v[0]),1);
    npoint_xfrm(&(l_bmax.v[0]),&(b->xfrm),&(l_bmax.v[0]),1);

    /* add back the center, plus the translation vector */

    for (i=0;i<3;i++)
    {
	l_bmin.v[i] = l_bmin.v[i] + b->translate_vect.v[i] + b->center.v[i];
	l_bmax.v[i] = l_bmax.v[i] + b->translate_vect.v[i] + b->center.v[i];
/***
	l_bmin.v[i] += b->translate_vect.v[i] + b->center.v[i];
	l_bmax.v[i] += b->translate_vect.v[i] + b->center.v[i];
****/
    }

    if (a_undefined && b_undefined) /* compute the boxes. */
    {
	fprintf(stderr," dual compute and union not ready yet. \n");
    }
    else if (a_undefined && !b_undefined) 
    {
	VCOPY(bmin,&(l_bmin));
	VCOPY(bmax,&(l_bmax));
    }
    else if (b_undefined && !a_undefined)
    {
	VCOPY(bmin,&(l_amin));
	VCOPY(bmax,&(l_amax));
    }
    else
    {
	for (i=0;i<3;i++)
        {
	    bmin->v[i] = MIN(l_amin.v[i],l_bmin.v[i]);
	    bmin->v[i] = MIN(bmin->v[i],l_bmax.v[i]);
	    bmin->v[i] = MIN(bmin->v[i],l_amax.v[i]);
	    
	    bmax->v[i] = MAX(l_amin.v[i],l_bmin.v[i]);
	    bmax->v[i] = MAX(bmax->v[i],l_amax.v[i]);
	    bmax->v[i] = MAX(bmax->v[i],l_bmax.v[i]);
	}
    }
}

static void
local_sync_root_box(rmonsterobj *r)
{
    int i,j;
    vertex_3d tmin,tmax;
    rmonsterobj *c;

    /* this routine is incomplete.  it only gets the first generation of
       root descendants */

    for (i=0;i<3;i++)
    {
	r->bmin.v[i] = r->bmax.v[i] = 0.;
    }
    
    for (i=0;i<r->nchildren;i++)
    {
	c = r->children[i];
	union_bounding_boxes(r,c,&(r->bmin),&(r->bmax));
    }

#if 0
    /**
      * recompute the center point so that rotations happen
      * "nicely".  this might not be the best way, and is a hack,
      * oh well.
   **/

    for (i=0;i<3;i++)
	r->center.v[i] = r->bmin.v[i] + (r->bmax.v[i] - r->bmin.v[i]) * 0.5;
#endif
}

/**
  * here, we're going to tranverse downward through the object
  * tree with the intent of recomputing the roots min/max box.
  * this is necessary to do since the user may move objects around,
  * then expect the root object to encompass them all in subsequent
  * transformations.
**/
void
sync_root_box()
{
    int i;

    local_sync_root_box(&root);
    
}

void
get_bounding_box(rmonsterobj *r,
		 vertex_3d *bmin,
		 vertex_3d *bmax)
{
    /**
      * this routine will return the bounding box for the object "r",
      * subject to whatever local transformations are applied to "r",
      * such as scaling, rotations and translations.
    **/

    vertex_3d box[2];
    int i,j;

    VCOPY(box,&(r->bmin));
    VCOPY(box+1,&(r->bmax));

    /* translate the box back to the center point of the object. */

    for (i=0;i<2;i++)
    {
	for (j=0;j<3;j++)
	    box[i].v[j] -= r->center.v[j];
    }

    /* apply transformation matrix */
    npoint_xfrm((KGEOM_VERTEX_TYPE *)&(box[0].v[0]),
		&(r->xfrm),
		(KGEOM_VERTEX_TYPE *)&(box[0].v[0]), 2);

    /* now, add back in the center point and the object translational comp. */
    
    for (i=0;i<2;i++)
    {
	for (j=0;j<3;j++)
	    box[i].v[j] += r->center.v[j] + r->translate_vect.v[j];
    }

    VCOPY(bmin,box);
    VCOPY(bmax,box+1);
}

void
geom_xform_points(matrix4x4 *child_xfrm,
		  vertex_3d *child_tv,
		  vertex_3d *child_center,
		  vertex_3d *in,
		  vertex_3d *out,
		  matrix4x4 *parent_xfrm,
		  vertex_3d *parent_tv,
		  int n)
/**
  * this routine is used to apply geometric transformations to lists of points.
  * it is called from the rendering routines in render.c, as well as
  * the xor bounding box code in rmonster_mouse.c.
  *
  * the order of operations is as follows:
  * 1. translate the points back to the local origin
  * 2. apply scaling then rotational transformations
  * 3. translate by -1 * local_origin + object translation
  * 4. apply the transformation specified by the parent.
  *    this will be represented by the composite matrix:
  *        [Scale*Rotate + Translate]
  *
  * to do: verify that all these operations can be collapsed into a single
  * matrix multiply 
**/
{
    vertex_3d work;
    matrix4x4 t;
   
    VCOPY(&work,child_center);

    /* translate to local origin */
#if 0
    work.v[0] += child_tv->v[0];
    work.v[1] += child_tv->v[1];
    work.v[2] += child_tv->v[2];
#endif
    npoint_diff(&(in->v[0]),&(work.v[0]),&(out->v[0]),n);

    /* apply scale and rotation operators */
    npoint_xfrm(&(out->v[0]),child_xfrm,&(out->v[0]),n);

    /* translate the origin back to the center point, and
       add translation, along with parents translation vector */

    work.v[0] += child_tv->v[0];
    work.v[1] += child_tv->v[1];
    work.v[2] += child_tv->v[2];

    npoint_add(&(out->v[0]),&(work.v[0]),&(out->v[0]),n);

    /* construst composite transformation matrix for the parent */
    identity_4x4(&t);

    if (parent_tv != NULL)
    {
	t.m[3][0] = parent_tv->v[0];
	t.m[3][1] = parent_tv->v[1];
	t.m[3][2] = parent_tv->v[2];
    }
    
    if (parent_xfrm != NULL)
	mmul_4x4(parent_xfrm,&t,&t);

    /* could put a check here for t == I and possibly avoid the
       following call */
    
    npoint_xfrm(&(out->v[0]),&t,&(out->v[0]),n);
			  
}

/**
  * need to move this routine to the geometry library when finished.
  * 8/11/94  wes
**/
int
lcompute_normals(float *verts,
		 float **normals,
		 int usize,
		 int vsize)
{
    int uv,vv;
    int i,j,k;
    vertex_3d *n;
    vertex_3d s1,s2;
    vertex_3d *v;
    double mag;

    /**
      * here, we compute normals on a per-vertex basis.  if per-facet
      * normals are desired, they could be derived from those
      * computed here, or from the raw vertices directly
    **/

    uv = usize;
    vv = vsize;

    /* alloc space for one normal per vertex */
    n = (vertex_3d *)kmalloc(sizeof(vertex_3d)*uv*vv);
    
    v = (vertex_3d *)verts;

    /* do general case first. */
    for (j=0;j<vv;j++)
    {
	for (i=0;i<uv;i++)
	{
	    if ((i != 0) && (i != uv-1) && (j != 0) && (j != vv-1))
	    {
		for (k=0;k<3;k++)
		{
		    s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
		    s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
		}
		vertex_unit(&s1,&mag);
		vertex_unit(&s2,&mag);
		vertex_cross(&s1,&s2,(n+i+j*uv));
	    }
	}
    }
    
    /* do the j=0 row. */
    j=0;
    for (i=0;i<uv;i++)
    {
	if ((i != 0) && (i != uv-1))
	{
	    for (k=0;k<3;k++)
	    {
		s1.v[k] = (v+i+1)->v[k] - (v+i-1)->v[k];
		s2.v[k] = (v+i+uv)->v[k] - (v+i)->v[k];
	    }
	    vertex_unit(&s1,&mag);
	    vertex_unit(&s2,&mag);
	    vertex_cross(&s1,&s2,(n+i));
	}
    }
    
    /* do the j=vv-1 row. */
    j=vv-1;
    for (i=0;i<uv;i++)
    {
	if ((i != 0) && (i != uv-1))
	{
	    for (k=0;k<3;k++)
	    {
		s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
		s2.v[k] = (v+i+j*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
	    }
	    vertex_unit(&s1,&mag);
	    vertex_unit(&s2,&mag);
	    vertex_cross(&s1,&s2,(n+i+j*uv));
	}
    }
    
    /* do the i=0 column */
    i=0;
    for (j=0;j<vv;j++)
    {
	if ((j != 0) && (j != vv-1))
	{
	    for (k=0;k<3;k++)
	    {
		s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i+j*uv)->v[k];
		s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
	    }
	    vertex_unit(&s1,&mag);
	    vertex_unit(&s2,&mag);
	    vertex_cross(&s1,&s2,(n+i+j*uv));
	}
    }
    
    /* do the i=uv column */
    i=uv-1;
    for (j=0;j<vv;j++)
    {
	if ((j != 0) && (j != vv-1))
	{
	    for (k=0;k<3;k++)
	    {
		s1.v[k] = (v+i+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
		s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
	    }
	    vertex_unit(&s1,&mag);
	    vertex_unit(&s2,&mag);
	    vertex_cross(&s1,&s2,(n+i+j*uv));
	}
    }
    /* now, do the four corners. these normals are computed as the
      cross product of the segments adjoining the 2 nearest points. */
    
    /* i==0,j==0 */
    i=0;
    j=0;
    for (k=0;k<3;k++)
    {
	s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i+j*uv)->v[k];
	s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+j*uv)->v[k];
    }
    vertex_unit(&s1,&mag);
    vertex_unit(&s2,&mag);
    vertex_cross(&s1,&s2,(n+i+j*uv));
    

    /* i=uv-1,j=0 */
    i=uv-1;
    j=0;
    for (k=0;k<3;k++)
    {
	s1.v[k] = (v+i+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
	s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+j*uv)->v[k];
    }
    vertex_unit(&s1,&mag);
    vertex_unit(&s2,&mag);
    vertex_cross(&s1,&s2,(n+i+j*uv));
    
    /* i=0,j=vv-1 */
    i=0;
    j=vv-1;
    for (k=0;k<3;k++)
    {
	s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i+j*uv)->v[k];
	s2.v[k] = (v+i+j*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
    }
    vertex_unit(&s1,&mag);
    vertex_unit(&s2,&mag);
    vertex_cross(&s1,&s2,(n+i+j*uv));
    
    /* i=uv-1,j=vv-1 */
    i=uv-1;
    j=vv-1;
    for (k=0;k<3;k++)
    {
	s1.v[k] = (v+i+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
	s2.v[k] = (v+i+j*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
    }
    vertex_unit(&s1,&mag);
    vertex_unit(&s2,&mag);
    vertex_cross(&s1,&s2,(n+i+j*uv));

    *normals = (float *)n;
    return(CHILL);
}

/**
  *
  * stuff to address object input.
  *
  * the notion is that there are up to N input for objects (N currently
  * #define'd to 4 (august 94).
  *
  * we keep static pointers around that point to the rmonster objs,
  * thus establishing logical links between the N inputs and the
  * rmonsterobjs in the object tree.  this will facilitate changing
  * the objects if either the input file changes in content or name.
  *
  * note that each of the input objects is a first-generation child
  * of the root object.
  *
**/
#if 0
static rmonsterobj *input_objects[NUMBER_RMONSTER_INPUT_OBJECT_PORTS]=
{
    NULL,
    NULL,
    NULL,
    NULL
};
#endif
void replace_input_object(int n,
			  rmonsterobj *r)
/**
  * n - index into input_objects[] array, ie, which input port is
  *     being tweaked.
  * r - the new rmonsterobj
**/
{
    rmonsterobj *t;
    vertex_3d union_box_min,union_box_max;
    int i;
    
    /**
      * free up the memory associated with the old object
    **/
    if (input_objects[n] != NULL)
    {
	/* free the old object */
	free_rmonsterobj(input_objects[n]);
	
	/**
	 * do a *copy, which avoids having to traverse through the
	 * object tree fixing up pointers to preserve the parent child
	 * relationships.
	 **/

	*(input_objects[n]) = *r;

	/**
	  * now, tweak the UI.
	**/
    }
    else /* virgin object pointer. */
    {
	input_objects[n] = r;
	add_child(&root,r);
	num_objects_in_tree++;
    }

    /**
      * update parent's bounding box to reflect extents of the
      * child's bounding box.  this is union type operation.
      * the root object is the parent, the child object is "r".
    **/

    union_bounding_boxes(&root,r,&union_box_min,&union_box_max);

    /**
      * compute the object's center point based upon bounding box.
    **/
    for (i=0;i<3;i++)
	root.center.v[i] = root.bmin.v[i] + 0.5 * (root.bmax.v[i] - root.bmin.v[i]);

    /**
      * rebuild the object list UI
    **/
      
    init_object_list_GUI(NULL);
    
    /** kfree(r) memory leak potential if this is commented out **/
}

static void
free_rmonsterobj(rmonsterobj *r)
{
    int i,n;
    rmonster_prim *p;

    n = r->nprims;
    p = r->prims;
    
    for (i=0;i<n;i++,p++)
	free_rmonster_prim(p);

    kfree(r->prims);
}

static void
free_rmonster_prim(rmonster_prim *p)
{
    if (p->verts)
	kfree(p->verts);

    if (p->prim_specifics.normals)
	kfree(p->prim_specifics.normals);
    
    if (p->rgb_color_list)
	kfree(p->rgb_color_list);
    
    if (p->rgba_color_list)
	kfree(p->rgba_color_list);

}

/**
  * action handler repsonsible for rereading an object when it`s
  * file contents changes.
**/
int
reread_input_object(xvobject dummy,
		    char *fname,
		    kaddr data)
{
    int *n,status;
    rmonsterobj *t;
    char work[256];

    n = (int *)data;

    kstrcpy(work,fname);
    
    t = create_rmonsterobj();
    status = convert_kobj_to_rmonsterobj(work,t);
    if (status == CHILL)
    {
	replace_input_object(*n,t);
	software_render_image();
    }
    else
	kerror("GEOMETRY","convert_kobj_to_rmonsterobj","The input object <%s> is problematic, and can't processed.",fname);


    /* else issue an error msg?? */
    return(TRUE);
}

int
get_object_bbox(rmonsterobj *r,
		vertex_3d *bmin,
		vertex_3d *bmax)
{
    VCOPY(bmin,&(r->bmin));
    VCOPY(bmax,&(r->bmax));
    return(TRUE);
}


int
set_object_bbox(rmonsterobj *r,
		vertex_3d *bmin,
		vertex_3d *bmax)
{
    /* range, etc., checking? */
    if (r == NULL)
    {
         /* issue insult */
         kerror("GEOMETRY","rmonster"," set_object_bbox: the input object is WHACKED, joser!");
    }
    VCOPY(&(r->bmin),bmin);
    VCOPY(&(r->bmax),bmax);

    /* auto-recompute center point?? */
    return(TRUE);
}

int
identity_current_object_matrix()
{
    rmonsterobj *tr;

    tr = get_current_object();
    identity_4x4(&(tr->xfrm));

    /* also, set translate vect to 0. */
    {
        int i;
	for (i=0;i<3;i++)
	    tr->translate_vect.v[i] = 0.;
    }
    return(TRUE);
}

