 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kslicer
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkslicer
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include "geometry/geometry.h"

static int _interpolated_slice PROTO((kobject, char *, int *, int, int,
			              float, int, kobject));

static int _integer_slice      PROTO((kobject, char *, int *, int, int,
			              float, int, kobject));

typedef int (*kslice_func) PROTO((kobject, char *, int *, int, int, float,
				  int, kobject));

kslice_func _kslicer_function[2] = { _interpolated_slice, _integer_slice };

#define _kslice_swap(a,b) { int tmp; tmp = a; a = b; b = tmp; }

static char * _direction PROTO((int dir));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkslicer - extract an (N-1)D slice from N-D data
* 
*       Purpose: This routine will generate an (N-1)D data set from an
*                N-D data set.  This is done by selecting a plane to
*                slice along and a slice number into that plane.
*                Planes are specified as being orthogonal to either
*                the width, height, depth, time, or element
*                dimenstions of the input data object.  The slice
*                plane can either be specified as a specific plane
*                number, or a percentage depth into the data along the
*                specified slice plane.
*
*                The user also has the option of reorienting the slice
*                to be aligned with the width-height plane.  This will
*                make it suitable for direct viewing with any of the
*                image viewing programs.  If the original data set
*                contains no location data, then explicit corner
*                markers may be added to position the data in space.
*
*         Input: src_obj - input object to be sliced
*
*		 plane - which slice plane to cut along?
*
*		 slice_flag   - TRUE if integer slice should be done
*		 slice        - integer slice number
*
*		 islice_flag  - TRUE if interpolated slice should be done
*		 islice       - interpolated slice number
*
* 		 pslice_flag  - TRUE if percentage slice should be done
*                pslice       - percentage slice depth
*
*		 reorient     - TRUE if largest face on slice
*				should be made to face the image plane
*
*		 add_locs     - TRUE if explicit location data should
*			 	be created for the final output
*
*        Output: dst_obj      - the output object to output slice to
*
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Steven Kubica
*          Date: Dec 13, 1994
*      Verified: 
*  Side Effects: 
* Modifications: *
****************************************************************/
/* -library_def */
lkslicer(
   kobject src_obj,
   int	   plane,
   int     force_int_slice,
   int     slice_flag,
   float   slice,
   int     pslice_flag,
   float   pslice,
   int     reorient,
   int     add_locs,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
   int val_avail;
   int msk_avail; 
   int loc_avail;
   int tim_avail;
   int map_avail;
   int *size = NULL;
   float slice_number = (float) slice;
   int   slice_function;

   /* identify which segments are present in the input. */
   val_avail = kpds_query_value(src_obj);
   msk_avail = kpds_query_mask(src_obj);
   loc_avail = kpds_query_location(src_obj);
   tim_avail = kpds_query_time(src_obj);
   map_avail = kpds_query_map(src_obj);


   /* create, and copy what I can here ---  the datatype in particular */
   if (val_avail)
   {
      kpds_create_value(dst_obj);
      kpds_copy_attribute(src_obj, dst_obj, KPDS_VALUE_DATA_TYPE);
   }

   if (msk_avail)
   {
      kpds_create_mask(dst_obj);
      kpds_copy_attribute(src_obj, dst_obj, KPDS_MASK_DATA_TYPE);
   }

   if (loc_avail)
   {
      kpds_copy_attribute(src_obj, dst_obj, KPDS_LOCATION_GRID);
      kpds_create_location(dst_obj);
      kpds_copy_attribute(src_obj, dst_obj, KPDS_LOCATION_DATA_TYPE);
   }

   if (tim_avail)
   {
      kpds_create_time(dst_obj);
      kpds_copy_attribute(src_obj, dst_obj, KPDS_TIME_DATA_TYPE);
   }

   /* slice the value and mask segments if they are present */
   if (val_avail)
   {
      kdms_get_attribute(src_obj, KDMS_SEGMENT_VALUE, KDMS_SIZE, &size);

      if (size == NULL)
      {
         kerror(NULL,"kslicer",
                     "Unable to retrieve the size of the value segment");
         
         return(FALSE);
      }

      /* if this is a percentage slice, then figure out the slice number */
      if (pslice_flag)
         slice_number = pslice * size[plane];

      /* if we want to force an integer slice, then round here */
      if (force_int_slice)
         slice_number = (float) ktrunc(slice_number + 0.5);

      /*  call either the integer or interpolated slicer */
      slice_function = (slice_number == (float) ktrunc(slice_number));

      /* slice the value */
      if (!_kslicer_function[slice_function](src_obj, KDMS_SEGMENT_VALUE, size,
					     5, plane, slice_number, reorient,
					     dst_obj))
      {
         kerror(NULL,"kslicer",
                     "Unable to slice the value segment in the %s direction "
		     "at slice plane %g", _direction(plane), slice_number);
         return(FALSE);
      }

      /* perform the mask slice, if it is present */
      if (msk_avail)
         if (!_kslicer_function[slice_function](src_obj, KDMS_SEGMENT_MASK, 
						size, 5, plane, slice_number,
					        reorient, dst_obj))
         {
            kerror(NULL,"kslicer",
                       "Unable to slice the mask segment in the %s direction "
	   	       "at slice plane %g", _direction(plane), slice_number);

            return(FALSE);
         }
   }
   else
   {
      return(FALSE);
   } 

   /* 
    *    We are assuming at this point that we've had value data, and
    *    the slice number is considered valid -- we will proceed under
    *    the assumption that the polymorphic model is being followed
    *    (i.e. all sizes and index orders are shared).
    *
    *    The assumption about not having value data is a limitation for
    *    now -- it should be fixed.
    */

   /* slice location here */
   if (loc_avail)
   {
      /* if the slice plane is along width, height, or depth, then slice */
      if (plane == 0 || plane == 1 || plane == 2)
      {
	 int grid;

	 /* -- get the location grid type -- */
	 kpds_get_attribute(src_obj, KPDS_LOCATION_GRID, &grid);

	 if (grid == KUNIFORM)
	 {
	    double beg_w, beg_h, beg_d, beg_t, beg_e;
	    double end_w, end_h, end_d, end_t, end_e;
	    int    wsize, hsize, dsize;
	    
	    if (!kpds_get_attribute(src_obj, KPDS_LOCATION_BEGIN, 
				    &beg_w, &beg_h, &beg_d, &beg_t, &beg_e) ||
		!kpds_get_attribute(src_obj, KPDS_LOCATION_END, 
				    &end_w, &end_h, &end_d, &end_t, &end_e) ||
		!kpds_get_attribute(src_obj, KPDS_LOCATION_SIZE, 
				    &wsize, &hsize, &dsize, NULL))
	    {
	       kerror(NULL,"kslicer",
		      "Unable to retrieve uniform coordinates.");
	       return(FALSE);
	    }

	    if (plane == 0)
	    {
	       beg_w = end_w = slice_number * ((end_w - beg_w + 1)/wsize);
	       wsize = 1;
	    }
	    
	    if (plane == 1)
	    {
	       beg_h = end_h = slice_number * ((end_h - beg_h + 1)/hsize);
	       hsize = 1;
	    }
	    
	    if (plane == 2)
	    {
	       beg_d = end_d = slice_number * ((end_d - beg_d + 1)/dsize);
	       dsize = 1;
	    }
	    
	    kpds_set_attribute(dst_obj, KPDS_LOCATION_BEGIN, 
			       beg_w, beg_h, beg_d, beg_t, beg_e);
	    kpds_set_attribute(dst_obj, KPDS_LOCATION_END,
			       end_w, end_h, end_d, end_t, end_e);
	    kpds_set_attribute(dst_obj, KPDS_LOCATION_SIZE,
			       wsize, hsize, dsize, -1);
	 }
	 else if (grid == KRECTILINEAR)
	 {
	    kerror(NULL,"kslicer",
		   "Unable to slice the rectilinear locaton -- yet.");
	    return(FALSE);
	 }
	 else if (grid == KCURVILINEAR)
	 {
	    int *rsize;
	    int  lsize[5];
	    
	    /* 	
	     *  important to remember that the dimension is *not* shared
	     *  with the size of the value segment -- we will make our
	     *  own location size array here.
	     */

	    kdms_get_attribute(src_obj, KDMS_SEGMENT_LOCATION, 
			       KDMS_SIZE, &rsize);

	    /* size should mostly match the value size, except for dimension */
	    lsize[0] = size[0];
	    lsize[1] = size[1];
	    lsize[2] = size[2];
	    lsize[3] = rsize[3];
	    
	    if (!_kslicer_function[slice_function](src_obj, 
						   KDMS_SEGMENT_LOCATION,
						   lsize, 4, plane, 
						   slice_number, reorient, 
						   dst_obj))
	    {
	       kerror(NULL,"kslicer",
		      "Unable to slice the locaton segment in the %s "
		      "direction at slice plane %g", _direction(plane), 
		      slice_number);
	       return(FALSE);
	    }
	 }
      }
      else /* otherwise all the location data applies to the slice */
      {
	 /* BUG : the copied locations will not line up with the value data
	    if the data is being reoriented to face the image plane... */
	 kpds_copy_location_data(src_obj, dst_obj, TRUE);
      } 
   }
         
   /* slice time here */
   if (tim_avail)
   {
      /* if the slice plane is along time, then slice */
      if (plane == 3)
      {
         /* since time is 1D, we have to have a separate copy of its size */
         int time_size[1];
         
         time_size[0] = size[3];

         if (!_kslicer_function[slice_function](src_obj, KDMS_SEGMENT_TIME,
					        time_size, 1, 0, slice_number,
				                FALSE, dst_obj))
         {
            kerror(NULL,"kslicer",
                     "Unable to slice the time segment "
	             "at slice plane %g", slice_number);
            return(FALSE);
         }
      }
      else /* otherwise all the time data applies to the slice */
      {
         kpds_copy_time_data(src_obj, dst_obj, TRUE);
      }
      
   }

   /* slice the map here */
   if (map_avail)
   {
      /* -- for now, just copy entire map -- */
      kpds_copy_map_data(src_obj, dst_obj, TRUE); 

      /* -- preserve the colorspace and has alpha attributes -- */
      if (!kcolor_copy_attributes(src_obj, dst_obj,
				  KCOLOR_COLORSPACE, 
				  KCOLOR_HAS_ALPHA, NULL))
      {
	 kerror(NULL,"kslicer",
		"Unable to copy colorspace or has_alpha attributes.");
	 kexit(KEXIT_FAILURE);
      }
   }

   /* -- add explicit location data if requested -- */
   if (add_locs)
   {
#if 0
      /* -- only add if we don't already have any location -- */
      if (!kpds_query_location(dst_obj))
      {
	 kpds_set_attribute(dst_obj, KPDS_LOCATION_GRID, KUNIFORM);
	 kpds_create_location(dst_obj);

	 /* -- set the location begin and end points -- */
	 kpds_set_attribute(dst_obj, KPDS_LOCATION_BEGIN, 
	 kpds_set_attribute(dst_obj, KPDS_LOCATION_END, 
      }
#endif
   }
   
   
   return(TRUE);
}

static char *
_direction(int dir)
{
   switch (dir)
   {
      case 0: return("width");
      case 1: return("height");
      case 2: return("depth");
      case 3: return("time");
      case 4: return("elements");
   }
   return(" ");
}

/*-----------------------------------------------------------
|
|  Routine Name: _integer_slice
|
|       Purpose: 
|
|         Input:
|
|        Output: none 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: May 21, 1994 17:06
| Modifications:
|
------------------------------------------------------------*/
static int
_integer_slice(
   kobject  src_obj,
   char    *segment,
   int     *size,
   int      dimension,
   int      plane,
   float    slice_number,
   int      reorient,
   kobject  dst_obj)
{
   kaddr data = NULL;
   int src_beg[5] = {0, 0, 0, 0, 0};
   int src_end[5] = {0, 0, 0, 0, 0};
   int dst_beg[5] = {0, 0, 0, 0, 0};
   int dst_end[5] = {0, 0, 0, 0, 0};
   int dst_siz[5] = {0, 0, 0, 0, 0};
   int islice = (int) slice_number;
   int i;

   /* force slice to be within range */
   if (islice < 1)
      islice = 1;
   if (islice > size[plane])
      islice = plane;

   /* put this thing back into 'C' indexing space */
   islice -= 1;

   /* the end point will be roughly the whole data set ... */
   for (i = 0; i < dimension; i++)
   {
      src_end[i] = size[i] - 1;
      dst_end[i] = size[i] - 1;
      dst_siz[i] = size[i];	 /* make a copy of the size while we're here */
   }

   /* ... not including the dimension we are slicing on */
   src_beg[plane] = islice;
   src_end[plane] = islice;

   /* get the data */
   data = kdms_get_data(src_obj, segment, src_beg, src_end, NULL);

   /* destination slice should be at the origin */
   dst_beg[plane] = 0;
   dst_end[plane] = 0;
   dst_siz[plane] = 1;  /* it's thickness is the thickness of one slice */

   if (reorient)
   {
      /* slicing along width or height -- 
	    swap it for the next direction with size greater than 1 */
      if (plane == 0 || plane == 1)
      { 
         int swap_plane = plane;

         swap_plane = (dst_siz[2] != 1) ? 2 : 
		      (dst_siz[3] != 1) ? 3 : dimension - 1; 

         _kslice_swap(dst_beg[plane], dst_beg[swap_plane]);
         _kslice_swap(dst_end[plane], dst_end[swap_plane]);
         _kslice_swap(dst_siz[plane], dst_siz[swap_plane]);

	 /* -- preserve index order -- */
	 if (plane == 0 && swap_plane > 1)
	 {
	    _kslice_swap(dst_beg[0], dst_beg[1])
	    _kslice_swap(dst_end[0], dst_end[1])
	    _kslice_swap(dst_siz[0], dst_siz[1])
	 }
      }
   }

   /* set the size on the output */
   kdms_set_attribute(dst_obj, segment, KDMS_SIZE, dst_siz);

   /* put the data */   
   return(kdms_put_data(dst_obj, segment, dst_beg, dst_end, data));
}

/*-----------------------------------------------------------
|
|  Routine Name: _interpolated_slice
|
|       Purpose: 
|
|         Input:
|
|        Output: none 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: May 21, 1994 17:06
| Modifications:
|
------------------------------------------------------------*/
static int
_interpolated_slice(
   kobject  src_obj,
   char    *segment,
   int     *size,
   int      dimension,
   int      plane,
   float    slice_number,
 int      reorient,
   kobject  dst_obj)
{
   
   return(FALSE);
}
/* -library_code_end */
