 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for gcuberille
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lgcuberille
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
typedef struct vertex3d
{
    float x,y,z;
} vertex;

typedef struct surface_colors_rgba
{
    float r,g,b,a;
} surface_colors_rgba;

typedef struct slice
{
    int usize,vsize;
    float **xpts,**ypts,**zpts;
    float **data;
    surface_colors_rgba **colors;
} slice;

static int khoros_init_input PROTO((kobject,int *,int *,int *));
static int khoros_init_output PROTO((kobject, char *));
static void malloc_slice PROTO((slice *,int,int));
static void free_slice PROTO((slice *));
static void load_slice PROTO((slice *, int,int,int,kobject,vis_cmap *));
static void slices_to_cuberille PROTO((slice *,slice *, float *));
static void accrue_bbox PROTO((vertex *,vertex *,vertex *,int));

static vertex *work_verts;
static surface_colors_rgba *work_colors;
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lgcuberille - Volume Visualization using Geometric Icons
* 
*       Purpose: This should be a complete description that anyone
*                could understand;  it should have acceptable grammar
*                and correct spelling.
*
*         Input: kobject inobj - contains volume data
*                kobject cmapobj - contains visualization colormap
*
*         Parms: float *shrinkage - ratio of cube size to grid block size
*                char *name - name to use for geometry object
*
*        Output: kobject outobj - contains output geometry data
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Wes Bethel
*          Date: Apr 13, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lgcuberille(kobject inobj,
	       kobject cmapobj,
	       kobject outobj,
	       float *shrinkage,
	       char *name)
/* -library_def_end */

/* -library_code */
{
    int u,v,w;
    int i,j,k;
    int status;
    slice *slice_this,*slice_next,*temp_slice;
    vis_cmap vmap;
    int num_cubes_per_slice,num_triangles_per_cube,num_verts_per_slice;
    int num_buffers_put=1;
    vertex bmin,bmax;
    
    /* do validity checks on input objects */
    status = khoros_init_input(inobj,&u,&v,&w);
    if (status != CHILL)
	return(WHACKED);

    status = lkobj_to_vismap(cmapobj,&vmap);
    if (status != CHILL)
    {
	kerror("GEOMETRY","lcuberille","Bad vis colormap data.  Aborting.");
	return(WHACKED);
    }
    
    /* configure output object */
    status = khoros_init_output(outobj,name);

    slice_this = (slice *)kmalloc(sizeof(slice));
    slice_next = (slice *)kmalloc(sizeof(slice));
    malloc_slice(slice_next,u,v);
    malloc_slice(slice_this,u,v);
    
    /* malloc work areas for vertices, colors */

    num_cubes_per_slice = (v-1)*(u-1);
    num_triangles_per_cube = 20; /* t-strip representation */
    num_verts_per_slice = num_cubes_per_slice * num_triangles_per_cube;
    
    work_verts = (vertex *)kmalloc(sizeof(vertex)*num_verts_per_slice);
    work_colors = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*num_verts_per_slice);
    
    load_slice(slice_this,0,u,v,inobj,&vmap);

    bmin.x = bmin.y = bmin.z = KGEOM_BIGNUM;
    bmax.x = bmax.y = bmax.z = KGEOM_SMALLNUM;

    /* loop over slices of input object */
    for (k=1;k<w;k++)
    {
#ifdef DEBUG
	fprintf(stderr," processing slice %d.\n",k);
#endif
	load_slice(slice_next,k,u,v,inobj,&vmap);

	slices_to_cuberille(slice_this,slice_next,shrinkage);
	
	kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,
			    num_buffers_put++);

	kgeom_set_attribute(outobj,KGEOM_TRIANGLES_CONNECTED,KGEOM_NUMBER_VERTICES,num_verts_per_slice);
	kgeom_put_data(outobj,KGEOM_TRIANGLES_CONNECTED,work_verts,
		       work_colors,NULL,NULL);

	accrue_bbox(&bmin,&bmax,work_verts,num_verts_per_slice);
	
	temp_slice = slice_this;
	slice_this = slice_next;
	slice_next = temp_slice;
    }
    
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_BOUNDING_BOX,&(bmin.x),&(bmax.x));
    bmin.x = bmin.x + (bmax.x - bmin.x) * 0.5;
    bmin.y = bmin.y + (bmax.y - bmin.y) * 0.5;
    bmin.z = bmin.z + (bmax.z - bmin.z) * 0.5;
    
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_CENTER,&(bmin.x));
    
    return(CHILL);
}

static void
accrue_bbox(vertex *min,
	    vertex *max,
	    vertex *vlist,
	    int n)
{
    /* code to compute the bounding box */
    register float *rf,test;
    register int i;
    
    /* X minimum */
    rf = &(vlist[0].x);
    test = min->x;
    for (i=0;i<n;i++,rf+=3)
    {
	if (*rf < test)
	    test = *rf;
    }
    min->x = test;

    /* X maximum */
    test = max->x;
    rf = &(vlist[0].x);
	
    for (i=0;i<n;i++,rf+=3)
    {
	if (*rf > test)
	    test = *rf;
    }
    max->x = test;

    /* Y minimum */
    rf = &(vlist[0].y);
    test = min->y;
    for (i=0;i<n;i++,rf+=3)
    {
	if (*rf < test)
	    test = *rf;
    }
    min->y = test;

    /* Y maximum */
    test = max->y;
    rf = &(vlist[0].y);
	
    for (i=0;i<n;i++,rf+=3)
    {
	if (*rf > test)
	    test = *rf;
    }
    max->y = test;
	
    /* Z minimum */
    rf = &(vlist[0].z);
    test = min->z;
    for (i=0;i<n;i++,rf+=3)
    {
	if (*rf < test)
	    test = *rf;
    }
    min->z = test;

    /* Z maximum */
    test = max->z;
    rf = &(vlist[0].z);
	
    for (i=0;i<n;i++,rf+=3)
    {
	if (*rf > test)
	    test = *rf;
    }
    max->z = test;
}

static int
khoros_init_input(kobject in,
		  int *u,
		  int *v,
		  int *w)
{
    int val_avail,dw,dh,dd,dt,de;
    int loc_avail;
    
    /* get dimensions of value data */
    val_avail = kpds_query_value(in);
    if (val_avail != FALSE)
    {
	kpds_get_attribute(in,KPDS_VALUE_SIZE,&dw,&dh,&dd,&dt,&de);

	/* do sanity check on size, type of data */

	if (de != 1)
	{
	    kerror("GEOMETRY","lcuberille","Cuberille can process only SCALAR (elements = 1) data.");
	    return(WHACKED);
	}
	if (dt != 1)
        {
	    kerror("GEOMETRY","lcuberille","Cuberille can process only T==1 data.");
	    return(WHACKED);
	}
	if ((dw < 2) || (dh < 2) || (dd < 2))
	{
	    kerror("GEOMETRY","lcuberille","Cuberille requires that width, height and depth ALL be greater than 1.");
	    return(WHACKED);
	}
	
	*u = dw;
	*v = dh;
	*w = dd;

	/* insist on "float" presentation of data */
	kpds_set_attribute(in,KPDS_VALUE_DATA_TYPE,KFLOAT);
    }
    else
    {
       kerror("GEOMETRY","lcuberille","gcuberille requires that the input contain value data.");
       return(WHACKED);
    }

    loc_avail = kpds_query_location(in);

    /* -- if we have value but no location, create location -- */
    if (val_avail && !loc_avail)  
    {
       /* -- create uniform location -> this can be accessed as
          -- curvilienar location later in this routine 
          -- */
       kpds_set_attribute(in, KPDS_LOCATION_GRID, KUNIFORM);
       kpds_create_location(in);
       
       /* -- create uniform location from (0,0,0)<->(dw,dh,dd) -- */
       kpds_set_attributes(in, 
                           KPDS_LOCATION_BEGIN, 0., 0., 0.,
                           KPDS_LOCATION_END, (double)dw, 
                           (double)dh, (double)dd, NULL);
    }

    /* -- assert that we want floating point location (x,y,z) data -- */
    kpds_set_attributes(in, 
                        KPDS_LOCATION_DATA_TYPE, KFLOAT,
                        KPDS_LOCATION_SIZE, -1, -1, -1, 3, NULL);
    return(CHILL);
}

static int
khoros_init_output(kobject kobj, char *name)
{
    /* the object has been created already.  here we set the
       attributes of interest. */

    kgeom_create_primitive_list(kobj);
    lset_common_geom_attributes(kobj);
    kgeom_set_attribute(kobj, KGEOM_OBJECT, KGEOM_NAME, name);
    kgeom_set_attribute(kobj,KGEOM_OBJECT,KGEOM_HAS_ALPHA,TRUE);
    
    return(CHILL);
}

static void
malloc_slice(slicep,u,v)
slice *slicep;
int u,v;
{
    int i,t;
    vertex *v2;
    float *d2;
    unsigned char *m2;
    float *fx,*fy,*fz;
    surface_colors_rgba *c2;

    slicep->usize = u;
    slicep->vsize = v;

    fx = (float *)kmalloc(sizeof(float)*u*v);
    slicep->xpts = (float **)kmalloc(sizeof(float *)*v);

    fy = (float *)kmalloc(sizeof(float)*u*v);
    slicep->ypts = (float **)kmalloc(sizeof(float *)*v);

    fz = (float *)kmalloc(sizeof(float)*u*v);
    slicep->zpts = (float **)kmalloc(sizeof(float *)*v);

    d2 = (float *)kmalloc(sizeof(float)*u*v);
    slicep->data = (float **)kmalloc(sizeof(float *)*v);

    c2 = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*u*v);
    slicep->colors = (surface_colors_rgba **)kmalloc(sizeof(surface_colors_rgba *)*v);

    for (i=0,t=0;i<v;i++,t+=u)
    {
	slicep->xpts[i] = fx + t;
	slicep->ypts[i] = fy + t;
	slicep->zpts[i] = fz + t;
	slicep->data[i] = d2 + t;
	slicep->colors[i] = c2 + t;
    }
}

static void
free_slice(slicep)
slice *slicep;
{
    int i;
    
    kfree(slicep->xpts[0]);
    kfree(slicep->ypts[0]);
    kfree(slicep->zpts[0]);
    kfree(slicep->data[0]);

    kfree(slicep->xpts);
    kfree(slicep->ypts);
    kfree(slicep->zpts);
    kfree(slicep->data);
    kfree(slicep->colors);
}

static void
load_slice(slicep,w,u,v,input,vmap)
slice *slicep;
int w,u,v;
kobject input;
vis_cmap *vmap;
{
    int i,j,index;
    float *data_ptr;
    surface_colors_rgba *color_ptr;

    /* grab a plane of data */
    kpds_set_attribute(input,KPDS_VALUE_POSITION,0,0,w,0,0);
    kpds_get_data(input,KPDS_VALUE_PLANE,(float *)&(slicep->data[0][0]));

    /* grab a plane of coordinates */
    kpds_set_attribute(input,KPDS_LOCATION_POSITION,0,0,w,0);
    kpds_get_data(input,KPDS_LOCATION_PLANE,(float *)&(slicep->xpts[0][0]));
    
    kpds_set_attribute(input,KPDS_LOCATION_POSITION,0,0,w,1);
    kpds_get_data(input,KPDS_LOCATION_PLANE,(float *)&(slicep->ypts[0][0]));

    kpds_set_attribute(input,KPDS_LOCATION_POSITION,0,0,w,2);
    kpds_get_data(input,KPDS_LOCATION_PLANE,(float *)&(slicep->zpts[0][0]));

    /* colorize each slice of data */
    
    for (j=0;j<v;j++)
    {
	data_ptr = &(slicep->data[j][0]);
	color_ptr = &(slicep->colors[j][0]);
	
	for (i=0;i<u;i++,data_ptr++,color_ptr++)
	{
	    index = lget_map_index(vmap,data_ptr);
	    lget_map_rgba(vmap,index,&(color_ptr->r),&(color_ptr->g),
			  &(color_ptr->b),&(color_ptr->a));
	}
    }
}

static void
slices_to_cuberille(slice *slice_this,
		    slice *slice_next,
		    float *shrink)
{
    /* in this routine, we take something which is basically a hexahedra
     and spit out geometry */

    vertex verts[20];
    vertex base[8];
    vertex avg;
    surface_colors_rgba base_colors[8];
    surface_colors_rgba colors[20],cell_color;
    int i,index;
    float *xp,*yp,*zp,*cp;
    int vertex_offset=0;

    int u,v;
    
#if 0
    cp = inf->data + cell_index * inf->veclen;
    cell_color.r = *cp++;
    cell_color.g = *cp++;
    cell_color.b = *cp;
#endif
    
    for (v=0;v<slice_this->vsize - 1;v++)
    {
	for (u=0;u<slice_this->usize - 1;u++)
	{

	    /* load base vertices */

	    base[0].x = slice_this->xpts[v][u];
	    base[1].x = slice_this->xpts[v][u+1];
	    base[2].x = slice_this->xpts[v+1][u+1];
	    base[3].x = slice_this->xpts[v+1][u];
	    
	    base[4].x = slice_next->xpts[v][u];
	    base[5].x = slice_next->xpts[v][u+1];
	    base[6].x = slice_next->xpts[v+1][u+1];
	    base[7].x = slice_next->xpts[v+1][u];
	    
	    base[0].y = slice_this->ypts[v][u];
	    base[1].y = slice_this->ypts[v][u+1];
	    base[2].y = slice_this->ypts[v+1][u+1];
	    base[3].y = slice_this->ypts[v+1][u];
	    
	    base[4].y = slice_next->ypts[v][u];
	    base[5].y = slice_next->ypts[v][u+1];
	    base[6].y = slice_next->ypts[v+1][u+1];
	    base[7].y = slice_next->ypts[v+1][u];
	    
	    base[0].z = slice_this->zpts[v][u];
	    base[1].z = slice_this->zpts[v][u+1];
	    base[2].z = slice_this->zpts[v+1][u+1];
	    base[3].z = slice_this->zpts[v+1][u];
	    
	    base[4].z = slice_next->zpts[v][u];
	    base[5].z = slice_next->zpts[v][u+1];
	    base[6].z = slice_next->zpts[v+1][u+1];
	    base[7].z = slice_next->zpts[v+1][u];

	    avg.x = avg.y = avg.z = 0.;
	    for (i=0;i<8;i++)
	    {
		avg.x += base[i].x;
		avg.y += base[i].y;
		avg.z += base[i].z;
	    }

	    if (*shrink != 1.)		/* do shrinkage if requested */
	    {
		register float scale;
	
		scale = *shrink;
		avg.x *= 0.125;
		avg.y *= 0.125;
		avg.z *= 0.125;
	
		for (i=0;i<8;i++)
		{
		    base[i].x = avg.x + (base[i].x - avg.x)* scale;
		    base[i].y = avg.y + (base[i].y - avg.y)* scale;
		    base[i].z = avg.z + (base[i].z - avg.z)* scale;
		}
	    }

	    base_colors[0] = slice_this->colors[v][u];
	    base_colors[1] = slice_this->colors[v][u+1];
	    base_colors[2] = slice_this->colors[v+1][u+1];
	    base_colors[3] = slice_this->colors[v+1][u];
	    
	    base_colors[4] = slice_next->colors[v][u];
	    base_colors[5] = slice_next->colors[v][u+1];
	    base_colors[6] = slice_next->colors[v+1][u+1];
	    base_colors[7] = slice_next->colors[v+1][u];

	    index = 0;
	    
	    colors[index] = base_colors[3];
	    verts[index++] = base[3];
	    
	    colors[index] = base_colors[3];
	    verts[index++] = base[3];
	    
	    colors[index] = base_colors[7];
	    verts[index++] = base[7];
	    
	    colors[index] = base_colors[2];
	    verts[index++] = base[2];
	    
	    colors[index] = base_colors[6];
	    verts[index++] = base[6];
	    
	    colors[index] = base_colors[1];
	    verts[index++] = base[1];
	    
	    colors[index] = base_colors[5];
	    verts[index++] = base[5];
	    
	    colors[index] = base_colors[0];
	    verts[index++] = base[0];
	    
	    colors[index] = base_colors[4];
	    verts[index++] = base[4];
	    
	    colors[index] = base_colors[4];
	    verts[index++] = base[4];
	    
	    colors[index] = base_colors[5];
	    verts[index++] = base[5];
	    
	    colors[index] = base_colors[5];
	    verts[index++] = base[5];
	    
	    colors[index] = base_colors[6];
	    verts[index++] = base[6];
	    
	    colors[index] = base_colors[4];
	    verts[index++] = base[4];
	    
	    colors[index] = base_colors[7];
	    verts[index++] = base[7];
	    
	    colors[index] = base_colors[0];
	    verts[index++] = base[0];
	    
	    colors[index] = base_colors[3];
	    verts[index++] = base[3];
	    
	    colors[index] = base_colors[1];
	    verts[index++] = base[1];
	    
	    colors[index] = base_colors[2];
	    verts[index++] = base[2];
	    
	    colors[index] = base_colors[2];
	    verts[index++] = base[2];
	    /* now, copy over to holding area */
	    kmemcpy(&(work_verts[vertex_offset].x),
		    &(verts[0].x),sizeof(vertex)*index);
    
	    kmemcpy(&(work_colors[vertex_offset].r),
		    &(colors[0].r),sizeof(surface_colors_rgba)*index);
    
#if 0
	    kmemcpy((char *)&(color_buffer[vertex_offset].r),
		    (char *)&(colors[0].r),sizeof(color_vertex)*index);
#endif    
	    vertex_offset += index;

	}
    }
    
}
/* -library_code_end */
