

/*********************************************************************

Copyright (C) 1993, 1994 Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

*********************************************************************/

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


#include <geometry.h>
#include "gextents.h"

/**
  *
  * the vertices of the box are in the following order, where index
  * 0 (zero) corresponds to the "min" location, and index 7 corresponds
  * to the "max" location.
  *
  * the extents box is dispatched off to geom services as a series of
  * disjoint line segments, a total of 12 in all.  the x-lines are
  * first, followed by the y-lines, followed by the z-lines.
  *
  * if the user chooses the have a "colored" box, the x-lines will be
  * red, the y-lines will be green, and the z-lines will be cyan (blue
  * is too hard to see).
  *
  *                   6                 7
  *                   -----------------
  *                  /|              /|
  *              2  /---------------/3|
  *                 | |             | |
  *                 | |4            | |
  *                 | +-------------|-+ 5
  *                 |/              |/
  *               0 +---------------+ 1
  *
  * the box type flag will affect the way in which the box is computed.
  * a type of "min/max" will always produce a rectangular box, the corners
  * of which represent the absolute min/max of the location data.
  *
  * a box type of "bounding hull" will cause a different type of box to
  * be produced.  this box more accurately reflects that shape of the grid
  * defined by the location data, and is especially useful for curvilinear
  * grids.  the hull produced by this box type will cause a number of
  * polyline segments to be generated, one for each of the 12 edges of
  * logical grid.  the number of segments is a function of the amount of
  * data - this box takes up much more space than the min/max box.
  *
  * for nonsensical situations, such as trying to make a hull from
  * scatter (1-d, 3 space) data, the default recovery procedure will be
  * to create a min/max box without issuing an error message.
  *
  * NOTE: if no location data is present, or if the grid is a rectilinear
  * one, the results of the two box types will be identical.
**/
  

static rgb_vertex box_colors[12]=
{
    {1.,0.,0.},
    {1.,0.,0.},
    {1.,0.,0.},
    {1.,0.,0.},
    {0.,1.,0.},
    {0.,1.,0.},
    {0.,1.,0.},
    {0.,1.,0.},
    {0.,1.,1.},
    {0.,1.,1.},
    {0.,1.,1.},
    {0.,1.,1.}
};

static void make_hull PROTO((kobject,kobject,int,int,int,int));
static void make_minmax_box PROTO((kobject,kobject,int,int,int,int));

/*-----------------------------------------------------------
| 
|  Routine Name: make_extents_box
| 
|       Purpose: compute the extents, create the geometry
| 
|         Input: kobject - containing the data from which to make the box.
| 
|        Output: kobject - containing the geometry representing the box.
|    Written By: wes
|          Date: May 19, 1993
| Modifications: 
|                *** 14 June 1993, wes
|                make use of lget_coords() library call, remove
|                _all_ data services references to location data.
|
|                *** 6 July 1993 wes
|                add box type parameter and supporting code.
| 
------------------------------------------------------------*/
int make_extents_box(kobject input,   /* input data */
		     kobject output,  /* output geometry */
		     int     style,   /* colored vs. white flag */
		     char   *name,    /* name of geom object */
		     int     type)    /* hull vs. min/max flag */
{
    int has_colors;
    int val_avail,loc_avail;
    int dw,dh,dd,dt,de;  /* dimensions of value data. */
    int do_hull;

    /**
      * if style is 1, then the user hasn't specified anything
      * for the style, so they get a "white" box (white by the
      * fact that when no colors are specified, the default
      * object color is white).  if the style is 2 (not equal
      * to 1 at this point in time), then the box
      * gets colors.  the colors are such that x-lines are red,
      * y lines are green and z lines are cyan.
    **/
    
    has_colors = (style == 1) ? 0 : 1;

    /* check the availability of value data */
    val_avail = kpds_query_value(input);
    loc_avail = kpds_query_location(input);

    /* get the dimensions of the value data */
    if (val_avail != FALSE)
    {
	kpds_get_attribute(input,KPDS_VALUE_SIZE,&dw,&dh,&dd,&dt,&de);
#if 0
	fprintf(kstderr," data dims: w,h,d,t,e = %d,%d,%d,%d,%d \n",dw,dh,dd,dt,de);
#endif
	/**
	  * figure out if there's 3D of data, and if the user has asked
	  * for a hull.  if so, set the internal flag so that gets
	  * computed.  otherwise, set the flag so that a min/max box
	  * gets computed instead.  no warning msg gets issues.
	**/
	if ((dw != 1) && (dh != 1) && (dd != 1) && (type==2))
	    do_hull = 1;
	else
	    do_hull = 0;
    }
    else
    {
	/* no value data -- see if there's location data */
	/* loc_avail = kpds_query_location(input);  << do this earlier */
	if (loc_avail != FALSE)
	{
	    kpds_get_attribute(input,KPDS_LOCATION_SIZE,&dw,&dh,&dd,&dt);
#if 0
	    fprintf(kstderr," location dims: w,h,d,t,e = %d,%d,%d,%d,%d \n",dw,dh,dd,dt,de);
#endif
	    /**
	     * figure out if there's 3D of data, and if the user has asked
	     * for a hull.  if so, set the internal flag so that gets
	     * computed.  otherwise, set the flag so that a min/max box
	     * gets computed instead.  no warning msg gets issues.
	     **/
	    if ((dw != 1) && (dh != 1) && (dd != 1) && (type==2))
		do_hull = 1;
	    else
		do_hull = 0;

	}
	else
	{
/* KERROR CAT: "geom_annotate" */
	    kerror("GEOMETRY","gextents","there is no value or location data in the input file.");
	    return(WHACKED);
	}
    }

    /* -- if we have value but no location, create location -- */
    if (val_avail && !loc_avail)  
    {
       /* -- create uniform location -> this can be accessed as
          -- curvilienar location later in this routine 
	  -- */
       kpds_set_attribute(input, KPDS_LOCATION_GRID, KUNIFORM);
       kpds_create_location(input);
       
       /* -- create uniform location from (0,0,0)<->(dw,dh,dd) -- */
       kpds_set_attributes(input, 
			   KPDS_LOCATION_BEGIN, 0., 0., 0.,
			   KPDS_LOCATION_END, (double)dw, 
			   (double)dh, (double)dd, NULL);
    }

    /* -- assert that we want floating point location (x,y,z) data -- */
    kpds_set_attributes(input, 
			KPDS_LOCATION_DATA_TYPE, KFLOAT,
			KPDS_LOCATION_SIZE, -1, -1, -1, 3, NULL);


    /* make assertions to geometry services before stuffing any data */

    kgeom_create_primitive_list(output);

    lset_common_geom_attributes(output);

    /* set the layout of color information to per-segment */
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_LAYOUT,KPER_LINE);

    /* this is erroneous - hulls use multiple polyline primitives,
       while min/max boxes use only a ssingle primitive consisting
       of a bunch of disjoint line segments. */

    /* do the name */
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NAME,name); 

    if (do_hull == 1)
	make_hull(input,output,dw,dh,dd,has_colors);
    else
	make_minmax_box(input,output,dw,dh,dd,has_colors);

    return(CHILL);
}

static void
make_minmax_box(kobject input,
		kobject output,
		int dw,
		int dh,
		int dd,
		int has_colors)
{
    vertex_3d min,max,t,center;
    vertex_3d box[24];    /* holds the vertices for the 12 box segments */
    float *color_info;
    float *plane;
    int i,j,k,l;
    
    for (i=0;i<3;i++)
    {
	min.v[i] = KGEOM_BIGNUM;
	max.v[i] = KGEOM_SMALLNUM;
    }

    /* -- get planes of location -- */
    plane = NULL;
    for (k=0;k<dd;k++)
    {
       for (l=0;l<3;l++)
       {
	  /* -- look for min/max points for plane k -- */
	  kpds_set_attribute(input, KPDS_LOCATION_POSITION, 0, 0, k, l);
	  plane = kpds_get_data(input, KPDS_LOCATION_PLANE, plane);
	  for (j=0;j<dh*dw;j++)
	  {
	     if (plane[j] > max.v[l])
		max.v[l] = plane[j];
	     if (plane[j] < min.v[l])
		min.v[l] = plane[j];
	  }
       }
    }
    kfree(plane);

#if 0
    for (k=0;k<dd;k++)
	for (j=0;j<dh;j++)
	    for (i=0;i<dw;i++)
	    {
		lget_coords(input,i,j,k,&t);

		for (l=0;l<3;l++)
		{
		    if (t.v[l] > max.v[l])
			max.v[l] = t.v[l];
		    if (t.v[l] < min.v[l])
			min.v[l] = t.v[l];
		}
	    }
#endif

    for (i=0;i<3;i++)
	center.v[i] = (max.v[i] + min.v[i]) * 0.5;

#if 0
    fprintf(kstderr," min point is %f,%f,%f\n",min.v[0],min.v[1],min.v[2]);
    fprintf(kstderr," max point is %f,%f,%f\n",max.v[0],max.v[1],max.v[2]);
#endif
    /* do line from u=0,v=0,w=0 to u=1,v=0,w=0 */
    memcpy((char *)(box+0),(char *)&min,sizeof(vertex_3d));
    memcpy((char *)(box+1),(char *)&min,sizeof(vertex_3d));
    box[1].v[0] = max.v[0];
    
    /* do line from u=0,v=1,w=0 to u=1,v=1,w=0 */
    memcpy((char *)(box+2),(char *)&min,sizeof(vertex_3d));
    box[2].v[1] = max.v[1];
    memcpy((char *)(box+3),(char *)&max,sizeof(vertex_3d));
    box[3].v[2] = min.v[2];
    
    /* do line from u=0,v=0,w=1 to u=1,v=0,w=1 */
    memcpy((char *)(box+4),(char *)&min,sizeof(vertex_3d));
    box[4].v[2] = max.v[2];
    memcpy((char *)(box+5),(char *)&max,sizeof(vertex_3d));
    box[5].v[1] = min.v[1];
    
    /* do line from u=0,v=1,w=1 to u=1,v=1,w=1 */
    memcpy((char *)(box+6),(char *)&max,sizeof(vertex_3d));
    box[6].v[0] = min.v[0];
    memcpy((char *)(box+7),(char *)&max,sizeof(vertex_3d));

    

    /* ok, now do the y lines */
    
    /* do line from u=0,v=0,w=0 to u=0,v=1,w=0 */
    memcpy((char *)(box+8),(char *)&min,sizeof(vertex_3d));
    memcpy((char *)(box+9),(char *)&min,sizeof(vertex_3d));
    box[9].v[1] = max.v[1];
    
    /* do line from u=1,v=0,w=0 to u=1,v=1,w=0 */
    memcpy((char *)(box+10),(char *)&min,sizeof(vertex_3d));
    box[10].v[0] = max.v[0];
    memcpy((char *)(box+11),(char *)&max,sizeof(vertex_3d));
    box[11].v[2] = min.v[2];
    
    /* do line from u=0,v=0,w=1 to u=0,v=1,w=1 */
    memcpy((char *)(box+12),(char *)&min,sizeof(vertex_3d));
    box[12].v[2] = max.v[2];
    memcpy((char *)(box+13),(char *)&max,sizeof(vertex_3d));
    box[13].v[0] = min.v[0];
    
    /* do line from u=1,v=0,w=1 to u=1,v=1,w=1 */
    memcpy((char *)(box+14),(char *)&max,sizeof(vertex_3d));
    box[14].v[1] = min.v[1];
    memcpy((char *)(box+15),(char *)&max,sizeof(vertex_3d));

    
    /* now do all the Z lines */
    
    /* do line from u=0,v=0,w=0 to u=0,v=0,w=1 */
    memcpy((char *)(box+16),(char *)&min,sizeof(vertex_3d));
    memcpy((char *)(box+17),(char *)&min,sizeof(vertex_3d));
    box[17].v[2] = max.v[2];
    
    /* do line from u=1,v=0,w=0 to u=1,v=0,w=1 */
    memcpy((char *)(box+18),(char *)&min,sizeof(vertex_3d));
    box[18].v[0] = max.v[0];
    memcpy((char *)(box+19),(char *)&max,sizeof(vertex_3d));
    box[19].v[1] = min.v[1];
    
    /* do line from u=0,v=1,w=0 to u=0,v=1,w=1 */
    memcpy((char *)(box+20),(char *)&min,sizeof(vertex_3d));
    box[20].v[1] = max.v[1];
    memcpy((char *)(box+21),(char *)&max,sizeof(vertex_3d));
    box[21].v[0] = min.v[0];
    
    /* do line from u=1,v=1,w=0 to u=1,v=1,w=1 */
    memcpy((char *)(box+22),(char *)&max,sizeof(vertex_3d));
    box[22].v[2] = min.v[2];
    memcpy((char *)(box+23),(char *)&max,sizeof(vertex_3d));

    /**
      * we need to tell geometry services about this object.
    **/

    /* ok, now put the data into the primitive */

    if (has_colors)
	color_info = (float *)box_colors;
    else
	color_info = NULL;
    
    /* state that we have only one primitive in the object */
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,1);
    
    /* set the position of the primitive that we wish to work with */
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_PRIMITIVE_POSITION,0);


    kgeom_set_attribute(output,KGEOM_POLYLINE_DISJOINT,
			KGEOM_NUMBER_VERTICES,24);

    kgeom_put_data(output,KGEOM_POLYLINE_DISJOINT,box,color_info);

    kgeom_set_attributes(output,KGEOM_OBJECT,
			 KGEOM_BOUNDING_BOX,&(min.v[0]),&(max.v[0]),
			 KGEOM_CENTER,&(center.v[0]),
			 NULL);
}

static void
make_hull(kobject input,
	  kobject output,
	  int dw,
	  int dh,
	  int dd,
	  int has_colors)
{
    int max_segs,i,j;
    vertex_3d *v_work,*c_work;
    vertex_3d min,max,t,center;
    int num_prims = 1;

    c_work=NULL;
    
    max_segs = (dw > dh) ? dw:dh;
    max_segs = (dd > max_segs) ? dd:max_segs;

    v_work = (vertex_3d *)kmalloc(sizeof(vertex_3d)*max_segs);
    if (has_colors)
	/* NOTE: the max_segs*2 business is a temp hack till per-segment
	   color is supported at the storage layer. */
	c_work = (vertex_3d *)kmalloc(sizeof(vertex_3d)*max_segs*2);

    for (i=0;i<3;i++)
    {
	min.v[i] = KGEOM_BIGNUM;
	max.v[i] = KGEOM_SMALLNUM;
    }

    /* do the u-row at v=w=0. */
    for (i=0;i<dw;i++)
    {
	lget_coords(input,i,0,0,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,
			KGEOM_NUMBER_PRIMITIVES,num_prims++);

    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,
			KGEOM_NUMBER_VERTICES,dw);

    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);


    /* do the u-row at v=1,w=0. */
    for (i=0;i<dw;i++)
    {
	lget_coords(input,i,dh-1,0,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,
			KGEOM_NUMBER_PRIMITIVES,num_prims++);

    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,
			KGEOM_NUMBER_VERTICES,dw);

    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);


    /* do the u-row at v=0,w=1. */
    for (i=0;i<dw;i++)
    {
	lget_coords(input,i,0,dd-1,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,
			KGEOM_NUMBER_PRIMITIVES,num_prims++);

    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,
			KGEOM_NUMBER_VERTICES,dw);

    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);

    /* do the u-row at v=1,w=1. */
    for (i=0;i<dw;i++)
    {
	lget_coords(input,i,dh-1,dd-1,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,
			KGEOM_NUMBER_PRIMITIVES,num_prims++);

    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,
			KGEOM_NUMBER_VERTICES,dw);

    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);


    /* do the v-row at u=0,w=0. */
    for (i=0;i<dh;i++)
    {
	lget_coords(input,0,i,0,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors+4),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,
			KGEOM_NUMBER_PRIMITIVES,num_prims++);

    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,
			KGEOM_NUMBER_VERTICES,dh);

    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);

    /* do the v-row at u=1,w=0. */
    for (i=0;i<dh;i++)
    {
	lget_coords(input,dw-1,i,0,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors+4),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,
			KGEOM_NUMBER_PRIMITIVES,num_prims++);

    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,
			KGEOM_NUMBER_VERTICES,dh);

    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);


    /* do the v-row at u=0,w=1. */
    for (i=0;i<dh;i++)
    {
	lget_coords(input,0,i,dd-1,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors+4),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,
			KGEOM_NUMBER_PRIMITIVES,num_prims++);

    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,
			KGEOM_NUMBER_VERTICES,dh);

    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);


    /* do the v-row at u=1,w=1. */
    for (i=0;i<dh;i++)
    {
	lget_coords(input,dw-1,i,dd-1,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors+4),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,num_prims++);
    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,dh);
    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);


    /* do the w-row at u=0,v=0. */
    for (i=0;i<dd;i++)
    {
	lget_coords(input,0,0,i,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors+8),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,num_prims++);
    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,dd);
    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);


    /* do the w-row at u=1,v=0. */
    for (i=0;i<dd;i++)
    {
	lget_coords(input,dw-1,0,i,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors+8),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,num_prims++);
    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,dd);
    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);

    /* do the w-row at u=0,v=1. */
    for (i=0;i<dd;i++)
    {
	lget_coords(input,0,dh-1,i,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors+8),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,num_prims++);
    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,dd);
    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);

    /* do the w-row at u=1,v=1. */
    for (i=0;i<dd;i++)
    {
	lget_coords(input,dw-1,dh-1,i,&t);

	for (j=0;j<3;j++)
	{
	    if (t.v[j] > max.v[j])
		max.v[j] = t.v[j];
	    if (t.v[j] < min.v[j])
		min.v[j] = t.v[j];
	}
	memcpy((char *)(v_work+i),(char *)&t,sizeof(vertex_3d));
	if (has_colors)
	    memcpy((char *)(c_work+i),(char *)(box_colors+8),sizeof(vertex_3d));
    }

    /* set the number of primitives, which one we're working on, and the
       number of vertices. */
    
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,num_prims++);
    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,dd);
    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,v_work,c_work);


    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_BOUNDING_BOX,&(min.v[0]),&(max.v[0]));

    for (i=0;i<3;i++)
	center.v[i] = (min.v[i] + max.v[i]) * 0.5;
    
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_CENTER,&(center.v[0]));
    
    kfree(v_work);
    if (has_colors)
	kfree(c_work);
}
