 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>  	        String Manipulation / Utility Routines
   >>>>
   >>>>  Private:
   >>>>		spc_create_strings_mapcol_list()
   >>>>		spc_create_strings_category_list()
   >>>>		spc_create_strings_cluster_list()
   >>>>		spc_striptext()
   >>>>		spc_read_eqn_string_from_file()
   >>>>		spc_find_legend_from_classname()
   >>>>   Static:
   >>>>		string_member()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

#include "spectrum.h"

static int string_member PROTO((char *, char **, int));

/*-----------------------------------------------------------
|
|  Routine Name: spc_create_strings_mapcol_list
|
|       Purpose: Creates the list of strings naming map columns to be
|                passed to xvu_run_list_wait() by run_files()
|         Input: none
|        Output: none
|       Returns: The list of strings naming map columns
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications: 
|
------------------------------------------------------------*/

char **spc_create_strings_mapcol_list(void)
{
	int  i;
	char **list;
	char temp[KLENGTH];

	list = (char **) kcalloc(1, spc_map_colnum * sizeof(char *));
	for (i = 0; i < spc_map_colnum; i++)
	{
	    ksprintf(temp, "Band %d (%s)", i+1, spc_mapcol_names[i]);
	    list[i] = kstring_copy(temp, NULL);
	}
	return(list);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_create_strings_category_list
|
|       Purpose: Creates  the list of strings naming categories to be
|                passed to xvu_run_list_wait() by other routines
|         Input: prepend     - supply 1st element of list if desired
|                listflag    - indicates type of list; one of:
|                              SPC_HIDDEN    (hidden categories)
|                              SPC_DISPLAYED (displayed categories)
|                              SPC_NOTEMPTY  (categories w/ > 0 clusters)
|                              SPC_ALL       (all categories)
|                excllist    - list of names to exclude (if any)
|                exclsize    - size of excllist array
|        Output: size        - returns size of created list
|       Returns: The list of strings naming categories
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

char **spc_create_strings_category_list(
    char  *prepend,
    int   listflag,
    int   *size,
    char  **excllist,
    int   exclsize)
{
	int         i;
	char        **list;
	LegendEntry *legend_ptr;
	    
	list = (char **) kcalloc(1, sizeof(char *));
	i = 0;
	
	/* 
	 * prepend an item if desired 
         */
	if ((prepend != NULL) && (i == 0))
	{
	    list[i++] = kstring_copy(prepend, NULL);
	    list = (char **) krealloc(list, (i+1)*sizeof(char *));
	}
	
	if ((spc_legend_list != NULL) && (spc_legend_list->next != NULL))
	    list[0] = "ALL Classes";

	/*
	 *  go through list of categorys, picking out the ones we want
	 */
	legend_ptr = spc_legend_list;
	while (legend_ptr != NULL)
	{
	     /* 
	      * create list of hidden class categories 
	      */
	     if ((legend_ptr->hidden) && 
                 (listflag == SPC_HIDDEN) &&
	        (!(string_member(legend_ptr->classname, excllist, exclsize))))
	        list[i++] = kstring_copy(legend_ptr->classname, NULL);
	
	     /* 
	      * create list of displayed class categories 
	      */
	    else if ((!legend_ptr->hidden) && 
                     (listflag == SPC_DISPLAYED) &&
		     (!(string_member(legend_ptr->classname, excllist, exclsize))))
	        list[i++] = kstring_copy(legend_ptr->classname, NULL);
	
	     /* 
	      * create list of non-empty class categories 
              */
	    else if ((legend_ptr->clusternum > 0) && 
		     (listflag == SPC_NOTEMPTY) &&
		     (!(string_member(legend_ptr->classname, excllist, exclsize))))
	        list[i++] = kstring_copy(legend_ptr->classname, NULL);
	
	     /* 
              * create list of all class categories 
              */
	    else if ((listflag == SPC_ALL) && 
		    (!(string_member(legend_ptr->classname, excllist, exclsize))))
	        list[i++] = kstring_copy(legend_ptr->classname, NULL);
	
	    legend_ptr = legend_ptr->next;
	    list = (char **) krealloc(list, (i+1) *sizeof(char *));
	}
	*size = i;
	
	return(list);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_create_strings_cluster_list
|
|       Purpose: Creates the list of strings naming cluster numbers
|                to be re-assigned to another class, from the unique
|                cluster values that appear in the polygon ROI that
|		 was extracted from the image.
|
|         Input: unique_values - integer array of clusters to include in list
|                size          - size of unique_values array
|        Output: none
|       Returns: allocates array of strings naming unique cluster numbers
|    Written By: Danielle Argiro
|          Date: Feb 15, 1994
| Modifications: 
|
------------------------------------------------------------*/

char **spc_create_strings_cluster_list(
    int  *unique_values,
    int  size)
{
	int  i, cluster, offset = 0;
	char temp[KLENGTH];
	char **list;

	if (size == 0)
	    return(NULL);

	if (size > 1) offset = 1;
        list = (char **) kcalloc(1, (size+offset) * sizeof(char *));
	if (size > 1)
	    list[0] = kstrdup("ALL Clusters");

        for (i = offset; i < size+offset; i++)
        {
	    cluster = unique_values[i-offset];
            if (spc_legend_lookup[cluster] != NULL)
	 	ksprintf(temp, "Cluster Number %d (class %s)", cluster,
			 (spc_legend_lookup[cluster])->classname);
	    else ksprintf(temp, "Cluster Number %d (unassigned)", cluster);
            list[i] = kstring_copy(temp, NULL);
        }

        return(list);
}


/*-----------------------------------------------------------
|
|  Routine Name: spc_striptext
|       Purpose: strips a text string of capital letters & whitespace
|         Input: string
|        Output: none
|       Returns: the stripped string
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
---------------------------------------------------------------*/
char *spc_striptext(
    char *string)
{
        char *stripped;
        char temp[KLENGTH];
        int  i, j;

        j = 0;
        for (i = 0; i < kstrlen(string); i++)
        {
            if (string[i] != ' ' && string[i] != '\t' && string[i] != '\n')
                temp[j++] = string[i];
        }
        temp[j] = '\0';
        stripped = kstring_lower(temp, NULL);
        return(stripped);

}

/*-----------------------------------------------------------
|
|  Routine Name: spc_read_eqn_string_from_file
|
|       Purpose: Given a filename, reads an equation out of a file, 
|                and returns the equation.  Used on the Display subform,
|                allows user to type in the name of a file containing
|                the equation for the Red, Green, or Blue column rather 
|                than the equation itself.
| 
|         Input: filename - name of file containing eqn
|        Output: none
|       Returns: Returns equation string on success, NULL on failure.
|    Written By: Danielle Argiro
|          Date: May 17, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
---------------------------------------------------------------*/

char *spc_read_eqn_string_from_file(
    char *filename)
{
	int   i;
	kfile *file;
	char  c, equation[10*KLENGTH];
	
	if (file = kfopen(filename, "r"))
	{
	    i = 0;
	    while (!(kfeof(file)))
	    {
	        c = kfgetc(file);
	        if (c != '\n' && c != '\t')
	            equation[i++] = c;
	    }
	    equation[i] = '\0';
	    kfclose(file);
	    return(kstring_copy(equation, NULL));
	}
	else return(NULL);
}


/*-----------------------------------------------------------
|
|  Routine Name: spc_find_legend_from_classname
|       Purpose: Given the identifying text, returns the legend node
|                it is associated with (or NULL if it is not associated w/ any)
|         Input: text - the text to look for
|        Output: none
|       Returns: The associated legend_ptr on success, NULL on failure
|    Written By: Danielle Argiro
|          Date: June 13, 1993
| Modifications: 
|
---------------------------------------------------------------*/
LegendEntry *spc_find_legend_from_classname(
    char *text)
{
	int         classname_token;
        LegendEntry *legend_ptr;

	classname_token = kstring_to_token(text);
        legend_ptr = spc_legend_list;
        while (legend_ptr != NULL)
        {
           if (legend_ptr->classname_token == classname_token)
                return(legend_ptr);
           legend_ptr = legend_ptr->next;
        }
        return(legend_ptr);
}


/*-----------------------------------------------------------
|
|  Routine Name: string_member
|
|       Purpose: Sees if the string passed in is a member 
|                of the list of strings passed in.
|         Input: string - the string to look for
|                list   - array of strings to look in.
|                size   - size of the array of strings to look in
|        Output: None
|       Returns: TRUE if string is in list, FALSE otherwise.
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
static int string_member(
    char *string,
    char **list,
    int  size)
{
	int i;

	if ((list == NULL) || (size == 0)) return(FALSE);

	for (i = 0; i < size; i++)
	{
	    if ((kstrcmp(string, list[i])) == 0)
		return(TRUE);
	}
	return(FALSE);
}


