 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>  		     	Output Utilities
   >>>>
   >>>>  Private:
   >>>>			spc_write_output_image()
   >>>>			spc_write_output_grouped_image()
   >>>>			spc_set_output_maps()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

#include "spectrum.h"

extern XColor *ColorGetXColors   PROTO((Widget));
static int spc_set_output_maps PROTO((kobject));

/*-----------------------------------------------------------
|
|  Routine Name: spc_write_output_image
|
|       Purpose: Writes out the image & map data
|
|         Input: filename - name of file to write out
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro  & Jeremy Worley
|          Date: 
| Modifications: 
|
------------------------------------------------------------*/

int spc_write_output_image(
    char *filename)
{
	int     i, rw, rh, rd, rt, re, region_num; 
	int     width, height, depth, ttime, elements;
	kaddr   region = NULL;
	kobject out_image;
		      

	/* see if legend file already exists/prompt to over-write */
	if (!(koverwrite(KSTANDARD, filename)))
	    return(FALSE);

	/* we will be writing out an image w/ value data & a map */
	out_image = kpds_open_object(filename, KOBJ_WRITE);
	kpds_create_value(out_image);
	kpds_create_map(out_image);

	/* 
	 * output image will be the same size, type as global image 
         */
	kpds_set_attribute(spc_image, KPDS_VALUE_DATA_TYPE, spc_image_type);
	kpds_set_attribute(out_image, KPDS_VALUE_DATA_TYPE, spc_image_type);
	kpds_get_attribute(spc_image, KPDS_VALUE_SIZE, 
			   &width, &height, &depth, &ttime, &elements);
	kpds_set_attribute(out_image, KPDS_VALUE_SIZE,
			   width, height, depth, ttime, elements);

	/*
	 * want to be efficient wrt writing data; get optimal region
	 * size from classed image, and use it for output image.
 	 */
	kpds_get_attribute(spc_image, KPDS_VALUE_OPTIMAL_REGION_SIZE,
		           &rw, &rh, &rd, &rt, &re, &region_num);
	kpds_set_attribute(out_image, KPDS_VALUE_REGION_SIZE,
                           rw, rh, rd, rt, re);
	kpds_set_attribute(spc_image, KPDS_VALUE_REGION_SIZE,
                           rw, rh, rd, rt, re);
	/* 
	 * write out the value data of the image using optimal region size
	 */
	kpds_set_attribute(spc_image, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0);
	for (i = 0; i < region_num; i++)
	{
	    region = kpds_get_data(spc_image, KPDS_VALUE_REGION, region);
	    kpds_put_data(out_image, KPDS_VALUE_REGION, region);
	}
	kfree(region);

	spc_set_output_maps(out_image);

	/* write out the image */
	(void) kpds_close_object(out_image);

	return(TRUE);

}


/*-----------------------------------------------------------
|
|  Routine Name: spc_write_output_grouped image
|
|       Purpose: Writes out a grouped image
|
|         Input: filename - name of file to write out
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro & Jeremy Worley
|          Date: July 16, 1994
| Modifications:
|
------------------------------------------------------------*/


int spc_write_output_grouped_image(
    char *filename)
{
        int           max_class_id, class_id, count, i, j, cluster; 
	int           width, height, depth, ttime, elements;
	int           rw, rh, rd, rt, re, region_num, region_size;
        kobject       out_image;
	unsigned char triple[3];
	LegendEntry   *legend_ptr;
	int           *region = NULL;

        /* see if image file already exists/prompt to over-write */
        if (!(koverwrite(KSTANDARD, filename)))
            return(FALSE);

        /* we will be writing out an image w/ value data & a map */
        out_image = kpds_open_object(filename, KOBJ_WRITE);
        kpds_create_value(out_image);

	/* output image same size, type as global image */
        kpds_get_attribute(spc_image, KPDS_VALUE_SIZE,
                           &width, &height, &depth, &ttime, &elements);
        kpds_set_attribute(out_image, KPDS_VALUE_SIZE,
                           width, height, depth, ttime, elements);

	/* reading the current image as integers */
	kpds_set_attribute(spc_image, KPDS_VALUE_DATA_TYPE, KINT);

	/* 
	 * i'll be writing out unsigned bytes (executive decision that user will
         * not have more than 256 classes) but, i'll be providing integers
         * (obtained from the current image, set to integers agove) to
         * kpds_put_data() setting KPDS_VALUE_COUPLING to KUNCOUPLED allows
         * me to do this.
         */
	kpds_set_attribute(out_image, KPDS_VALUE_DATA_TYPE, KUBYTE);
        kpds_set_attributes(out_image, 
			    KPDS_COUPLING, KUNCOUPLED, 
			    KPDS_VALUE_DATA_TYPE, KINT,
			    NULL);

        /*
         * want to be efficient wrt writing data; get optimal region
         * size from classed image, and use it for output image.
         */
        kpds_get_attribute(spc_image, KPDS_VALUE_OPTIMAL_REGION_SIZE,
                           &rw, &rh, &rd, &rt, &re, &region_num);
        kpds_set_attribute(out_image, KPDS_VALUE_REGION_SIZE,
                           rw, rh, rd, rt, re);
        kpds_set_attribute(spc_image, KPDS_VALUE_REGION_SIZE,
                           rw, rh, rd, rt, re);

	/*
	 * write out the value data of the image using optimal region size
	 */
	kpds_set_attribute(spc_image, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0);
	for (i = 0; i < region_num; i++)
	{
	    region = (int *) kpds_get_data(spc_image, 
					   KPDS_VALUE_REGION, region);
	    region_size = rw*rh*rd*rt*re;
	    for (j = 0; j < region_size; j++)
	    {
		cluster = region[j];
		if (spc_classes[cluster] == -1)
		{
		    kerror(NULL, "spc_write_output_grouped_image", 
			   "Found (at least one) cluster not assigned to a class.  Please complete the classification process before attempting to write out a grouped image (note that you may use the 'Catch-All' class to dump all unassigned clusters into the same class).");
		   kpds_close_object(out_image);
		   kunlink(filename);
		   return(FALSE);
		}
		region[j] = spc_classes[cluster];
	    }
	    kpds_put_data(out_image, KPDS_VALUE_REGION, region);
	}
	kfree(region);

	/* 
	 * write the colormap defining the current appearance of the image 
	 */
	
	/* first, have to find the largest class id, that is the map size */
	max_class_id = 0;
	legend_ptr = spc_legend_list;
	while (legend_ptr != NULL)
	{
	    if (legend_ptr->class > max_class_id)
		max_class_id = legend_ptr->class;
	    legend_ptr = legend_ptr->next;
	}
	
	/* create the map segment */
        kpds_create_map(out_image);
	kpds_set_attributes(out_image, 
			    KPDS_MAP_DATA_TYPE, KUBYTE,
			    KPDS_MAP_SIZE, 3, max_class_id+1, 1, 1, 1,
			    NULL);

	kpds_set_attribute(out_image, KPDS_MAP_POSITION, 0, 0, 0, 0, 0);
	
	/* 
	 * now, fill out the map, which may have "holes" if an incremental
	 * class id is not being used 
	 */
	class_id = 0;
	count = 0;
	while (count < spc_legend_classnum)
	{
	    legend_ptr = spc_find_class_from_id(class_id);
	    if (legend_ptr != NULL)
	    {
	        triple[0] = (unsigned char) legend_ptr->redval;
	        triple[1] = (unsigned char) legend_ptr->greenval;
	        triple[2] = (unsigned char) legend_ptr->blueval;
		count++;
	    }
	    else 
	    {
	  	triple[0] = (unsigned char) 0;
		triple[1] = (unsigned char) 0;
		triple[2] = (unsigned char) 0;
	    }
	    kpds_put_data(out_image, KPDS_MAP_LINE, triple);

	    class_id++;
	}

	/* write out the image */
	(void) kpds_close_object(out_image);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_write_output_maps
|
|       Purpose: Writes out ONLY the map data
|
|         Input: filename - name of file to write out
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro & Jeremy Worley
|          Date: July 16, 1994
| Modifications: 
|
------------------------------------------------------------*/


int spc_write_output_maps(
    char *filename)
{
        int status;

	status = xvw_set_attribute(spc_display->workspace, 
				   XVW_COLOR_SAVEMAP, filename);
        return(status);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_set_output_maps
|
|       Purpose: Enters the map data for an output file:
|                count column,
|                class column,
|                covariance matrix   (if applicable),
|                covariance diagonal (if applicable),
|
|         Input: out_image - data object representing output file
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro & Jeremy Worley
|          Date: July 16, 1994
| Modifications: 
|
------------------------------------------------------------*/

static int spc_set_output_maps(
    kobject out_image)
{
	int     column_offset, i, j, extra_columns;
	double  *covar_data;
		      
        /* 2 extra columns for Count and Class columns */
	extra_columns = 2;  
	if (spc_map_contents & SPC_COVAR_MATRIX) 
	     extra_columns += spc_matrix_size;
	else if (spc_map_contents & SPC_COVAR_DIAG)
	     extra_columns += spc_diag_size;

	/* set number of map columns: map colnum + count + class */
	kpds_set_attribute(out_image, KPDS_MAP_SIZE, 
			   spc_map_colnum + extra_columns, 
			   spc_map_rownum, 1, 1, 1);
	
	/* write out the mean vectors in the map columns */
	kpds_set_attribute(out_image, KPDS_MAP_DATA_TYPE, KDOUBLE);
	kpds_set_attribute(out_image, KPDS_MAP_POSITION, 0, 0, 0, 0, 0);
	kpds_set_attribute(spc_image, KPDS_MAP_REGION_SIZE, 
			   spc_map_colnum, 1, 1, 1, 1);

	for (i = 0; i < spc_map_rownum; i++)
	{
	    for (j = 0; j < spc_map_colnum; j++)
	    {
	      kpds_set_attribute(out_image, KPDS_MAP_POSITION, j, i, 0, 0, 0);
	      kpds_put_data(out_image, KPDS_MAP_POINT, (kaddr) &spc_maps[i][j]);
	    }
	}

	column_offset = spc_map_colnum;

	kpds_set_attribute(out_image, KPDS_MAP_POSITION, 
			   column_offset, 0, 0, 0, 0);

	/* always enter count column */
	kpds_set_attribute(out_image, KPDS_COUPLING, KUNCOUPLED);
	kpds_set_attribute(out_image, KPDS_MAP_DATA_TYPE, KINT);
	spc_map_contents = spc_map_contents | SPC_COUNT;
	kpds_set_attribute(out_image, KPDS_MAP_REGION_SIZE, 
			   1, spc_map_rownum, 1, 1, 1);
	kpds_put_data(out_image, KPDS_MAP_REGION, spc_count);
	column_offset++;

/*
kprintf("--- Count Column ---\n");
for (i = 0; i < spc_map_rownum; i++)
{
    if (spc_count[i] > 0)
        kprintf("%d: %d\n", i, spc_count[i]);
}
kprintf("\n\n");
*/

	/* enter class info (if any) */
	if (spc_classes != NULL)
	{

kprintf("--- Class Column ---\n");
for (i = 0; i < spc_map_rownum; i++)
if (spc_classes[i] != -1)
        kprintf("spc_classes[%d] = %d\n", i, spc_classes[i]);
kprintf("\n\n");

	    spc_map_contents = spc_map_contents | SPC_CLASS;
	    kpds_put_data(out_image, KPDS_MAP_REGION, spc_classes);
	    column_offset++;
	}
	else spc_map_contents = spc_map_contents & (~SPC_CLASS);

	/* enter covariance matrix if maps came in with one */
	kpds_set_attribute(out_image, KPDS_MAP_DATA_TYPE, KDOUBLE);
	if (spc_map_contents & SPC_COVAR_MATRIX)
        {
	    kpds_set_attribute(out_image, KPDS_MAP_INTERPOLATE, KNONE);
	    kpds_set_attribute(out_image, KPDS_MAP_REGION_SIZE, 
                               spc_matrix_size, 1, 1, 1, 1);
	    for ( i = 0; i < spc_map_rownum; i++ )
            {
	        covar_data = (double *) 
			     kcalloc(1, spc_matrix_size*sizeof(double));
	        for (j = 0; j < spc_matrix_size; j++)
		    covar_data[j] = spc_covar_matrix[i][0][j];

	        kpds_set_attribute(out_image, KPDS_MAP_POSITION, 
			           column_offset,i,0,0,0);
		kpds_put_data(out_image, KPDS_MAP_REGION,
                                (kaddr) covar_data);
		kfree(covar_data);
	    }

	}

	/* enter covariance diagonal if maps came in with one */
	else if (spc_map_contents & SPC_COVAR_DIAG)
        {
	    for ( i = 0; i < spc_diag_size; i++ )
	    {
		for (j = 0; j < spc_map_rownum; j++)
                {
	            kpds_set_attribute(out_image, KPDS_MAP_POSITION,
				       column_offset+i, j, 0, 0, 0); 
		    kpds_put_data(out_image, KPDS_MAP_POINT,
				  (kaddr) &spc_covar_diag[i][j]);
                }
	    }
	    column_offset += spc_diag_size;
	}

	
	/* write out the number of map columns as an attribute */
	(void) kpds_create_object_attr(out_image, "colorMapColnum", 
				       1, 1, KINT, TRUE, TRUE);
	(void) kpds_set_attribute(out_image, "colorMapColnum", spc_map_colnum);
	
	/* write out the number of map contents mask as an attribute */
	(void) kpds_create_object_attr(out_image, "colorMapContents",
	 			       1, 1, KULONG, TRUE, TRUE);
	kpds_set_attribute(out_image, "colorMapContents", spc_map_contents);

	return(TRUE);
}
