
/*
 *   Copyright, 1991, The Regents of the University of California.
 *   This software was produced under a U.S. Government contract
 *   (W-7405-ENG-36) by the Los Alamos National Laboratory, which is
 *   operated by the University of California for the U.S. Department
 *   of Energy.  The U.S. Government is licensed to use, reproduce,
 *   and distribute this software.  Neither the Government nor the
 *   University makes any warranty, express or implied, or assumes
 *   any liability responsibility for the use of this software.
 */

#include "spectrum.h"

/*
 *  SM_cfactor ()     Calculate the Cholesky factorization of a symmetric,
 *                    positive definite matrix.  The matrix is assumed to 
 *                    be an SM type matrix, allocated via the SM_get_matrix
 *                    routine, and NONE OF THE LOWER TRIANGULAR ELEMENTS
 *                    are ever modified.
 *
 *                    Upon routine exit, the matrix A is assumed to contain
 *                    an UPPER TRIANGULAR MATRIX R (the zero elements below the
 *                    diagonal SHOULD NEVER BE INDEXED due to the structure
 *                    of SM type matrices) such that:
 *
 *                                 A = transpose (R) * R
 *
 *  WARNING:  LITTLE ERROR CHECKING IS PERFORMED BY THIS ROUTINE AT THE MOMENT!
 *            INPUT MATRIX A *MUST* BE POSITIVE DEFINITE UPON ENTRY.
 *
 *  Written by:  Patrick M. Kelly
 *  Date:        2/25/91
 *
 *  MODIFICATIONS:
 *
 *     7/31/91  Patrick M. Kelly
 *
 *              Changed routine to return an integer instead of a void
 *              routine.  Now returns a 1 upon successful factorization, 
 *              and 0 otherwise.
 *
 *     8/19/94  Dave Modl
 *
 *              Converted to Khoros 2.0
 */

int
SM_cfactor(
  int      dim,
  double ** A
  )

/*
int dim;			Dimension of the problem
double **A;			The SM matrix to factor
*/

{
	int i, j, k;		/* Loop control */

	for ( k = 0 ; k < dim ; k ++ ) {

		if ( A [k][k] <= 0.0 ) return ( 0 );

		A [k][k] = sqrt ( A [k][k] ) ;
		for ( i = k+1 ; i < dim ; i ++ ) {
			A [k][i] = A [k][i] / A [k][k] ;
			for ( j = k+1 ; j <= i ; j ++ ) {
				A [j][i] -= A [k][i] * A [k][j] ;
			}
		}

	}

	return ( 1 );
}

/*
 *  SM_comb_stats ()  Combine the statistics for two clusters of data.
 *
 *                   Cluster 3 <---  Cluster 1 + Cluster 2
 *
 *       (NOTE:  It is OK to call this routine with the same parameter
 *               in more than one place.  For example, you can add
 *               cluster 1 data "into" cluster 2)
 *
 *  Written by:  Patrick M. Kelly
 *  Date:        11/5/90
 *
 *  MODIFICATION HISTORY:
 *
 *       11/16/90  Patrick M. Kelly
 *                 Re-arranged factors to reduce numerical round-off
 *                 errors.
 *
 *     August 15, 1993     Patrick M. Kelly
 *                         Corrected bug where we were checking ( num_2 = 0 )
 *                         instead of ( num_2 == 0 ).
 *
 *     8/19/94  Dave Modl
 *
 *              Converted to Khoros 2.0
 */
void
SM_comb_stats(
  int         dim,
  int         num_1,
  int         num_2,
  int       * num_3,
  double     * m1,
  double     * m2,
  double     * m3,
  double    ** C1,
  double    ** C2,
  double    ** C3
  )

/*
int dim;						Problem Dimension
int num_1, num_2, *num_3;		Number of data points
double *m1, *m2, *m3;			Mean vectors
double **C1, **C2, **C3;			SM covariance matrices
*/

{
	int i, j;					/* Loop Control */
	double num_1_f;				/* Floating point rep. of num_1 */
	double num_2_f;				/* Floating point rep. of num_2 */
	double l1, l2, l3, l4, l5;	/* Temporary */

	num_1_f = (double) num_1;
	num_2_f = (double) num_2;

	l1 = num_1_f - 1.0 ;
	l2 = num_2_f - 1.0 ;
	l5 = num_1_f + num_2_f;
	l3 = num_1_f * num_2_f / l5 ;
	l4 = num_1_f + num_2_f - 1.0 ;

	/*
	 *  If one cluster is empty, just return the other -- nothing to do.
	 */

	if ( num_1 == 0 ) {

		*num_3 = num_2;
		for ( i = 0 ; i < dim ; i ++ ) {
			m3 [i] = m2 [i];
			for ( j = i ; j < dim ; j ++ ) {
				C3 [i][j] = C2 [i][j] ;
			}
		}
		return;

	} else if ( num_2 == 0 ) {

		*num_3 = num_1;
		for ( i = 0 ; i < dim ; i ++ ) {
			m3 [i] = m1 [i];
			for ( j = i ; j < dim ; j ++ ) {
				C3 [i][j] = C1 [i][j] ;
			}
		}
		return;

	}

	/*
	 *  Create the new covariance matrix.
	 *
	 *  (NOTE:  REMEMBER TO ONLY ACCESS THE UPPER TRIANGLE ELEMENTS
	 *          OF THE MATRICES!  SEE SM_MATRIX ROUTINES)
	 */
	for ( i = 0 ; i < dim ; i ++ ) {
	for ( j = i ; j < dim ; j ++ ) {

		C3 [i][j] = l1 * C1 [i][j] + l2 * C2 [i][j] +
				l3 * (m1 [i] - m2 [i]) * (m1 [j] - m2 [j]);

		C3 [i][j] /= l4 ;

	}
	}

	/*
	 *  Create the new mean vector.
	 */
	for ( i = 0 ; i < dim ; i ++ ) {

		m3 [i] =  ( num_1_f * m1 [i] + num_2_f  * m2 [i] ) / l5;

	}

	/*
	 *  Set number of data points in the cluster.
	 */
	*num_3 = num_1 + num_2;

	return;
}

/*
 *  SM_csolve ()      Solve a system of linear equations given the Cholesky
 *                    decomposition of the symmetric, positive definite
 *                    coefficient matrix A.  The matrix is assumed to
 *                    be an SM type matrix, allocated via the SM_get_matrix
 *                    routine, and NONE OF THE LOWER TRIANGULAR ELEMENTS
 *                    are ever modified.
 *
 *  WARNING:  NO ERROR CHECKING IS PERFORMED BY THIS ROUTINE AT THE MOMENT!
 *
 *  Written by:  Patrick M. Kelly
 *  Date:        2/25/91
 *
 *     8/19/94  Dave Modl
 *
 *              Converted to Khoros 2.0
 */

void
SM_csolve(
  int      dim,
  double ** R,
  double  * b
  )

/*
int dim;		Dimension of the problem
double **R;		Upper triangular matrix of Cholesky factorization
double *b;		(IN) Right hand side   (OUT) Solution vector
*/

{
	int i, j;		/* Loop control */

	/*
	 *      Solve Ax = b        A = transpose (R) * R
	 */

	/*
	 *  FORWARD SUBSTITUTION      Ly = b    L = transpose (R)
	 */
	for ( i = 0 ; i < dim ; i ++ ) {
		for ( j = 0 ; j < i ; j ++ ) {
			b [i] -= R [j][i] * b [j];
		}
		b [i] /= R [i][i];
	}

	/*
	 *  BACKWARD SUBSTITUTION      Ux = y   U = R
	 */
	for ( i = dim-1 ; i >= 0 ; i -- ) {
		for ( j = i+1 ; j < dim ; j ++ ) {
			b [i] -= R [i][j] * b [j];
		}
		b [i] /= R [i][i];
	}
	
	return;
}

/*
 *  SM_get_matrix ()  Allocate memory for a symmetric matrix.  The
 *                    matrix returned is compact (only the upper
 *                    portion of the matrix is actually allocated)
 *                    and needs to be treated with care!
 *
 *  NOTES:  This technique is useful for programming in C, simply
 *          because arrays are actually pointers to pointers to.....
 *          and for this reason, this code should not be implemented
 *          like this in say a FORTRAN program if speed of program
 *          execution is more important than size requirements.
 *
 *  WARNING:  BECAUSE OF THE WAY THE POINTERS ARE ARRANGED FOR THE
 *            RETURNED MATRIX, ONLY THE UPPER POSITIONS SHOULD BE
 *            ACCESSED AND THE LOWER PORTIONS SHOULD *NEVER* BE 
 *            ACCESSED !!!!!!!!!  THUS, THE COLUMN INDEX SHOULD
 *            *ALWAYS* BE GREATER THAN OR EQUAL TO THE ROW INDEX
 *            WHEN ACCESSING THE MATRIX.
 *
 *  Written by:  Patrick M. Kelly
 *
 *  Date:  10/3/90
 *
 *     8/19/94  Dave Modl
 *
 *              Converted to Khoros 2.0
 */

double **
SM_get_matrix(
  int dim		/* The dimension of the matrix */
  )

{
	int row;				/* Loop control */
	int place;			/* Position in contiguous memory */
	int size;				/* Size of allocated block */
	double *contig;			/* Matrix values */
	double **row_indices;	/* Row indices into contig */

	/*
	 *  STEP I:  Allocate the memory for ALL of the matrix values 
	 *           contiguously.  The number of values stored will 
	 *           be:
	 *
	 *               dim  +  (dim - 1)  +  (dim - 2)  +  ....  +  1
	 *           OR
	 *               (dim + 1) * dim / 2
	 */
	size = ( ( dim + 1 ) * dim ) / 2;
	contig = (double *) kmalloc ( (unsigned) ( sizeof (double) * size ) );
	if ( contig == NULL ) {
		kfprintf (kstderr,"SM_get_matrix ()  MEMORY ALLOCATION FAILURE\n");
		exit ( -1 );
	}

	/*
	 *  STEP II:  Allocate memory for row indices.
	 */
	row_indices = (double **) kmalloc ((unsigned) (sizeof (double *) * dim ));
	if ( row_indices == NULL ) {
		kfprintf (kstderr,"SM_get_matrix ()  MEMORY ALLOCATION FAILURE\n");
		exit ( -1 );
	}

	/*
	 *  STEP III:  Set up the row pointers into the contiguous memory
	 *             block.
	 */
	for ( row = 0 ; row < dim ; row ++ ) {
		place = ( row * ( 2 * dim - row - 1 ) ) / 2 ;
		row_indices [row] = & ( contig [ place ] ) ;
	}

	return ( row_indices );
}

int
SM_copy_matrix(
  int           dim,
  double      ** A,
  double      ** B
  )
{
  int           i, j;

	for ( i = 0 ; i < dim ; i ++ ) {
	for ( j = i ; j < dim ; j ++ ) {
		B [i][j] = A [i][j] ;
	}
	}

  return 1;
}

