 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>		
   >>>>	                 Internal Menu routines
   >>>>   Static:
   >>>>			xvw_setinfo_menuform()
   >>>>			xvw_getinfo_menuform()
   >>>>			xvw_setinfo_menudata()
   >>>>			xvw_getinfo_menudata()
   >>>>  Private:
   >>>>			xvw_initinfo_menu()
   >>>>			xvw_update_menu()
   >>>>			xvw_set_menuinfo()
   >>>>			xvw_get_menuinfo()
   >>>>   Public:
   >>>>			xvw_activate_menu()
   >>>>			xvw_inactivate_menu()
   >>>>			xvw_activated_menu()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kcms/kcms.h>
#include <xvobjects/ManagerP.h>

static klist *menuform   = NULL;
static klist *menudata   = NULL;
static klist *menuactive = NULL;
static klist *menulist   = NULL;

static kfunc_kaddr ManagerCreateMenu = NULL;
static kfunc_int   ManagerUpdateMenu = NULL;
static kfunc_int   ManagerMapMenu    = NULL;
static kfunc_int   ManagerUnmapMenu  = NULL;



/*-----------------------------------------------------------
|
|  Routine Name: xvw_setinfo_menuform
|
|       Purpose: Sets the menu form information associated with
|		 a particular object class.
|
|         Input: identifier - the object class to retrieve the information for
|        	 menu_form  - if not NULL sets the menu form specification
|
|	Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 25, 1994
| Modifications:
|
------------------------------------------------------------*/

static int xvw_setinfo_menuform(
   kaddr  identifier,
   kstring menu_form)
{
	klist *list;

	if ((list = klist_locate(menuform, identifier)) == NULL)
	{
	   menuform = klist_add(menuform, identifier, menu_form);
	   list = klist_locate(menuform, identifier);
	}
	list->client_data = (kaddr) menu_form;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_getinfo_menuform
|
|       Purpose: Retrieves the menu form information associated with
|		 a particular object class.
|
|         Input: identifier - the object class to retrieve the information for
|
|        Output: menu_form - if not NULL returns the menu form specification
|	Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 19, 1992 9:13
| Modifications:
|
------------------------------------------------------------*/

static int xvw_getinfo_menuform(
   kaddr  identifier,
   kstring *menu_form)
{
	klist *list;

	if ((list = klist_locate(menuform, identifier)) == NULL ||
	    list->client_data == NULL)
	{
	   if (menu_form) *menu_form = NULL;
	   return(FALSE);
	}

	if (menu_form) *menu_form = (String) list->client_data;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_setinfo_menudata
|
|       Purpose: Sets the menu client data information associated with
|		 a particular object class.
|
|         Input: identifier - the object class to retrieve the information for
|        	 menu_form  - if not NULL sets the menu client data
|
|	Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 25, 1994
| Modifications:
|
------------------------------------------------------------*/

static int xvw_setinfo_menudata(
   kaddr  identifier,
   kaddr  menu_data)
{
	klist *list;

	if ((list = klist_locate(menudata, identifier)) == NULL)
	{
	   menudata = klist_add(menudata, identifier, menu_data);
	   list = klist_locate(menudata, identifier);
	}
	list->client_data = (kaddr) menu_data;
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvw_getinfo_menudata
|
|       Purpose: Retrieves the menu client data information associated with
|		 a particular object class.
|
|         Input: identifier - the object class to retrieve the information for
|
|        Output: menu_data - if not NULL returns the menu client data
|	Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Aug 19, 1992 9:13
| Modifications:
|
------------------------------------------------------------*/

static int xvw_getinfo_menudata(
   kaddr identifier,
   kaddr *menu_data)
{
	klist *list;

	if ((list = klist_locate(menudata, identifier)) == NULL ||
	     list->client_data == NULL)
	{
	   if (menu_data) *menu_data = NULL;
	   return(FALSE);
	}

	if (menu_data) *menu_data = (kaddr) list->client_data;
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_menuinfo()
|
|       Purpose: Returns the menu information 
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_MENU_FORMFILE, XVW_MENU_CLIENTDATA
|                calldata  - pointer to contents of return string
|
|        Output:
|    Written By: Mark Young
|          Date: Jan 06, 1993 13:59
| Modifications:
|
------------------------------------------------------------------*/

int xvw_get_menuinfo(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	if (kstrcmp(attribute, XVW_MENU_FORMFILE) == 0)
	{
	   return(xvw_getinfo_menuform(object, calldata) ||
		  xvw_getinfo_menuform(xvw_class(object), calldata));
	}
	else if (kstrcmp(attribute, XVW_MENU_CLIENTDATA) == 0)
	{
	   return(xvw_getinfo_menudata(object, calldata) ||
		  xvw_getinfo_menudata(xvw_class(object), calldata));
	}
	return(FALSE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_menuinfo()
|
|       Purpose: Sets the menu information
|	         text field <= string
|
|         Input: object    - the object to have contents set
|                attribute - XVW_MENU_FORMFILE, XVW_MENU_CLIENTDATA
|                calldata  - pointer to contents of string
|
|        Output:
|    Written By: Mark Young
|          Date: Jan 06, 1993 13:59
| Modifications:
|
------------------------------------------------------------------*/

int xvw_set_menuinfo(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	String  *menu_form = (String *) calldata;
	kaddr   *menu_data = (kaddr *) calldata;

	if (kstrcmp(attribute, XVW_MENU_FORMFILE) == 0)
	   return(xvw_setinfo_menuform(object, *menu_form));
	else if (kstrcmp(attribute, XVW_MENU_CLIENTDATA) == 0)
	   return(xvw_setinfo_menudata(object, *menu_data));

	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_initinfo_menu
|
|       Purpose: Initialize the menu build and update routines.
|		 These routines control the creation and update
|		 of the object/gadget internal menus.
|
|         Input: create_menu   - 
|                activate_menu - 
|
|        Output: none
|	Returns: none
|
|    Written By: Mark Young
|          Date: Aug 19, 1992 9:13
| Modifications:
|
------------------------------------------------------------*/

void xvw_initinfo_menu(
   kfunc_kaddr create_menu,
   kfunc_int   update_menu,
   kfunc_int   map_menu,
   kfunc_int   unmap_menu)
{
	if (create_menu != NULL)
	   ManagerCreateMenu = create_menu;

	if (update_menu != NULL)
	   ManagerUpdateMenu = update_menu;

	if (map_menu != NULL)
	   ManagerMapMenu = map_menu;

	if (unmap_menu != NULL)
	   ManagerUnmapMenu = unmap_menu;
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_update_menu
|
|       Purpose:
|
|         Input: object     - the object in which we wish to invoke the
|			      menu for.
|
|        Output: none
|	Returns: none
|
|    Written By: Mark Young
|          Date: Aug 19, 1992 9:13
| Modifications:
|
------------------------------------------------------------*/

void xvw_update_menu(
   xvobject object)
{
	klist *temp;
	String menu_form = NULL;
	static klist *active = NULL;


	/*
	 *  Don't update unless the object's menu has been activated
	 */
	if (menuactive == NULL || (temp = klist_locate(menuactive,
			(kaddr) object)) == NULL)
	{
	   return;
	}

	/*
	 *  Print DEBUG message
	 */
	kinfo(KDEBUG,"Inside update_menu routine for object '%s'\n\n",
			xvw_name(object));

	/*
	 *  Sanity check to make sure that the build and update menu
	 *  routines have been initialized
	 */
	if (!ManagerUpdateMenu)
	{
	   kinfo(KSTANDARD,"xvw_update_menu:  Update Routine has not been \
initialized!  Failed to bring up menus for '%s'", xvw_name(object));
	   return;
	}

	/*
	 *  Go ahead and call the default menu routines to build and
	 *  update the menu structures for both manager objects and manager
	 *  objects.
	 */
	if (!xvw_getinfo_menuform(object, &menu_form) &&
	    !xvw_getinfo_menuform(xvw_class(object), &menu_form))
	{
	   kinfo(KSTANDARD, "xvw_update_menu:  Unable to find menu \
specification associated with the object '%s'", xvw_name(object));
	   return;
	}

	/*
	 *  Call the menu activate routine so that the menu is activated
	 */
	if (!klist_locate(active, object) &&
	    (temp = klist_locate(menulist, menu_form)) != NULL)
	{
	   active = klist_insert(active, (kaddr) object, NULL, KLIST_TAIL,TRUE);
	   (*ManagerUpdateMenu)(object, temp->client_data);
	   active = klist_delete(active, (kaddr) object);
	}
}

/************************************************************
*
*  Routine Name: xvw_activate_menu - pop up the internal menuform for an
*                                    object.
*
*       Purpose: Pops up the internal menuform for a GUI or visual object.
*                Has no effect if the internal menuform is already mapped.  
*                Will give an error message if there is no internal menuform 
*                associated with the object.
*
*         Input: object - the object for which to display menuform.
*        Output: 
*       Returns:
*  Restrictions:
*    Written By: Mark Young
*          Date: Aug 19, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_activate_menu(
   xvobject object)
{
	kaddr  menu;
	klist  *temp;
	char   path[KLENGTH];
	String menu_form = NULL;


	/*
	 *  Sanity check
	 */
	if (!object)
	   return;

	/*
	 *  Print DEBUG message
	 */
	kinfo(KDEBUG,"Inside xvw_activate_menu routine for object '%s'\n\n",
			xvw_name(object));

	/*
	 *  Sanity check to make sure that the build and update menu
	 *  routines have been initialized
	 */
	if (!ManagerCreateMenu)
	{
	   kinfo(KSTANDARD,"xvw_activate_menu:  Build Menu Routine have not \
been initialized!  Failed to bring up menus for '%s'",
			xvw_name(object));
	   return;
	}

	/*
	 *  Go ahead and call the default menu routines to build and
	 *  update the menu structures for both manager objects and manager
	 *  objects.
	 */
	if (!xvw_getinfo_menuform(object, &menu_form) &&
	    !xvw_getinfo_menuform(xvw_class(object), &menu_form))
	{
	   kinfo(KSTANDARD, "xvw_activate_menu:  Unable to find menu \
specification associated with the object '%s'", xvw_name(object));
	   return;
	}

	/*
	 *  See if the menu_form has been created already.  If not create
	 *  it and add it to the menulist.
	 */
	if ((temp = klist_locate(menulist, menu_form)) == NULL)
	{
           if (!kfullpath(menu_form, NULL, path) ||
               kaccess(path, R_OK) != 0)
	   {
	      kinfo(KXVLIB,"xvw_activate_menu:  Unable to find menu '%s'.",
			menu_form);
	      return;
	   }

	   if ((menu = (*ManagerCreateMenu)(object, path)) == NULL)
	   {
	      kinfo(KXVLIB, "xvw_activate_menu: Unable to build menu '%s'.",
			menu_form);
	      return;
	   }
	   menulist = klist_add(menulist, menu_form, menu);
	}
	else
	   menu = temp->client_data;

	if (!ManagerMapMenu || !(*ManagerMapMenu)(object, menu))
	{
	   kinfo(KXVLIB, "xvw_activate_menu: Unable to map menu '%s'.",
			menu_form);
	   return;
	}

	if ((temp = klist_locate_clientdata(menuactive, (kaddr) menu)) != NULL)
	   menuactive = klist_delete(menuactive, klist_identifier(temp));

	menuactive = klist_add(menuactive, (kaddr) object, (kaddr) menu);
	xvw_update_menu(object);
}

/************************************************************
*
*  Routine Name: xvw_inactivate_menu - pop down the internal menuform for an
*                                      object.
*
*       Purpose: Pops down the internal menuform for a GUI or visual object.
*                Has no effect if the internal menuform is not mapped.  
*		 Will give an error message if there is no internal menuform
*                associated with the specified object.
*
*         Input: object - the object for which to pop down the menuform.
*        Output: 
*       Returns: 
*  Restrictions:
*    Written By: Mark Young
*          Date: Aug 19, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_inactivate_menu(
   xvobject object)
{
	klist   *temp;
	String  menu_form = NULL;


	/*
	 *  Sanity check
	 */
	if (!object)
	   return;

	/*
	 *  Print DEBUG message
	 */
	kinfo(KDEBUG,"Inside xvw_inactivate_menu routine for object '%s'\n\n",
			xvw_name(object));

	/*
	 *  Go ahead and call the default menu routines to build and
	 *  update the menu structures for both manager objects and manager
	 *  objects.
	 */
	if (!xvw_getinfo_menuform(object, &menu_form) &&
	    !xvw_getinfo_menuform(xvw_class(object), &menu_form))
	{
	   kinfo(KSTANDARD, "xvw_inactivate_menu:  Unable to find menu \
specification associated with the object '%s'", xvw_name(object));
	   return;
	}

	/*
	 *  Call the menu activate routine so that the menu is activated
	 */
	temp = klist_locate(menulist, menu_form);
	if (temp != NULL && ManagerUnmapMenu != NULL)
	{
	   if ((*ManagerUnmapMenu)(object, temp->client_data) == TRUE)
	      menuactive = klist_delete(menuactive, (kaddr) object);
	}
}


/************************************************************
*
*  Routine Name: xvw_activated_menu - see if the internal menuform for an
*                                     object is currently mapped
*
*       Purpose: Indicates whether or not the internal menuform 
*                for a GUI or visual object is currently displayed.
*
*         Input: object - the GUI or visual object to query
*        Output: TRUE if the object has its internal menuform up;
*                FALSE if the menuform is not up, or if the object
*                has no menuform associated with it.
*       Returns: 
*  Restrictions:
*    Written By: Mark Young
*          Date: Aug 19, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvw_activated_menu(
   xvobject object)
{
	/*
	 *  Sanity check
	 */
	if (!object)
	   return(FALSE);

	/*
	 *  Check the menu active list to see if the object's menu has
	 *  currently activated
	 */
	if (!klist_locate(menuactive, (kaddr) object))
	   return(FALSE);
	else return(TRUE);
}
