 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>		    File Change Detect Routines
   >>>>
   >>>>   These routines are used to detect whether a file
   >>>>   has been changed.  The file detect routine will
   >>>>   call the programmer's callback routine when the
   >>>>	  file has been detected to be changed.  The program-
   >>>>	  mer also specifies how often we are supposed
   >>>>   to check the file for being modified.
   >>>>
   >>>>   Static:
   >>>>             check_detectfile()
   >>>>  Private:
   >>>>
   >>>>   Public:
   >>>>             xvw_add_detectfile()
   >>>>             xvw_remove_detectfile()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


typedef struct _xvw_file
{
	xvobject  object;
	XtIntervalId id;
	time_t    time;
	char	  *filename;
	unsigned long interval;
	kfunc_int routine;
	kaddr	  client_data;
} xvw_file;

static klist *file_list = NULL;


/*-----------------------------------------------------------
|
|  Routine Name: check_detectfile - checks to see if a file has changed
|
|       Purpose: Checks to see if a file has changed.  If so
|                we call the user's callback routine with the
|                id and client data.  If the user returns TRUE
|		 then we update the time otherwise we will check
|		 the file again with the same time.
|
|         Input: data - our xvw_file structure
|                id   - the current timeout id
|
|        Output:
|	Returns:
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 10:42
| Modifications: Converted from check_detectfile) in Khoros 1.0 (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void check_detectfile(
   kaddr        data,
   XtIntervalId *id)
{
	struct stat buf, tmp_buf;
	kfunc_int routine;
	xvw_file  *entry = (xvw_file *) data;


	/*
	 *  Get the stat's for the entry filename, unless we have a force
	 *  callback.  If a force callback that simply do the callback no
	 *  matter what state we are in.
	 */
	if (stat(entry->filename, &buf) != -1)
	{
	   /*
	    *  check to see if the file has been modified since the last
	    *  time we read it.
	    */
	   if (entry->time != buf.st_mtime)
	   {
	      /*
	       *  since the file has been modified call the user's callback
	       *  routine and update the read time.
	       */
	      routine    = entry->routine;

	      tmp_buf.st_mtime = entry->time;
	      entry->time = buf.st_mtime;
	      if (!routine(entry->object, entry->filename, entry->client_data))
	         entry->time = tmp_buf.st_mtime;
	      else if (stat(entry->filename, &buf) != -1)
		 entry->time = buf.st_mtime;
	   }
	}

	/*
	 *  Re-add check_detectfile to the time out handler.
	 */
	entry->id = XtAppAddTimeOut(xvw_appcontext(NULL), entry->interval,
			            check_detectfile, data);
}


/************************************************************
*
*  Routine Name: xvw_add_detectfile - add a (file) input handler to an object
*
*       Purpose: Causes a detection mechanism to be installed on the
*                specified file.  After the specified interval of time
*                has elapsed, the file will be checked for any modification.
*		 If a modification to the file is detected, then the specified
*                input handler is called.
*
*		 The input handler can be associated with an object, so that
*		 file detection is automatically discontinued when the object is
*		 destroyed.  If NULL is passed for the object, then the input
*                handler is added to the global file detection list.
*
*	         !The detect file callback must be declared in the following
*		 !form:\f(CW
*	         !int input_handler(
*	         !     xvobject object,
*	         !     char *filename,
*	         !     kaddr client_data)\fP
*
*		 !\fIobject -\fP 
*		 !    If \fIxvw_add_detectfile()\fP is called with a particular 
*                !    xvobject, that object will be passed into the input handler.
*
*		 !\fIfilename -\fP
*                !    This is the name of the file being monitored for change.
*
*
*                !\fIclient_data -\fP
*                !    The pointer to the client data, used to pass parameters
*                !    from the application to the input handler.
*
*         Input: object      - object on which to place the input handler.  
*                              Pass NULL if the input handler is to be invoked 
*                              in general, and not associated with any 
*                              particular object.
*		 filename    - the name of the file to be monitored.
*                argtime     - the interval (in seconds) at which the file 
*                              should be checked for changes.
*                routine     - the input handler routine to be called when 
*                              change to the file is detected.
*                client_data - pointer to client data that
*                              will be passed to input handler
*        Output:
*	Returns: 
*  Restrictions: 
*    Written By: Mark Young
*          Date: Jul 15, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_add_detectfile(
   xvobject  object,
   char      *filename,
   double    argtime,
   kfunc_int routine,
   kaddr     client_data)
{
	char *fullpath;
	xvw_file *entry;
	struct stat buf;

	/*
	 *  Get the last modified time of the file in question.
	 */
	if ((fullpath = kfullpath(filename, NULL, NULL)) == NULL)
	{
	   errno = KINVALID_FILE;
	   kerror("xvutils", "xvw_add_detectfile", 
		  "Invalid filename '%s' found.", !filename ? "(NULL)" :
		 filename);
	   return;
	}

	if ((entry = (xvw_file *) kcalloc(1, sizeof(xvw_file))) == NULL)
	{
	   kerror("xvutils", "xvw_add_detectfile", 
		  "Cannot allocate memory for xvw_file structure");
	   return;
	}

	if (stat(fullpath, &buf) != -1)
	   entry->time = buf.st_mtime;
	else
	   entry->time = time(NULL);


	/*
	 *  update the rest of detect structure info.  We also need
	 *  compute the update interval in terms of millaseconds instead
	 *  of seconds.
	 */
	entry->object = object;
	entry->interval = argtime * 1000;
	entry->filename = fullpath;
	entry->routine  = routine;
	entry->client_data  = client_data;
	entry->id = XtAppAddTimeOut(xvw_appcontext(NULL), entry->interval,
			            check_detectfile, (kaddr) entry);

	/*
	 *  Add the entry to the detect file handler list.  If the object is
	 *  NULL then we add it to the global file list.
	 */
	if (!object)
	{
	   file_list = klist_insert(file_list, (kaddr)
			kstring_to_token(filename), entry, KLIST_HEAD, TRUE);
	}
	else
	{
	   object->detectfile = klist_insert(object->detectfile, (kaddr)
			kstring_to_token(filename), entry, KLIST_HEAD, TRUE);
	}
}


/************************************************************
*
*  Routine Name: xvw_remove_detectfile - remove a (file) input handler 
*					 from an object
*
*       Purpose: Causes the detection mechanism previously installed with
*                \fIxvw_add_detectfile()\fP to be removed from the specified
*                file descriptor.
*
*                If the input handler was associated with an object, the
*                file detection will be automatically removed when the object
*                is destroyed. Alternatively, it can be removed before the
*		 object is destroyed by using this routine.  If NULL was 
*                passed to \fIxvw_add_detectfile()\fP for the object, then 
*                you must call this routine if you need to remove the installed 
*                input handler.
*
*         Input: object      - object in which to remove the detect file
*			       handler, if NULL then removed from the global
*			       list.
*		 filename    - the name of the file that was being monitored.
*                routine     - the input handler routine which was being
*			       called when change to the file was detected.
*                client_data - pointer to private application data that
*                              was being passed to action handler
*        Output:
*	Returns:
*  Restrictions: 
*    Written By: Mark Young
*          Date: Jul 15, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
/* ARGSUSED */
void xvw_remove_detectfile(
   xvobject  object,
   char      *filename,
   kfunc_int routine,
   kaddr     client_data)
{
	int	 name;
	xvw_file *entry;
	klist	 *temp, **list = !object ? &file_list : &object->detectfile;

	temp = *list; name = kstring_to_token(filename);
	while ((temp = klist_locate(temp, (kaddr) name)) != NULL)
	{
	   entry = (xvw_file *) klist_clientdata(temp);
	   if (routine == entry->routine && client_data == entry->client_data)
	   {
	      XtRemoveTimeOut(entry->id);
	      kfree(entry->filename); kfree(entry);
	      temp = klist_delete(temp, temp->identifier);
	      *list = klist_head(temp);
	      return;
	   }
	   temp = klist_next(temp);
	}
}
