 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>               Select xvobject                         <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                xvu_create_select()		      <<<<              
   >>>>                xvu_choose()		              <<<<              
   >>>>   Static:                                             <<<<
   >>>>                xvu_cancel_select()		      <<<<              
   >>>>                xvu_choose_callback()		      <<<<              
   >>>>   Public:                                             <<<<
   >>>>                xvu_select_wait()		      <<<<              
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	

static int  xvu_select_count = 0;
static int  xvu_select_done  = FALSE;
static int  xvu_select_sig;

static void xvu_choose_callback	PROTO((xvobject,kaddr,kaddr));
static void xvu_cancel_select	PROTO((xvobject, kaddr));
static void xvu_destroy_select	PROTO((Select_Info *));

/*-----------------------------------------------------------
|  Routine Name: xvu_choose
|
|       Purpose: xvu_choose produces standardized list selections for
|                X-based Khoros library routines and X based Khoros
|                applications.  It should be set up as the choose
|                handler for all X-based list items.
|
|         Input: choices - an array of strings containing the items
|                                  to select from.
|                nchoices   -  The number of items in the choices
|                default_index - The index number to the default item.
|                message       - grammatically correct, clear explanation of
|                                the error that occurred. This can be formatted
|                                like a (void) printf statement.
|
|        Output: return_string  - string that holds the selected item.
|                                If it's NULL, it kmallocs the space necessary,
|                                and returns the string.
|                return_index  - This is the index of the item selected.
|
|        Returns: TRUE upon success and FALSE upon failure.
|
|    Written By: Danielle Argiro & Tom Sauer
|          Date: Oct 29, 1992 
------------------------------------------------------------*/
int xvu_choose(
    char **choices,
    int  nchoices,
    int  default_index,
    int  *return_index,
    char **return_string,
    char *message)
{
	*return_index = xvu_select_wait(message,"",choices,nchoices);

	if (*return_index == 0)
	{
		if (*return_string != NULL)
			**return_string = '\0';
	}
	else if (*return_string == NULL)
		*return_string = kstring_copy(choices[*return_index-1], NULL);
	else
		kstrcpy(*return_string, choices[*return_index-1]);

	return TRUE;
}

/************************************************************
*
*  Routine Name: xvu_select_wait - pop up select object (N choices); 
*                                  wait for response
*
*       Purpose: Creates a "select" object which has an information
*                string, N buttons, plus a "cancel" button.
*
*         Input: stmt          - string containing information about the 
*                                various choices
*                label         - short label for top of form;
*                                NULL will indicate use of default "Select One".
*		 button_labels - array of strings to serve as button labels.
*		 button_num    - number of buttons to create, also size of
*				  the button_labels array.
*
*        Output:
*	Returns: Value returned to caller will be:
*		       1 if user selected button1 
*		       2 if user selected button2 
*		       3 if user selected button3 
*		       N if user selected buttonN.
*		       0 if there was an error creating selection object,
*			 or the user clicked on CANCEL.
*
*		  This routine will NOT return to the calling program until 
*		  one of the buttons is chosen.
*  Restrictions:
*    Written By: Danielle Argiro
*          Date: May 20, 1992 13:58
*      Verified:
*  Side Effects:
* Modifications: 
*
*************************************************************/

int xvu_select_wait(
    char  *stmt,
    char  *label,
    char  **button_labels,
    int   button_num)
{
	knotify	handler;

	xvu_select_done = FALSE;

	if (xvw_display(NULL) == NULL)
	{
		kinfo(KSTANDARD, "%s\n", stmt);
		return TRUE;
	}

	handler = kset_choosehandler(NULL);
	if (!(xvu_create_select(stmt, label, button_labels, button_num))) 
		return FALSE;

	(void) kset_choosehandler(handler);
	while(!xvu_select_done)
		xvw_process_event(); 

	return(xvu_select_sig);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvu_create_select()
|
|       Purpose: Creates the selection object, with a label, text, 
|                N acknowledgement buttons, and a cancel button.
|
|         Input: select_prompt - the message to appear on selection object
|		 button_labels - array of strings for labelling buttons
|		 button_num    - number of buttons to create
|
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro 
|          Date: Jul 16, 1992
| Modifications:
|
------------------------------------------------------------*/

int xvu_create_select(
    char *select_prompt,
    char *dummy,
    char **strings,
    int  nstrings)
{
	int		indx, i, charnum, max_charwidth = 0;
	float		min_height;
	char		name[KLENGTH];
	char		label_string[KLENGTH];
	char		buffer[5*KLENGTH];
	Select_Info	*select_info;
	xvobject	manager, label, cancel, list, offset;

	/*
	 *  error checks
	 */
	if (nstrings < 1)
	{
	    kerror("xvutils", "xvu_create_select", "nstrings must be > 0\n");
	    return(FALSE);
	}
	else if (strings == NULL)
	{
	    kerror("xvutils", "xvu_create_select", "strings array is NULL");
	    return(FALSE);
	}
	else
	{
	    for (i = 0; i < nstrings; i++)
	    {
		if (strings[i] == NULL)
		{
		    kerror("xvutils", "xvu_create_select", 
			   "One or more NULL strings");
		    return(FALSE);
		}
	    }
	}

	/* create identifying object name */
	ksprintf(name,"%d_select", xvu_select_count++);

	/* allocate select info structure */
	if ((select_info = (Select_Info *) 
			    kcalloc(1, sizeof(Select_Info))) == NULL)
	    return(FALSE);
	

	/* create the selection box's toplevel object */
	select_info->toplevel = xvw_create_transient_shell(name, NULL, NULL);
	xvw_set_attribute(select_info->toplevel, XVW_SHELL_RESIZE, FALSE);
	select_info->num = nstrings;
	select_info->prompt = kstrdup(select_prompt);

	/* create manager widget */
	manager = xvw_create_manager(select_info->toplevel, "chooseManager");

	/* create cancel button & install cancellation callback */
	cancel = xvw_create_button(manager,"cancel");
	xvw_set_attributes(cancel,
		XVW_LABEL,	"CANCEL",
		XVW_LEFT_OF,	NULL,
		XVW_BELOW,	NULL,
		NULL);
	xvw_add_callback(cancel, XVW_BUTTON_SELECT, 
			 xvu_cancel_select, select_info);

	/*
	 *  need to know the length of the longest string, to see
	 *  how wide to make the select object. max out at 50 characters.
	 */
	i = 0;
	for (i = 0; i < nstrings; i++)
	{
	    if (kstrlen(strings[i]) > max_charwidth) 
		max_charwidth = kstrlen(strings[i]);
	}
	max_charwidth = kmax(max_charwidth, kstrlen(select_info->prompt));
	if (max_charwidth > 50) max_charwidth = 50;

	/* 
	 * create the labelstr object w/ prompt.  if the prompt is too long,
	 * we may need a series of vertically laid out labelstring objects. 
	 */
	i = 0; 
	charnum = kstrlen(select_info->prompt);
	offset = cancel;
	ksprintf(buffer, " %s", select_info->prompt);
	while (i <= charnum)
	{
	    /* 
	     * the following nasty little 10 lines of code are to cause
	     * the labels for the labelstrings to wrap neatly on words.
	     */
	    label_string[0] = '\0';
	    if (kstrlen(buffer) < i+50)
		indx = kstrlen(buffer) - i;
	    else
	    {
	        indx = 50;
	        while ((buffer[i+indx] != ' ') && (indx > 0)) indx--;
	        if (indx == 0) indx = 50;
	    }
	    (void) kstring_ncopy(&buffer[i], indx, label_string);

	    /*
	     * create labelstr with (what might be only part of) prompt as label
	     */
	    label = xvw_create_labelstr(manager,"label");
            xvw_set_attributes(label,
		    XVW_LABEL,		  label_string,
		    XVW_CHAR_WIDTH,	  (float) max_charwidth,
                    XVW_BORDER_WIDTH, 	  0,
                    XVW_MAP_WHEN_MANAGED, TRUE,
		    XVW_LABEL_JUSTIFY,	  KLABEL_JUSTIFY_CENTER,
		    XVW_BELOW,   	  offset,
                    NULL);
	    offset = label;
	    i += indx;
	}

	/* create list object in which to display choices */
	min_height = nstrings > 10 ? 10.0 : (float) (nstrings + 1);
	list = xvw_create_list(manager,"list");
	xvw_set_attributes(list,
		XVW_BELOW,		offset,
		XVW_CHAR_MIN_HEIGHT,	min_height,
		XVW_VERT_DIST,    	15,
		XVW_TACK_EDGE,		KMANAGER_TACK_ALL,
		NULL);

	/* change the list according to specifications passed in */
	xvw_change_list(xvw_retrieve_list(list),
			strings, nstrings, FALSE);

	/* add callback to be fired when they choose an item from the list */
	xvw_add_callback(xvw_retrieve_list(list), XVW_LIST_ITEM_SELECT,
			 xvu_choose_callback, select_info);

	xvw_add_protocol(select_info->toplevel, "WM_DELETE_WINDOW", 
			 xvu_cancel_select, select_info->toplevel);

	/* pop up the select object */
	xvw_place(select_info->toplevel, NULL);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvu_choose_callback()
|
|       Purpose: Destroys select object when the user clicks on
|                one of the N acknowledgement buttons.
|
|         Input: object     - the acknowledgement button
|                client_data - holds the button info
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Jul 16, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_choose_callback(
    xvobject object,
    kaddr    client_data,
    kaddr    call_data)
{
	xvw_list_struct	*list_struct = (xvw_list_struct *) call_data;

	xvu_select_sig  = list_struct->list_index + 1;
	xvu_destroy_select((Select_Info *) client_data);
	xvu_select_done = TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: xvu_cancel_select()
|
|       Purpose: Destroys select object when the user clicks on
|                the "cancel" button.
|
|         Input: object     - the acknowledgement button
|                client_data - toplevel object
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Jul 16, 1992
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_cancel_select(
     xvobject object,
     kaddr    client_data)
{
	xvu_destroy_select((Select_Info *) client_data);
        xvu_select_done	= TRUE;
	xvu_select_sig	= 0;
}

/*-----------------------------------------------------------
|  Routine Name: xvu_destroy_select()
|
|       Purpose: Destroys a select object, after unmapping it first.
|                kfree()s all associated memory.
|
|         Input: select_info	- pointer to the select info structure
|    Written By: Neil Bowers
|          Date: Sep 23 1993
------------------------------------------------------------*/
static void xvu_destroy_select(
     Select_Info *select_info)
{
	Display	*display = xvw_display(select_info->toplevel);

	xvw_remove_protocol(select_info->toplevel, "WM_DELETE_WINDOW", 
				xvu_cancel_select, select_info->toplevel);
        xvw_unmap(select_info->toplevel);
        xvw_destroy(select_info->toplevel);
	XFlush(display);
	xvw_sync(FALSE);
	kfree(select_info->prompt);
        kfree(select_info);
}
