 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>	     Single-Item-Selection Popup List Utility         <<<<
   >>>>                                                       <<<<
   >>>>  Private:					      <<<<
   >>>>   Static:					      <<<<
   >>>>                create_list_object()	              <<<<              
   >>>>                select_item()   	          	      <<<<              
   >>>>                                                       <<<<
   >>>>                return_user_defined_item()	      <<<<              
   >>>>                cancel_list()   	         	      <<<<              
   >>>>   Public:					      <<<<
   >>>>                xvu_run_list_wait()		      <<<<              
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	

#define MaxNumLinesInViewport 35

static int               xvu_list_done   = FALSE;
static int               xvu_list_count  = 1;
static xvw_list_struct  *xvu_list_return = NULL;

static xvobject create_list_object PROTO((char **, int, char *, char *, int));

static void     return_user_defined_item  PROTO((xvobject, kaddr, kaddr));
static void     select_item               PROTO((xvobject, kaddr, kaddr));
static void     cancel_list               PROTO((xvobject, kaddr, kaddr));


/************************************************************
*
*  Routine Name: xvu_run_list_wait - display list, wait for single choice
*
*       Purpose: Takes an array of strings, and uses them to create a pop-up
*                list object.  The user is allowed to select a string from 
*                the list; the xvw_list_struct specifying the user's selection
*                is then returned to the caller.
*
*                An xvw_list_struct is defined as follows:
*                ! typedef struct _xvw_list_struct 
*                ! {
*                !     char   *string;
*                !     int    list_index;
*                !
*                ! } xvw_list_struct;
*
*
*                A default list item can be provided if desired.
*                If appropriate, the popup list object may also include
*                a text parameter box in which the user may enter their
*                own choice.  This routine will block the calling program until
*                the user chooses an item from the list, or clicks on "cancel".
*
*	  Input: list         - array of strings to be displayed in the list
*		 size         - size of the 'list' array
*		 prompt       - prompt to display at top of list object
*		 label        - label for the list object
*		 def_index    - index of the item in the list to return (in
*                               xvw_list_struct form) if the user clicks on 
*                               "Cancel". Specify default of -1 if the default 
*                               is to be NULL.  Be aware that indices begin at 
*                               zero (0), and run to (size - 1).
*		 user_defined - flag indicating if a string object should be
*			        created at the bottom of the list object to 
*                               allow the user to enter a string that is not 
*                               part of the list array.  If user defined items
*                               are allowed, and the user specifies their own
*                               list item, the xvw_list_struct returned will
*                               have a list_index of -1, indicating that the
*                               user's choice was not a member of the original
*                               list.
*        Output:
*	Returns: If the user selects a list item, a pointer to a xvw_list_struct
*                containing the string selected from the list by the user, and 
*                the index of that string in the list.  If user_defined is 
*                passed as TRUE and the user enters their own item, the
*                xvw_list_struct returned will contain the string entered by
*                the user and a list_index of -1.  If the user selects "Cancel",
*                and a default index between 0 and (size-1) was provided,
*                the xvw_list_struct containing the default index and the 
*                associated string will be returned.  If the user selects
*                "Cancel", and the default index provided was -1, NULL is
*                returned.
*
*  Restrictions:
*    Written By: Danielle Argiro
*          Date: May 19, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*
*************************************************************/


xvw_list_struct *xvu_run_list_wait(
   char *list[],
   int  size,
   char *prompt,
   char *label,
   int  def_index,
   int  user_defined)
{
	xvobject         toplevel;

	xvu_list_return = NULL;

	/*
	 *  default provided may be a valid index (0 to size-1)
	 */
	if ((def_index < -1) || (def_index > size-1))
        {
	    errno = KCALL;
	    kerror("xvutils", "xvu_run_list_wait",
		   "Default index provided must be -1 for NULL default, or between 0 and %d for a valid default from the list.", size-1);
	    return(NULL);
	}

	xvu_list_done = FALSE;

	/*
	 *  create the list object
	 */
	xvw_busy(NULL, TRUE);
	toplevel  = create_list_object(list, size, prompt, label, user_defined);
	if (toplevel == NULL) return(NULL);

	/*
	 *  pop up list & wait for response
	 */
	while (!xvu_list_done)
	   xvw_process_event();      

       /*
	*  user selected an item; destroy list object
        */
	xvw_unmap(toplevel);
	xvw_destroy(toplevel); 
	xvw_busy(NULL, FALSE);

	/*
	 *  global list_return still NULL implies that the user
         *  clicked on "Cancel";  malloc and initialize default list_return
         *  if caller provided a valid default
	 */
	if ((xvu_list_return == NULL) && (def_index != -1))
	{
	    xvu_list_return = (xvw_list_struct *) 
		 	       kcalloc(1, sizeof(xvw_list_struct));
	    xvu_list_return->list_index = def_index;
	    xvu_list_return->string = kstrdup(list[def_index]);
	    return(xvu_list_return);
	}

	/*
         *  caller provided a -1 for the default, indicating no default;
         *  return NULL.
	 */
	else if ((xvu_list_return == NULL) && (def_index >= 0))
	    return(NULL);

	/*
 	 *  list_return != NULL => user selected a list item; return it.
	 */
	return(xvu_list_return);
}

/*------------------------------------------------------------
|
|   Routine Name: create_list_object()
|
|        Purpose: Creates the list object.  It takes an array of strings, 
|                 and uses each string as an item in the list. The prompt 
|                 and label strings are used to complete the list object.
|
|          Input: list         - the array of strings to be displayed as 
|                                elements of the list.
|		  size         - the size of the 'list' array
|		  prompt       - prompt to display at top of the list object
|		  label        - the label of the list object
|		  user_defined - flag indicating if a string object should be
|			        included to allow the user to enter a string
|				that is not part of the list array.
|
|        Output: Returns the toplevel of the list object
|
|    Written By: Danielle Argiro, Stephanie Hallett, Mark Young
|          Date: May 19, 1994
| Modifications: 
|
|
--------------------------------------------------------------------*/

static xvobject create_list_object(
   char **list,
   int  size,
   char *prompt,
   char *label,
   int  user_defined)
{
	xvobject toplevel;	  /* toplevel object */
	xvobject back;		  /* backplane for list object */
	xvobject label_object;	  /* label object */
	xvobject prompt_object;	  /* prompt object */
	xvobject list_object;	  /* compound list object */
	xvobject actual_list;	  /* list component of compound list object */
	xvobject cancel_object;	  /* button to cancel list object*/
	xvobject string_wid;      /* button to enter user-specified string */
	xvobject offset = NULL;   /* offset object */
	char     name[KLENGTH];
	int      i, max_strlen; 
	float    width, height; 

	if (list == NULL)
	{  
	    errno = KNULL_PARAMETER;
 	    kerror("xvutils", "xvu_create_list_object","'list' cannot be null");
	    return(NULL);
	}

	if (label == NULL)
	    label = kstring_copy(" ", NULL);

	if (prompt == NULL)
	    prompt = kstring_copy("Choose from:", NULL);

	/*
	 * create identifying object name
	 */
	ksprintf(name, "%d_list", xvu_list_count++);

	/* 
	 * create the list's toplevel object and add it to the
	 * list used by journal playback.
	 */
	toplevel = xvw_create_transient_shell(name, NULL, NULL);
	xvw_add_protocol(toplevel, "WM_DELETE_WINDOW", 
			 cancel_list, toplevel);

	/*
	 * Compute the longest selection length. Then set the width & 
         * height of the manager backplane accordingly.
	 */
	max_strlen = kmax(kstrlen(prompt), kstrlen(label) + kstrlen("cancel"));
	for (i = 0; i < size; i++)
	{
	    if (kstrlen(list[i]) > max_strlen) 
		max_strlen = kstrlen(list[i]);
	}
	width  = (float) max_strlen;
	if (size > MaxNumLinesInViewport)
	    height = (float) MaxNumLinesInViewport;
	else  height = (float) size;

	/* 
	 * create backplane object 
	 */ 
	back = xvw_create_manager(toplevel, "back");
	xvw_set_attribute(back, XVW_CHAR_WIDTH, width);

	/* 
	 * create cancel button to upper left hand corner
	 */
	width = 7.0;
	cancel_object = xvw_create_button(back, "cancel");
        xvw_set_attributes(cancel_object,
		XVW_LABEL,         "Cancel",       /* button label           */
		XVW_LEFT_OF,       NULL,           /* to far R               */
		XVW_BELOW,         NULL,           /* upper far R            */
		NULL);
	xvw_add_callback(cancel_object, XVW_BUTTON_SELECT, 
			 cancel_list, toplevel);

	/* 
	 * create the label object 
	 */
        label_object = xvw_create_label(back, "label_object");
        xvw_set_attributes(label_object,
		XVW_LABEL,        label,       /* label                       */
		XVW_BORDER_WIDTH, 0,           /* no border                   */
		XVW_RIGHT_OF,     offset,      /* R of use (multsel) or NULL  */
		XVW_BELOW,        NULL,        /* at the top                  */
		XVW_LEFT_OF,      cancel_object, /* L of cancel               */
		NULL);

	/*
	 * create the prompt object
	 */
	prompt_object = xvw_create_label(back, "prompt");
        xvw_set_attributes(prompt_object,
		XVW_LABEL,        prompt,         /* label                   */
		XVW_BORDER_WIDTH, 0,              /* no border               */
		XVW_BELOW,        cancel_object,  /* under "cancel", "use"   */
	        XVW_LEFT_OF,      NULL,           /* centered                */
	        XVW_RIGHT_OF,     NULL,           /* centered                */
		NULL);

	/*
	 * create the list object
	 */
	list_object = xvw_create_list(back, "list");
	xvw_set_attributes(list_object, 
		   XVW_BELOW,        prompt_object,  /* below prompt      */
		   XVW_TACK_EDGE,    KMANAGER_TACK_HORIZ,/* centered      */
	           XVW_CHAR_HEIGHT,  height + 1.0,  
		   NULL);

	/*
	 *  retrieve the actual list component of the list object,
	 *  so as to set the list contents & install callback.
	 */
	actual_list = xvw_retrieve_list(list_object);
	xvw_change_list(actual_list, list, size, FALSE);
	xvw_add_callback(actual_list, XVW_LIST_ITEM_SELECT, 
			 select_item, NULL);

	/* 
	 *  if specified, create text object in which user 
	 *  may specify their own string response 
	 */
	if (user_defined)
	{
	    string_wid = xvw_create_text(back, "user_defined");
	    xvw_set_attributes(string_wid,
		XVW_TACK_EDGE,	 KMANAGER_TACK_HORIZ,/* tack horizontal   */
		XVW_BELOW,       list_object,       /* under list object */
		NULL);

	    xvw_add_action(string_wid, "<Key>Return", 
			   return_user_defined_item, list_object, TRUE);
	}

	xvw_place(toplevel, NULL);
	return(toplevel);
}

/*------------------------------------------------------------
|
|  Routine Name: select_item()
|
|       Purpose: Gets the selected item from the list object
|                and sets global xvu_list_return.
|
|         Input: object      - the ascii text object
|                client_data - unused
|                call_data   - pointer to list structure containing
|                              the chosen item
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 19, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void select_item(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvw_list_struct *list_return = (xvw_list_struct *) call_data;

	/* 
	 * allocate xvu_list_return, and set values to the xvw_list_struct 
         * passed in as call_data, indicating the user's selection
	 */
	xvu_list_return = (xvw_list_struct *)
                           kcalloc(1, sizeof(xvw_list_struct));
	xvu_list_return->list_index = list_return->list_index;
	xvu_list_return->string = kstrdup(list_return->string);
	xvu_list_done = TRUE;
}

/*------------------------------------------------------------
|
|  Routine Name: return_user_defined_item()
|
|       Purpose: When user defined items are allowed in the
|                list, this is the action handler that returns
|                the list item defined by the user.
|
|         Input: object      - the ascii text object
|                client_data - not used
|                call_data   - not used
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 19, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void return_user_defined_item(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	char           *string_return;

	xvw_get_attribute(object, XVW_TEXT_STRING, &string_return);

	/*
	 *  simply make up a list_return structure
	 */
	xvu_list_return = (xvw_list_struct *) 
			   kcalloc(1, sizeof(xvw_list_struct));

	/*
	 *  index set to -1, indicating that this was not originally
         *  a member of the list;  string set to whatever the user entered.
	 */
	xvu_list_return->list_index = -1;
	xvu_list_return->string = kstrdup(string_return);
	xvu_list_done = TRUE;
}

/*------------------------------------------------------------
|
|  Routine Name: cancel_list()
|
|       Purpose: Cancels list object by by setting global variable
|                xvu_list_done (but xvu_list_return is still NULL).
|
|         Input: object      - the "Cancel" button
|                client_data - unused
|                call_data   - unused
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 19, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void cancel_list(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvu_list_done   = TRUE;
	xvu_list_return = NULL;
}
