 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		        Exit xvobject                           <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                xvu_create_exit()		      <<<<              
   >>>>   Static:                                             <<<<
   >>>>                xvu_exit_yes()		      	      <<<<              
   >>>>                xvu_exit_no()		      	      <<<<              
   >>>>                xvu_exit_canc()		      	      <<<<              
   >>>>   Public:                                             <<<<
   >>>>                xvu_exit_wait()			      <<<<              
   >>>>                                                       <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	

static int xvu_exit_count = 0;
static int xvu_exit_done  = FALSE;
static int xvu_exit_sig;

static void xvu_exit_yes  PROTO((xvobject, kaddr, kaddr));
static void xvu_exit_no   PROTO((xvobject, kaddr, kaddr));
static void xvu_exit_canc PROTO((xvobject, kaddr, kaddr));

/************************************************************
*
*  Routine Name: xvu_exit_wait - pop up an exit message (3 buttons); wait for response
*
*       Purpose: Creates and maps a pop-up exit object which has
*                three buttons representing three choices. The user must
*                click on one of the buttons;  a status representing
*                which of the three buttons was chosen will be returned
*                to the application.  The user must click on one of the
*                three buttons before control will be returned to the
*                application program.
*
*        Input:   stmt       - string containing instructions for the user.
*                 label      - short label for top of exit object;
*                              NULL will give default of "Exiting".
*		  yes_label  - label for first button
*                              NULL will give default of "Yes".
*		  no_label   - label for first button
*                              NULL will give default of "No".
*		  canc_label - label for first button
*                              NULL will give use default of "Cancel".
*
*        Output:
*	Returns:       2 if user selected yes_label  (Yes), 
*		       1 if user selected no_label   (No), 
*		       0 if user selected canc_label (Cancel).  
*
*  Restrictions: 
*    Written By: Danielle Argiro & Stephanie Hallett
*          Date: Jul 9, 1992 14:23
*      Verified:
*  Side Effects:
* Modifications: Converted from Khoros 1.0 (DA)
*
*************************************************************/
int xvu_exit_wait(
   char *stmt,
   char *label,
   char *yes_label,
   char *no_label,
   char *canc_label)
{
	xvu_exit_done = FALSE;

        /*
         *  xvu_exit_wait supports non-X displays
         */
	if (xvw_display(NULL) == NULL)
	{
	    kinfo(KSTANDARD, "%s\n", stmt);
	    return(TRUE);
	}

	if (!(xvu_create_exit(stmt,label,yes_label,no_label,canc_label)))
	    return(FALSE);

	while(!xvu_exit_done)
	    xvw_process_event(); 

	return(xvu_exit_sig);
}


/*------------------------------------------------------------
|
|  Routine Name: xvu_create_exit()
|
|       Purpose: Creates the exit object.
|
|         Input: exit_string   - the exit message 
|                exit_label    - short label for top of object
|                                NULL will give default of "Exiting"
|		 yes_label     - label for acknowledgement button
|                                NULL will give default of "Yes"
|		 no_label      - label for acknowledgement button
|                                NULL will give default of "No"
|		 canc_label    - label for acknowledgement button
|                                NULL will give default of "Cancel"
|
|        Output: Returns TRUE on success, FALSE on failure
|          Date: July 9, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/

#define TEXT_WIDTH  37
#define TEXT_HEIGHT 7

int xvu_create_exit(
   char *exit_string,
   char *exit_label,
   char *yes_label,
   char *no_label,
   char *canc_label)
{
	xvobject toplevel;	/* toplevel object for message */
	xvobject back;		/* backplane for exit object */
	xvobject label;	        /* label describing exit message */
	xvobject mesg;	        /* text object containing exit message */
	xvobject yes_wid; 	/* button to select to go on (1) */
	xvobject no_wid; 	/* button to select to stop action (0) */
	xvobject canc_wid; 	/* button to select to cancel action (-1) */
	float  width, height, yes_width, no_width, canc_width, label_width;
	char   name[KLENGTH];

	/*
	 * create identifying object name
	 */
	ksprintf(name, "%d_exit", xvu_exit_count++);

	/*
	 * create the exit box's toplevel object, and add it
	 * to the list of toplevels that is used with journal playback.
	 */
        toplevel = xvw_create_transient_shell(name, NULL, NULL);

        /*
         * create the backplane object
         */
        back = xvw_create_manager(toplevel, "back");

	if (exit_label == NULL)
	    exit_label = kstring_copy("Exiting:", NULL);
	label_width = (float) (kstrlen(exit_label)+1);

	if (yes_label == NULL)
	    yes_label = kstring_copy("Yes", NULL);
	yes_width = (float) (kstrlen(yes_label)+1);

	if (no_label == NULL)
	    no_label = kstring_copy("No", NULL);
	no_width = (float) (kstrlen(no_label)+1);

	if (canc_label == NULL)
	    canc_label = kstring_copy("Cancel", NULL);
	canc_width = (float) (kstrlen(canc_label)+1);

	width =  (float) (kmax(TEXT_WIDTH, 
		  (int) (yes_width + no_width + canc_width + label_width)));
	/* 
	 * create the label object 
	 */
        label = xvw_create_label(back, "label");
        xvw_set_attributes(label,
		XVW_LABEL,         exit_label,         /* label      */
                XVW_BORDER_WIDTH,  0,                  /* no border  */
                XVW_BELOW,         NULL,               /* at top     */
                NULL);

	/*
	 * create "yes" object (returns 2)
	 */
	yes_wid = xvw_create_button(back, "yes");
        xvw_set_attributes(yes_wid,
		XVW_LABEL,           yes_label, /* button label  */
		XVW_RIGHT_OF,        label,   	/* R of label */
		XVW_BELOW,           NULL,   	/* at top    */
		NULL);

	xvw_add_callback(yes_wid, XVW_BUTTON_SELECT,
		  	 xvu_exit_yes, toplevel);
	/*
	 * create "no" object (returns 1)
	 */
	no_wid = xvw_create_button(back, "no");
        xvw_set_attributes(no_wid,
		XVW_LABEL,       no_label,      /* button label  */
		XVW_RIGHT_OF,    yes_wid, 	/* R of "yes"    */
		XVW_BELOW,     	 NULL,  	/* at top    */
		NULL);
	xvw_add_callback(no_wid, XVW_BUTTON_SELECT, 
		 	 xvu_exit_no, toplevel);

	/*
	 * create "cancel" object (returns 0)
	 */
	canc_wid = xvw_create_button(back, "canc");
        xvw_set_attributes(canc_wid,
		XVW_LABEL,       canc_label,    /* button label */
		XVW_RIGHT_OF,    no_wid,  	/* far right */
		XVW_BELOW,     	 NULL,  	/* at top    */
                NULL);

	xvw_add_callback(canc_wid, XVW_BUTTON_SELECT,
			 xvu_exit_canc, toplevel);


	/*
	 * create the text object w/ exit mesg
	 */
	height = (float) kmax(TEXT_HEIGHT, 
			     xvu_figure_height(exit_string, (int) width));
	mesg = xvw_create_text(back, "exit_mesg");
        xvw_set_attributes(mesg,
		XVW_TEXT_MULTILINE,   TRUE,	     /* multi-line    */
                XVW_TEXT_STRING,      exit_string,   /* text          */
                XVW_CHAR_WIDTH,       width,         /* set width     */
                XVW_CHAR_HEIGHT,      height,        /* set height    */
                XVW_BELOW,            no_wid,        /* below "no"    */
                XVW_TEXT_EDIT_TYPE,   KTEXT_READ, /* read-only     */
                XVW_TEXT_WRAP,        KTEXT_WRAP_WORD, /* wrap on word  */
                NULL);

        /*
         *  map object & return
         */
	xvw_add_protocol(toplevel, "WM_DELETE_WINDOW", xvu_exit_canc,
                         toplevel);
	xvw_place(toplevel, NULL);
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvu_exit_yes()
|
|       Purpose: Destroys exit object when the user clicks on
|                the "yes" button; returns status of 2.
|
|         Input: object      - the "Yes" acknowledgement button
|                client_data - stores the toplevel object
|                call_data   - unused
|        Output: none
|          Date: Jul 9, 1992         
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_exit_yes(
   xvobject object,
   kaddr  client_data,
   kaddr  call_data)
{
	xvobject toplevel = (xvobject) client_data;

	xvw_unmap(toplevel);
	xvw_destroy(toplevel);
	xvu_exit_done = TRUE;
	xvu_exit_sig = 2;
}

/*------------------------------------------------------------
|
|  Routine Name: xvu_exit_no()
|
|       Purpose: Destroys exit object when the user clicks on
|                the "no" button; returns status of 1.
|
|         Input: object      - the "No" acknowledgement button
|                client_data - stores the toplevel object
|                call_data   - unused
|
|        Output: none
|          Date: Jul 9, 1992 
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_exit_no(
   xvobject object,
   kaddr  client_data,
   kaddr  call_data)
{
	xvobject toplevel = (xvobject) client_data;

	xvw_unmap(toplevel);
	xvw_destroy(toplevel);
	xvu_exit_done = TRUE;
	xvu_exit_sig = 1;
}

/*------------------------------------------------------------
|
|  Routine Name: xvu_exit_canc()
|
|       Purpose: Destroys exit object when the user clicks on
|                the "Cancel" button; returns status of 0.
|
|         Input: object      - the "Cancel" acknowledgement button
|                client_data - toplevel object
|                call_data   - unused
|        Output: none
|          Date: July 9, 1992 
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_exit_canc(
   xvobject object,
   kaddr  client_data,
   kaddr  call_data)
{
	xvobject toplevel = (xvobject) client_data;

	xvw_remove_protocol(toplevel, "WM_DELETE_WINDOW", 
			    xvu_exit_canc, toplevel);
	xvw_unmap(toplevel);
	xvw_destroy(toplevel);
	xvu_exit_done = TRUE;
	xvu_exit_sig = 0;
}
