 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>	              Khoros Viewport Widget
   >>>>  Private:
   >>>>   Static:
   >>>> 		ClassInitialize()
   >>>> 		Initialize()
   >>>> 		Realize()
   >>>> 		Resize()
   >>>> 		ConstraintInitialize()
   >>>> 		RecomputeScrollbarPlacement()
   >>>> 		VerticalCallback()
   >>>> 		HorizontalCallback()
   >>>> 		ResizeHandler()
   >>>>   Public:
   >>>>			xvw_create_viewport()
   >>>>	
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvobjects/ViewportP.h>

static void     ClassInitialize PROTO((void));
static void	Initialize      PROTO((Widget, Widget, ArgList, Cardinal *));
static void	Resize          PROTO((Widget));
static void	Realize         PROTO((Widget, XtValueMask *, 
                                       XSetWindowAttributes *));
static Boolean	SetValues       PROTO((Widget, Widget, Widget, ArgList,
				       Cardinal *));
static void ConstraintInitialize PROTO((Widget, Widget, ArgList, Cardinal *));
 
/*
 *  Declaration of scrolling routines that will actually move the plane
 *  window.
 */
static void	RecomputeScrollbarPlacement PROTO((Widget));
static void	VerticalCallback          PROTO((xvobject, kaddr, kaddr));
static void	HorizontalCallback        PROTO((xvobject, kaddr, kaddr));
static void	ResizeHandler	    	    PROTO((xvobject, kaddr, XEvent *));


/*-------------------------------------------------------------------*
|
|   Full class attributes
|
--------------------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_VP_ALLOW_VERT,           NULL,   XtRInt,      XtRBoolean},
{XVW_VP_ALLOW_HORIZ,          NULL,   XtRInt,      XtRBoolean},
{XVW_VP_FORCE_VERT,           NULL,   XtRInt,      XtRBoolean},
{XVW_VP_FORCE_HORIZ,          NULL,   XtRInt,      XtRBoolean},
{XVW_VP_USE_BOTTOM,           NULL,   XtRInt,      XtRBoolean},
{XVW_VP_USE_RIGHT,            NULL,   XtRInt,      XtRBoolean},
{XVW_VP_XOFFSET,              NULL,   XtRInt,      XtRPosition},
{XVW_VP_YOFFSET,              NULL,   XtRInt,      XtRPosition},
{XVW_VP_HORIZONTAL_SCROLLBAR, NULL,   XtRObject,   NULL},
{XVW_VP_VERTICAL_SCROLLBAR,   NULL,   XtRObject,   NULL},
{XVW_VP_CLIP_OBJECT,	      NULL,   XtRObject,   NULL},
{XVW_VP_PLANE_OBJECT,	      NULL,   XtRObject,   NULL},

{XVW_CANVAS_GRID,	"vpPlaneObject.canvasGrid",     XtRInt,   NULL},
{XVW_CANVAS_GRIDSIZE,	"vpPlaneObject.canvasGridsize", XtRInt,   NULL},
{XVW_CANVAS_PIXEL,	"vpPlaneObject.canvasPixel",    XtRPixel, NULL},
/*
{XVW_CHILDREN,	        "vpPlaneObject.children",    XtRPointer,  NULL},
{XVW_NUM_CHILDREN,      "vpPlaneObject.numChildren", XtRPointer,  NULL},
*/
{XVW_FOREGROUND_COLOR,  "vpPlaneObject.foreground",  XtRString,   XtRPixel},
{XVW_BACKGROUND_COLOR,  "vpPlaneObject.background",  XtRString,   XtRPixel},
{XVW_FOREGROUND_PIXEL,  "vpPlaneObject.foreground",  XtRPixel,    NULL},
{XVW_BACKGROUND_PIXEL,  "vpPlaneObject.background",  XtRPixel,    NULL},
{XVW_CURSORNAME,        "vpPlaneObject.cursor",      XtRString,	  XtRCursor},
{XVW_CURSOR,            "vpPlaneObject.cursor",      XtRCursor,   NULL},
{XVW_BACKGROUND_PIXMAP,	"vpPlaneObject.backgroundPixmap",
		 XtRPixmap,   NULL},
{XVW_BACKGROUND_PIXMAPFILE,"vpPlaneObject.backgroundPixmap",
		XtRFilename, XtRPixmap},
};


/*-------------------------------------------------------------------*
|
|   Full class record constant
|
--------------------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwViewportWidgetRec, viewport.field)

static XtResource resources[] = { 
{XVW_VP_ALLOW_VERT, NULL, XtRBoolean, sizeof(Boolean),
        offset(allow_vertical), XtRImmediate, (XtPointer) TRUE},
{XVW_VP_ALLOW_HORIZ, NULL, XtRBoolean, sizeof(Boolean),
        offset(allow_horizontal), XtRImmediate, (XtPointer) FALSE},
{XVW_VP_FORCE_VERT, NULL, XtRBoolean, sizeof(Boolean),
        offset(force_vertical), XtRImmediate, (XtPointer) TRUE},
{XVW_VP_FORCE_HORIZ, NULL, XtRBoolean, sizeof(Boolean),
        offset(force_horizontal), XtRImmediate, (XtPointer) FALSE},
{XVW_VP_USE_BOTTOM, NULL, XtRBoolean, sizeof(Boolean),
        offset(use_bottom), XtRImmediate, (XtPointer) TRUE},
{XVW_VP_USE_RIGHT, NULL, XtRBoolean, sizeof(Boolean),
        offset(use_right), XtRImmediate, (XtPointer) FALSE},
{XVW_VP_XOFFSET, NULL, XtRPosition, sizeof(Position),
        offset(xoffset), XtRImmediate, 0},
{XVW_VP_YOFFSET, NULL, XtRPosition, sizeof(Position),
        offset(yoffset), XtRImmediate, 0},
{XVW_VP_VERTICAL_SCROLLBAR, NULL, XtRObject, sizeof(xvobject),
        offset(vertical), XtRImmediate, (XtPointer) NULL},
{XVW_VP_HORIZONTAL_SCROLLBAR, NULL, XtRObject, sizeof(xvobject),
        offset(horizontal), XtRImmediate, (XtPointer) NULL},
{XVW_VP_CLIP_OBJECT, NULL, XtRObject, sizeof(xvobject),
        offset(clip), XtRImmediate, (XtPointer) NULL},
{XVW_VP_PLANE_OBJECT, NULL, XtRObject, sizeof(xvobject),
        offset(plane), XtRImmediate, (XtPointer) NULL},
{XVW_DEF_HORIZ_DIST, NULL, XtRPosition, sizeof(Position),
        XtOffsetOf(XvwViewportWidgetRec, manager.horiz_offset),
        XtRImmediate, (XtPointer) 0},
{XVW_DEF_VERT_DIST, NULL, XtRPosition, sizeof(Position),
        XtOffsetOf(XvwViewportWidgetRec, manager.vert_offset),
        XtRImmediate, (XtPointer) 0},
};
#undef offset


/*-------------------------------------------------------------------*
|
|   Class Declaration for Viewport Widget
|
--------------------------------------------------------------------*/

#define superclass (&xvwManagerWidgetClassRec)

XvwViewportWidgetClassRec xvwViewportWidgetClassRec =
{
  {
    (WidgetClass) superclass,		/* superclass		  */	
    "Viewport",				/* class_name		  */
    sizeof(XvwViewportWidgetRec),	/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    Realize,				/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    TRUE,				/* compress_motion	  */
    XtExposeCompressMaximal,		/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    NULL,				/* destroy		  */
    NULL,				/* resize		  */
    XtInheritExpose,			/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    NULL,				/* tm_table		  */
    XtInheritQueryGeometry,		/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* CoreClass fields initialization */
  {
    NULL,                    		/* geometry_manager	  */
    XtInheritChangeManaged,		/* change_managed	  */
    XtInheritInsertChild,		/* insert_child	  	  */
    XtInheritDeleteChild,		/* delete_child	  	  */
    NULL,				/* extension	 	  */
  },  /* CompositeClass fields initialization */
  {
    NULL,				    /* subresources	  */
    0,					    /* subresources_count */
    sizeof(XvwViewportWidgetConstraintsRec),/* constraint_size	  */
    ConstraintInitialize,		    /* initialize	  */
    NULL,				    /* destroy		  */
    NULL,				    /* set_values	  */
    NULL,				    /* extension	  */
  },  /* ConstraintClass fields initialization */
  {
    XtInheritLayout,                        /* child layout routine  */
    XtInheritChangeSel,			    /* change selected proc   */
    XtInheritEraseSel,			    /* erase selected proc    */
    XtInheritRefreshSel,		    /* refresh selection proc */
    Resize,				    /* resize		      */
    XtInheritGeometryManager,		    /* geometry_manager	      */
  },  /* XvwManagerWidgetClass fields initialization */
  {
    NULL,  		                    /* extension          */
  },  /* XvwViewportWidgetClass fields initialization */
};
#undef superclass

  /* for public consumption */
WidgetClass xvwViewportWidgetClass = (WidgetClass) &xvwViewportWidgetClassRec;


/*-------------------------------------------------------------------*
|
|   Miscelleanous defines for Class and Constraint Declarations
|
--------------------------------------------------------------------*/

#undef kwidget
#undef kwidgetclass
#undef kconstraint

#define kwidget(widget)	    (XvwViewportWidget) (widget)
#define kwidgetclass(widget) (XvwViewportWidgetClass) (widget)->core.widget_class
#define kconstraint(widget)  (XvwViewportWidgetConstraints) (widget)->core.constraints


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an
|                instance of a XvwViewportWidget class has been created.
|                This is where the attributes for the class are
|		 initialized. 
|
|         Input: None
|
|        Output: None
|       Returns: None 
|
|    Written By: Mark Young
|          Date: Oct 19, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ClassInitialize(void)
{
	xvw_init_attributes(xvwViewportWidgetClass, attributes,
		XtNumber(attributes), NULL, 0, NULL);
	xvw_load_resources("$DESIGN/objects/library/xvobjects/app-defaults/Viewport");
}

/*-----------------------------------------------------------
|
|  Routine Name: Initialize
|
|       Purpose: This method will create all the subparts to the
|		 viewport widget.  They consist of a horizontal &
|		 vertical scrollbar, a clip window, and a
| 	 	 plane widget in which we want all future children 
|		 to be placed.  It will also add callbacks to the
|		 scrollbars allowing them to move the plane widget.
|	
|	         Warning:  Child must be created absolutely last since 
|		           all future creation of widgets will be 
|			   re-mapped into that of the plane rather 
|			   than the viewport widget itself.  This
|	                   allows us to by-pass the re-parenting of any 
|			   viewport children.
|
|         Input: request  - the requested viewport
|		 new      - the viewport widget 
|                args     - not used 
|                nun_args - not used 
|    Written By: Mark Young
|          Date: Jan 29, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Initialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwViewportWidget	xwid   = kwidget(new);
	XvwViewportWidgetClass  class  = kwidgetclass(new);
	xvobject		object = xvw_object(new);

	/*
	 * The scrollbars must be create first otherwise they will be remapped
	 * into plane object instead of the viewport.
	 */
	xwid->viewport.override   = TRUE;
	xwid->viewport.top        = NULL;
	xwid->viewport.left       = NULL;
	xwid->viewport.right      = NULL;
	xwid->viewport.bottom     = NULL;
	xwid->viewport.clip       = xvw_create_manager(object, "clip");
	xwid->viewport.plane      = xvw_create_manager(xwid->viewport.clip,
						       "plane");
	xwid->viewport.horizontal = xvw_create_scrollbar(object, "horizontal");
	xwid->viewport.vertical   = xvw_create_scrollbar(object, "vertical");

	/*
	 *  Make sure the vertical scrollbar extends the height of the
	 *  viewport widget.
	 */
	xvw_set_attributes(xwid->viewport.vertical,
			   XVW_ORIENTATION,  KSCROLLBAR_ORIENT_VERT,
			   XVW_TACK_EDGE,    KMANAGER_TACK_VERT,
			   XVW_WIDTH,	     16,
			   XVW_MAP_WHEN_MANAGED, FALSE,
			   XVW_SCROLL_VALUE, 0.0,
			   XVW_SCROLL_MIN,   0.0,
			   XVW_SCROLL_MAX,   1.0,
		   	   NULL);

	/*
	 *  Make sure the horizontal scrollbar extends the length of the
	 *  viewport widget.
	 */
	xvw_set_attributes(xwid->viewport.horizontal,
			   XVW_ORIENTATION, KSCROLLBAR_ORIENT_HORIZ,
			   XVW_TACK_EDGE,   KMANAGER_TACK_HORIZ,
			   XVW_HEIGHT,	    16,
			   XVW_MAP_WHEN_MANAGED, FALSE,
			   XVW_SCROLL_VALUE, 0.0,
			   XVW_SCROLL_MIN,   0.0,
			   XVW_SCROLL_MAX,   1.0,
			   NULL);

	/*
	 *  Make sure that the clipping widget extends the viewport area
	 *  left-over by the scrollbars
	 */
	xvw_set_attributes(xwid->viewport.clip,
			   XVW_TACK_EDGE,     KMANAGER_TACK_ALL,
			   XVW_BORDER_WIDTH,  0,
			   XVW_DEF_HORIZ_DIST,0,
			   XVW_DEF_VERT_DIST, 0,
			   XVW_HORIZ_DIST,    0,
			   XVW_VERT_DIST,     0, 
			   NULL);

	xvw_set_attributes(xwid->viewport.plane,
			   XVW_XPOSITION,	  0,
			   XVW_YPOSITION,	  0,
			   XVW_BORDER_WIDTH,	  0,
			   NULL);

	/*
	 *  Add the callbacks for the vertical and horizontal scrollbars that
	 *  will be used to move the plane widget.
	 */
	xvw_insert_callback(xwid->viewport.vertical, XVW_SCROLL_CONT_MOTION,
			FALSE, VerticalCallback, (XtPointer) xwid);
	xvw_insert_callback(xwid->viewport.vertical, XVW_SCROLL_INCR_MOTION,
			FALSE, VerticalCallback, (XtPointer) xwid);
	xvw_insert_callback(xwid->viewport.horizontal, XVW_SCROLL_CONT_MOTION,
			FALSE, HorizontalCallback, (XtPointer) xwid);
	xvw_insert_callback(xwid->viewport.horizontal, XVW_SCROLL_INCR_MOTION,
			FALSE, HorizontalCallback, (XtPointer) xwid);

	xvw_add_event(xwid->viewport.plane, StructureNotifyMask,
			ResizeHandler, (XtPointer) xwid);
	xvw_add_event(xwid->viewport.clip, StructureNotifyMask,
			ResizeHandler, (XtPointer) xwid);

	/*
	 *  This routine will position the scrollbars according to whether
	 *  they should be located on the top, left, right or bottom edge.
	 *  The routine will also unmap scrollbars that are no longer
	 *  needed.
	 */
	RecomputeScrollbarPlacement(new);

	xvw_set_attributes(xwid->viewport.plane,
		XVW_BACKGROUND_PIXMAP, xwid->core.background_pixmap,
		XVW_BACKGROUND,	       xwid->core.background_pixel,
		XVW_FOREGROUND,	       xwid->manager.foreground_pixel,
		XVW_CANVAS_GRID,       xwid->manager.gridtype,
		XVW_CANVAS_GRIDSIZE,   xwid->manager.gridsize,
		XVW_CANVAS_PIXEL,      xwid->manager.grid_pixel,
		NULL);
	xwid->core.background_pixmap = XtUnspecifiedPixmap;
	xwid->viewport.override = FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: Realize
|
|       Purpose: This method will set up the window attributes
|                and create a window for the XvwViewportWidget. 
|                The desired initial window attributes are passed 
|                in via a mask and list of specification attributes.
|		 The gc is also initialized here.
|
|         Input: widget     - the widget to realize
|                valuemask  - determines the window attributes being passed in
|                attributes - the list of window attributes to be set
|    Written By: Mark Young
|          Date: Oct 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Realize(
   Widget               widget,
   XtValueMask          *valuemask,
   XSetWindowAttributes *winattrib)
{
        XvwViewportWidget xwid = kwidget(widget);

	(*xvwViewportWidgetClass->core_class.superclass->core_class.realize)
		(widget, valuemask, winattrib);
	RecomputeScrollbarPlacement(widget);
}

/*-----------------------------------------------------------
|
|  Routine Name: ConstraintInitialize
|
|       Purpose: When a new child of the viewport is created,
|		 tell the child who its parent is. 
|                
|         Input: request  - widget child with requested constraints
|                new      - widget child with actual constraints
|		 args     - not used
|		 num_args - not used
|    Written By: Mark Young
|          Date: Oct 12, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ConstraintInitialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num)
{
	XvwViewportWidget xwid = kwidget(request->core.parent);


	if (xwid->viewport.plane != NULL &&
            xvw_widget(xwid->viewport.plane) != new &&
	    xwid->viewport.override == FALSE &&
	    XtClass(new) != InputOnlyWidgetClass)
	{
	   new->core.parent = xvw_widget(xwid->viewport.plane);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: This method is used to set the public values
|                of a XvwViewportWidget instance.  The public values
|                which can be changed are all related to the scrollbars. 
|
|         Input: current - the widget containing current settings
|                request - the widget containing requested settings
|                new     - the widget processed through all set values methods
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues (
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num)
{
	XvwViewportWidget cxwid = kwidget(current);
	XvwViewportWidget nxwid = kwidget(new);


	if (cxwid->viewport.allow_vertical != nxwid->viewport.allow_vertical  ||
	   cxwid->viewport.allow_horizontal!=nxwid->viewport.allow_horizontal ||
	   cxwid->viewport.force_vertical != nxwid->viewport.force_vertical   ||
	   cxwid->viewport.force_horizontal!=nxwid->viewport.force_horizontal ||
	   cxwid->viewport.use_bottom != nxwid->viewport.use_bottom           ||
	   cxwid->viewport.use_right != nxwid->viewport.use_right	      ||
	   cxwid->viewport.xoffset != nxwid->viewport.xoffset		      ||
	   cxwid->viewport.yoffset != nxwid->viewport.yoffset		      ||
	   cxwid->core.border_width != nxwid->core.border_width)
	{
	   RecomputeScrollbarPlacement(new);
	}
	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: Resize
|
|       Purpose:
|
|         Input: 
|    Written By: Mark Young
|          Date: Aug 19, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Resize(
   Widget widget)
{
	XvwViewportWidget xwid = kwidget(widget);

	RecomputeScrollbarPlacement(widget);
	(*xvwViewportWidgetClass->core_class.superclass->core_class.resize)
                (widget);
}

/*-----------------------------------------------------------
|
|  Routine Name: RecomputeScrollbarPlacement 
|
|       Purpose: This routine will calculate the location
|		 of the scrollbars on the viewport widget. 
|                
|         Input: widget - the viewport widget with the scrollbars
|    Written By: Mark Young
|          Date: Oct 10, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RecomputeScrollbarPlacement(
   Widget widget)
{
	XvwViewportWidget xwid = kwidget(widget);

	int offset = 0;
	unsigned int bw;
	int width  = xwid->core.width;
	int height = xwid->core.height;
	double position, total, length;

	xvobject vertical = xwid->viewport.vertical;
	xvobject horizontal = xwid->viewport.horizontal;
	Widget	 clip = xvw_widget(xwid->viewport.clip);
	Widget	 plane = xvw_widget(xwid->viewport.plane);
	xvobject top    = xwid->viewport.top;
	xvobject left   = xwid->viewport.left;
	xvobject right  = xwid->viewport.right;
	xvobject bottom = xwid->viewport.bottom;


	/*
	 *  Make sure that the horizontal and vertical scrollbars should be
	 *  visible.
	 */
	if ((plane->core.width <= clip->core.width &&
	     xwid->viewport.force_horizontal == FALSE)  ||
	    xwid->viewport.allow_horizontal == FALSE)
	{
	   xvw_unmap(horizontal);
	   horizontal = NULL;
	}
	else
	   xvw_map(horizontal);

	if ((plane->core.height <= clip->core.height &&
	     xwid->viewport.force_vertical == FALSE)  ||
	    xwid->viewport.allow_vertical == FALSE)
	{
	   xvw_unmap(vertical);
	   vertical = NULL;
	}
	else
	   xvw_map(vertical);

	/*
	 *  Force the clipping widget to use the viewport area left-over
	 *  from the scrollbars.
	 */
	if (xwid->viewport.use_bottom)
	   xvw_set_attributes(xwid->viewport.clip,
                           XVW_ABOVE,  horizontal,
                           XVW_BELOW,  top,
                           NULL);
	else
	   xvw_set_attributes(xwid->viewport.clip,
                           XVW_ABOVE,  bottom,
                           XVW_BELOW,  horizontal,
                           NULL);

	if (xwid->viewport.use_right)
	   xvw_set_attributes(xwid->viewport.clip,
                           XVW_RIGHT_OF, left,
                           XVW_LEFT_OF,  vertical,
                           NULL);
	else
	   xvw_set_attributes(xwid->viewport.clip,
                           XVW_RIGHT_OF, vertical,
                           XVW_LEFT_OF,  right,
                           NULL);

	/*
	 *  See if the vertical scrollbar is to use the right or left
	 *  edge of the viewport widget.
	 *
	 *  Make sure the vertical scrollbar extends the height of the
	 *  viewport widget.
	 */
	if (xwid->viewport.use_right)
	{
	   xvw_set_attributes(xwid->viewport.vertical,
	   		   XVW_RIGHT_OF, KMANAGER_UNDEFINED,
	   		   XVW_LEFT_OF,  right,
		   	   NULL);

	   xvw_set_attributes(xwid->viewport.horizontal,
			   XVW_LEFT_OF, vertical,
			   XVW_RIGHT_OF, left,
		   	   NULL);
	}
	else
	{
	   xvw_set_attributes(xwid->viewport.vertical,
	   		   XVW_RIGHT_OF, left,
	   		   XVW_LEFT_OF,  KMANAGER_UNDEFINED,
		   	   NULL);

	   xvw_set_attributes(xwid->viewport.horizontal,
			   XVW_RIGHT_OF, vertical,
			   XVW_LEFT_OF, right,
		   	   NULL);
	}

	/*
	 *  See if the horizontal scrollbar is to use the top or bottom
	 *  edge of the viewport widget.
	 *
	 *  Make sure the horizontal scrollbar extends the length of the
	 *  viewport widget.
	 */
	if (xwid->viewport.use_bottom)
	{
	   xvw_set_attributes(xwid->viewport.horizontal,
	   		   XVW_BELOW, KMANAGER_UNDEFINED,
	   		   XVW_ABOVE, bottom,
		   	   NULL);

	   xvw_set_attributes(xwid->viewport.vertical,
	   		   XVW_BELOW, top,
	   		   XVW_ABOVE, horizontal,
		   	   NULL);
	}
	else
	{
	   xvw_set_attributes(xwid->viewport.horizontal,
	   		   XVW_BELOW, top,
	   		   XVW_ABOVE, KMANAGER_UNDEFINED,
		   	   NULL);
	   xvw_set_attributes(xwid->viewport.vertical,
	   		   XVW_BELOW, horizontal,
	   		   XVW_ABOVE, bottom,
		   	   NULL);
	}

	/*
	 *  According to our xoffset horizontally position the scrollbar
	 *  and scrollbar increment.
	 */
	total = plane->core.width; length = clip->core.width;
	if (total < length) total = length;
	
	if ((double) xwid->viewport.xoffset+length > total)
	   xwid->viewport.xoffset = total - length;

	xvw_geometry(xwid->viewport.horizontal, NULL, NULL, NULL, NULL, &bw);
	bw = kmin(bw, xwid->core.border_width); offset = (bw > 0) ? -bw : 0;

	if (total <= length) 
	   position = 0.0;
	else
	   position = xwid->viewport.yoffset/(total-length);

	xwid->viewport.xscale = length/total;
	xvw_set_attributes(xwid->viewport.horizontal,
		XVW_SCROLL_VALUE, position,
		XVW_SCROLL_INCR,  xwid->viewport.xscale,
		XVW_HORIZ_DIST,   offset,
		XVW_VERT_DIST,    offset,
		NULL);

	/*
	 *  According to our yoffset vertically position the scrollbar
	 *  and scrollbar increment.
	 */
	total = plane->core.height; length = clip->core.height;
	if (total < length) total = length;

	if ((double) xwid->viewport.yoffset+length > total)
	   xwid->viewport.yoffset = total - length;

	xvw_geometry(xwid->viewport.vertical, NULL, NULL, NULL, NULL, &bw);
	bw = kmin(bw, xwid->core.border_width); offset = (bw > 0) ? -bw : 0;

	if (total <= length) 
	   position = 0.0;
	else
	   position = xwid->viewport.yoffset/(total-length);

	xwid->viewport.yscale = length/total;
	xvw_set_attributes(xwid->viewport.vertical,
		XVW_SCROLL_VALUE, position,
		XVW_SCROLL_INCR,  xwid->viewport.yscale,
		XVW_HORIZ_DIST,   offset,
		XVW_VERT_DIST,    offset, 
		NULL);

	/*
	 *  According to our xoffset & yoffset position the plane.
	 */
	XtMoveWidget(plane, -xwid->viewport.xoffset, -xwid->viewport.yoffset);
}

/*-----------------------------------------------------------
|
|  Routine Name: VerticalCallback
|
|       Purpose: This routine will move the plane widget vertically 
|		 by an amount proportional to the movement of the 
|                viewport's vertical scrollbar.
|
|         Input: object      - not used 
|		 client_data - the viewport widget
|		 call_data   - float representing the scroll position
|    Written By: Mark Young 
|          Date: Oct 10, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void VerticalCallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	double position = *(double *) call_data;
	XvwViewportWidget xwid = kwidget(client_data);

	Position ypos;
	Widget clip  = xvw_widget(xwid->viewport.clip);
	Widget plane = xvw_widget(xwid->viewport.plane);


	/*
	 *  The movement is really a float double represents the percental
	 *  position of the vertical scrollbar.  Therefore we need to move
	 *  the plane widget vertically (height) relative to the clipping
	 *  widget.
	 */
	ypos = ((int) plane->core.height - (int) clip->core.height) * position;
	if (ypos < 0)
	{
	   ypos = 0;
	   xvw_set_attribute(object, XVW_SCROLL_VALUE, 0.0);
	}
	else if (ypos + (int) clip->core.height > (int) plane->core.height)
	{
	   ypos = plane->core.height - ((int) clip->core.height);
	   xvw_set_attribute(object, XVW_SCROLL_VALUE, 1.0);
	}
	xwid->viewport.yoffset = ypos;
	XtMoveWidget(plane, plane->core.x, -ypos);
}

/*-----------------------------------------------------------
|
|  Routine Name: HorizontalCallback
|
|       Purpose: This routine will move the plane widget horizontally 
|                by an amount proportional to the movement of the 
|                viewport's horizontal scrollbar.
|
|         Input: object      - not used 
|                client_data - the viewport widget
|                call_data   - float representing the scroll position
|    Written By: Mark Young 
|          Date: Oct 10, 1992
|
------------------------------------------------------------*/
/* ARGSUSED */
static void HorizontalCallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	double position = *(double *) call_data;
	XvwViewportWidget xwid = kwidget(client_data);

	Position xpos;
	Widget clip  = xvw_widget(xwid->viewport.clip);
	Widget plane = xvw_widget(xwid->viewport.plane);


	/*
	 *  The movement is really a float that represents the percental
	 *  position of the horizontal scrollbar.  Therefore we need to move
	 *  the plane widget horizontally (width) relative to the clipping
	 *  widget.
	 */
	xpos = ((int) plane->core.width - (int) clip->core.width) * position;
	if (xpos < 0)
	{
	   xpos = 0;
	   xvw_set_attribute(object, XVW_SCROLL_VALUE, 0.0);
	}
	else if (xpos + (int) clip->core.width > (int) plane->core.width)
	{
	   xpos = plane->core.width - ((int) clip->core.width);
	   xvw_set_attribute(object, XVW_SCROLL_VALUE, 1.0);
	}
	xwid->viewport.xoffset = xpos;
	XtMoveWidget(plane, -xpos, plane->core.y);
}

/*-----------------------------------------------------------
|
|  Routine Name: ResizeHandler
|
|       Purpose: This routine will call the RecomputeScrollbarPlacement()
|		 routine when the plane has changed size.  It is used to
|		 update whether or not to map the vertical and horizontal
|		 scrollbars, as well as update the length of the scrollbars.
|
|         Input: object      - not used 
|                client_data - the viewport widget
|                event       - the event that caused us to be called
|    Written By: Mark Young 
|          Date: Dec 30, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ResizeHandler(
   xvobject object,
   kaddr    client_data,
   XEvent   *event)
{
	Widget widget = (Widget) client_data;
	XvwViewportWidget xwid = kwidget(client_data);

	Widget plane, clip;
	double total, length, xscale, yscale;
	xvobject vertical   = xwid->viewport.vertical;
	xvobject horizontal = xwid->viewport.horizontal;

	if (event->type == ConfigureNotify)
	{
	   clip  = xvw_widget(xwid->viewport.clip);
	   plane = xvw_widget(xwid->viewport.plane);

	   /*
	    *  Compute the increment value for the horizontal scrollbar.
	    */
	   total = plane->core.width; length = clip->core.width;
	   if (total < length) total = length;

	   if ((double) xwid->viewport.xoffset+length > total)
	      xwid->viewport.xoffset = total - length;

	   xscale = length/total;
	   if (xwid->viewport.xscale != xscale)
	   {
	      xwid->viewport.xscale = xscale;
	      xvw_set_attribute(horizontal, XVW_SCROLL_INCR, xscale);
	      if ((xscale < 1.0  && !xvw_check_mapped(horizontal)) ||
		  (xscale == 1.0 && xvw_check_mapped(horizontal)))
	      {
		 RecomputeScrollbarPlacement(widget);
	      }
	   }

	   /*
	    *  Do the vertical scrollbar.
	    */
	   total = plane->core.height; length = clip->core.height;
	   if (total < length) total = length;

	   if ((double) xwid->viewport.yoffset+length > total)
	      xwid->viewport.yoffset = total - length;

	   yscale = length/total;
	   if (xwid->viewport.yscale != yscale)
	   {
	      xwid->viewport.yscale = yscale;
	      xvw_set_attribute(vertical, XVW_SCROLL_INCR, yscale);
	      if ((yscale < 1.0  && !xvw_check_mapped(vertical)) ||
		  (yscale == 1.0 && xvw_check_mapped(vertical)))
	      {
		 RecomputeScrollbarPlacement(widget);
	      }
	   }
	}
}


/************************************************************
*
*  Routine Name: xvw_create_viewport - create a viewport object
*
*       Purpose: Creates a viewport widget. The \fIviewport\fP 
*                provides a way of restricting a potentially overlarge
*                display to a predefined area, while at the same time
*                allowing its contents (consisting of other GUI 
*                objects) to occupy as much space as necessary.
*
*                The viewport object is made up of four other GUI
*                objects: a "plane" object, a clip object, a vertical
*                scrollbar, and a horizontal scrollbar.
*
*                The plane object component of the viewport object
*                implements a "virtual area", only part of which can
*                be seen in the viewport.  The plane object is the actual 
*                parent of the viewport's children;  in this way, the
*                the object(s) created inside the viewport can take up 
*                more space than the viewport itself. 
*
*                The clip object component of the viewport is what
*                dictates the size of the viewport itself.  By clipping
*                out all portions of the plane object that lie outside
*                the bounds of the viewport size, the contents of the
*                viewport are only visible when they are in the predefined
*                range.
*
*                The viewport contains horizontal and vertical scrollbars;
*                these scrollbars are used to control which portion of the
*                plane object is visible in the viewport.  
*
*         Input: parent - parent of the viewport object; NULL will cause
*                         a default toplevel to be created automatically
*                name   - name with which to reference viewport object
*
*        Output: 
*       Returns: The viewport object on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 14, 1993 16:47
*      Verified: 
*  Side Effects:
* Modifications:
*
*******************************************************************/

xvobject xvw_create_viewport(
   xvobject parent,
   char     *name)
{
	xvobject object;


	object = xvw_create(parent, FALSE, TRUE, name, xvwViewportWidgetClass);
	return(object);
}
