 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>	              Khoros Workspace Widget Utilities
   >>>>   Static:
   >>>>			SaveNode()
   >>>>			SaveGlyph()
   >>>>			SaveConditional()
   >>>>			SaveProcedure()
   >>>>			SaveLoop()
   >>>>			SaveConnections()
   >>>>			SaveExternalConnections()
   >>>>  Private:
   >>>>			WorkspaceSave()
   >>>>   Public:
   >>>>	
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvlang/WorkspaceP.h>
#include <xvlang/GlyphP.h>


/*-----------------------------------------------------------
|
|  Routine Name: LocateExported - located the exported connection
|
|       Purpose: This routine actually returns the exported connection number
|
|         Input: glyph - the glyph to save the connections
|		 ident - the glyph identifier
|		 export - whether external connections should be exported
|		 selections - the other selections that are being saved
|		 num_sels    - the number of selections in the selection array
|        Output: exported - the exported connections
|		 num_exported - the number of exported connections
|		 file  - the file the information is suppose to be written to
|		
|    Written By: Mark Young
|          Date: Nov 12, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int LocateExported(
   xvobject object,
   xvobject *sels,
   int      num_sels,
   xvobject **exported,
   int      *num_exported)
{
	int      num;
	xvobject parent = xvw_parent(object);

	if (karray_locate((char **) sels, (kaddr) parent, num_sels) != -1)
	   return(-1);

	if ((num = karray_locate((char **) *exported, (kaddr) parent,
			*num_exported)) == -1)
	{
	   num = *num_exported;
	   *exported = (xvobject *) karray_add((char **) *exported,
			(kaddr) parent, num);
	   (*num_exported)++;
	}
	return(num);
}

/*-----------------------------------------------------------
|
|  Routine Name: SaveNode - Save the glyph/control object
|
|       Purpose: This routine does the actually saving of a glyph/control to a
|		 workspace file.
|
|         Input: object - the glyph/control to be saved
|		 ident  - the glyph/control identifier
|		 export - whether external connections should be exported
|		 gui_parameters - whether exported parameters should be saved
|		 selections - the other selections that are being saved
|		 num_sels    - the number of selections in the selection array
|        Output: file  - the file the information is suppose to be written to
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void SaveNode(
    xvobject object,
    int	     ident,
    int	     export,
    int	     gui_parameters,
    xvobject *selections,
    int      num_sels,
    kfile    *file,
    char     *type)
{
	XvwGlyphWidget xglyph = (XvwGlyphWidget) xvw_widget(object);

	kform        *form;
	kselection   *selection;
        kform_struct **extensions;
	klist        *list, *tblist = NULL;
	xvobject port, parent, *connections;
	char	 temp[KLENGTH], command[KLENGTH], variables[KLENGTH];
	char     *tbname, *oname, *title, *name, *var, *filename, *tmp, **info;
        int      i, j, num, xpos, ypos, temporary, selected, tbname_token,
		 oname_token;

	xvw_get_attributes(object,
		XVW_GLYPH_TBNAME, &tbname,
		XVW_GLYPH_ONAME,  &oname,
		XVW_GLYPH_FORM,   &form,
		XVW_NODE_NAME,    &title,
		XVW_XPOSITION,    &xpos,
		XVW_YPOSITION,    &ypos,
		NULL);

	if (form != NULL)
	   tmp = kvf_construct_partial_cmd(form, form->subform, NULL, FALSE);
	else
	   tmp = kstrdup("(unknown)");

	variables[0] = '\0';
	kstrcpy(command, tmp);
	for (i = 0; i < xglyph->glyph.isize; i++)
	{
	   port = xglyph->glyph.iports[i];
	   if (!PortGetInfo(port, &var, &filename, &temporary, &selected,
		&num, &connections))
	   {
	      continue;
	   }

	   if (temporary == FALSE && selected && num == 0 && filename != NULL)
	   {
	      ksprintf(temp, " -%s %s", var, filename);
	      kstrcat(command, temp);
	   }

	   for (j = 0; j < num && gui_parameters == TRUE; j++)
	   {
              xvw_get_attribute(connections[j],XVW_CONNECTION_BEGIN,&port);
	      parent = xvw_parent(port);
	      if (karray_locate((char **) selections, (kaddr) parent,
			num_sels) == -1)
	      {
		 kstrcat(variables, " ");
		 kstrcat(variables, var);
/*
	         xvw_copy_selection(xglyph->glyph.wkspgui, NULL, NULL,
				xglyph->glyph.isels[i], NULL);
 */
		 break;
	      }
	   }
	}

	for (i = 0; i < xglyph->glyph.osize; i++)
	{
	   port = xglyph->glyph.oports[i];
	   if (!PortGetInfo(port, &var, &filename, &temporary, NULL,
		&num, &connections))
	   {
	      continue;
	   }

	   if (temporary == FALSE && filename != NULL)
	   {
	      ksprintf(temp, " -%s %s", var, filename);
	      kstrcat(command, temp);
	   }

	   for (j = 0; j < num && gui_parameters == TRUE; j++)
	   {
              xvw_get_attribute(connections[j], XVW_CONNECTION_END, &port);
	      parent = xvw_parent(port);
	      if (karray_locate((char **) selections, (kaddr) parent,
			num_sels) == -1)
	      {
		 kstrcat(variables, " ");
		 kstrcat(variables, var);
/*
	         xvw_copy_selection(xglyph->glyph.wkspgui, NULL, NULL,
				xglyph->glyph.osels[i], NULL);
 */
		 break;
	      }
	   }
	}

	extensions = kvf_get_extensions(form, &num);
	for (i = 0; i < num; i++)
	{
	    selection = extensions[i]->Selptr->extensions[0];
	    xvf_get_attribute(selection->back_kformstruct, XVF_VARIABLE, &var);
	    kstrcat(variables, " ");
	    kstrcat(variables, var); kfree(var);

	    xvf_get_attribute(extensions[i], XVF_VARIABLE, &var);
	    kstrcat(variables, "!");
	    kstrcat(variables, var); kfree(var);
	}
	kfree(extensions);

	/*
	 *  Make sure to initialize the toolbox, object, and icon name
	 */
	xvw_get_contents(NULL, &tblist);
	tbname_token = kstring_to_token(tbname);
	oname_token  = kstring_to_token(oname);
	if ((list = klist_locate(tblist, (kaddr) tbname_token)) != NULL &&
	    (list = (klist *) klist_clientdata(list))           != NULL &&
	    (list = klist_locate(list, (kaddr) oname_token))    != NULL)
	{
	   info = (char **) klist_clientdata(list);
	   if (title != NULL && kstrcmp(title, info[0]) != 0)
	      name = title;
	   else
	      name = "";
	}
	tbname = !tbname ? "(unknown)" : tbname;
	oname  = !oname  ? "(unknown)" : oname;
	title  = !title  ? "(unknown)" : title;

	/*
	 *  Need to make sure that the command doesn't contain new lines.
	 *  If they do then we will want to substitute these with \\n newline
	 *  tokens.
	 */
	kstring_replace(command, "\n", "\\n", command);
	kfprintf(file,"\n#  %s '%s'\n%s:%s:%s:%s:%d:%d:%d:%s:%s\n\n", type,
		title, type, tbname, oname, name, ident, xpos, ypos,
		variables, command);
}

/*-----------------------------------------------------------
|
|  Routine Name: SaveGlyph - Save the glyph
|
|       Purpose: This routine does the actually saving of a glyph to a
|		 workspace file.
|
|         Input: glyph - the glyph to be saved
|		 ident - the glyph identifier
|		 export - whether external connections should be exported
|		 selections - the other selections that are being saved
|		 gui_parameters - whether exported parameters should be saved
|		 num_sels    - the number of selections in the selection array
|        Output: file  - the file the information is suppose to be written to
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void SaveGlyph(
    xvobject glyph,
    int	     ident,
    int	     export,
    int	     gui_parameters,
    xvobject *selections,
    int      num_sels,
    kfile    *file)
{
	SaveNode(glyph, ident, export, gui_parameters, selections,
			num_sels, file, "Glyph");
}

/*-----------------------------------------------------------
|
|  Routine Name: SaveConditional - Save the control glyph
|
|       Purpose: This routine does the actually saving of a glyph to a
|		 workspace file.
|
|         Input: glyph - the glyph to be saved
|		 ident - the glyph identifier
|		 export - whether external connections should be exported
|		 gui_parameters - whether exported parameters should be saved
|		 selections - the other selections that are being saved
|		 num_sels    - the number of selections in the selection array
|        Output: file  - the file the information is suppose to be written to
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void SaveConditional(
    xvobject glyph,
    int	     ident,
    int	     export,
    int	     gui_parameters,
    xvobject *selections,
    int      num_sels,
    kfile    *file)
{
	SaveNode(glyph, ident, export, gui_parameters, selections,
			num_sels, file, "Conditional");
}

/*-----------------------------------------------------------
|
|  Routine Name: SaveProcedure - Save the procedure
|
|       Purpose: This routine does the actually saving of a procedure to a
|		 workspace file.
|
|         Input: procedure  - the glyph to be saved
|		 ident      - the procedure identifier
|		 export - whether external connections should be exported
|		 gui_parameters - whether exported parameters should be saved
|		 selections - the other selections that are being saved
|		 num_sels    - the number of selections in the selection array
|        Output: file  - the file the information is suppose to be written to
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void SaveProcedure(
    xvobject procedure,
    int      ident,
    int      export,
    int	     gui_parameters,
    xvobject *selections,
    int      num_sels,
    kfile    *file)
{
	kform	 *form;
	int      xpos, ypos;
	char     *command, *name;
	xvobject workspace = NULL;

	xvw_get_attributes(procedure,
		XVW_GLYPH_FORM,   &form,
		XVW_NODE_NAME,    &name,
		XVW_XPOSITION,    &xpos,
		XVW_YPOSITION,    &ypos,
		NULL);

	name = !name ? "(unknown)" : name;
	if (form != NULL)
	   command = kvf_form_to_cmd(form, form->subform, NULL, FALSE);
	else
	   command = "(unknown)";

	/*
	 *  Oh No!  It's a procedure, so we need to do the nasty.
	 */
	kfprintf(file,"  '%s'\nProcedureBegin:%s:%d:%d:%d:%s\n\n", name,
			name, ident, xpos, ypos, command);

	xvw_get_attribute(procedure,XVW_PROCEDURE_WORKSPACE_OBJECT,&workspace);
	if (workspace != NULL)
	{
	   WorkspaceSave(xvw_widget(workspace), file, FALSE, FALSE, FALSE,
			FALSE, NULL, NULL);
	}
	kfprintf(file,"ProcedureEnd:%s\n\n", name);
}

/*-----------------------------------------------------------
|
|  Routine Name: SaveLoop - Save the loop
|
|       Purpose: This routine does the actually saving of a loop to a
|		 workspace file.
|
|         Input: loop       - the glyph to be saved
|		 ident      - the loop identifier
|		 export - whether external connections should be exported
|		 gui_parameters - whether exported parameters should be saved
|		 selections - the other selections that are being saved
|		 num_sels    - the number of selections in the selection array
|        Output: file  - the file the information is suppose to be written to
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void SaveLoop(
    xvobject loop,
    int      ident,
    int      export,
    int	     gui_parameters,
    xvobject *selections,
    int      num_sels,
    kfile    *file)
{
	kform	 *form;
	int      xpos, ypos;
	xvobject workspace = NULL;
	char     *command, *name, *oname;

	xvw_get_attributes(loop,
		XVW_GLYPH_FORM,   &form,
		XVW_GLYPH_ONAME,  &oname,
		XVW_NODE_NAME,    &name,
		XVW_XPOSITION,    &xpos,
		XVW_YPOSITION,    &ypos,
		NULL);

	name = !name ? "(unknown)" : name;
	if (form != NULL)
	   command = kvf_form_to_cmd(form, form->subform, NULL, FALSE);
	else
	   command = "(unknown)";

	/*
	 *  Oh No!  It's a loop, so we need to do the nasty.
	 */
	kfprintf(file,"  '%s'\nLoopBegin:%s:%s:%d:%d:%d:%s\n\n", name, name,
			oname, ident, xpos, ypos, command);

	xvw_get_attribute(loop, XVW_PROCEDURE_WORKSPACE_OBJECT, &workspace);
	if (workspace != NULL)
	{
	   WorkspaceSave(xvw_widget(workspace), file, FALSE, FALSE, FALSE,
			FALSE, NULL, NULL);
	}
	kfprintf(file,"LoopEnd:%s\n\n", name);
}

/*-----------------------------------------------------------
|
|  Routine Name: SaveConnections - Save the connections
|
|       Purpose: This routine does the actually saving of a glyph's connections
|		 to a workspace file.
|
|         Input: glyph - the glyph to save the connections
|		 ident - the glyph identifier
|		 export - whether external connections should be exported
|		 gui_parameters - whether exported parameters should be saved
|		 selections - the other selections that are being saved
|		 num_sels    - the number of selections in the selection array
|        Output: file  - the file the information is suppose to be written to
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void SaveConnections(
    xvobject object,
    int      ident,
    int      export,
    int      gui_parameters,
    xvobject *selections,
    int      num_sels,
    kfile    *file)
{
	XvwGlyphWidget xglyph = (XvwGlyphWidget) xvw_widget(object);

	char     *ivar, *ovar, *name = NULL;
	int      i, j, inum, onum, num;
	xvobject *connections, oport, iport, parent;


	if (xglyph->glyph.osize <= 0 && xglyph->glyph.onum <= 0)
	   return;

	xvw_get_attribute(object, XVW_NODE_NAME, &name);
	kfprintf(file,"\n#  '%s'\n", name);

	for (i = 0, onum = ident; i < xglyph->glyph.osize; i++)
	{
	   oport = xglyph->glyph.oports[i];
	   if (!PortGetInfo(oport, &ovar, NULL, NULL, NULL,
		&num, &connections))
	   {
	      continue;
	   }

	   for (j = 0; j < num; j++)
	   {
	      iport = NULL;
	      xvw_get_attribute(connections[j], XVW_CONNECTION_END,&iport);
	      parent = xvw_parent(iport);

	      inum = karray_locate((char **) selections,(kaddr)parent,num_sels);
	      if (inum != -1)
	      {
		 if (PortGetInfo(iport, &ivar, NULL, NULL, NULL, NULL, NULL))
		 {
	            kfprintf(file, "NodeConnection:%d:%s:%d:%s:\n", onum, ovar,
			inum, ivar);
		 }
		 continue;
	      }
	   }
	   kfprintf(file,"\n");
	}

	for (i = 0, onum = ident; i < xglyph->glyph.onum; i++)
	{
	   iport = NULL;
	   xvw_get_attribute(xglyph->glyph.oconnections[i],
			XVW_CONNECTION_END, &iport);
	   parent = xvw_parent(iport);
	   inum = karray_locate((char **) selections, (kaddr) parent, num_sels);
	   if (inum != -1)
	      kfprintf(file, "ControlConnection:%d:%d:\n", onum, inum);
	}
	kfprintf(file,"\n");
}

/*-----------------------------------------------------------
|
|  Routine Name: SaveExternalConnections - Save the external connections
|
|       Purpose: This routine does the actually saving of a glyph's external
|		 connections to a workspace file.
|
|         Input: glyph - the glyph to save the connections
|		 ident - the glyph identifier
|		 export - whether external connections should be exported
|		 gui_parameters - whether exported parameters should be saved
|		 selections - the other selections that are being saved
|		 num_sels    - the number of selections in the selection array
|        Output: exported - the exported connections
|		 num_exported - the number of exported connections
|		 file  - the file the information is suppose to be written to
|		
|    Written By: Mark Young
|          Date: Nov 12, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void SaveExternalConnections(
    xvobject object,
    int      ident,
    int      export,
    int	     gui_parameters,
    xvobject *sels,
    int      num_sels,
    xvobject **exported,
    int      *num_exported,
    kfile    *file)
{
	XvwGlyphWidget xglyph = (XvwGlyphWidget) xvw_widget(object);

	int      i, j, inum, onum, num;
	xvobject *connections, out, in;
	char     *ivar, *ovar, *name = NULL;


	/*
	 *  Don't even bother for glyph's with no connections...
	 */
	if (xglyph->glyph.isize <= 0 && xglyph->glyph.inum <= 0 &&
	    xglyph->glyph.osize <= 0 && xglyph->glyph.onum <= 0)
	   return;

	xvw_get_attribute(object, XVW_NODE_NAME, &name);
	kfprintf(file,"\n#  '%s'\n", name);

	for (i = 0, onum = ident; i < xglyph->glyph.osize; i++)
	{
	   out = xglyph->glyph.oports[i];
	   if (!PortGetInfo(out, &ovar, NULL, NULL, NULL, &num, &connections))
	      continue;

	   for (j = 0; j < num; j++)
	   {
	      in = NULL;
	      xvw_get_attribute(connections[j], XVW_CONNECTION_END, &in);
	      if (!PortGetInfo(in, &ivar, NULL, NULL, NULL, NULL, NULL))
		 continue;

	      inum = LocateExported(in, sels, num_sels, exported, num_exported);
	      if (inum == -1) continue;
	      kfprintf(file, "ExternalConnectionNode:0:%d:%s:%d:%s:\n",
				onum, ovar, inum, ivar);
	   }
	}
	kfprintf(file,"\n");

	for (i = 0, inum = ident; i < xglyph->glyph.isize; i++)
	{
	   in = xglyph->glyph.iports[i];
	   if (!PortGetInfo(in, &ivar, NULL, NULL, NULL, &num, &connections))
	      continue;

	   for (j = 0; j < num; j++)
	   {
	      out = NULL;
	      xvw_get_attribute(connections[j], XVW_CONNECTION_BEGIN, &out);
	      if (!PortGetInfo(out, &ovar, NULL, NULL, NULL, NULL, NULL))
		 continue;

	      onum = LocateExported(out, sels, num_sels, exported,num_exported);
	      if (onum == -1) continue;
	      kfprintf(file, "ExternalConnectionNode:1:%d:%s:%d:%s:\n",
				onum, ovar, inum, ivar);
	   }
	}
	kfprintf(file,"\n");

	for (i = 0, onum = ident; i < xglyph->glyph.onum; i++)
	{
	   out = xglyph->glyph.oconnections[i]; in = NULL;
	   xvw_get_attribute(out, XVW_CONNECTION_END, &in);

	   inum = LocateExported(in, sels, num_sels, exported, num_exported);
           if (inum == -1) continue;
	   kfprintf(file, "ExternalConnectionControl:0:%d:%d:\n", onum, inum);
	}

	for (i = 0, inum = ident; i < xglyph->glyph.inum; i++)
	{
	   in = xglyph->glyph.iconnections[i]; out = NULL;
	   xvw_get_attribute(in, XVW_CONNECTION_BEGIN, &out);

	   onum = LocateExported(out, sels, num_sels, exported, num_exported);
           if (onum == -1) continue;
	   kfprintf(file, "ExternalConnectionControl:1:%d:%d:\n", onum, inum);
	}
	kfprintf(file,"\n");
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceSave
|
|       Purpose: WorkspaceSave is used to save the glyph's within a
|		 workspace.
|
|         Input: widget - the workspace widget
|		 file   - the file pointer in which to write the workspace to
|		 header - whether we should write the workspace header
|		 export - whether external connections should be exported
|		 selected - save only the currently selected glyphs
|		 gui_parameters - whether exported parameters should be saved
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Nov 30, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int WorkspaceSave(
   Widget widget,
   kfile  *file,
   int    header,
   int    export,
   int    selected,
   int    gui_parameters,
   xvobject **exported,
   int	    *num_exported)
{
	XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) widget;

	time_t   tval;
        int      i, num, gui;
	xvobject object, *selections;
	char	 *filename, *username;


	if (header == TRUE)
	{
	   tval = time(NULL);
	   if ((username = kgetenv("USER")) == NULL)
	      username = "(unknown)";

	   filename = kfile_filename(kfileno(file));
	   kfprintf(file, "# Khoros Visual Programming Workspace\n#\n# %s \
workspace file (%s) was created \n# on %s# by user %s\n#\n",kprog_get_program(),
		filename, ctime(&tval), username);
	}

	/*
	 *  Get the info needed for saving the workspace
	 */
	gui        = gui_parameters;
	object     = xwid->viewport.plane;

	/*
	 *  Save the variables to the file
	 */
	if (export == FALSE)
	{
	   kfprintf(file,"VariablesBegin:\n");
	   kexpr_variables_print((long) xvw_object(widget), file);
	   kfprintf(file,"VariablesEnd:\n\n");
	}

	/*
	 *  Save the Glyph/Procedure/Control information
	 */
	kfprintf(file,"#GlyphBegin\n\n");
	selections = WorkspaceChildren(xvw_object(widget), selected, &num);
	for (i = 0; i < num; i++)
	{
	   if (xvw_check_subclass(selections[i], GlyphWidgetClass) == FALSE)
	      continue;

	   if (xvw_check_subclass(selections[i], LoopWidgetClass))
	      SaveLoop(selections[i], i, export, gui, selections, num, file);
	   else if (xvw_check_subclass(selections[i], ProcedureWidgetClass))
	      SaveProcedure(selections[i], i, export,gui, selections, num,file);
	   else if (xvw_check_subclass(selections[i], ConditionalWidgetClass))
	      SaveConditional(selections[i], i, export,gui,selections,num,file);
	   else
	      SaveGlyph(selections[i], i, export, gui, selections, num, file);
 	}
	kfprintf(file,"#GlyphEnd\n");

	/*
	 *  Save the Connections associated with the above
	 *  Glyph/Procedure/Control object
	 */
	kfprintf(file,"#ConnectionBegin\n\n");
	for (i = 0; i < num; i++)
	{
	   if (xvw_check_subclass(selections[i], GlyphWidgetClass) == FALSE)
	      continue;

	   SaveConnections(selections[i], i, export, gui, selections, num,file);
 	}
	kfprintf(file,"#ConnectionEnd\n\n");

	/*
	 *  Save the Exported Connections associated with the above
	 *  Glyph/Procedure/Control object if "export" is set to TRUE
	 */
	if (export == TRUE)
	{
	   if (exported != NULL)
	      *exported = NULL;
	   if (num_exported != NULL)
	      *num_exported = 0;

	   kfprintf(file,"#ExternalConnectionBegin\n\n");
	   for (i = 0; i < num; i++)
	   {
	      if (xvw_check_subclass(selections[i], GlyphWidgetClass) == FALSE)
	         continue;

	      SaveExternalConnections(selections[i], i, export, gui,
			selections, num, exported, num_exported, file);
 	   }
	}
	kfprintf(file,"#ExternalConnectionEnd\n\n");

	/*
	 *  Save the annotations to the file
	 */
	xvw_save_annotations(object, file);

	/*
	 *  Cleanup, since we are done
	 */
	kfree(selections);
	return(TRUE);
}
