 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                Axis Utility Routines
   >>>>
   >>>>  Private:
   >>>>			parse_scale_factor()
   >>>>			find_scale_factor()
   >>>>			find_decimal_places()
   >>>>			max_min_linear()
   >>>>			nothing_linear()
   >>>>			number_steps_linear()
   >>>>			size_number_linear()
   >>>>			step_size_linear()
   >>>>			max_min_log()
   >>>>			nothing_log()
   >>>>			number_steps_log()
   >>>>			size_number_log()
   >>>>			step_size_log()
   >>>>			fracround()
   >>>>			truncate_double()
   >>>>			numdecs()
   >>>>			numdigs()
   >>>>			first_digit()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

#define TEN	10.0

/*
*******************************************************************

	General subroutine

	C source filename:		truncate_double.c

	revision	date			programmer
	--------	----			----------
	   0.0		August 8, 1985		John M. Salas
						Mike Conley

******************************************************************

Synopsis :

	#include <math.h>

	double truncate_double(x,char_precision)

Description :

	This routine rounds a number to the number of characters 
	given by char_precision and returns the result.  It does not 
	take into account the existence of a minus sign or decimal 
	point.  (E.g., if x == 345.12573 and char_precision == 6, 
	truncate_double(x) == 345.126.)

Routines called by this routine :

	fracround		- General subroutine
	numdigs			- General subroutine
	numdecs			- General subroutine
	abs			- standard C math library routine
	pow			- standard C math library routine

Returned values :

        KMAXFLOAT: character precision is out of bounds
	truncate_double(x) (see above)

******************************************************************
*/

double truncate_double(
   double x,
   int    char_precision)
{
  int k;
  int shift;


  if(char_precision < 2 || char_precision > 15) return(KMAXFLOAT);

  shift = char_precision + numdecs(x) - 1;

  for (k = 0; k < abs(shift); k++)
  {
    if (shift > 0) x *= 10.0;
    else
      x /= 10.0;
  }

  x = fracround(x);

  for(k = 0; k < abs(shift); k++)
  {
    if(shift > 0.0) x /= 10.0;
    else
      x *= 10.0;
  }

  return(x);
}

/*
*******************************************************************

	Plot Package subroutine

	C source filename:	parse_scale_factor.c

	revision	date			programmer
	--------	----			----------
	   0.0		May 13, 1985		John M. Salas

******************************************************************

Synopsis :

	void parse_scale_factor(tack_it_on, exponent, label)

	int tack_it_on, exponent;
	char *label;

Description :

	     This routine will attach a scale factor to the label
	designated by axis for the specific plot_area.  Its value
	is passed in exponent.

Routines called by this routine :

	none.

Returned values :

	0 - no error.

Plot Package variables changed by this routine :

	none.

******************************************************************
*/

void parse_scale_factor(
   int  tack_it_on,
   int  exponent,
   char *label)
{
  if (tack_it_on)
  {
    if ( label == NULL )
    {
      label = (char *) kmalloc(20 * sizeof(char));
      if ( label == NULL )
      {
         XtWarning("Axis Object: can not malloc room to append scale factor to the axis label.\n");
	 return;
      }
    }

    ksprintf(label," (10e%d)", exponent);
  }
  else
  {
    if ( label != NULL )
    {
      kfree(label);
      label = NULL;
    }
  }
  return;
}


/*
*******************************************************************

	Plot Package subroutine

	C source filename:		find_scale_factor.c

	revision	date			programmer
	--------	----			----------
	   0.0		June 6, 1985		John M. Salas
	   0.1		July 29, 1985		John M. Salas

******************************************************************

Synopsis :

	find_scale_factor(characters_displayed,
			  maximum,minimum,step_size,decimal_places,
			  multiplier)

	int characters_displayed,*decimal_places;
	double maximum,minimum,step_size,*multiplier;
	char *label;

Description :

	     This routine will set a scale factor if needed and 
	call parse_scale_factor() to attach it to the end of the
	appropriate axis label.  If will also find the number of
	significant decimal places for the numerical labels.

	0.1 - Corrected the errors when determining the number of
	      decimal places in the label.

Routines called by this routine :

	find_decimal_places - Plot Package routine.
	parse_scale_factor  - Plot Package routine.

Returned values :

	0 - no error.

Plot Package variables changed by this routine :

	none.

******************************************************************
*/

void find_scale_factor(
   int    characters_displayed,
   double maximum,
   double minimum,
   double step_size,
   int    *decimal_places,
   double *multiplier,
   char   *label)
{
  int power_ten = 0;

  double power_fraction,power_integer;
  double lower_limit;
  double upper_limit;

  power_ten = 0;

  if(fabs(minimum) == 0.0)
    lower_limit = 1.0;
  else
    lower_limit = minimum;

  if(fabs(maximum) == 0.0)
    upper_limit = 1.0;
  else
    upper_limit = maximum;

  if((power_fraction = characters_displayed - 2) > 6.0)
    power_fraction = 6.0;

  power_integer = characters_displayed - 2;

  if((fabs(upper_limit) < pow(TEN,power_integer) &&
     fabs(upper_limit) > pow(1.0/TEN,power_fraction)) &&
     (fabs(lower_limit) < pow(TEN,power_integer) &&
     fabs(lower_limit) > pow(1.0/TEN,power_fraction)))
  {
    *multiplier = 1.0;
  }
  else
  {
    if(fabs(maximum) < pow(1.0/TEN,power_fraction) ||
       fabs(minimum) < pow(1.0/TEN,power_fraction))
    {
      if(maximum == 0.0)
      {
	power_ten = (int) floor(log10(1.0/fabs(minimum)));
      }
      else
      {
	power_ten = (int) floor(log10(1.0/fabs(maximum)));
      }
      *multiplier = pow(TEN,(double) (power_ten));
      power_ten = -power_ten;
    }
    else
    {
      power_ten = (int) floor(log10(fabs(maximum)));
      if(power_ten < ((int) floor(log10(fabs(minimum)))))
      power_ten = (int) floor(log10(fabs(minimum)));
      *multiplier = pow(1.0/TEN,(double) (power_ten));
    }
  }
/*
   Finds out the decimal places in the label.
*/
  maximum = *multiplier * maximum;
  minimum = *multiplier * minimum;
  step_size = *multiplier * step_size;
  find_decimal_places(characters_displayed, maximum, minimum,
		      step_size, decimal_places);

  parse_scale_factor((power_ten != 0), power_ten, label);

  return;
}

/*
*******************************************************************

	General subroutine

	C source filename:		find_decimal_places.c

	revision	date			programmer
	--------	----			----------
	   0.0		May 13, 1985		John M. Salas

******************************************************************

Synopsis :

	#define TEN			10.0

	find_decimal_places(characters_displayed,maximum,minimum,step,
			    decimal_places)

	int characters_displayed,*decimal_places;
	double maximum,minimum,step;

Description:

	     This routine finds the largest number of decimal places 
	which fit in the range determined from characters_displayed.
	The value is found for the three variables maximum, minimum,
	and step.

Routines called by this routine :

	none.

Returned values :

	none.

******************************************************************
*/
#define TEN			10.0

void find_decimal_places(
   int    characters_displayed,
   double maximum,
   double minimum,
   double step,
   int    *decimal_places)
{
  int j;
  int integer_places;
  int temp_integer_places;

  double abs_temp_min,abs_temp_max,abs_temp_step;
  double temp_value_int;
  double power;
  double remainder_min,remainder_max,remainder_step;
  double temp_value;

/*
   The following calculates the integer part of the numbers to
   be displayed.  If the integer part is zero it must be ignored,
   therefore it is set to 1.
*/
  if(fabs(maximum) > 1.0)
    integer_places = (int) floor(fabs(log10(fabs(maximum)))) + 1;
  else
    integer_places = 1;

  if(fabs(minimum) > 1.0)
    temp_integer_places = (int) floor(fabs(log10(fabs(minimum)))) + 1;
  else
    temp_integer_places = 1;

  if(integer_places < temp_integer_places)
    integer_places = temp_integer_places;

  power = characters_displayed - integer_places;

  abs_temp_min = (double) floor(fabs(minimum) + pow(1.0/TEN,power));
  abs_temp_max = (double) floor(fabs(maximum) + pow(1.0/TEN,power));
  abs_temp_step = (double) floor(fabs(step) + pow(1.0/TEN,power));

  remainder_min = fabs(fabs(minimum) - abs_temp_min);
  if(remainder_min < pow(1.0/TEN,power)) remainder_min = 1.0;

  remainder_max = fabs(fabs(maximum) - abs_temp_max);
  if(remainder_max < pow(1.0/TEN,power)) remainder_max = 1.0;

  remainder_step = fabs(fabs(step) - abs_temp_step);
  if(remainder_step < pow(1.0/TEN,power)) remainder_step = 1.0;
/*
   The loops uses the remainder values from above to find the decimalplaces
   to which the number is displayed.
*/
  *decimal_places = 0;

  for(j = 0;j <= (characters_displayed - integer_places);j++)
  {
    temp_value = pow(TEN,(double)j) * remainder_min;
    temp_value_int = (double) floor(temp_value + pow(1.0/TEN,power));
    power = characters_displayed - integer_places - j;
    if(fabs(temp_value - temp_value_int) >= 1.0) break;
    if(fabs(temp_value - temp_value_int) < pow(1.0/TEN,power)) break;
  }

  if(*decimal_places < j) *decimal_places = j;

  for(j = 0;j <= (characters_displayed - integer_places);j++)
  {
    temp_value = pow(TEN,(double)j) * remainder_max;
    temp_value_int = (double) floor(temp_value + pow(1.0/TEN,power));
    power = characters_displayed - integer_places - j;
    if(fabs(temp_value - temp_value_int) >= 1.0) break;
    if(fabs(temp_value - temp_value_int) < pow(1.0/TEN,power)) break;
  }

  if(*decimal_places < j) *decimal_places = j;

  for(j = 0;j <= (characters_displayed - integer_places);j++)
  {
    temp_value = pow(TEN,(double)j) * remainder_step;
    temp_value_int = (double) floor(temp_value + pow(1.0/TEN,power));
    power = characters_displayed - integer_places - j;
    if(fabs(temp_value - temp_value_int) >= 1.0) break;
    if(fabs(temp_value - temp_value_int) < pow(1.0/TEN,power)) break;
  }

  if(*decimal_places < j) *decimal_places = j;

  return;
}


/*
*******************************************************************

	General subroutine

	C source filename:		max_min_linear.c

	revision	date			programmer
	--------	----			----------
	   0.0		May 31, 1985		John M. Salas

******************************************************************

Synopsis :

	#define TEN			10.0

	max_min_linear(step_size,number_steps,minimum,maximum)

	double *number_steps,*step_size,minimum,maximum;

Description:

	     This routine will find the step_size and number_steps
	for the range specified by maximum and minimum.  It set-up
	the step to be orders of magnitude of 1, 2, and 5.  This
	may result in number_steps to have a fractional part.

Routines called by this routine :

	none.

Returned values :

	0 - no error.
	1 - if minimum >= maximum.

******************************************************************
*/
#define TEN			10.0

max_min_linear(
   double *step_size,
   double *number_steps,
   double minimum,
   double maximum)
{
  double temp_step_size;
  double power;

  if(minimum >= maximum) return(1);

  if((temp_step_size = (maximum - minimum)/(*number_steps)) < 1)
  {
    power = pow(TEN,ceil(log10(1.0/(temp_step_size))));
  }
  else
  {
    power = pow(1.0/TEN,floor(log10(temp_step_size)));
  }

  *step_size = power * (maximum - minimum)/(*number_steps);
  *step_size = truncate_double(*step_size,4);

  if(*step_size > 0.0 && *step_size <= 1.0)
  {
    *step_size = 1.0;
  }
  else if(*step_size > 1.0 && *step_size <= 2.0)
  {
    *step_size = 2.0;
  }
  else if(*step_size > 2.0 && *step_size <= 5.0)
  {
    *step_size = 5.0;
  }
  else if(*step_size > 5.0 && *step_size <= 10.0)
  {
    *step_size = 10.0;
  }

  *step_size *= 1.0/power;
/*
 * Calls truncate_double to establish a step size that has 2 digit accuracy,
 * so that round-off error is eleminated.
*/
  *step_size = truncate_double(*step_size,2);
  *number_steps = (maximum - minimum)/(*step_size);

  return(0);
}

/*
*******************************************************************

	General subroutine

	C source filename:		nothing_linear.c

	revision	date			programmer
	--------	----			----------
	   0.0		May 28, 1985		John M. Salas

******************************************************************

Synopsis :

	#define TEN				10.0

	nothing_linear(step_size,number_steps,minimum,maximum)

	double *number_steps,*step_size,*minimum,*maximum;

Description :

	     This routine will find a step_size that is an order of
	magnitude of 1, 2, and 5, for the range given by maximum and 
	minimum.  Then it will set-up the minimum and maximum such
	that when divided by the step_size the result will be an
	integer value.  Then the number_steps is calculated.  All
	four values are returned.  NOTE: Some value must be passed
	in maximum, minimum, and number_steps.

Routines called by this routine :

	0 - no error.
	1 - if number_steps <= 0.
	2 - if minimum >= maximum.

Returned values :

	none.

******************************************************************
*/
#define TEN				10.0

nothing_linear(
   double *step_size,
   double *number_steps,
   double *minimum,
   double *maximum)
{
  double temp_step_size;
  double power;
  double max_divide,min_divide;

  if(*number_steps <= 0.0) return(1);
  if(*minimum >= *maximum) return(2);

  if((temp_step_size = (*maximum - *minimum)/(*number_steps)) < 1)
  {
    power = pow(TEN,ceil(log10(1.0/temp_step_size)));
  }
  else
  {
    power = pow(1.0/TEN,floor(log10(temp_step_size)));
  }

  *step_size = power * temp_step_size;
  *step_size = truncate_double(*step_size,4);

  if(*step_size > 0.0 && *step_size <= 1.0)
  {
    *step_size = 1.0;
  }
  else if(*step_size > 1.0 && *step_size <= 2.0)
  {
    *step_size = 2.0;
  }
  else if(*step_size > 2.0 && *step_size <= 5.0)
  {
    *step_size = 5.0;
  }
  else if(*step_size > 5.0 && *step_size <= 10.0)
  {
    *step_size = 10.0;
  }

  *step_size *= 1.0/power;
/*
 * Calls truncate_double to establish a step size that has 2 digit accuracy,
 * so that round-off error is eleminated.
*/
  *step_size = truncate_double(*step_size,2);

  max_divide = ceil(*maximum/(*step_size));
  min_divide = floor(*minimum/(*step_size));

  *maximum = max_divide * *step_size;
  *minimum = min_divide * *step_size;
  *number_steps = max_divide - min_divide;

  return(0);
}


/*
*******************************************************************

	General subroutine

	C source filename:		number_steps_linear.c

	revision	date			programmer
	--------	----			----------
	   0.0		May 13, 1985		John M. Salas

******************************************************************

Synopsis :

	#define TEN			10.0

	number_steps_linear(step_size,number_steps,minimum,maximum)

	double number_steps,*step_size,*minimum,*maximum;

Description:

	     This routine will calculate a step_size, a new minimum,
	and a new maximum for a specified number_steps.  The step_size
	will be an order of magnitude of 1, 2, and 5.  The minimum
	and maximum will be set-up such that when divided by the
	step_size the result will be an integer value.  NOTE: Some
	value must be passed in maximum, minimum, and number_steps.

Routines called by this routine :

	none.

Returned values :

	0 - no error.
	1 - if number_steps <= 0.
	2 - if minimum >= maximum.

******************************************************************
*/
#define TEN			10.0

number_steps_linear(
   double  *step_size,
   double  number_steps,
   double  *minimum,
   double  *maximum,
   Boolean nice_min_max)
{
  double power;
  double divisible,temp_step_size;

  if(number_steps <= 0.0) return(1);
  if(*minimum >= *maximum) return(2);

  if((temp_step_size = fabs((*maximum - *minimum)/number_steps)) < 1)
  {
    power = pow(TEN,ceil(log10(1.0/(temp_step_size))));
  }
  else
  {
    power = pow(1.0/TEN,floor(log10(temp_step_size)));
  }

  if ( nice_min_max )
  {
    *step_size = power * (*maximum - *minimum)/number_steps;
    *step_size = truncate_double(*step_size,4);

    if(*step_size > 0.0 && *step_size <= 1.0)
    {
      *step_size = 1.0;
    }
    else if(*step_size > 1.0 && *step_size <= 2.0)
    {
      *step_size = 2.0;
    }
    else if(*step_size > 2.0 && *step_size <= 5.0)
    {
      *step_size = 5.0;
    }
    else if(*step_size > 5.0 && *step_size <= 10.0)
    {
      *step_size = 10.0;
    }

    *step_size *= 1.0/power;
/*
 * Calls truncate_double to establish a step size that has 2 digit accuracy,
 * so that round-off error is eleminated.
*/
    *step_size = truncate_double(*step_size,2);

    divisible = floor(*minimum/(*step_size));
    *minimum = *step_size * divisible;
    *maximum = *step_size * number_steps + *minimum;
  }
  else
  {
    *step_size = (*maximum - *minimum)/number_steps;
  }

  return(0);
}


/*
*******************************************************************

	General subroutine

	C source filename:		size_number_linear.c

	revision	date			programmer
	--------	----			----------
	   0.0		May 13, 1985		John M. Salas

******************************************************************

Synopsis :

	size_number_linear(step_size,number_steps,minimum,maximum)

	double number_steps,step_size,*minimum,*maximum;

Description:

	     This routine will set the maximum and minimum using
	the step_size and number_steps specified.  The minimum
	is the bases for calculating the values.  The new maximum
	and minimum are set such that when divided by the step_size
	the result will be an integer value.
	NOTE: Some range must be specified by maximum and minimum.

Routines called by this routine :

	0 - no error.
	1 - if step_size <= 0.
	2 - if number_steps <= 0.
	3 - if minimum >= maximum.

Returned values :

	none.

******************************************************************
*/

size_number_linear(
   double step_size,
   double number_steps,
   double *minimum,
   double *maximum)
{
  double divisible;

  if(step_size <= 0.0) return(1);
  if(number_steps <= 0.0) return(2);
  if(*minimum >= *maximum) return(3);

  *minimum = truncate_double(*minimum,2);
  divisible = floor(*minimum/step_size);
  divisible = truncate_double(divisible,2);
  *minimum = step_size * divisible;

  *maximum = (step_size * number_steps) + *minimum;

  return(0);
}


/*
*******************************************************************

	General subroutine

	C source filename:		step_size_linear.c

	revision	date			programmer
	--------	----			----------
	   0.0		May 13, 1985		John M. Salas

******************************************************************

Synopsis :

	step_size_linear(step_size,number_steps,minimum,maximum)

	double *number_steps,step_size,*minimum,*maximum;

Description:

	     This routine will calculate the maximum, minimum, and
	number_steps for a specified step_size for the range passed
	in maximum and minimum.  The new maximum and minimum are
	set such that when divided by the step_size the result will
	be an integer value.
	NOTE: Some range must be specified by maximum and minimum.

Routines called by this routine :

	0 - no error.
	1 - if step_size <= 0.
	2 - if minimum >= maximum.

Returned values :

	none.

******************************************************************
*/

step_size_linear(
   double  step_size,
   double  *number_steps,
   double  *minimum,
   double  *maximum,
   Boolean nice_min_max)
{
  double divisible;

  if(step_size <= 0.0) return(1);
  if(*minimum >= *maximum) return(2);

  if ( nice_min_max )
  {
    divisible = floor(*minimum/step_size);
    *minimum = step_size * divisible;

    divisible = ceil(*maximum/step_size);
    *maximum = step_size * divisible;
  }

  *number_steps = (*maximum - *minimum)/step_size;

  return(0);
}

/*
*******************************************************************

	General subroutine

	C source filename:			

	revision	date			programmer
	--------	----			----------
	   0.0		June 3, 1985		Mike Conley

******************************************************************

Synopsis : 

	max_min_log(&step_size, &number_steps, &minimum, &maximum)

	double number_steps,step-size, minimum, maximum;

Description : Given a user-defined minimum and maximum, this routine
	      computes the nearest matches to these limits which 
	      are multiples of powers of ten.  It then computes
	      step_size and number_steps.

Routines called by this routine : nothing_log

Returned values :

	0 - no error.
	1 - if negative value passed in minimum.
	2 - if minimum >= maximum.

******************************************************************
*/
max_min_log(
   double *step_size,
   double *number_steps,
   double *minimum,
   double *maximum)
{
  if(*minimum <= 0.0) return(1);
  if(*minimum >= *maximum) return(2);
  /*
     This routine performs the exact same function as nothing_log(),
     since minimum and maximum will be reset if they are not 
     multiples of powers of the log base.  Therefore, we call
     nothing_log().
  */
  nothing_log(step_size, number_steps, minimum, maximum);

  return(0);
}
/*
*******************************************************************

	General subroutine

	C source filename: nothing_log.c

	revision	date			programmer
	--------	----			----------
	   0.0		June 3, 1985		Mike Conley

******************************************************************

Synopsis : nothing_log (&step_size, &number_steps, &minimum, &maximum);

	   double number_steps, step_size, maximum, minimum;

Description : Default logarithmic plot axis-bounding routine.
	      Using only the data maximum and minimum, this routine
	      computes the actual max and min to be used in drawing
	      the axis, the number of cycles, and the log base
	      (step_size).

Routines called by this routine : none

Returned values :

	0 - no error.
	1 - if negative value passed in minimum.
	2 - if minimum >= maximum.

******************************************************************
*/

#define TEN 10.0

nothing_log(
   double *step_size,
   double *number_steps,
   double *minimum,
   double *maximum)
{
  double minfac, maxfac, nextten;

  if(*minimum <= 0.0) return(1);
  if(*minimum >= *maximum) return(2);
  /*
     Set step-size to 10.  (NOTE:  The program currently only supports
     base-10 log plots, and automatically sets step-size to 10 during
     initialization.  This step is placed here for future use, when
     other bases may be used.)
  */
  *step_size = 10.0;

  /*
     Compute magnitude of minimum and maximum.
  */
  minfac = fracround(pow(TEN, floor(log10(*minimum))));
  maxfac = fracround(pow(TEN, floor(log10(*maximum))));

  /*
     New minimum is now set to largest multiple of a power of ten 
     less than or equal to the old minimum; new maximum is smallest 
     multiple of a power of ten greater than or equal to the 
     old maximum.
  */
  nextten = fracround(pow(10.0, ceil(log10(*minimum))));
  if ((nextten - *minimum) / nextten < 0.0001) *minimum = nextten;
  *minimum = floor(*minimum / minfac) * minfac;
  *maximum = ceil(*maximum / maxfac) * maxfac;

  /*
     Compute number of cycles included in new range.
  */
  *number_steps = log10(*maximum) - log10(*minimum);

  return(0);
}
/*
*******************************************************************

	General subroutine

	C source filename:  number_steps_log.c

	revision	date			programmer
	--------	----			----------
	   0.0		June 3, 1985		Mike Conley

******************************************************************

Synopsis : number_steps_log (step_size, number_steps, &minimum, &maximum);

	   double number_steps, step_size, minimum, maximum;

Description : The user specifies the number of cycles (number_steps) and
	      the log base (step_size) he desires.  The data supplies
	      maximum and minimum values.  Minimum value is reduced to the
	      largest power of ten smaller than the data minimum.  Maximum
	      is reset to number_steps cycles beyond the new minimum.
	      (E.g., if minimum == 0.3, maximum == 25.0, number_steps == 3.0,
	      and step_size == 10.0, the routine will set minimum to 0.1 and
	      maximum to 100.)

Routines called by this routine : none

Returned values :

	0 - no error.
	1 - if negative value passed in minimum.
	2 - if minimum >= maximum.
	3 - if number_steps is negative.

******************************************************************
*/

#define TEN 10.0

number_steps_log(
   double *step_size,
   double number_steps,
   double *minimum,
   double *maximum)
{
  double minfac;
  double power;

  if(*minimum <= 0.0) return(1);
  if(*minimum >= *maximum) return(2);
  if(number_steps <= 0.0) return(3);
  /*
     Compute magnitude of minimum.
  */
  power = floor(log10(*minimum));
  minfac = pow(TEN, power);

  /*
     New minimum is now set to largest multiple of a power of ten 
     less than or equal to the old minimum.
  */
  *minimum = floor(*minimum / minfac) * minfac;

  /*
     Compute new maximum for plot based on minimum plus number of 
     cycles desired.
  */
  *maximum = pow(TEN, number_steps) * *minimum;

  return(0);
}
/*
*******************************************************************

	General subroutine

	C source filename: size_number_log.c

	revision	date			programmer
	--------	----			----------
	   0.0		June 3, 1985		Mike Conley

******************************************************************

Synopsis : size_number_log (step_size, number_steps, &maximum, &minimum);

	   double number_steps, step_size, maximum, minimum;

Description : Given fixed log base (step_size) and number of cycles
	      (number_steps), this routine calculates an appropriate
	      minimum based on the data minimum and then sets the
	      maximum to fit the givens specifications.

Routines called by this routine : none

Returned values :

	0 - no error.
	1 - if minimum >= maximum.
	2 - number_steps <= 0.0.
	3 - step_size <= 0.0.

******************************************************************
*/

#define TEN 10.0

size_number_log(
   double step_size,
   double number_steps,
   double *minimum,
   double *maximum)
{
  
  double minfac;
  double power;

  if(*minimum <= 0.0) return(1);
  if(*minimum >= *maximum) return(2);
  if(number_steps <= 0.0) return(3);
  if(step_size <= 0.0) return(4);
  /*
     Compute magnitude of minimum.
  */
  power = floor(log10(*minimum));
  minfac = pow(TEN, power);

  /*
     New minimum is now set to largest multiple of a power of ten 
     less than or equal to the old minimum.
  */
  *minimum = floor(*minimum / minfac) * minfac;

  /*
     New maximum is set to minimum plus number_steps cycles.
  */
  *maximum = pow(TEN, number_steps) * *minimum;

  return(0);
}
/*
*******************************************************************

	General subroutine

	C source filename:		

	revision	date			programmer
	--------	----			----------
	   0.0		August 8, 1985		Mike Conley

******************************************************************

Synopsis :

	step_size_log(step_size,number_steps,minimum,maximum)

Description :

	This routine is meant to compute the parameters number_steps,
	minimum, and maximum, given the value of step_size.  For a
	logarithmic plot, step_size is actually the base of the
	logarithm to be used.  Currently, this value should always be 10,
	since alternate bases are not supported.  Consequently, this 
	routine does nothing but call the default parameter-setting 
	routine, nothing_log, which assumes a base of 10 as default.

Routines called by this routine :

	nothing_log		- General subroutine

Returned values :

	0 - normal termination
	1 - minimum is less than 0 for a logarithmic plot
	2 - number_steps is less than 0
	3 - step_size is less than 0

******************************************************************
*/
step_size_log(
   double step_size,
   double *number_steps,
   double *minimum,
   double *maximum)
{
  if(*minimum <= 0.0) return(1);
  if(*minimum >= *maximum) return(2);
  if(*number_steps <= 0.0) return(3);
  if(step_size <= 0.0) return(4);

  nothing_log(&step_size, number_steps, minimum, maximum);
  return(0);
}

/*
*******************************************************************

	General subroutine

	C source filename:		fracround.c

	revision	date			programmer
	--------	----			----------
	   0.0		July 12, 1985		Mike Conley

******************************************************************

Synopsis :  

	fracround(x);

	double x;

Description :  

	This routine rounds the first significant digit
	of a fractional number and returns the result.  
	If the number is greater than 1, it is rounded 
	normally.  (E.g., the number 0.002532 will be 
	rounded to 0.003.)

Routines called by this routine :  

	floor()			- standard C math library routine

Returned values :  

	fracround(x) (see above)

******************************************************************
*/
#define TRUE 1
#define FALSE 0

double fracround(
   double x)
{
  int i, e;
  short negative;

  negative = FALSE;
  if (x != 0.0)
  {
    if (x < 0.0)
    /*
       Retain the sign.
    */
    {
      negative = TRUE;
      x = -x;
    }
    e = 0;
    while (x < 1.0)
    /*
       Shift the first significant digit into the ones place.
    */
    {
      x = x * 10.0;
      e++;
    }
    /*
       Round.
    */
    x = floor(x +  0.5);
    if (x >= 10.0 && e > 0)
    /*
       If the rounding results in an increase in the number of
       significant digits (e.g., if 9.534 gets rounded to 10.0),
       decrease the shift amount by 1.
    */
    {
      x = 1.0;
      e--;
    }
    /*
       Shift back to the proper place.
    */
    for (i = e; i > 0; i--) x /= 10.0;

    if (negative) x = -x; /* Restore sign if necessary. */
  }
  return(x);
}
/*
*******************************************************************

	General subroutine

	C source filename:		numdecs.c

	revision	date			programmer
	--------	----			----------
	   0.0		August 8, 1985		Mike Conley

******************************************************************

Synopsis :

	int numdecs(x)

Description :

	This routine returns the number of decimals in
	a fractional number.  If the number is greater than 
	or equal to 1, it returns 0.  If x == 0, the answer
	is defined to be 1.  (E.g., if x == 0.00341, 
	numdecs(x) == 3.  If x == 3.221, numdecs(x) == 0.)

Routines called by this routine :

	abs()		- standard C math library routine

Returned values :

	numdecs(x) (see above)

******************************************************************
*/

int numdecs(
   double x)
{
  int e;

  if(x == 0) return(1);
  x = fabs(x);
  e = 0;
  while (x < 1)
  {
    x *= 10.0;
    e++;
  }
  return(e);
}

/*
*******************************************************************

	General subroutine

	C source filename:		numdigs.c

	revision	date			programmer
	--------	----			----------
	   0.0		August 8, 1985		Mike Conley

******************************************************************

Synopsis :


	int numdigs(x)

Description :

	This routine computes and returns the number of decimals in a
	fractional number or the number of integer digits in a number 
	greater than or equal to 1.  The value for 0 is defined to be
	1.  (E.g., if x == 0.0032232, numdigs(x) == 7.  If x == 323.23023, 
	numdigs(x) == 3.)

Routines called by this routine :

	none

Returned values :

	numdigs(x) (see above)

******************************************************************
*/

int numdigs(
   double x)
{
  int e;

  if(x == 0.0) return(1);
  x = fabs(x);
  e = 0;
  if (x < 1)
  {
    while (x < 1.0)
    {
      x *= 10.0;
      e++;
    }
  }
  else
  {
    while (x >= 1.0)
    {
      x /= 10.0;
      e++;
    }
  }
  return(e);
}

/*
*******************************************************************

	General subroutine

	C source filename:		firstdigit.c

	revision	date			programmer
	--------	----			----------
	   0.0		August 8, 1985		Mike Conley

******************************************************************

Synopsis :

	int firstdigit(x)

	double x;

Description :

	This routine takes a floating-point number, x, and
	returns the rounded value of the first significant
	digit.  If the number is greater than 1, it simply
	returns the rounded integer portion.  (E.g., if 
	x == 0.0003721, firstdigit(x) == 4.  If x == 12.5412, 
	firstdigit(x) == 13.)

Routines called by this routine :

	floor()			- standard C math library routine

Returned values :

	firstdigit(x) (see above)

******************************************************************
*/
#define TRUE 1
#define FALSE 0

int first_digit(
   double x)
{
  int negative;
  
  if (x < 0.0) 
  {
    negative = TRUE;
    x = -x;
  }
  else
    negative = FALSE;

  if ( x == 0.0 )
    return (0);

  if ( x < 1.0 )
  {
    while (x < 1.0) x = x * 10.0;
    x = floor(x + 0.5);
/*
  if (x >= 10.0) x = 1.0;
*/
  }
  else
  {
    while (x > 10.0) x = x / 10.0;
    x = floor(x + 0.5);
  }

  if (negative) x = -x;

  return((int) x);
}
