/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                Image Roi Routines
   >>>>
   >>>>	Private:
   >>>>			ImageGetRoi()
   >>>>			ImageSetRoi()
   >>>>	 Static:
   >>>>			flood_fill()
   >>>>			scanline_fill()
   >>>>			draw_line()
   >>>>			draw_circle()
   >>>>			draw_ellipse()
   >>>>			compute_extents()
   >>>>			extract_region()
   >>>>			extract_line()
   >>>>			extract_circle()
   >>>>			extract_ellipse()
   >>>>			extract_polyline()
   >>>>			extract_roi()
   >>>>  Public:
   >>>>			xvw_getroi_line()
   >>>>			xvw_getroi_rectangle()
   >>>>			xvw_getroi_circle()
   >>>>			xvw_getroi_ellipse()
   >>>>			xvw_getroi_polyline()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvisual/ImageP.h>


/*-----------------------------------------------------------
|
|  Routine Name: flood_fill - flood fill according to the values in the mask.
|
|       Purpose: This is a flood fill routine that is used to build masks
|		 according to the boundary information already stored.
|         Input: data - the data to perform the scanline fill on
|                w  - the width of the data array
|                h  - the height of the data array
|		 x  - the initial x position to start the fill
|		 y  - the initial y position to start the fill
|                fill - the fill value to set within the mask
|    Written By: Mark Young
|          Date: Jan 02, 1995
|
------------------------------------------------------------*/
/*ARGSUSED*/
static void flood_fill(
  char *data,
  unsigned int  w,
  unsigned int  h,
  int  x,
  int  y,
  int  fill)
{
	int j, k, indx = y*w;

	data[indx+x] = fill;
	j = x-1; k = x+1;

	while (j >= 0 && data[indx+j] == 0) data[indx + j--] = fill;
	while (k < w  && data[indx+k] == 0) data[indx + k++] = fill;

	for (x = j+1, indx = j+1 + y*w; x < k; x++, indx++)
	{
	   if (y > 0 && data[indx-w] == 0)
	      flood_fill(data, w, h, x, y-1, fill);
	   if (y < h-1 && data[indx+w] == 0)
	      flood_fill(data, w, h, x, y+1, fill);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: scanline_fill - scanline fill according to the values in the
|				 mask.
|
|       Purpose: This is a modified scanline fill routine that is used
|		 to build masks according to the boundary information
|		 already stored.
|         Input: data - the data to perform the scanline fill on
|		 width  - the width of the data array
|		 height - the height of the data array
|                fill - the fill value to set within the mask
|    Written By: Mark Young
|          Date: Dec 18, 1994
|
------------------------------------------------------------*/
/*ARGSUSED*/
static void scanline_fill(
  char *data,
  unsigned int w,
  unsigned int h,
  int  fill)
{
	char *tmp;
	int  i, j, set;

	for (i = 0, j = w*(h-1); i < w; i++, j++)
	{
	   if (data[i] == 0) flood_fill(data, w, h, i, 0, -1);
	   if (data[j] == 0) flood_fill(data, w, h, i, h-1, -1);
	}
	for (i = 0, j = w-1; i < h; i++)
	{
	   if (data[i*w] == 0) flood_fill(data, w, h, 0, i, -1);
	   if (data[i*w+j] == 0) flood_fill(data, w, h, j, i, -1);
	}

	for (j = 0, tmp = data; j < h; j++)
	{
	   for (i = 0, set = TRUE; i < w; i++)
	   {
	      if (*tmp++ == 0)
		 flood_fill(data, w, h, i, j, !set ? -1 : fill);
	   }
	}

	for (i = 0; i < w*h; i++)
	   data[i] = (data[i] == -1) ? 0 : data[i];
}

/*-----------------------------------------------------------
|
|  Routine Name: draw_line - scanline draw will rasterize draw
|				     in the mask.
|
|       Purpose: This is a modified bresenham algorithm that is used to draw
|		 a line into the mask, given a draw value.
|
|         Input: data - the data to draw the line into
|                width  - the width of the data array
|                height - the height of the data array
|                x1, y1 - the begin point
|                x2, y2 - the end point
|		 draw   - the value to set in the mask for drawing
|    Written By: Mark Young
|          Date: Dec 18, 1994
|
------------------------------------------------------------*/
/*ARGSUSED*/
static void draw_line(
  char *data,
  unsigned int width,
  unsigned int height,
  int  x1,
  int  y1,
  int  x2,
  int  y2,
  int  draw)
{
	int    i, position, dx, dy, xincr, yincr;

	dx = kabs(x2 - x1);
	dy = kabs(y2 - y1);
	xincr = (x1 < x2) ? 1 : -1;
	yincr = (y1 < y2) ? 1 : -1;

	if (dx > dy)
	{
	   position = 2*dy - dx;
	   for (i = 0; i <= dx; i++)
	   {
	      data[y1 * (int) width + x1] = draw; x1 += xincr;
	      if (position >= 0)
	      {
		 y1 += yincr;
		 position += 2 * (dy - dx);
	      }
	      else position += 2 * dy;
	   }
	}
	else
	{
	   position = 2*dx - dy;
	   for (i = 0; i <= dy; i++)
	   {
	      data[y1 * (int) width + x1] = draw; y1 += yincr;
	      if (position >= 0)
	      {
		 x1 += xincr;
		 position += 2 * (dx - dy);
	      }
	      else position += 2 * dx;
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: draw_circle - scanline draw circle will rasterize draw
|				     in the mask.
|
|       Purpose: This is a modified bresenham algorithm that is used to draw
|		 an circle into the mask, given a draw value.
|
|         Input: data - the data to draw the line into
|                width  - the width of the data array
|                height - the height of the data array
|                radius - the radius of the circle
|		 draw   - the value to set in the mask for drawing
|    Written By: Mark Young
|          Date: Dec 18, 1994
|
------------------------------------------------------------*/
/*ARGSUSED*/
static void draw_circle(
  char *data,
  unsigned int width,
  unsigned int height,
  int  xc,
  int  yc,
  int  radius,
  int  draw)
{
	int x, y, position;

	x = 0; y = radius; position = 3 - 2 * radius;
	while (x <= y)
	{
	   data[xc + x  +  (yc + y) * (int) width] = draw;
	   data[xc - x  +  (yc + y) * (int) width] = draw;
	   data[xc + x  +  (yc - y) * (int) width] = draw;
	   data[xc - x  +  (yc - y) * (int) width] = draw;
	   data[xc + y  +  (yc + x) * (int) width] = draw;
	   data[xc - y  +  (yc + x) * (int) width] = draw;
	   data[xc + y  +  (yc - x) * (int) width] = draw;
	   data[xc - y  +  (yc - x) * (int) width] = draw;
	   if (x == y)
	      break;

	   if (position < 0)
	      position += 4*x + 6;
	   else
	      position += 4*(x - y--) + 10;

	   x++;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: draw_ellipse - scanline draw ellipse will rasterize draw
|				     in the mask.
|
|       Purpose: This is a modified bresenham algorithm that is used to draw
|		 an ellipse into the mask, given a draw value.
|
|         Input: data - the data to draw the line into
|                width  - the width of the data array
|                height - the height of the data array
|                xc, yc - the center point
|                a, b   - the width & height (major / minor axis)
|		 draw   - the value to set in the mask for drawing
|    Written By: Mark Young
|          Date: Dec 18, 1994
|
------------------------------------------------------------*/
/*ARGSUSED*/
static void draw_ellipse(
  char *data,
  unsigned int width,
  unsigned int height,
  int  xc,
  int  yc,
  int  a,
  int  b,
  int  draw)
{
	double position;
	int    x, y, asqr, bsqr, done = FALSE;

	x = 0; y = b;
	asqr = ksqr(a);
	bsqr = ksqr(b);

	position = bsqr - asqr*b + asqr/4;
	data[xc + x  +  (yc + y) * (int) width] = draw;
	data[xc - x  +  (yc + y) * (int) width] = draw;
	data[xc + x  +  (yc - y) * (int) width] = draw;
	data[xc - x  +  (yc - y) * (int) width] = draw;
	while (asqr*(y - 0.5) > bsqr*(x + 1.0))
	{
	   if (position < 0)
	      position += bsqr*(2*x + 3);
	   else
	      position += bsqr*(2*x + 3) + asqr*(-2*y-- + 2);

	   x++;
	   data[xc + x  +  (yc + y) * (int) width] = draw;
	   data[xc - x  +  (yc + y) * (int) width] = draw;
	   data[xc + x  +  (yc - y) * (int) width] = draw;
	   data[xc - x  +  (yc - y) * (int) width] = draw;
	}

	position = bsqr*ksqr(x + 0.5) + asqr*ksqr(y - 1.0) - asqr*bsqr;
	while (y > 0)
	{
	   if (position < 0)
	      position += bsqr*(2*x++ + 2) + asqr*(-2*y + 3);
	   else
	      position += asqr*(-2*y + 3);

	   y--;
	   data[xc + x  +  (yc + y) * (int) width] = draw;
	   data[xc - x  +  (yc + y) * (int) width] = draw;
	   data[xc + x  +  (yc - y) * (int) width] = draw;
	   data[xc - x  +  (yc - y) * (int) width] = draw;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: compute_extents - compute extents from a set of points
|
|       Purpose: this routine computes the extents for a given set of points.
|		 It returns the begin and end points as well as the overall
|		 width and height in pixels.
|
|         Input: pts - the points
|		 num - the number of points
|        Output: x,y - the begin point
|		 w,h - the overall width & height in pixels
|    Written By: Mark Young
|          Date: Dec 18, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
static void compute_extents(
   XPoint *pts,
   int	  num,
   int	  *x,
   int	  *y,
   unsigned int	*w,
   unsigned int	*h)
{
	int i, xmax, ymax;

	*x = xmax = pts[0].x;
	*y = ymax = pts[0].y;
	for (i = 1; i < num; i++)
	{
	   if (*x > pts[i].x)
	      *x = pts[i].x;
	   else if (xmax < pts[i].x)
	      xmax = pts[i].x;

	   if (*y > pts[i].y)
	      *y = pts[i].y;
	   else if (ymax < pts[i].y)
	      ymax = pts[i].y;
	}
	*w = (xmax - *x) + 1;
	*h = (ymax - *y) + 1;
}

/*-----------------------------------------------------------
|
|  Routine Name: extract_region - extract the region
|
|       Purpose: This routine will extract the desired region and store it
|		 in the specified region of interest (roi) object.
|
|         Input: object - the object in which to get the ROI extent
|		 roi    - the roi to store the region in
|                x,y    - the roi's x&y position
|		 w,h    - the roi's width & height
|		 mask   - a mask if the object is not rectangular
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Jan 01, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */
static int extract_region(
   xvobject object,
   kobject  roi,
   int x,
   int y,
   unsigned int w,
   unsigned int h,
   char *mask)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	kobject  iobject = xwid->image.iobject;

	char     *tmp;
	kaddr    data;
	int      i, xpos, ypos, dpos, tpos, epos, tmpw, tmph, xoffset, yoffset,
	         ow, oh, d, t, e;


	/* clip to the width/height of the input object */
	kpds_get_attribute(iobject, KPDS_VALUE_SIZE, &ow, &oh, &d, &t, &e);
	tmpw = (x+w > ow) ? ow - x : w;
	tmph = (y+h > oh) ? oh - y : h;

	/* if multiple band extraction is false then extract the current band */
	if (xwid->image.roi_multiband == FALSE)
	{
	   d = t = 1;
	   e = xwid->image.image_depth;
	   epos = xwid->image.band_number % xwid->image.esize;
	   dpos = xwid->image.band_number/xwid->image.esize % xwid->image.dsize;
	   tpos = xwid->image.band_number/(xwid->image.esize*xwid->image.dsize)
				 % xwid->image.tsize;
	}
	else dpos = tpos = epos = 0;

	/*
	 *  Sanity check...
	 */
	if (tmpw < 0 || tmph < 0 || x+tmpw < 0 || y+tmph < 0)
	{
	   kerror(XVISUAL, "extract_region", "Extracted Region out of Bounds! \
Trying to extract the region (%d,%d) to (%d,%d), but the Image is of \
dimensions (%d,%d).\n\nAborting region extraction!", x, y, x+w, y+h, ow, oh);
	   return(FALSE);
	}

	/*
	 *  Set the position and region size in which to extract the
	 *  region from.
	 */
	x += xwid->image.xoffset;
	y += xwid->image.yoffset;
	kpds_set_attributes(iobject,
		KPDS_VALUE_POSITION, x, y, dpos, tpos, epos,
		KPDS_VALUE_REGION_SIZE, w, h, d, t, e,
		NULL);

	/*
	 *  Better make sure the mask exists if we are creating multiple roi's
	 *  or our policy is outside.
	 */
	if (!kpds_query_mask(roi) && (xwid->image.multiple_roi == TRUE ||
	     xwid->image.roi_policy == KIMAGE_ROI_OUTSIDE))
	{
	   kpds_create_mask(roi);
	   kpds_set_attribute(roi, KPDS_MASK_DATA_TYPE, KUBYTE);
	   if (xwid->image.roi_policy == KIMAGE_ROI_OUTSIDE)
	      kpds_initialize_mask(roi, 1.0);
	   else
	      kpds_initialize_mask(roi, 0.0);

	   if (xwid->image.roi_policy != KIMAGE_ROI_OUTSIDE)
	      kpds_initialize_value(roi, 0.0, 0.0);
	   else
	      kpds_copy_value_data(iobject, roi, FALSE);
	}
	else if (xwid->image.multiple_roi == FALSE &&
                 xwid->image.roi_policy != KIMAGE_ROI_OUTSIDE)
	{
	   kpds_initialize_value(roi, 0.0, 0.0);
	}

	/* reset x&y if the region we are getting in less than (0,0) */
	if (x < 0) xoffset = x, x = 0; else xoffset = 0;
	if (y < 0) yoffset = y, y = 0; else yoffset = 0;

	if (xwid->image.multiple_roi == FALSE &&
	    xwid->image.roi_policy != KIMAGE_ROI_OUTSIDE)
	{
	   xpos = ypos = 0;
	   kpds_set_attribute(roi, KPDS_VALUE_SIZE, tmpw+xoffset,
			tmph+yoffset, d, t, e);
	}
	else
	   xpos = x, ypos = y;

	/*
	 *  Set the position and region size in which to insert the
	 *  region into.
	 */
	kpds_set_attributes(roi,
		KPDS_VALUE_POSITION, xpos+xoffset, ypos+yoffset, 0, 0, 0,
		KPDS_VALUE_REGION_SIZE, w, h, d, t, e,
		NULL);

	/* Get the value data region and store it into out roi object */
	if ((data = kpds_get_data(iobject, KPDS_VALUE_REGION, NULL)) == NULL)
	   return(FALSE);
	kpds_put_data(roi, KPDS_VALUE_REGION, data); kfree(data);

	/*
	 *  In case there is a mask, we better put the mask as well.
	 */
	if (mask != NULL)
	{
	   if (!kpds_query_mask(roi))
	   {
	      kpds_create_mask(roi);
	      kpds_set_attributes(roi,
		   KPDS_SUBOBJECT_POSITION, x, y, 0, 0, 0,
		   KPDS_MASK_SIZE, tmpw+xoffset, tmph+yoffset, d, t, e,
		   KPDS_MASK_DATA_TYPE, KUBYTE,
		   NULL);
	      kpds_initialize_mask(roi, 0.0);
	   }

	   if (xwid->image.roi_policy ==  KIMAGE_ROI_OUTSIDE)
	      for (i = 0; i < w*h; i++) mask[i] = !mask[i] ? 1 : 0;

	   kpds_set_attributes(roi,
		KPDS_MASK_POSITION, xpos+xoffset, ypos+yoffset, 0, 0, 0,
		KPDS_MASK_REGION_SIZE, w, h, 1, 1, 1,
		NULL);

	   if (xwid->image.multiple_roi == TRUE)
	   {
	      tmp = kpds_get_data(roi, KPDS_MASK_REGION, NULL);
	      if (xwid->image.roi_policy == KIMAGE_ROI_OUTSIDE)
	         for (i = 0; i < w*h; i++) tmp[i] = !tmp[i] ? tmp[i] : mask[i];
	      else
	         for (i = 0; i < w*h; i++) tmp[i] = tmp[i] ? tmp[i] : mask[i];
	   }
	   else tmp = mask;

	   for (i = 0; i < d*t*e; i++)
	   {
	      kpds_set_attribute(roi, KPDS_MASK_POSITION, xpos+xoffset,
			ypos+yoffset, i%d, i/d%t, i/(d*t)%e);
	      kpds_put_data(roi, KPDS_MASK_REGION, tmp);
	   }
	   kpds_set_attribute(roi, KPDS_MASK_POSITION, 0, 0, 0, 0, 0);
	   if (tmp != mask) kfree(tmp);
	}
	kpds_set_attribute(roi, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: extract_line - extract the line
|
|       Purpose: This routine will extract the desired line and store it
|		 in the specified region of interest (roi) object.
|
|         Input: object - the object in which to get the ROI extent
|		 roi    - the roi to store the region in
|                x1,y1  - the roi's x&y position
|		 x2,y2  - the roi's width & height
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Jan 01, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */
static int extract_line(
   xvobject object,
   kobject  roi,
   int x1,
   int y1,
   int x2,
   int y2)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	kobject  iobject = xwid->image.iobject;

	kaddr  point = NULL;
	int    i, position, dx, dy, xincr, yincr;

	dx = kabs(x2 - x1);
	dy = kabs(y2 - y1);
	xincr = (x1 < x2) ? 1 : -1;
	yincr = (y1 < y2) ? 1 : -1;

	kpds_set_attributes(roi,
		KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
		KPDS_VALUE_SIZE, kmax(dx, dy), 1, 1, 1, 1,
		NULL);

	if (dx > dy)
	{
	   position = 2*dy - dx;
	   for (i = 0; i <= dx; i++)
	   {
	      kpds_set_attribute(iobject, KPDS_VALUE_POSITION, x1, y1, 0, 0, 0);
	      point = kpds_get_data(iobject, KPDS_VALUE_POINT, point);
	      kpds_put_data(roi, KPDS_VALUE_POINT, point);

	      x1 += xincr;
	      if (position >= 0)
	      {
		 y1 += yincr;
		 position += 2 * (dy - dx);
	      }
	      else position += 2 * dy;
	   }
	}
	else
	{
	   position = 2*dx - dy;
	   for (i = 0; i <= dy; i++)
	   {
	      kpds_set_attribute(iobject, KPDS_VALUE_POSITION, x1, y1, 0, 0, 0);
	      point = kpds_get_data(iobject, KPDS_VALUE_POINT, point);
	      kpds_put_data(roi, KPDS_VALUE_POINT, point);
	
	      y1 += yincr;
	      if (position >= 0)
	      {
		 x1 += xincr;
		 position += 2 * (dx - dy);
	      }
	      else position += 2 * dx;
	   }
	}
	kfree(point);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: extract_circle - extract the circle
|
|       Purpose: This routine will extract the desired circle and store it
|		 in the specified region of interest (roi) object.
|
|         Input: object - the object in which to get the ROI extent
|		 roi    - the roi to store the region in
|                x,y    - the roi's x&y position
|		 radius - the roi's radius
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Jan 01, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */
static int extract_circle(
   xvobject object,
   kobject  roi,
   int xc,
   int yc,
   int radius)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	kobject  iobject = xwid->image.iobject;

	kaddr  point = NULL;
	int    x, y, position;

	x = 0; y = radius; position = 3 - 2 * radius;
	kpds_set_attributes(roi,
		KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
		KPDS_VALUE_SIZE, radius*8, 1, 1, 1, 1,
		NULL);
	while (x <= y)
	{
/*
	   data[xc + x  +  (yc + y) * (int) width] = draw;
	   data[xc - x  +  (yc + y) * (int) width] = draw;
	   data[xc + x  +  (yc - y) * (int) width] = draw;
	   data[xc - x  +  (yc - y) * (int) width] = draw;
	   data[xc + y  +  (yc + x) * (int) width] = draw;
	   data[xc - y  +  (yc + x) * (int) width] = draw;
	   data[xc + y  +  (yc - x) * (int) width] = draw;
	   data[xc - y  +  (yc - x) * (int) width] = draw;
 */
	   if (x == y)
	      break;

	   if (position < 0)
	      position += 4*x + 6;
	   else
	      position += 4*(x - y--) + 10;

	   x++;
	}
	kfree(point);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: extract_roi - gets the roi 
|
|       Purpose: This routine will get the roi extents which will then
|		 can be used to get 
|
|         Input: object - the object in which to get the ROI extent
|		 roi    - the roi to store the region in
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Dec 30, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
static int extract_roi(
   xvobject object,
   kobject  roi,
   int roi_num)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);

	XPoint *pts  = NULL;
	unsigned int w, h, a, b, r;
	int x, y, x1, y1, x2, y2, num, erase;


	erase = (xwid->image.multiple_roi == FALSE);
	switch (xwid->image.roi_shape)
	{
	   case KIMAGE_ROI_POLYLINE:
	   case KIMAGE_ROI_FREEHAND:
	        if (xwid->image.roi_shape == KIMAGE_ROI_POLYLINE)
		   pts = xvw_getextent_polyline(object, erase, &num);
		else
		   pts = xvw_getextent_freehand(object, erase, &num);

		if (pts == NULL)
	           return(FALSE);

		xvw_getroi_polyline(object, roi, pts, num);
		break;

	   case KIMAGE_ROI_RECTANGLE:
	        if (!xvw_getextent_rectangle(object, erase, &x, &y, &w, &h))
	           return(FALSE);

	        xvw_getroi_rectangle(object, roi, x, y, w, h);
		break;

	   case KIMAGE_ROI_CIRCLE:
	        if (!xvw_getextent_circle(object, erase, &x, &y, &r))
	           return(FALSE);

		xvw_getroi_circle(object, roi, x, y, r);
		break;

	   case KIMAGE_ROI_ELLIPSE:
	        if (!xvw_getextent_ellipse(object, erase, &x, &y, &a, &b))
	           return(FALSE);

		xvw_getroi_ellipse(object, roi, x, y, a, b);
		break;

	   case KIMAGE_ROI_LINE:
	        if (!xvw_getextent_line(object, erase, &x1, &y1, &x2, &y2))
	           return(FALSE);

		xvw_getroi_line(object, roi, x1, y1, x2, y2);
		break;
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ImageGetRoi
|
|       Purpose: ImageGetRoi is used to get a region of interest according
|		 to the displayed image.
|
|         Input: object    - the object in which we will be getting the roi
|		 attribute - the attribute name XVW_IMAGE_ROI
|        Output: calldata  - pointer to retrieve the roi
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Dec 18, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
int ImageGetRoi(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	kobject *roi  = (kobject *) calldata;

	XEvent	 event;
	Window	 window;
	int      roi_num = 1;
	kobject  iobject = xwid->image.iobject;
	int      policy  = xwid->image.roi_policy;
	Boolean  multiple_roi = xwid->image.multiple_roi;


	/*
	 *  Get the ROI extent that we need in which to extract the data set
	 */
	*roi = kpds_create_object();
	kpds_copy_mask_data(iobject, *roi, FALSE);
	kpds_copy_object_attr(iobject, *roi);
	window = xvw_window(object);

	do
	{
	   if (!extract_roi(object, *roi, roi_num))
	      break;

	   roi_num++;
	   if (!xwid->image.multiple_roi) break;

	   /* yuck!!  no i'm not going to explain this, so please don't ask */
	   while (1)
	   {
	      XtAppNextEvent(xvw_appcontext(NULL), &event);
	      if (event.type == ButtonPress && event.xany.window == window)
		 break;

	      XtDispatchEvent(&event);
	   }
	} while (event.xbutton.button != Button3);

	if (multiple_roi == TRUE)
	   xvw_refresh(object);

	if (roi_num == 1)
	{
	   kpds_close_object(*roi); *roi = NULL;
	   return(FALSE);
	}
	kpds_copy_remaining_data(iobject, *roi);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ImageSetRoi
|
|       Purpose: ImageSetRoi is used to set the region of interest according
|		 to the displayed image.
|
|         Input: object    - the object in which we will be setting the roi
|		 attribute - the attribute name XVW_IMAGE_ROI
|        Output: calldata  - pointer to set the roi
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Dec 31, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
int ImageSetRoi(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	kobject *roi = (kobject *) calldata;
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);

	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_getroi_line - extracts a line region from the image 
*
*       Purpose: This routine extracts a line region from an image object.
*
*         Input: object - the image object to extract from
*                x1,y1  - the starting x&y point
*                x2,y2  - the ending x&y point
*
*       Returns: the newly created region on success, NULL otherwise
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Jan 04, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kobject xvw_getroi_line(
   xvobject object,
   kobject  roi,
   int	    x1,
   int	    y1,
   int	    x2,
   int	    y2)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	kobject  iobject = xwid->image.iobject;
	int      policy = xwid->image.roi_policy;

	int  x, y, status;
	char *mask = NULL;
	unsigned int w, h;
	kobject tmp = NULL;

	/*
	 *  Create the initial roi object
	 */
	if (roi == NULL)
	{
	   tmp = roi = kpds_create_object();
	   kdms_copy_object(iobject, roi, TRUE, TRUE);
	}

	/* Get the extent */
	w = kabs(x2-x1)+1; h = kabs(y2-y1)+1;
	x = kmin(x1,x2);   y = kmin(y1,y2);

	/* Draw the line and extract the region */
	mask = kcalloc(1, w*h);
	draw_line(mask, w, h, x1-x, y1-y, x2-x, y2-y, 1);

	if (xwid->image.roi_presentation != KIMAGE_ROI_SIGNAL)
	   status = extract_region(object, roi, x, y, w, h, mask);
	else
	   status = extract_line(object, roi, x, y, w, h);

	kfree(mask);
	if (status == FALSE)
	{
	   if (tmp) kpds_close_object(tmp);
	   return(NULL);
	}
	return(roi);
}

/************************************************************
*
*  Routine Name: xvw_getroi_rectangle - extracts a rectangular region
*					from the image 
*
*       Purpose: This routine extracts a rectangular region from an
*		 image object.
*
*         Input: object - the image object to extract from
*                x,y    - the x&y point
*                w,h    - the width & height
*
*       Returns: the newly created region on success, NULL otherwise
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Jan 04, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kobject xvw_getroi_rectangle(
   xvobject object,
   kobject  roi,
   int	    x,
   int	    y,
   int	    w,
   int	    h)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	kobject  iobject = xwid->image.iobject;
	int      policy = xwid->image.roi_policy;

	int  status;
	char *mask = NULL;
	kobject tmp = NULL;

	/*
	 *  Create the initial roi object
	 */
	if (roi == NULL)
	{
	   tmp = roi = kpds_create_object();
	   kdms_copy_object(iobject, roi, TRUE, TRUE);
	}

	if (policy == KIMAGE_ROI_OUTLINE)
	{
	   mask = kcalloc(1, w*h); x = w; y = h;
           draw_line(mask, w, h, 0, 0, x-1, 0, 1);
           draw_line(mask, w, h, x-1, 0, x-1, y-1, 1);
           draw_line(mask, w, h, x-1, y-1, 0, y-1, 1);
           draw_line(mask, w, h, 0, y-1, 0, 0, 1);
	}
	else if (xwid->image.multiple_roi == TRUE ||
		 policy == KIMAGE_ROI_OUTSIDE)
	{
	   mask = kcalloc(1, w*h);
	   flood_fill(mask, w, h, w/2, h/2, 1);
	}

        if (xwid->image.roi_presentation != KIMAGE_ROI_SIGNAL)
	   status = extract_region(object, roi, x, y, w, h, mask);
        else
           status = extract_line(object, roi, x, y, w, h);

	kfree(mask);
	if (status == FALSE)
	{
	   if (tmp) kpds_close_object(tmp);
	   return(NULL);
	}
	return(roi);
}

/************************************************************
*
*  Routine Name: xvw_getroi_circle - extracts a circle region from the image 
*
*       Purpose: This routine extracts a circle region from an image object.
*
*         Input: object - the image object to extract from
*                x,y    - the center x&y point
*                r      - the radius of the circle
*
*       Returns: the newly created region on success, NULL otherwise
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Jan 04, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kobject xvw_getroi_circle(
   xvobject object,
   kobject  roi,
   int	    x,
   int	    y,
   int	    r)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	kobject  iobject = xwid->image.iobject;
	int      policy = xwid->image.roi_policy;

	int     status;
	unsigned int w, h;
	char *mask  = NULL;
	kobject tmp = NULL;

	/*
	 *  Create the initial roi object
	 */
	if (roi == NULL)
	{
	   roi = kpds_create_object();
	   kdms_copy_object(iobject, roi, TRUE, TRUE);
	}

	/* Get the extent */
	w = h = r*2+1;

	/* Draw the circle and extract the region */
	mask = kcalloc(1, w*h);
	draw_circle(mask, w, h, r, r, r, 1);
	if (policy == KIMAGE_ROI_INSIDE || policy == KIMAGE_ROI_OUTSIDE)
	   flood_fill(mask, w, h, r, r, 1);

        if (xwid->image.roi_presentation != KIMAGE_ROI_SIGNAL)
           status = extract_region(object, roi, x-r, y-r, w, h, mask);
        else
           status = extract_line(object, roi, x, y, w, h);

	kfree(mask);
	if (status == FALSE)
	{
	   if (tmp) kpds_close_object(tmp);
	   return(NULL);
	}
	return(roi);
}

/************************************************************
*
*  Routine Name: xvw_getroi_ellipse - extracts a ellipse region from the image 
*
*       Purpose: This routine extracts a ellipse region from an image object.
*
*         Input: object - the image object to extract from
*                x,y    - the center x&y point
*                a,b    - the major/minor axis
*
*       Returns: the newly created region on success, NULL otherwise
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Jan 04, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kobject xvw_getroi_ellipse(
   xvobject object,
   kobject  roi,
   int	    x,
   int	    y,
   int	    a,
   int	    b)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	kobject  iobject = xwid->image.iobject;
	int      policy = xwid->image.roi_policy;

	int  status;
	unsigned int w, h;
	char  *mask = NULL;
	kobject tmp = NULL;

	/*
	 *  Create the initial roi object
	 */
	if (roi == NULL)
	{
	   tmp = roi = kpds_create_object();
	   kdms_copy_object(iobject, roi, TRUE, TRUE);
	}

	/* Get the extent */
	w = a*2+1; h = b*2+1;

	/* Draw the ellipse and extract the region */
	mask = kcalloc(1, w*h);
	draw_ellipse(mask, w, h, a, b, a, b, 1);
	if (policy == KIMAGE_ROI_INSIDE || policy == KIMAGE_ROI_OUTSIDE)
	   flood_fill(mask, w, h, a, b, 1);

        if (xwid->image.roi_presentation != KIMAGE_ROI_SIGNAL)
           status = extract_region(object, roi, x-a, y-b, w, h, mask);
        else
           status = extract_line(object, roi, x-a, y-b, w, h);

	kfree(mask);
	if (status == FALSE)
	{
	   if (tmp) kpds_close_object(tmp);
	   return(NULL);
	}
	return(roi);
}

/************************************************************
*
*  Routine Name: xvw_getroi_polyline - extracts a polyline region from
*				the image 
*
*       Purpose: This routine extracts a polyline region from an image object.
*
*         Input: object - the image object to extract from
*                pts	- the polyline points to be extracted
*                num    - the number of points
*
*       Returns: the newly created region on success, NULL otherwise
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Jan 04, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kobject xvw_getroi_polyline(
   xvobject object,
   kobject  roi,
   XPoint   *pts,
   int	    num)
{
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	kobject  iobject = xwid->image.iobject;
	int      policy = xwid->image.roi_policy;

	unsigned int w, h;
	char  *mask = NULL;
	kobject tmp = NULL;
	int  i, x, y, status;

	/*
	 *  Create the initial roi object
	 */
	if (roi == NULL)
	{
	   tmp = roi = kpds_create_object();
	   kdms_copy_object(iobject, roi, TRUE, TRUE);
	}

	/* Get the extent */
	compute_extents(pts, num, &x, &y, &w, &h);

	/* Draw the polyline and extract the region */
	mask = kcalloc(1, w*h);
	for (i = 1; i < num; i++)
	{
	   draw_line(mask, w, h, pts[i-1].x - x, pts[i-1].y - y,
                        pts[i].x - x, pts[i].y - y, 1);
	}

	if (policy == KIMAGE_ROI_INSIDE || policy == KIMAGE_ROI_OUTSIDE)
	{
	   draw_line(mask, w, h, pts[0].x - x, pts[0].y - y,
                        pts[num-1].x - x, pts[num-1].y - y, 1);
	   scanline_fill(mask, w, h, 1);
	}

        if (xwid->image.roi_presentation != KIMAGE_ROI_SIGNAL)
           status = extract_region(object, roi, x, y, w, h, mask);
        else
           status = extract_line(object, roi, x, y, w, h);

	kfree(mask);
	if (status == FALSE)
	{
	   if (tmp) kpds_close_object(tmp);
	   return(NULL);
	}
	return(roi);
}
