/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                 3D Contour Drawing Utilities
   >>>>
   >>>>  Private:
   >>>>			X3D_draw_contour()
   >>>>			X3D_get_contour_levels()
   >>>>   Static:
   >>>>			display_contour()
   >>>>			compute_contour()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: display_contour
|
|       Purpose:
|
|         Input: id       -
|                contours -
|                num      -
|                color    -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young & Scott Wilson
|          Date:
| Modifications:
|
------------------------------------------------------------*/

static void display_contour(
   int    id,
   Coord  *contours,
   int    num,
   XColor *color)
{
        X3DGraphics     *graphics;
        XSegment        *segments;
        int             size;


        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           kfprintf (kstderr,"X3D_display_contour:\n");
           kfprintf (kstderr,"\t unknown graphics id\n");
           return;
        }

        size   = num/2;
        if ((segments = (XSegment *) kmalloc ((unsigned) size *
                         sizeof(XSegment))) == NULL)
        {
           kfprintf (kstderr,"X3D_display_contour:\n");
           kfprintf (kstderr,"\t not enough memory\n");
           return;
        }

        if ((_X3D_convert_wc_to_dc_seg(graphics, graphics->matrix, contours,
                        segments, num)) == FALSE)
        {
           kfree(segments);
           return;
        }
        X3D_draw[graphics->device].draw_color(graphics, color, FALSE);
        X3D_draw[graphics->device].segments(graphics, segments, size);
        kfree(segments);
}


/*-----------------------------------------------------------
|
|  Routine Name: compute_contour
|
|       Purpose:
|
|         Input: id	-
|		 coords	-
|		 rows	-
|		 cols	-
|		 level	-
|		 color	-
|		 minz	-
|		 maxz	-
|		 base	-
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young & Scott Wilson
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void compute_contour(
   int            id,
   Coord          *coords,
   int            rows,
   int            cols,
   Real           level,
   XColor         *color,
   register  Real *minz,
   register  Real *maxz,
   Real           base)
{
	int	 size, i, j, c, n, num;
	register Coord *pt0, *pt1, *pt2, *pt3, *contours; 
        Coord box[4];

	size = rows * cols;
	if ((contours = (Coord *) kmalloc(sizeof(Coord) * size)) == NULL)
	{
	   kfprintf (kstderr,"X3D_compute_contour:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	/*
	 *  Initialize the box Z value to the specified base.  We do this
	 *  now rather than for each and every contour point.
	 */
	for (i = 0; i < 4; i++)
	   box[i].z = base;

	pt0 = coords;
	pt1 = pt0 + 1;
	pt2 = coords + rows;
	pt3 = pt2 + 1;
	num = 0;

	for (i = 0; i < cols -1; i++)
	{
	   for (j = 0; j < rows -1; j++)
	   {
	      if (*minz != *maxz && level >= *minz && level <=  *maxz)
	      {
                 c = 0;

                 /*
                  *  Check for intersections and find coords.
                  */
                 if ((pt0->z <= level && pt2->z >= level) ||
		     (pt0->z >= level && pt2->z <= level))
                 {
		    box[c].x   = kintercept(pt0->x, pt2->x, pt0->z, pt2->z, level);
		    box[c++].y = kintercept(pt0->y, pt2->y, pt0->z, pt2->z, level);
                 }

                 if ((pt1->z <= level && pt0->z >= level) ||
		     (pt1->z >= level && pt0->z <= level))
                 {
		    box[c].x   = kintercept(pt0->x, pt1->x, pt0->z, pt1->z, level);
		    box[c++].y = kintercept(pt0->y, pt1->y, pt0->z, pt1->z, level);
                 }

                 if ((pt3->z <= level && pt2->z >= level) || 
		     (pt3->z >= level && pt2->z <= level))
                 {
		    box[c].x   = kintercept(pt2->x, pt3->x, pt2->z, pt3->z, level);
		    box[c++].y = kintercept(pt2->y, pt3->y, pt2->z, pt3->z, level);
                 }

                 if ((pt3->z <= level && pt1->z >= level) ||
		     (pt3->z >= level && pt1->z <= level))
                 {
		    box[c].x   = kintercept(pt1->x, pt3->x, pt1->z, pt3->z, level);
		    box[c++].y = kintercept(pt1->y, pt3->y, pt1->z, pt3->z, level);
		 }

                 /*
                  *  Perform plotting chores for this box
                  */
                 if (c > 1)
                 {
                    for (n = 0; n < (c -1); n++) 
                    {
		       contours[num++] = box[n];
		       contours[num++] = box[n +1];
		    }  
 
		    if ((num + 6) >= size)
		    {
		       display_contour(id, contours, num, color);
		       num = 0;
		    }
		 }
	      }
	      pt0++; pt1++; pt2++; pt3++;
	      minz++; maxz++;
	   }
	   pt0++; pt1++; pt2++; pt3++;
	   minz++; maxz++;
	}
	display_contour(id, contours, num, color);
	kfree(contours);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_contour
|
|       Purpose: Produces a contour map of an arbitrary function
|		 given in array points.  This routine uses a grid square
|		 scan algorithm where each grid square is analyzed
|		 for a zero crossing on each edge of the square at the
|		 desired contour level.  For each crossing, an interpolation
|		 is performed to find the location of intersection with
|		 the grid square.  Two intersections implies a line.
|		 Four intersections implies a saddle point or ridge.
|		 The intersection coordinates are stored in coordinate array
|		 and are plotted according to the above criteria.
|
|         Input: id         - X3D graphics structure ID
|		 coords     - array of world coordinate points
|		 rows	    - number of points per row
|		 size       - number of points
|		 levels	    - array of Real number containing the desired
|		 	      contour levels.
|		 colors	    - array of xcolors containing the color to be
|			      associated with each level
|	         num_levels - number of desired contour levels
|	 	 base	    - a boolean flag indicating if the contours
|			      are to be projected to the bottom of the window
|			      or at the desired contour level.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Scott Wilson
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_contour(
   int    id,
   Coord  *coords,
   int    rows,
   int    size,
   Real   *levels,
   XColor *colors,
   int    color_offset,
   int    num_levels,
   int    base)
{
        X3DGraphics     *graphics;
	register Coord	*pt0, *pt1, *pt2, *pt3;
	register Real	*zmin, *zmax, *minz, *maxz;

	Real	 wc_zmin, wc_zmax, base_level;
	int	 i, j, cols, indx, free_levels = FALSE;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_contour:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	/*
	 *   Compute the minimum and maximum values for each set of
	 *   contour points.  This will be used by X3D_compute_contour
	 *   for each contour level.  We use it as a quick check to see
	 *   if a contour exist for that given level. 
	 */
	zmin   = (Real *) kmalloc(sizeof(Real) * size);
	zmax   = (Real *) kmalloc(sizeof(Real) * size);
	minz   = zmin;
	maxz   = zmax;

	if ((!zmin) || (!zmax))
	{
	   kfprintf (kstderr,"X3D_draw_contour:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	cols = size/rows;
	pt0 = coords;
	pt1 = pt0 + 1;
	pt2 = coords + rows;
	pt3 = pt2 + 1;

	/*
	 *  Need to find the overall min & max value for Z.  This information
	 *  is used if levels is NULL.  We then create our own levels evenly
	 *  distributed between the wc_zmin and wc_zmax.
	 */
	wc_zmin = wc_zmax = kmin4(pt0->z, pt1->z, pt2->z, pt3->z);

	for (i = 0; i < cols -1; i++)
	{
           for (j = 0; j < rows -1; j++)
           {
	       *minz = kmin4(pt0->z, pt1->z, pt2->z, pt3->z);
	       *maxz = kmax4(pt0->z, pt1->z, pt2->z, pt3->z);

	       if (*minz < wc_zmin)  wc_zmin = *minz;
	       if (*maxz > wc_zmax)  wc_zmax = *maxz;

	       pt0++; pt1++; pt2++; pt3++;
	       minz++; maxz++;
	   }
	   pt0++; pt1++; pt2++; pt3++;
	   minz++; maxz++;
	}

	if (levels == NULL)
	{
	   free_levels = TRUE;
	   levels = X3D_get_contour_levels(wc_zmin, wc_zmax, 0.0, num_levels);
	   if (levels == NULL)
	   {
	      kfree(zmin); kfree(zmax);
	      return;
	   }
	}

	for (i = 0; i < num_levels; i++)
	{
	   if (base)
	      base_level = graphics->wc_min.z;
	   else
	      base_level = levels[i];

	   indx = i+color_offset;
	   compute_contour(id, coords, rows, cols, levels[i], &colors[indx],
			   zmin, zmax, base_level);
	}

	/*
	 *  Cleanup time.
	 */
	if (free_levels) kfree(levels);
	kfree(zmin); kfree(zmax);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_contour_levels
|
|       Purpose: Creates contour levels
|
|         Input: min_level - The minimum contour level.
|		 max_level - The maximum contour level.
|		 offset    - The offset value to add the given contour
|			     levels.
|		 num        -  Number of contour levels to be generated.
|
|        Output: none
|
|       Returns: A real array containing the defined contour levels.
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Real *X3D_get_contour_levels(
   Real min_level,
   Real max_level,
   Real offset,
   int  num)
{
	int	i;
	Real	factor, *levels;

	if ((levels = (Real *) kmalloc(sizeof(Real) * num)) == NULL)
	{
	   kinfo(KSTANDARD, "X3D_get_contour_levels:  Could not alloc \
contour %d levels\n", num);
	   return(NULL);
	}

	factor = (max_level - min_level)/(num -1);
	for (i = 0; i < num; i++)
	{
	    levels[i] = min_level + i * factor + offset;
	}
	return(levels);
}
