/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                 3D Bezier Drawing Utilities
   >>>>
   >>>>  Private:
   >>>>			X3D_draw_bezier()
   >>>>			X3D_blend_patch()
   >>>>			X3D_bezier_surface()
   >>>>			_X3D_draw_patch()
   >>>>			_X3D_fill_patch()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"

int bezier_step = 16;

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_bezier - draws a 3D mesh
|
|       Purpose Uses X11 graphics routines to connect a series
|		of polylines to make a 3D plot in a mesh pattern.
|		This is done by drawing both lines along both
|		the columns and rows.
|
|         Input: id        - X3D graphics structure ID
|                pts       - array of world coordinate points
|                row_size  - number of points in each polyline
|                size      - number of points
|                fg        - foreground color
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_bezier(
   int    id,
   Coord  *pts,
   int    row_size,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
	Coord		control[4];
	int		i, col_size;
	Real		unit_step, step = 0.0;

	col_size = size/row_size;
	if (col_size != 4 && row_size != 4)
	{
	   kfprintf (kstderr,"X3D_draw_bezier:\n");
	   kfprintf (kstderr,"\t Only able to plot 4x4 plots (a patch)\n");
	   return;
	}

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_bezier:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	unit_step = 1.0/bezier_step;
	for (i = 0; i <= bezier_step; i++)
	{
	   X3D_blend_patch(pts[0],pts[1],pts[2],pts[3],step, &control[0]);
	   X3D_blend_patch(pts[4],pts[5],pts[6],pts[7],step, &control[1]);
	   X3D_blend_patch(pts[8],pts[9],pts[10],pts[11],step, &control[2]);
	   X3D_blend_patch(pts[12],pts[13],pts[14],pts[15],step, &control[3]);
	   _X3D_draw_patch(graphics, control[0], control[1], control[2],
			     control[3], bezier_step);

	   X3D_blend_patch(pts[0],pts[4],pts[8],pts[12],step, &control[0]);
	   X3D_blend_patch(pts[1],pts[5],pts[9],pts[13],step, &control[1]);
	   X3D_blend_patch(pts[2],pts[6],pts[10],pts[14],step, &control[2]);
	   X3D_blend_patch(pts[3],pts[7],pts[11],pts[15],step, &control[3]);
	   _X3D_draw_patch(graphics, control[0], control[1], control[2],
			     control[3], bezier_step);

	   step += unit_step;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_blend_patch
|
|       Purpose: 
|
|         Input: pt0	 -
|		 pt1	 -
|		 pt2	 -
|		 pt3	 -
|		 step	 -
|		 control -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_blend_patch(
   Coord          pt0,
   Coord          pt1,
   Coord          pt2,
   Coord          pt3,
   register Real  step,
   Coord          *control)
{
	register Real	t, t0, t1, t2, t3;

	t   = (1 - step);
	t0  = t*t*t;
	t1  = 3*step*t*t;
	t2  = 3*step*step*t;
	t3  = step*step*step;

	control->x = pt0.x*t0 + pt1.x*t1 + pt2.x*t2 + pt3.x*t3;
	control->y = pt0.y*t0 + pt1.y*t1 + pt2.y*t2 + pt3.y*t3;
	control->z = pt0.z*t0 + pt1.z*t1 + pt2.z*t2 + pt3.z*t3;
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_draw_patch
|
|       Purpose: 
|
|         Input: graphics    -
|                pt0	     -
|                pt1	     -
|                pt2	     -
|                pt3	     -
|                bezier_step -
|
|        Output: none
|                argument5 - explanation
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void _X3D_draw_patch(
   X3DGraphics *graphics,
   Coord       pt0,
   Coord       pt1,
   Coord       pt2,
   Coord       pt3,
   int         bezier_stepsize)
{
	int	i, size;
	Coord	*coords;
	XPoint	*points;
	Real	unit_step, step = 0.0;


	size = bezier_stepsize + 1;
	if ((coords = (Coord *) kmalloc(size * sizeof (Coord))) == NULL)
	{
	   kfprintf (kstderr,"X3D_display_patch:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if ((points = (XPoint *) kmalloc(size * sizeof (XPoint))) == NULL)
	{
	   kfprintf (kstderr,"X3D_display_patch:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	unit_step = 1.0/bezier_step;
	for (i = 0; i <= bezier_step; i++)
	{
	   X3D_blend_patch(pt0, pt1, pt2, pt3, step, &coords[i]);
	   step += unit_step;
	}

	if((_X3D_convert_wc_to_dc(graphics, graphics->matrix, coords, points,
			size)) == FALSE)
	{
	   kfree(coords);
	   kfree(points);
	   return;
	}
	X3D_draw[graphics->device].lines(graphics, points, size,
			CoordModeOrigin);
	
	kfree(coords);
	kfree(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_bezier_surface
|
|       Purpose:
|
|         Input: id         - X3D graphics structure ID
|                pts        - array of world coordinate points
|                row_size   - number of points in each polyline
|                size       - number of points
|                levels     -
|                colors     -
|                num_levels -
|                wcmin	    -
|                wcmax	    -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_bezier_surface(
   int    id,
   Coord  *pts,
   int    row_size,
   int    size,
   Real   *levels,
   XColor *colors,
   int    num_levels,
   Coord  wcmin,
   Coord  wcmax)
{
	X3DGraphics	*graphics;
	Coord		control[16];
	int		i, col_size;
	Real		unit_step, step = 0.0;

	col_size = size/row_size;
	if (col_size != 4 && row_size != 4)
	{
	   kfprintf (kstderr,"X3D_bezier_surface:\n");
	   kfprintf (kstderr,"\t Only able to plot 4x4 plots (a patch)\n");
	   return;
	}

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_bezier_surface:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	unit_step = 1.0/bezier_step;
	X3D_blend_patch(pts[0],pts[1],pts[2],pts[3],step, &control[0]);
	X3D_blend_patch(pts[4],pts[5],pts[6],pts[7],step, &control[1]);
	X3D_blend_patch(pts[8],pts[9],pts[10],pts[11],step, &control[2]);
	X3D_blend_patch(pts[12],pts[13],pts[14],pts[15],step, &control[3]);

	X3D_blend_patch(pts[0],pts[4],pts[8],pts[12],step, &control[4]);
	X3D_blend_patch(pts[1],pts[5],pts[9],pts[13],step, &control[5]);
	X3D_blend_patch(pts[2],pts[6],pts[10],pts[14],step, &control[6]);
	X3D_blend_patch(pts[3],pts[7],pts[11],pts[15],step, &control[7]);
	step += unit_step;
	for (i = 1; i <= bezier_step; i++)
	{
	   X3D_blend_patch(pts[0],pts[1],pts[2],pts[3],step, &control[8]);
	   X3D_blend_patch(pts[4],pts[5],pts[6],pts[7],step, &control[9]);
	   X3D_blend_patch(pts[8],pts[9],pts[10],pts[11],step, &control[10]);
	   X3D_blend_patch(pts[12],pts[13],pts[14],pts[15],step, &control[11]);

	   X3D_blend_patch(pts[0],pts[4],pts[8],pts[12],step, &control[12]);
	   X3D_blend_patch(pts[1],pts[5],pts[9],pts[13],step, &control[13]);
	   X3D_blend_patch(pts[2],pts[6],pts[10],pts[14],step, &control[14]);
	   X3D_blend_patch(pts[3],pts[7],pts[11],pts[15],step, &control[15]);
	   _X3D_fill_patch(graphics, control, bezier_step, levels, colors,
			  num_levels, wcmin, wcmax);

	   for (i = 0; i < 8; i++)
	      control[i] = control[i + 8];

	   step += unit_step;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_fill_patch
|
|       Purpose:
|
|         Input: graphics    -
|                pts         - array of world coordinate points
|                bezier_step -
|                levels      -
|                colors      -
|                num_levels  -
|                wcmin	     -
|                wcmax	     -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void _X3D_fill_patch(
   X3DGraphics *graphics,
   Coord       *pts,
   int         bezier_stepsize,
   Real        *levels,
   XColor      *colors,
   int         num_levels,
   Coord       wcmin,
   Coord       wcmax)
{
	int	i, j, m, indx, size;
	Coord	*coords;
	Real	unit_step, step;


	size = bezier_stepsize +1;
	if ((coords = (Coord *) kmalloc(size * size * sizeof (Coord))) == NULL)
	{
	   kfprintf (kstderr,"X3D_display_patch:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	m = 0;
	unit_step = 1.0/bezier_stepsize;
	for (i = 0; i <= bezier_stepsize; i++)
	{
	   step = 0.0;
	   for (j = 0; j <= bezier_stepsize; j++)
	   {
	       indx = i*size + j;
	       X3D_blend_patch(pts[indx],pts[indx+1],pts[indx+2],pts[indx+3],
			       step, &coords[m++]);
	       step += unit_step;
	   }
	}
	/* Color offset of 0?  Not sure...    - SK */
	X3D_shade_phong(graphics->id, coords, 4, 16, levels, colors, 0,
			num_levels, wcmin, wcmax, KPLOT3D_SHADE_NORMAL);
	kfree(coords);
}
