 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>     Routines For Getting Components of GUI items      <<<<
   >>>>                                                       <<<<
   >>>>  Private:      xvf_get_backplane()                    <<<<
   >>>>                xvf_get_title_back()                   <<<<
   >>>>                xvf_get_line()                         <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:      xvf_get_xvobject()                     <<<<
   >>>>                xvf_get_kformstruct()                  <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"

static kselection *
search_selections_for_xvobject PROTO((kselection *, xvobject));


/************************************************************
*
*
*  Routine Name: xvf_get_xvobject - return desired xvobject component of kformstruct
*
*       Purpose: Given a pointer to a kformstruct, and an "item_part" indicator,
*                returns the desired xvobject associated with the selection 
*                represented by the kformstruct, so that the application
*                may make changes to its GUI items using xvw_set_attributes()
*                which may not be supported by xvf_set_attributes().  
*
*                NOTE: Please see xvf_set_attributes() before using this more 
*                      difficult method of implementing GUI changes.  
*
*                Depending on the type of GUI item, it may be made up of one 
*                or many xvobjects.  For example, an action button consists 
*                only of a button GUI object.  A Float selection, however, will 
*                have at least a backplane, a label object, and a text object. 
*                If it is optional, it will also have a small button object 
*                that serves as the optional box;  if it is "live", it will 
*                have a pixmap object to display the stylized <cr> sign,
*                and it may also contain a scrollbar object.  The "item_part"
*                argument is used to specify which part of the GUI item is
*                desired.
*
*                For example, when passed a kformstruct associated with an 
*                Action Button and an "item_part" of XVF_BUTTON_OBJ, 
*                xvf_get_xvobject() will return the xvobject which is the GUI 
*                button object.  When passed a kformstruct associated with an 
*                InputFile selection, and an "item_part" of XVF_BACKPLANE, 
*                xvf_get_xvobject() will return the xvobject which serves as 
*                the backplane for the InputFile selection.  If an "item_part"
*                is requested which does not exist in the GUI item represented
*                by the kformstruct, an error message will be printed and NULL
*                will be returned.
*
* ! Legal Item Parts Include:
*
* ! XVF_BACKPLANE   : The backplane of the GUI item.
* ! XVF_OPT_OBJ     : The optional box of a GUI item (optional selection only).
* ! XVF_BUTTON_OBJ  : The button on a GUI item (GUI buttons, cycles, lists, etc)
* ! XVF_LABEL_OBJ   : The label on a GUI item
* ! XVF_TEXT_OBJ    : The text box of a GUI selection 
* ! XVF_SCROLL_OBJ  : Scrollbar of a Float, Integer, or Double selection
* ! XVF_PIXMAP_OBJ  : Stylized <cr> pixmap of a "live" selection
*
*         Input: kformstruct - Pointer to the generic structure representing
*                              the GUI item's node in the form tree; appropriate
*                              candidates are automatically generated in 
*                              "form_info.h" by conductor.
*
*                item_part   - Indicates the part of the gui desired.
*
*                create      - the xvobject in question may not have been 
*                              created yet (ie, if a subform or a pane has 
*                              not been mapped yet). Passing FALSE indicates
*                              that NULL should be returned for the xvobject
*                              if it has not yet been created;  passing TRUE
*                              specifies that the object should be created
*                              and then returned.
*
*        Output: 
*       Returns: The xvobject associated with the kformstruct on success. 
*                If 'create' is TRUE, NULL is returned only on failure;
*                If 'create' is FALSE, NULL is returned when the xvobject
*                has not yet been created.
*  Restrictions:
*    Written By: Danielle Argiro
*          Date: March 16, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
************************************************************/
xvobject xvf_get_xvobject(
    kform_struct *kformstruct,
    int          item_part,
    int          create)
{
	xvobject obj = NULL;
	xvobject back = NULL;

	if (!(xvf_check_item_part(kformstruct, item_part)))
	    return(NULL);

	back = xvf_get_backplane(kformstruct, create);

	if (item_part == XVF_BACKPLANE)
	    obj = back;

	else if (item_part == XVF_OPT_OBJ)
	    obj = kformstruct->Selptr->opt_object;

	else if (item_part == XVF_TEXT_OBJ)
	    obj = kformstruct->Selptr->value_object;

	else if (item_part == XVF_SCROLL_OBJ)
	    obj = kformstruct->Selptr->scroll;

	else if (item_part == XVF_PIXMAP_OBJ)
	    obj = kformstruct->Selptr->pix_object;

	else if (item_part == XVF_LABEL_OBJ)
	{
	    if ((kformstruct->flag == KUIS_STARTMASTER) ||
	        (kformstruct->flag == KUIS_STARTPANE)   ||
		(kformstruct->flag == KUIS_STARTGUIDE))
	        obj = kformstruct->Controlptr->label_object;
	    else obj = kformstruct->Selptr->label_object;
	}

	else if (item_part == XVF_BUTTON_OBJ)
	{
	    if (kformstruct->flag == KUIS_SUBFORMBUTTON)
		obj = kformstruct->Subformptr->button;
	    else if (kformstruct->flag == KUIS_GUIDEBUTTON)
		obj = kformstruct->Guideptr->button;
	    else if (kformstruct->flag == KUIS_STARTPANE)
		obj = kformstruct->Controlptr->back_guide->button;
	    else if  (kformstruct->flag == KUIS_STARTSUBMENU)
		obj = kformstruct->Selptr->label_object;
	    else obj = kformstruct->Selptr->value_object;
	}
	else
	{
	    kerror("xvforms", "xvf_get_xvobject",
		   "Unknown 'item_part' passed in."); 
	    return(NULL);
	}
	return(obj);

}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_get_backplane
|
|       Purpose: Returns the backplane associated with a kformstruct.
|
|         Input: kformstruct - pointer to the generic kformstruct structure
|                create      - if TRUE, the subform or pane will be
|                              created if it is has not been already; false
|                              says return NULL if the subform or pane
|                              has not been created.
|        Output: none
|       Returns: Returns the backplane xvobject on success, NULL on failure
|          Date: March 16, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

xvobject xvf_get_backplane(
   kform_struct *kformstruct,
   int          create)
{
	kform      *form;
	ksubform   *subform;
	kcontrol   *pane; 
	kcontrol   *master; 
	kselection *selection;
	xvobject   back = NULL;
	int	   map_pane;

	if (kformstruct->type == KSELECTION) 
	{
	    if (kformstruct->Selptr->back != NULL)
		back = kformstruct->Selptr->back;
	    else
	    {
		selection = kformstruct->Selptr;
		if (selection->back_control->type == KPANE)
		{
		    pane = selection->back_control;
		    if ((pane->back == NULL) && (create))
		    {
			if (!pane->back_subform->selected)
			    map_pane = FALSE;
			else map_pane = pane->back_guide->selected;
		        xvf_create_pane(pane, map_pane); 
		    }
		}
		else if (selection->back_control->type == KGUIDEPANE)
	        {
	            subform = selection->back_subform;
	            if ((subform->back == NULL) && (create))
	                xvf_create_subform(subform, -1, -1, subform->selected);
	        }
		else
	        {
		    form = selection->back_form;
	            if ((form->master != NULL)       && 
			(form->master->back == NULL) &&
		        (create))
	        	xvf_create_mainform(form, -1, -1);
	        }
		back = selection->back;
	    }
	}

	else if (kformstruct->type == KSUBFORM)
	{
	    subform = kformstruct->Subformptr;
	    if ((subform->back == NULL) && (create))
                xvf_create_subform(subform, -1, -1, subform->selected);
            back = subform->back;
	}
	else if ((kformstruct->type == KGUIDE) ||
                 (kformstruct->type == KPANE))
	{
	    if (kformstruct->type == KPANE)
	        pane = kformstruct->Controlptr;
	    else pane = kformstruct->Guideptr->pane;

	    if ((pane->back == NULL) && (create))
	    {
		if (!pane->back_subform->selected)
		    map_pane = FALSE;
		else map_pane = pane->back_guide->selected;
		xvf_create_pane(pane, map_pane); 
	    }
	    back = pane->back;
	}
	else if (kformstruct->type == KGUIDEPANE)
	{
	    subform = kformstruct->Controlptr->back_subform;
	    if ((subform->back == NULL) && (create))
	        xvf_create_subform(subform, -1, -1, subform->selected);
	    back = subform->guidepane->back;
	}
	else if (kformstruct->type == KMASTER)
	{
	    master = kformstruct->Controlptr;
	    if ((master->back == NULL) && (create))
	        xvf_create_mainform(master->back_form, -1, -1);
	    back = master->back;
	}
	return(back);
}

/*------------------------------------------------------------
|
|  Routine Name: xvf_get_line
|
|       Purpose: Given a generic kform_struct, returns the 
|		 UIS line associated with the form, subform,
|		 guide, pane, or selection.
|         Input: kforstruct - generic kform_struct
|        Output: The UIS line associated with kform_struct
|          Date: March 16, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
char *xvf_get_line(
   kform_struct *kformstruct)
{
	if (kformstruct->type == KSELECTION)
	   return(kformstruct->Selptr->line);

	else if ((kformstruct->type == KPANE)    ||
		 (kformstruct->type == KGUIDEPANE) ||
		 (kformstruct->type == KPANE))
	   return(kformstruct->Controlptr->control_line);

	else if (kformstruct->type == KGUIDE)
	   return(kformstruct->Guideptr->line);

	else if (kformstruct->type == KSUBFORM)
	{
	   if (kformstruct->flag == KUIS_STARTSUBFORM)
	       return(kformstruct->Subformptr->control_line);
	   else 
	       return(kformstruct->Subformptr->line);
	}

	else if (kformstruct->type == KFORM)
	   return(kformstruct->Formptr->control_line);
	return(NULL);
}



/************************************************************
*
*  Routine Name: xvf_get_kformstruct - return kformstruct associated w/ xvobject
*
*       Purpose: Written for cantata, this routine takes an xvobject and
*                a form tree, and searches through the form tree for the
*                master, subform, guidepane, pane, or selection to which
*                the xvobject belongs (for example, the xvobject might be the 
*                label of a cycle selection, the text parameter box of an
*                input file selection, the title on a master form, etc).
*                If the xvobject is found in the form tree, the kformstruct
*                corresponding to the GUI item on which it is found is returned.
*
*         Input: form        - form tree to search for kformstruct 
*                xvobject    - xvobject for which to find kformstruct
*        Output:
*       Returns: A pointer to the kformstruct on success, NULL on failure.
*  Restrictions:
*    Written By: Danielle Argiro
*          Date: Jul 05, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kform_struct *xvf_get_kformstruct(
    kform    *form,
    xvobject object)
{
        ksubform   *subform;
        kguide     *guide;
        kselection *selection;

        if (form == NULL)
            return(NULL);

        if (form->master != NULL)
        {
	    if ((form->master->back == object) ||
		(form->master->label_object == object))
		return(form->master->back_kformstruct);

            subform = form->master->subform_list;
            selection = search_selections_for_xvobject(form->master->sel_list, 
						       object);
	    if (selection != NULL)
                return(selection->back_kformstruct);
        }
        else subform = form->subform;

        while (subform != NULL)
        {
	    if ((subform->button == object)       ||
		(subform->toplevel == object)     ||
		(subform->label_object == object) ||
		(subform->back == object))
		return(subform->back_kformstruct);

            if (subform->guidepane != NULL)
            {
	        if ((subform->guidepane->back == object) ||
		    (subform->guidepane->label_object == object))
		     return(subform->guidepane->back_kformstruct);

                selection = 
	 	   search_selections_for_xvobject(subform->guidepane->sel_list, 
						  object);
	        if (selection != NULL)
                    return(selection->back_kformstruct);

                guide = subform->guidepane->guide_list;
            }
            else guide = subform->guide;

            while (guide != NULL)
            {
		if (guide->button == object)
		    return(guide->back_kformstruct);

                if (guide->pane != NULL)
		{
	            if ((guide->pane->back == object) ||
		        (guide->pane->label_object == object))
			return(guide->pane->back_kformstruct);

                    selection = 
	 	       search_selections_for_xvobject(guide->pane->sel_list, 
						      object);
	            if (selection != NULL)
                        return(selection->back_kformstruct);
		}
                guide = guide->next;
            }
            subform = subform->next;
        }
        return(NULL);
}

static kselection *search_selections_for_xvobject(
    kselection *sel_list,
    xvobject   object)
{
	kselection *selection;
	kselection *group = NULL;
	
	selection = sel_list;
	while (selection != NULL)
	{
	    if ((selection->back == object)         ||
		(selection->value_object == object) ||
		(selection->opt_object == object)   ||
		(selection->label_object == object) ||
		(selection->pix_object == object)   ||
		(selection->scroll == object))
		return(selection);

            if (selection->group_next != NULL)
            {   
		group = search_selections_for_xvobject(selection->group_next,
                                                       object);
                if (group != NULL) return(group);
            }

	    selection = selection->next;
	}
	return(NULL);
}
