/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>             Attribute Actions For Printing UIS        <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                 xvf_get_print_uis()                   <<<<
   >>>>                 xvf_set_print_uis()                   <<<<
   >>>>                 xvf_get_print_subform()               <<<<
   >>>>                 xvf_set_print_subform()               <<<<
   >>>>                 xvf_get_print_pane()                  <<<<
   >>>>                 xvf_set_print_pane()                  <<<<
   >>>>                                                       <<<<
   >>>>                 xvf_get_uis_location()                <<<<
   >>>>                 xvf_set_uis_location()                <<<<
   >>>>                 xvf_get_subform_file()                <<<<
   >>>>                 xvf_set_subform_file()                <<<<
   >>>>                 xvf_get_pane_file()                   <<<<
   >>>>                 xvf_set_pane_file()                   <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include "attributes.h"


/*-----------------------------------------------------------
|
|
|       Routine: xvf_get_print_uis
|       Purpose: Initializes the filename for XVF_PRINT_UIS to NULL, as
|                we don't know what the name of the file should be.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_PRINT_UIS ("xvf_print_uis")
|        Output: calldata    - passes back UIS filename (NULL)
|
|          Date: March 23, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_print_uis(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char **uis_filename;

        /* set the pointer to be returned */
        uis_filename = (char **) calldata;

	/* initialize to NULL */
	*uis_filename = NULL;

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_print_uis
|
|       Purpose: Called via the non-editable menuform of a Pane, 
|                Guidepane, or Master, this routine writes out the UIS
|                file (which, presumably, was recently changed
|                by the user, to the extent that they are allowed).
|                This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_PRINT_UIS ("xvf_print_uis")
|                calldata    - passes in the filename to print UIS into
|        Output: none
|
|          Date: March 23, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_print_uis(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int status;

	xvw_busy(NULL, TRUE);
	status = kvf_set_print_uis(kformstruct, KVF_PRINT_UIS, calldata);
	xvw_busy(NULL, FALSE);

	return (status);
}


/*-----------------------------------------------------------
|
|
|       Routine: xvf_get_print_pane
|       Purpose: Sets the filename for the *.pane file. If there
|                is no *.pane file associated with pane, sets value to NULL.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_PRINT_PANE ("xvf_print_pane")
|        Output: calldata    - passes back UIS filename 
|
|          Date: March 23, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_print_pane(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_print_pane(kformstruct, KVF_PRINT_PANE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_print_pane
|
|       Purpose: Called via the editable menuform of a Pane, this routine 
|                writes out the *.pane file to the specified file.  
|                This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_PRINT_PANE ("xvf_print_pane")
|                calldata    - passes in the filename to print UIS into
|        Output: none
|
|          Date: March 23, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_print_pane(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int status;
	
	xvw_busy(NULL, TRUE);
	status = kvf_set_print_pane(kformstruct, KVF_PRINT_PANE, calldata);
	xvw_busy(NULL, FALSE);

	if (!status) return(FALSE);

        if (kformstruct->callback_list != NULL)
            xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|
|       Routine: xvf_get_print_subform
|       Purpose: Sets the filename for the *.subform file. If there
|                is no *.subform file associated with subform, sets value to NULL.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_PRINT_SUBFORM ("xvf_print_subform")
|        Output: calldata    - passes back UIS filename (NULL)
|
|          Date: Sept 22, 1993
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_print_subform(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_print_subform(kformstruct, KVF_PRINT_SUBFORM, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_print_subform
|
|       Purpose: Called via the editable menuform of a Subform, this routine 
|                writes out the *.subform file.  This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_PRINT_SUBFORM ("xvf_print_subform")
|                calldata    - passes in the filename to print UIS into
|        Output: none
|
|          Date: Sept 22, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_print_subform(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int status;
	
	xvw_busy(NULL, TRUE);
	status = kvf_set_print_subform(kformstruct, KVF_PRINT_SUBFORM, calldata);
	xvw_busy(NULL, FALSE);

	if (!status) return(FALSE);

        if (kformstruct->callback_list != NULL)
            xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|
|       Routine: xvf_get_uis_location
|       Purpose: Sets the location for a *.form, *.pane, or *.subform file 
|                to be written out.  UIS location is stored in 
|                form->uis_location.  Since this routine is so simple, it 
|                replicates code in kvf_get_uis_location() rather than
|                calling it.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_UIS_LOCATION ("xvf_uis_location")
|                calldata    - passes in UIS location
|        Output: none
|          Date: June 15, 1993
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_uis_location(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        char  **uis_location = (char **) calldata;

        *uis_location = kstrdup(kformstruct->Formptr->uis_location);
        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_uis_location
|
|       Purpose: Sets the directory location for UIS files that might
|                be written out: *.form, *.pane, and *.subform.  This
|                directory is stored in form struct as form->uis_location.
|                Since this routine is so simple, it replicates code in 
|                kvf_set_uis_location() rather than calling it.
|
|         Input: kformstruct - kformstruct associated with form ptr
|                attribute   - XVF_UIS_LOCATION ("xvf_uis_location")
|        Output: calldata    - passes back UIS location
|          Date: June 15, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_uis_location(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        char  **uis_location = (char **) calldata;

        kformstruct->Formptr->uis_location = kstrdup(*uis_location);
        return(TRUE);
}


/*-----------------------------------------------------------
|
|
|       Routine: xvf_get_pane_file
|       Purpose: Gets the name of the *.pane file associated with the pane.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_PANE_FILE ("xvf_pane_file")
|        Output: calldata    - passes back UIS filename (NULL)
|
|          Date: June 10, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_pane_file(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
      return (kvf_get_pane_file(kformstruct, KVF_PANE_FILE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_pane_file
|
|       Purpose: Sets the name of the *.pane file associated with the pane.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_PANE_FILE ("xvf_pane_file")
|                calldata    - passes in the filename to print UIS into
|        Output: none
|
|          Date: June 10, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_pane_file(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_pane_file(kformstruct, KVF_PANE_FILE, calldata)))
            return(FALSE);

        if (kformstruct->callback_list != NULL)
            xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|
|       Routine: xvf_get_subform_file
|       Purpose: Gets the name of the *.subform file associated with the subform.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_SUBFORM_FILE ("xvf_subform_file")
|        Output: calldata    - passes back UIS filename (NULL)
|
|          Date: June 10, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_subform_file(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
      return (kvf_get_subform_file(kformstruct, KVF_SUBFORM_FILE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_subform_file
|
|       Purpose: Sets the name of the *.subform file associated with
|                the subform.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_SUBFORM_FILE ("xvf_subform_file")
|                calldata    - passes in the filename to print UIS into
|        Output: none
|
|          Date: June 10, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_subform_file(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        if (!(kvf_set_subform_file(kformstruct, KVF_SUBFORM_FILE, calldata)))
            return(FALSE);

        if (kformstruct->callback_list != NULL)
            xvf_call_gui_callbacks(kformstruct, attribute, calldata);

        return(TRUE);
}
