 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>       	   OLIT Abstract Widget Set Utilities	      <<<<
   >>>>                                                       <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>         Button Utilities:                             <<<<
   >>>>		   xvw_get_buttonshape()                      <<<<
   >>>>		   xvw_set_buttonshape()                      <<<<
   >>>>            xvw_update_button()			      <<<<
   >>>>                                                       <<<<
   >>>>         Label Utilities:                              <<<<
   >>>>		   xvw_get_labeltype()                        <<<<
   >>>>		   xvw_set_labeltype()                        <<<<
   >>>>                                                       <<<<
   >>>>         Text Utilities:                               <<<<
   >>>>            xvw_get_textstring()                       <<<<
   >>>>            xvw_set_textstring()                       <<<<
   >>>>            xvw_get_textmulti()                        <<<<
   >>>>            xvw_set_textmulti()                        <<<<
   >>>>                                                       <<<<
   >>>>         Scrollbar Utilities:                          <<<<
   >>>>            xvw_get_scrollbar()                        <<<<
   >>>>            xvw_set_scrollbar()                        <<<<
   >>>>            xvw_set_orientation()                      <<<<
   >>>>            xvw_update_scroll_value()                  <<<<
   >>>>                                                       <<<<
   >>>>         List Utilities:                               <<<<
   >>>>            xvw_get_list_num()                         <<<<
   >>>>            xvw_set_list_num()                         <<<<
   >>>>            xvw_set_list_highlight()                   <<<<
   >>>>            xvw_set_list_unhighlight()                 <<<<
   >>>>            xvw_update_list_struct()                   <<<<
   >>>>                                                       <<<<
   >>>>         Misc Utilities:                               <<<<
   >>>>            xvw_get_misc()                             <<<<
   >>>>            xvw_set_misc()                             <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>            xvw_change_list()                          <<<<
   >>>>            xvw_reverse_colors()                       <<<<
   >>>>            xvw_retrieve_list()                        <<<<
   >>>>            xvw_retrieve_menu()                        <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

#ifdef KOLIT_DEF

extern klist   *xvw_menuwidget_list;


/*################################################################
#
#    Button Utilities
#
################################################################*/

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_buttonshape()
|
|       Purpose: Gets the current button shape of the button object
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_BUTTON_SHAPE
|                calldata  - pointer to contents of return button shape
|
|        Output: none
|    Written By: Mark Young & John Salas
|          Date: Feb 15, 1993 11:52
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_buttonshape(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int *shape = (int *) calldata;

	if (xvw_class(object) == rectButtonWidgetClass)
	   *shape = KBUTTON_SHAPE_RECT;
	else if (xvw_class(object) == oblongButtonWidgetClass)
	   *shape = KBUTTON_SHAPE_OVAL;
	else
	{
            kerror("xvwidgets", "xvw_get_buttonshape",
	          "The following object is not either a rectangular or oval \
button '%s'.  Returning NONE for the button shape.", xvw_name(object));
	   *shape = NONE;
	   return(FALSE);
	}
	return(TRUE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_buttonshape()
|
|       Purpose: Sets the current button shape of the button object.
|
|         Input: object    - the object to have contents set
|                attribute - XVW_BUTTON_SHAPE
|                calldata  - pointer to contents of button shape
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Nov 16, 1992 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_buttonshape(
   xvobject object,
   char   *attribute,
   kaddr  calldata)
{
	int *shape = (int *) calldata;

	Widget widget = xvw_widget(object);
	WidgetClass wclass = xvw_class(object);


	if (*shape == KBUTTON_SHAPE_RECT &&
	    wclass == oblongButtonWidgetClass)
	   xvw_recreate(object, rectButtonWidgetClass);
	else if (*shape == KBUTTON_SHAPE_OVAL &&
                 wclass == rectButtonWidgetClass)
	   xvw_recreate(object, oblongButtonWidgetClass);
	else if (wclass != oblongButtonWidgetClass &&
		 wclass != rectButtonWidgetClass)
	{
            kerror("xvwidgets", "xvw_set_buttonshape",
	          "The following desired button shape can be either \
rectangular or oval.  Ignoring current button shape request.", 
		   xvw_name(object));
	   return(FALSE);
	}
	return(TRUE);
}


/*-------------------------------------------------------------
|
|  Routine Name: xvw_update_button()
|
|       Purpose: updates the button during a XVW_BUTTON_SELECT callback.
|
|         Input: object    - the object that the callback was triggered on
|		 type      - the type of the callback
|                call_data - pointer to the callback data
|
|        Output: none
|    Written By: Mark Young
|          Date: Jun 06, 1993 14:37
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_update_button(
   xvobject object,
   char     *type,
   kaddr    call_Data)
{
	Boolean set;
	Widget  widget = xvw_widget(object);


	XtVaGetValues(widget, XtNset, &set, NULL);
	XtVaSetValues(widget, XtNset, !set, NULL);
}


/*################################################################
#
#   Label utilities
#
################################################################*/

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_labeltype()
|
|       Purpose: Gets the current label type of the label object
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_LABEL_TYPE
|                calldata  - pointer to contents of return label type
|
|        Output: none
|    Written By: Mark Young & John Salas
|          Date: Feb 15, 1993 11:53
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_labeltype(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int *type = (int *) calldata;


	if (xvw_class(object) == staticTextWidgetClass)
	   *type = KLABEL_TYPE_LABEL;
	else if (xvw_class(object) == pixmapWidgetClass)
	   *type = KLABEL_TYPE_PIXMAP;
	else
	{
            kerror("xvwidgets", "xvw_get_labeltype",
	          "The following object is not either a rectangular or oval \
label '%s'.  Returning NONE for the label type.", xvw_name(object));
	   *type = NONE;
	   return(FALSE);
	}
	return(TRUE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_labeltype()
|
|       Purpose: Sets the current label type of the label object.
|
|         Input: object    - the object to have contents set
|                attribute - XVW_LABEL_TYPE
|                calldata  - pointer to contents of label type
|
|        Output: none
|    Written By: Mark Young & John Salas
|          Date: Feb 15, 1993 11:53
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_labeltype(
   xvobject object,
   char   *attribute,
   kaddr  calldata)
{
	int *type = (int *) calldata;
	WidgetClass wclass = xvw_class(object);


	if (*type == KLABEL_TYPE_LABEL && wclass == pixmapWidgetClass)
	   xvw_recreate(object, staticTextWidgetClass);
	else if (*type == KLABEL_TYPE_PIXMAP && wclass == staticTextWidgetClass)
	   xvw_recreate(object, pixmapWidgetClass);
	else if (wclass != staticTextWidgetClass &&
		 wclass != pixmapWidgetClass)
	{
            kerror("xvwidgets", "xvw_set_labeltype",
	          "The following desired label type can be either a \
label or pixmap.  Ignoring current label type request.", xvw_name(object));
	   return(FALSE);
	}
	return(TRUE);
}

/*################################################################
#
#   Text Utilities
#
################################################################*/


/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_textstring()
|
|       Purpose: Allocates & returns the contents of a text object
|	         text field => string
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_TEXT_STRING
|                calldata  - pointer to contents of return string
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Nov 16, 1992 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_textstring(
   xvobject object,
   char   *attribute,
   kaddr  calldata)
{
	Widget widget = xvw_widget(object);
	char **string_return = (char **) calldata;


	if (!(OlTextEditCopyBuffer((TextEditWidget) widget, string_return)))
	{
            kerror("xvobject", "xvw_get_textstring",
		   "Can't copy textedit buffer for %s", xvw_name(object));
	    *string_return = NULL; 
	    return(FALSE);
	}
	return(TRUE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_textstring()
|
|       Purpose: Replaces the contents of a text object
|	         text field <= string
|
|         Input: object    - the object to have contents set
|                attribute - XVW_TEXT_STRING
|                calldata  - pointer to contents of string
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Nov 16, 1992 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_textstring(
   xvobject object,
   char   *attribute,
   kaddr  calldata)
{
	int    type;
	char   *fullpath;
	Widget widget = xvw_widget(object);
	char   **string = (char **) calldata;

        xvw_get_attribute(object, XVW_TEXT_TYPE, &type);
	
	/*
	 * string contained in a file 
	 */
        if (type == KTEXT_TYPE_FILE)
	{
	    fullpath = kfullpath(*string, NULL, NULL);
            /* file not there, or cannot be opened */
            if (kaccess(fullpath,R_OK))
            {
                kerror("xvwidgets", "xvw_set_textstring",
                        "Invalid file");
                return(FALSE);
            }
	    XtVaSetValues(xvw_widget(object), 
			  XtNsource,     fullpath, 
			  XtNeditType,   OL_TEXT_READ, 
			  NULL);
	}

	/*
	 * string verbetim 
	 */
	else
        {
	    OlTextEditClearBuffer((TextEditWidget) widget);
	    if (!(OlTextEditInsert((TextEditWidget) widget, *string,
			    kstrlen(*string))))
	    {
                kerror("xvobject", "xvw_set_textstring",
	              "Can't replace textedit buffer for %s", xvw_name(object));
	        return(FALSE);
	    }
	    (void) XtVaSetValues(widget, XtNdisplayPosition, 0, NULL);
	}
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_textmulti()
|
|       Purpose: Gets the current text multiline type of the label object
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_TEXT_MULTILINE
|                calldata  - pointer to contents of return text multiline type
|
|        Output: none
|    Written By: Mark Young & John Salas
|          Date: Feb 15, 1993 11:53
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_textmulti(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int num;
	int *multiline = (int *) calldata;


	XtVaGetValues(xvw_widget(object), XtNlinesVisible, &num, NULL);
	*multiline = (num > 1) ? TRUE : FALSE;
	return(TRUE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_textmulti()
|
|       Purpose: Sets the current text multiline of the text object.
|
|         Input: object    - the object to have contents set
|                attribute - XVW_TEXT_MULTILINE
|                calldata  - pointer to contents of text multiline type
|
|        Output: none
|    Written By: Mark Young & John Salas
|          Date: Feb 15, 1993 11:53
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_textmulti(
   xvobject object,
   char   *attribute,
   kaddr  calldata)
{
	int *multiline = (int *) calldata;


	if (*multiline)
	{
	   XtVaSetValues(xvw_widget(object),
		 	 XtNleftMargin,	     2,
		 	 XtNrightMargin,     2,
		 	 XtNtopMargin,       2,
		 	 XtNbottomMargin,    2,
		 	 XtNlinesVisible,    5,
		 	 XtNdisplayPosition, 0,
		 	 XtNcursorPosition,  0,
			 NULL);
	   xvw_set_attribute(object, XVW_CHAR_MAX_HEIGHT, 100.0);
	}
	else
	{
	   xvw_set_attribute(object, XVW_CHAR_MAX_HEIGHT, 1.0);
	   XtVaSetValues(xvw_widget(object),
		 	 XtNleftMargin,	     1,
		 	 XtNrightMargin,     1,
		 	 XtNtopMargin,       0,
		 	 XtNbottomMargin,    0,
		 	 XtNlinesVisible,    1,
		 	 XtNdisplayPosition, 0,
		 	 XtNcursorPosition,  0,
			 NULL);
	}
	return(TRUE);
}
/*################################################################
#
#   Scrollbar Utilities
#
################################################################*/

typedef struct _xvw_scrollbar
{
	double scroll_min;
	double scroll_max;
	double scroll_incr;
	double scroll_value;
	int    increment_set;
} xvw_scrollbar;

klist *scrollbar_list = NULL;

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_scrollbar()
|
|       Purpose: Allocates & returns the contents of a scrollbar
|	         scrollbar field => {min,value,incr,max} value
|	         for one of the attributes:
|
|			XVW_SCROLL_MIN
|			XVW_SCROLL_MAX
|			XVW_SCROLL_VALUE
|			XVW_SCROLL_INCR
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above value
|                calldata  - pointer to contents of return string
|
|        Output: none
|    Written By: Mark Young
|          Date: Nov 16, 1992 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_scrollbar(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int           tmpval;
	klist         *entry;
	xvw_scrollbar *temp;
	double        *retval = (double *) calldata;

	if ((entry = klist_locate(scrollbar_list, (kaddr) object)) == NULL)
	{
           kerror("xvobject", "xvw_get_scrollbar",
		   "Can't retrieve %s for %s", attribute, xvw_name(object));
	   return(FALSE);
	}
	temp = (xvw_scrollbar *) entry->client_data;

	if (kstrcmp(attribute, XVW_SCROLL_MIN) == 0)
	   *retval = temp->scroll_min;
	else if (kstrcmp(attribute, XVW_SCROLL_MAX) == 0)
	   *retval = temp->scroll_max;
	else if (kstrcmp(attribute, XVW_SCROLL_INCR) == 0)
	   *retval = temp->scroll_incr;
	else if (kstrcmp(attribute, XVW_SCROLL_VALUE) == 0)
	{
	   XtVaGetValues(xvw_widget(object), XtNsliderValue, &tmpval, NULL);
	   temp->scroll_value = (((temp->scroll_max - temp->scroll_min) *
 				kmin(tmpval,100))/100.0) + temp->scroll_min;
	   *retval = temp->scroll_value;
	}
	else
	{
           kerror("xvobject", "xvw_get_scrollbar",
		   "Unknown attribute '%s' for scrollbar object %s",
		   attribute, xvw_name(object));
	   return(FALSE);
	}
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_set_scrollbar()
|
|       Purpose: Sets the contents of a scrollbar
|	         for one of the attributes:
|
|			XVW_SCROLL_MIN
|			XVW_SCROLL_MAX
|			XVW_SCROLL_VALUE
|			XVW_SCROLL_INCR
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above value
|                calldata  - pointer to contents of return string
|
|        Output: none
|    Written By: Mark Young
|          Date: Nov 16, 1992 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_scrollbar(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	klist         *entry;
	Arg           arg[KLENGTH];
	int           i, value = 0, incr = 1;
	double        diff; 
	xvw_scrollbar *temp;
	double        *setval = (double *) calldata;
	Widget        widget = xvw_widget(object);


	if ((entry = klist_locate(scrollbar_list, (kaddr) object)) == NULL)
	{
	   temp = (xvw_scrollbar *) kcalloc(1, sizeof(xvw_scrollbar));
	   temp->scroll_max   = 1.0;
	   temp->scroll_incr  = 0.1;
	   temp->scroll_min   =
	   temp->scroll_value = 0.0;
	   temp->increment_set = FALSE;
	   scrollbar_list = klist_add(scrollbar_list, (kaddr) object, 
				      (kaddr) temp);
	}
	else temp = (xvw_scrollbar *) entry->client_data;

	if (kstrcmp(attribute, XVW_SCROLL_MIN) == 0)
	{
	   temp->scroll_min = *setval;
	   if (!temp->increment_set)
	      temp->scroll_incr = (temp->scroll_max - temp->scroll_min)*0.1;
	}
	else if (kstrcmp(attribute, XVW_SCROLL_MAX) == 0)
	{
	   temp->scroll_max = *setval;
	   if (!temp->increment_set)
	      temp->scroll_incr = (temp->scroll_max - temp->scroll_min)*0.1;
	}
	else if (kstrcmp(attribute, XVW_SCROLL_VALUE) == 0)
	   temp->scroll_value = *setval;
	else if (kstrcmp(attribute, XVW_SCROLL_INCR) == 0)
	   temp->scroll_incr = *setval, temp->increment_set = TRUE;
	else
	{
           kerror("xvobject", "xvw_set_scrollbar",
		   "Unknown attribute '%s' for scrollbar object %s",
		   attribute, xvw_name(object));
	   return(FALSE);
	}

	/*
	 *  Compute percent value relative to the slider's min and max
	 */
	diff = kabs(temp->scroll_max - temp->scroll_min);
	incr  = krange(1, (temp->scroll_incr/diff) * 100, 100);
	value = ((temp->scroll_value - temp->scroll_min)/diff) * 100;
	value = krange(0, value, 100);

	i = 0;
	XtSetArg(arg[i], XtNsliderMin,	      0);	i++;
	XtSetArg(arg[i], XtNsliderMax,	      100+incr);i++;
	XtSetArg(arg[i], XtNproportionLength, incr);	i++;
	XtSetArg(arg[i], XtNgranularity,      incr);	i++;
	XtSetArg(arg[i], XtNsliderValue,      value);	i++;
	XtSetValues(widget, arg, i);
	return(TRUE);
}


/*--------------------------------------------------------------
|
|  Routine Name: xvw_set_orientation()
|
|       Purpose: Sets the orientation of a scrollbar
|	         for the attribute:
|
|			XVW_ORIENTAITON
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above value
|                calldata  - pointer to contents of return string
|
|        Output: none
|    Written By: Mark Young
|          Date: Nov 16, 1992 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_orientation(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	OlDefine olit_orient;
	OlDefine tmp_orient;
	int      *orientation = (int *) calldata;

	if (*orientation == KSCROLLBAR_ORIENT_VERT)
	    tmp_orient = OL_VERTICAL;
	else tmp_orient = OL_HORIZONTAL;

	XtVaGetValues(xvw_widget(object), XtNorientation, &olit_orient, NULL);
	if (tmp_orient != (int) olit_orient)
	{
	   XtVaSetValues(xvw_widget(object), XtNorientation, *orientation,NULL);
	   xvw_recreate(object, scrollbarWidgetClass);
	   XtVaSetValues(xvw_widget(object), XtNproportionLength, 1, NULL);
	}
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvw_update_scroll_value()
|
|       Purpose: Sets the double value of the scroll
|                according to the Motif widget set.
|
|         Input: object      - the scrollbar object
|                type        - the type of callback
|                call_data   - the structure passed in as the call_data
|                              for a scrollbar widget by MOTIF
|                value - scrolbar value 
|        Output: none
|       Returns: none
|    Written By: Mark Young & Danielle Argiro
|          Date: June 1, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_update_scroll_value(
    xvobject object,
    char     *type,
    kaddr    call_data,
    double   *value)
{
	OlScrollbarVerify *cb = (OlScrollbarVerify *) call_data;

	XtVaSetValues(xvw_widget(object),XtNsliderValue,cb->new_location,NULL);
	xvw_get_attribute(object, XVW_SCROLL_VALUE, value);
}


/*################################################################
#
#    List Utilities
#
################################################################*/

typedef struct
{
        int         size;
	OlListToken *token;
} xvw_liststruct;
 
/*------------------------------------------------------------
|
|  Routine Name: xvw_get_list_num()
|
|       Purpose: Gets a list according to the list attributes.
|		 The attributes handled by the
|
|                       XVW_LIST_NUMBER
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above value
|                calldata  - pointer to contents of return string
|
|        Output: none
|    Written By: Danielle Argiro and Mark Young
|          Date: Nov 25, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_list_num(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int *num = (int *) calldata;

	int i = 0;
	xvw_liststruct *list = NULL;
	Widget widget = xvw_widget(object);


	if (kstrcmp(attribute, XVW_LIST_SIZE) == 0)
	{
	   XtVaGetValues(widget, XtNuserData, &list, NULL);
	   if (list != NULL) i = list->size;
	   *num = i;
	}
        else
        {
           kerror("xvobject", "xvw_get_list_num",
                   "Unknown attribute '%s' for list object %s",
                   attribute, XtName(xvw_widget(object)));
           return(FALSE);
        }
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_set_list_num()
|
|       Purpose: Sets a list according to the list attributes.
|		 The attributes handled by the
|
|                       XVW_LIST_NUMBER
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above value
|                calldata  - pointer to contents of return string
|
|        Output: none
|    Written By: Danielle Argiro and Mark Young
|          Date: Nov 25, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_list_num(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int *num = (int *) calldata;

	int size;
	int i = 0;
	xvw_liststruct *list = NULL;

	OlListToken (*AddItem)();
	void (*UpdateView)(), (*DeleteItem)();
	Widget widget = xvw_widget(object);


	if (kstrcmp(attribute, XVW_LIST_SIZE) == 0)
	{
	   XtVaGetValues(widget,
		XtNapplAddItem,    &AddItem,
		XtNapplUpdateView, &UpdateView,
		XtNapplDeleteItem, &DeleteItem,
		XtNuserData,       &list,
		NULL);

	   if (list == NULL)
	      return(TRUE);

	   xvw_get_attribute(object, XVW_LIST_SIZE, &size);
	   (*UpdateView)(widget, FALSE);
	   for (i = *num; i < size && list->token[i] != NULL; i++)
	   {
	      (void) (*DeleteItem)(widget, list->token[i]);
	      list->token[i] = NULL;
	   }
	   (*UpdateView)(widget, TRUE);
	}
        else
        {
           kerror("xvobject", "xvw_set_list_num",
                   "Unknown attribute '%s' for list object %s",
                   attribute, XtName(xvw_widget(object)));
           return(FALSE);
        }
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_get_list_highlight()
|
|       Purpose: Gets the highlighted element of the list.
|
|         Input: object    - the object to have highlighted element obtained
|                attribute - should be XVW_LIST_HIGHLT_ELEM
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_list_highlight(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int         i, size;
	int         *indx = (int *) calldata;
        OlListItem  *list_item;
	xvw_liststruct *list = NULL;

	*indx = -1;
        if (kstrcmp(attribute, XVW_LIST_HIGHLT_ELEM) == 0)
        {
	    XtVaGetValues(xvw_widget(object), XtNuserData, &list, NULL);
	    xvw_get_attribute(object, XVW_LIST_SIZE, &size);

	   if (list == NULL)
	      return(TRUE);

            for (i = 0; i < size && list->token[i] != NULL; i++)
	    {
                list_item = OlListItemPointer(list->token[i]);
                if (list_item && list_item->attr & OL_LIST_ATTR_CURRENT)
		{
		    *indx = i;
		    return(TRUE);
	 	}
	    }
        }
        else
        {
           kerror("xvobject", "xvw_get_list_highlight",
                   "Unknown attribute '%s' for list object %s",
                   attribute, xvw_name(object));
           return(FALSE);
        }
        return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_set_list_highlight()
|
|       Purpose: Sets the highlighted element of the list.
|
|         Input: object    - the object to have list element highlighted
|                attribute - should be XVW_LIST_HIGHLT_ELEM
|                calldata  - passes to index of element to be highlighted
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 15, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_list_highlight(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
        int		i, *indx = (int *) calldata;
	void		(*UpdateView)();
	void		(*applViewItem)();
	OlListItem	*list_item;
	xvw_liststruct  *list = NULL;

        if (kstrcmp(attribute, XVW_LIST_HIGHLT_ELEM) == 0)
        {
           XtVaGetValues(xvw_widget(object),
                XtNapplViewItem,   &applViewItem,
		XtNapplUpdateView, &UpdateView,
		XtNuserData,	   &list,
                NULL);
 
           if (list == NULL)
              return(TRUE);

	   list_item = OlListItemPointer(list->token[*indx]);
	   list_item->attr = list_item->attr | OL_LIST_ATTR_CURRENT;
	   (*UpdateView)(xvw_widget(object), TRUE);
	   (*applViewItem)(xvw_widget(object), list->token[*indx]);
        }
        else
        {
           kerror("xvobject", "xvw_set_list_highlight",
                  "Unknown attribute '%s' for list object %s",
                  attribute, xvw_name(object));
           return(FALSE);
        }

        return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_set_list_unhighlight()
|
|       Purpose: Unhighlights an element of the list.
|
|         Input: object    - the object to have list element highlighted
|                attribute - should be XVW_LIST_UNHIGHLT_ELEM
|                calldata  - passes to index of element to be unhighlighted
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 15, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_list_unhighlight(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
        int         i, *indx = (int *) calldata;
	void       (*UpdateView)();
	xvw_liststruct *list = NULL;
	OlListItem  *list_item;

        if (kstrcmp(attribute, XVW_LIST_UNHIGHLT_ELEM) == 0)
        {
           XtVaGetValues(xvw_widget(object),
		XtNapplUpdateView, &UpdateView,
		XtNuserData,	   &list,
                NULL);

	   if (list == NULL)
	      return(TRUE);

           list_item = OlListItemPointer(list->token[*indx]);
	   list_item->attr = list_item->attr & 0xfffdffff;
           (*UpdateView)(xvw_widget(object), TRUE);
        }
        else
        {
           kerror("xvobject", "xvw_set_list_unhighlight",
                  "Unknown attribute '%s' for list object %s",
                  attribute, xvw_name(object));
           return(FALSE);
        }
        return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_update_list_struct()
|
|       Purpose: Sets the fields in the xvw_list_struct structure
|                according to the Olit widget set.
|
|         Input: object      - the list object
|		 type        - the type of callback
|                call_data   - the structure passed in as the call_data 
|                              for a list widget by OLIT
|		 list_struct - xvwidgets list structure to be set
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: June 1, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_update_list_struct(
    xvobject        object,
    char	    *type,
    kaddr           call_data,
    int		    *double_click,
    xvw_list_struct *list_struct)
{
        /* in KOLIT, list item is returned as a token */
        OlListToken token;
        OlListItem *list_item;

        /*
         *  translate information passed back in KOLIT's "token" form
         *  into the xvw_list_struct
         */
        token = (OlListToken) call_data;
        list_item = OlListItemPointer(token);
        list_struct->string = kstring_copy(list_item->label, NULL);
        list_struct->list_index = (list_item->attr & 0x00ff);
}


/*################################################################
#
#   Miscellaneous utilities
#
################################################################*/


/*------------------------------------------------------------
|
|  Routine Name: xvw_get_misc()
|
|       Purpose: Gets miscellaneous attributes of various xvobjects.
|
|         Input: object    - the object to have attribute obtained
|                attribute - one of the miscellaneous attributes, eg,
|                            XVW_TEXT_EDIT_TYPE
|        Output: calldata  - passes back the miscellaneous value to be set
|    Written By: Mark Young Danielle Argiro
|          Date: June 7, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_misc(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	Widget widget = xvw_widget(object);
	int    olit_value;
	int    *value = (int *) calldata;

	if (kstrcmp(attribute, XVW_TEXT_TYPE) == 0)
	{
            XtVaGetValues(widget, XtNsourceType, &olit_value, NULL);
	    if (olit_value == OL_STRING_SOURCE)
	        *value = KTEXT_TYPE_STRING;
	    else if (olit_value == OL_DISK_SOURCE)
	        *value = KTEXT_TYPE_FILE;
        }
	else if (kstrcmp(attribute, XVW_TEXT_EDIT_TYPE) == 0)
        {
            XtVaGetValues(widget, XtNeditType, &olit_value, NULL);
	    if (olit_value == OL_TEXT_READ)
	        *value = KTEXT_READ;
	    else if (olit_value == OL_TEXT_EDIT)
	        *value = KTEXT_EDIT;
        }
	else if (kstrcmp(attribute, XVW_TEXT_WRAP) == 0)
        {
            XtVaGetValues(widget, XtNwrapMode, &olit_value, NULL);
	    if (olit_value == OL_WRAP_WHITE_SPACE)
	        *value = KTEXT_WRAP_WORD;
	    else if (olit_value == OL_WRAP_ANY)
	        *value = KTEXT_WRAP_LINE;
	    else if (olit_value == OL_WRAP_OFF)
	        *value = KTEXT_WRAP_NEVER;
	}

        else if (kstrcmp(attribute, XVW_TEXT_SCROLL_VERT) == 0)
        {
            /*XtVaGetValues(widget, XtNverticalSB, &olit_value, NULL);*/
	    if (olit_value == TRUE)
                *value = KSCROLLBAR_ALWAYS;
	    else if (olit_value == FALSE)
                *value = KSCROLLBAR_NEVER;
        }

        else if (kstrcmp(attribute, XVW_TEXT_SCROLL_HORIZ) == 0)
        {
            /*XtVaGetValues(widget, XtNhorizontalSB, &olit_value, NULL);*/
	    if (olit_value == TRUE)
                *value = KSCROLLBAR_ALWAYS;
	    else if (olit_value == FALSE)
                *value = KSCROLLBAR_NEVER;
        }

        else if (kstrcmp(attribute, XVW_LABEL_JUSTIFY) == 0)
        {
	    XtVaGetValues(widget, XtNgravity, &olit_value, NULL);

	    if (olit_value == WestGravity)
                *value = KLABEL_JUSTIFY_LEFT;
	    else if (olit_value == EastGravity)
                *value = KLABEL_JUSTIFY_RIGHT;
	    else if (olit_value == CenterGravity)
                *value = KLABEL_JUSTIFY_CENTER;
        }
	else if (kstrcmp(attribute, XVW_PIXMAP_JUSTIFY) == 0)
	{
            *value = KLABEL_JUSTIFY_CENTER;
	}

        else if (kstrcmp(attribute, XVW_BUTTON_TYPE) == 0)
        {
            XtVaGetValues(widget, XtNlabelType, &olit_value, NULL);
	    if (olit_value == OL_STRING)
                *value = KLABEL_TYPE_LABEL;
	    else if (olit_value == OL_IMAGE)
                *value = KLABEL_TYPE_PIXMAP;
	}
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_set_misc()
|
|       Purpose: Sets miscellaneous attributes of various xvobjects.
|
|         Input: object    - the object to have attribute set
|                attribute - one of the miscellaneous attributes, eg,
|                            XVW_TEXT_EDIT_TYPE
|                calldata  - passes in the miscellaneous value to be set
|
|        Output: none
|    Written By: Mark Young Danielle Argiro
|          Date: June 7, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_misc(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	Widget widget = xvw_widget(object);
	int    *value = (int *) calldata;
	
	if (kstrcmp(attribute, XVW_TEXT_TYPE) == 0)
	{
	    if (*value == KTEXT_TYPE_STRING)
	    {
		XtVaSetValues(xvw_widget(object),
			      XtNsourceType, OL_STRING_SOURCE,
			      NULL);
	    }
	    else if (*value == KTEXT_TYPE_FILE)
	    {
		XtVaSetValues(xvw_widget(object),
			      XtNeditType,   OL_TEXT_READ,
			      XtNsourceType, OL_DISK_SOURCE,
			      NULL);
	    }
	    else
	    {
		errno = KCALL;
		kerror("xvwidgets", "xvw_set_misc",
		       "Invalid value passed in for XVW_TEXT_TYPE");
		return(FALSE);
	    }
	}

	else if (kstrcmp(attribute, XVW_TEXT_EDIT_TYPE) == 0)
        {  

	    if (*value == KTEXT_READ)
		XtVaSetValues(widget, XtNeditType, OL_TEXT_READ, NULL);
	    else if (*value == KTEXT_EDIT)
		XtVaSetValues(widget, XtNeditType, OL_TEXT_EDIT, NULL);
	    else if (*value == KTEXT_APPEND)
		XtVaSetValues(widget, XtNeditType, OL_TEXT_EDIT, NULL);
	    else
	    {
		errno = KCALL;
		kerror("xvwidgets", "xvw_set_misc",
		       "Invalid value passed in for XVW_TEXT_EDIT_TYPE");
		return(FALSE);
	    }
	}

	else if (kstrcmp(attribute, XVW_TEXT_WRAP) == 0)
	{
	    if (*value == KTEXT_WRAP_WORD)
                XtVaSetValues(widget, XtNwrapMode, OL_WRAP_WHITE_SPACE, NULL);
            else if (*value == KTEXT_WRAP_LINE)
                XtVaSetValues(widget, XtNwrapMode, OL_WRAP_ANY, NULL);
            else if (*value == KTEXT_WRAP_NEVER)
                XtVaSetValues(widget, XtNwrapMode, OL_WRAP_OFF, NULL);
	    else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_WRAP");
                return(FALSE);
            }

	}

	else if (kstrcmp(attribute, XVW_TEXT_SCROLL_VERT) == 0)
        {
/*
	    if (*value == KSCROLLBAR_ALWAYS)
                XtVaSetValues(widget, XtNverticalSB, TRUE, NULL);
	    else if (*value == KSCROLLBAR_IFNEEDED)
                XtVaSetValues(widget, XtNverticalSB, TRUE, NULL);
	    else if (*value == KSCROLLBAR_NEVER)
                XtVaSetValues(widget, XtNverticalSB, FALSE, NULL);
	    else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_SCROLL_VERT");
                return(FALSE);
            }
 */
	}

	else if (kstrcmp(attribute, XVW_TEXT_SCROLL_HORIZ) == 0)
        {
/*
	    if (*value == KSCROLLBAR_ALWAYS)
                XtVaSetValues(widget, XtNhorizontalSB, TRUE, NULL);
	    else if (*value == KSCROLLBAR_IFNEEDED)
                XtVaSetValues(widget, XtNhorizontalSB, TRUE, NULL);
	    else if (*value == KSCROLLBAR_NEVER)
                XtVaSetValues(widget, XtNhorizontalSB, FALSE, NULL);
	    else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_SCROLL_HORIZ");
                return(FALSE);
            }
 */
	}
	
	else if (kstrcmp(attribute, XVW_LABEL_JUSTIFY) == 0)
	{
	    if (*value == KLABEL_JUSTIFY_LEFT)
                XtVaSetValues(widget, XtNgravity, WestGravity, NULL);
	    else if (*value == KLABEL_JUSTIFY_RIGHT)
                XtVaSetValues(widget, XtNgravity, EastGravity, NULL);
	    else if (*value == KLABEL_JUSTIFY_CENTER)
                XtVaSetValues(widget, XtNgravity, CenterGravity, NULL);
            else 
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for %s", attribute);
                return(FALSE);
	    }
	}

	else if (kstrcmp(attribute, XVW_PIXMAP_JUSTIFY) == 0)
	{
	    if (*value != KLABEL_JUSTIFY_CENTER)
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Attribute %s is not supported for Pixmap label;  Only center justification is allowed.", attribute);
                return(FALSE);
	    }
	}

	else if (kstrcmp(attribute, XVW_BUTTON_TYPE) == 0)
        {
            if (*value == KBUTTON_TYPE_LABEL)
               XtVaSetValues(widget, XtNlabelType, OL_STRING, NULL);
            else if (*value == KBUTTON_TYPE_PIXMAP)
               XtVaSetValues(widget, XtNlabelType, OL_IMAGE, NULL);
            else 
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_BUTTON_TYPE");
                return(FALSE);
	    }
	}
	return(TRUE);
}

/*################################################################
#
#    Public Utilities - public headers in ../xvwidgets/widgets.c
#
################################################################*/


/*-----------------------------------------------------------
|
|  Routine Name: xvw_change_list()
|
|       Purpose: Changes the contents of a list object
|                while it is displayed
|
|         Input: object  - the list object to have contents changed
|                entries - new contents of list object
|                size    - size of the new list (# entries)
|                resize  - TRUE if list object should try to resize
|                          itself after the change, FALSE otherwise.
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: Jul 17, 1992 14:59
| Modifications:
|
-------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_change_list(
   xvobject object,
   char   **entries,
   int    size,
   int    resize)
{
	int    i, num_entries;
	Widget widget = xvw_widget(object);

	OlListItem item;
	OlListToken (*AddItem)();
	xvw_liststruct *list = NULL;
	void (*UpdateView)(), (*DeleteItem)();
	static int width = 16;


        XtVaGetValues(widget,
                XtNapplAddItem,    &AddItem,
                XtNapplUpdateView, &UpdateView,
                XtNapplDeleteItem, &DeleteItem,
                XtNuserData,       &list,
                NULL);

	(*UpdateView)(widget, FALSE);
	if (list != NULL)
	{
	   for (i = 0; i < list->size && list->token[i] != NULL; i++)
	      (void) (*DeleteItem)(widget, list->token[i]);
	}
	else
	{
	   list = (xvw_liststruct *) kmalloc(sizeof(xvw_liststruct));
	   if (list == NULL)
	   {
	      kerror("xvwidgets", "xvw_change_list",
                  "Unable to allocate internal list structure.");
	      return;
	   }
	   list->size  = 0;
	   list->token = NULL;
	   XtVaSetValues(widget, XtNuserData, list, NULL);
	}

	if (resize)
	{
	   for (i = 0; i < size; i++)
	      width = kmax(width, kstrlen(entries[i]));

	   xvw_set_attributes(object,
			XVW_CHAR_WIDTH,  (float) width, 
			XVW_CHAR_HEIGHT, (float) kmin(size, 16),
			NULL);
	}

	if (size != list->size)
	{
	   list->token = (OlListToken *) krealloc(list->token,
			size * sizeof(OlListToken));
	   list->size = size;
	}

	for (i = 0 ; i < size; i++)
	{
	   item.label_type = OL_STRING;
	   item.label	   = kstrdup(entries[i]);
	   item.attr	   = (OlBitMask) i;
	   item.mnemonic   = 0;
	   list->token[i] = (*AddItem)(widget, 0, 0, item);
	}
	(*UpdateView)(widget, TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_reverse_colors
|
|       Purpose: Reverses the foreground and background colors
|                on a button object to indicate whether it is selected.
|
|         Input: object - the button object on which to reverse colors
|                value  - TRUE if button should be changed to look selected,
|                         FALSE if button should be changed to look un-selected.
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: Jul 17, 1992 15:02
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_reverse_colors(
   xvobject object,
   int    value)
{
	Widget widget = xvw_widget(object);


	(void) XtVaSetValues(widget, XtNset, (Boolean) value, NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_retrieve_list
|
|       Purpose: This is actually a dummy routine;
|                no interceding widget btwn the backplane
|                and the widget for OLIT.
|
|         Input: list - the object for which the list is desired
|        Output: none
|       Returns: The list object.
|    Written By: Danielle Argiro
|          Date: April 21, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
xvobject xvw_retrieve_list(
    xvobject list)
{
        return(list);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_retrieve_menu
|
|       Purpose: Given a compound menubutton object, returns the
|                menu associated with the menubutton
|
|         Input: menubutton - the menubutton for which the menu is desired
|        Output:
|       Returns: The menu object on success, NULL on failure
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications:
|
------------------------------------------------------------------*/

xvobject xvw_retrieve_menu(
    xvobject menubutton)
{
        xvobject menu;
        klist    *menuwidget_list;

        menuwidget_list = klist_locate(xvw_menuwidget_list, menubutton);
        if (menuwidget_list != NULL)
        {
           menu = (xvobject) klist_clientdata(menuwidget_list);
           return(menu);
        }
        else return(NULL);
}

#else
/*
 * These are dummy function calls so that if the widget set is not supported
 * on the machine a message is printed out.
 */

/* ARGSUSED */
void xvw_change_list(
   xvobject object,
   char   **list,
   int    size,
   int    resize)
{
   kfprintf(kstderr,"\nThe Olit widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return;
}

/* ARGSUSED */
void xvw_reverse_colors(
   xvobject object,
   int    value)
{
   kfprintf(kstderr,"\nThe Olit widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return;
}

/* ARGSUSED */
xvobject xvw_retrieve_list(
    xvobject list)
{
   kfprintf(kstderr,"\nThe Olit widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return(NULL);
}

/* ARGSUSED */
xvobject xvw_retrieve_menu(
    xvobject menubutton)
{
   kfprintf(kstderr,"\nThe Olit widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return(NULL);
}
#endif /* KOLIT_DEF */
/* Don't add after the endif */
