 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>       	   MOTIF Abstract Widget Set Utilities	      <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>         Menu Utilities:                               <<<<
   >>>>            xvw_popup_motif_menu()                     <<<<
   >>>>                                                       <<<<
   >>>>         Button Utilities:                             <<<<
   >>>>            xvw_get_buttonshape()                      <<<<
   >>>>            xvw_set_buttonshape()                      <<<<
   >>>>                                                       <<<<
   >>>>         Text Utilities:                               <<<<
   >>>>            xvw_get_text()                             <<<<
   >>>>            xvw_set_text()                             <<<<
   >>>>            xvw_get_texttype()                         <<<<
   >>>>            xvw_set_texttype()                         <<<<
   >>>>            xvw_get_textmulti()                        <<<<
   >>>>            xvw_set_textmulti()                        <<<<
   >>>>                                                       <<<<
   >>>>         Scrollbar Utilities:                          <<<<
   >>>>            xvw_get_scrollbar()                        <<<<
   >>>>            xvw_set_scrollbar()                        <<<<
   >>>>            xvw_set_orientation()                      <<<<
   >>>>            xvw_update_scroll_value()                  <<<<
   >>>>                                                       <<<<
   >>>>         List Utilities:                               <<<<
   >>>>            xvw_get_list_num()                         <<<<
   >>>>            xvw_set_list_num()                         <<<<
   >>>>            xvw_set_list_highlight()                   <<<<
   >>>>            xvw_set_list_unhighlight()                 <<<<
   >>>>            xvw_update_list_struct()                   <<<<
   >>>>                                                       <<<<
   >>>>         Misc Utilities:                               <<<<
   >>>>            xvw_get_misc()                             <<<<
   >>>>            xvw_set_misc()                             <<<<
   >>>>                                                       <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>            xvw_change_list()                          <<<<
   >>>>            xvw_reverse_colors()                       <<<<
   >>>>            xvw_retrieve_list()                        <<<<
   >>>>            xvw_retrieve_menu()                        <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

#ifdef KMOTIF_DEF

extern klist   *xvw_textwidget_list;
extern klist   *xvw_menuwidget_list;
extern klist   *xvw_listwidget_list;


/*################################################################
#
#   Menu Utilities
#
################################################################*/


/*--------------------------------------------------------------
|
|  Routine Name: xvw_popup_motif_menu()
|
|       Purpose: Pops up the motif menu
|
|         Input: widget      - the menubutton widget
|                client_data - passes in the menu to pop up
|                event       - the ButtonPress event
|        Output: none
|    Written By: Danielle Argiro & John Salas
|          Date: April 21, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_popup_motif_menu(
   Widget widget,
   kaddr  client_data,
   kaddr  call_data)
{
        Widget menu = (Widget) client_data;
	XmPushButtonCallbackStruct *cb;

	int x, y;
	unsigned int height;
	xvobject object = xvw_object(widget);

	xvw_translate_coords(object, 0, 0, &x, &y);
	xvw_geometry(object, NULL, NULL, NULL, &height, NULL);

	cb = (XmPushButtonCallbackStruct *) call_data;
	cb->event->xbutton.x_root = x;
	cb->event->xbutton.y_root = y + height + 5;

        XmMenuPosition(menu, (XButtonPressedEvent *) cb->event);
        XtManageChild(menu);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_menubutton()
|
|       Purpose: Gets the menubutton from the Motif row/col widget.
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_LABEL, XVW_BORDER_WIDTH
|                calldata  - pointer to contents of return text type
|
|        Output: none
|    Written By: Mark Young
|          Date: Jan 5, 1994
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_menubutton(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	XmString text;
	xvobject *children;

	if ((children = xvw_children(object, xmCascadeButtonGadgetClass,NULL))
							== NULL)
	   return(FALSE);

	if (kstrcmp(attribute, XVW_LABEL) == 0)
	{
	   XtVaGetValues(xvw_widget(children[0]), XmNlabelString, &text, NULL);
	   XmStringGetLtoR(text, "charset1", calldata);
	}
	else
	   return(FALSE);

	kfree(children);
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_set_menubutton()
|
|       Purpose: Sets the menubutton from the Motif row/col widget.
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_LABEL, XVW_BORDER_WIDTH
|                calldata  - pointer to contents of return text type
|
|        Output: none
|    Written By: Mark Young
|          Date: Jan 5, 1994
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_menubutton(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	kstring  string = *(kstring *) calldata;

	XmString text;
	xvobject *children;

	if ((children = xvw_children(object, xmCascadeButtonGadgetClass,NULL))
							== NULL)
	   return(FALSE);

	text = XmStringCreate(string, "charset1");
	XtVaSetValues(xvw_widget(children[0]), XmNlabelString, text, NULL);
	kfree(children);
	return(TRUE);
}


/*################################################################
#
#   Button Utilities
#
################################################################*/

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_buttonshape()
|
|       Purpose: Gets the current button shape of the button object
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_BUTTON_SHAPE
|                calldata  - pointer to contents of return button shape
|
|        Output: none
|    Written By: Danielle Argiro & John Salas
|          Date: April 12, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_buttonshape(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int *shape = (int *) calldata;

	*shape = KBUTTON_SHAPE_RECT;
	return(TRUE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_buttonshape()
|
|       Purpose: Sets the current button shape of the button object.
|
|         Input: object    - the object to have contents set
|                attribute - XVW_BUTTON_SHAPE
|                calldata  - pointer to contents of button shape
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: April 12, 1993 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_buttonshape(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int *shape = (int *) calldata;

	/* in Motif, the button shape is rectangular. period. */

	return(TRUE);
}


/*################################################################
#
#   Text Utilities
#
################################################################*/

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_text()
|
|       Purpose: Gets the text from the Motif compound text structure
|                associated with the text object
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_TEXT_STRING
|                calldata  - pointer to contents of return text type
|
|        Output: none
|    Written By: Danielle Argiro & John Salas
|          Date: April 12, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_text(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int  type;
	char **string = (char **) calldata;

	xvw_get_attribute(object,  XVW_TEXT_TYPE, &type);
	XtVaGetValues(xvw_widget(object), XmNvalue, string, NULL);
	return(TRUE);
}


/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_text()
|
|       Purpose: Sets the text in the Motif compound text structure
|                associated with the text object
|
|         Input: object    - the object to have text type set
|                attribute - XVW_TEXT_STRING
|                calldata  - pointer to contents of text type
|
|        Output: none
|    Written By: Danielle Argiro & John Salas
|          Date: April 12, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_text(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	Widget widget;
	int    type, edit_mode;
	char   *file_buffer, **string = (char **) calldata;

	widget = xvw_widget(object);
	xvw_get_attribute(object,  XVW_TEXT_TYPE, &type);
	if (type == KTEXT_TYPE_STRING)
	{
	    if (*string == NULL)
	        *string = "";

	    XmTextSetString(widget, *string);
	    XtVaGetValues(widget, XmNeditMode, &edit_mode, NULL);
	    if (edit_mode != XmMULTI_LINE_EDIT)
	       XmTextSetInsertionPosition(widget, kstrlen(*string));

	    return(TRUE);
	}
	else
	{
	    /* open the file in question */
            XtVaSetValues(widget, XmNeditable, FALSE, NULL);

            /* file not there, or cannot be opened */
            if (kaccess(*string, R_OK))
            {
                kerror("xvwidgets", "xvw_set_text", "Invalid file");
                return(FALSE);
            }
	    /* read in the contents of the file */
	    file_buffer = kreadfile(*string, NULL);
		
	    /* display file contents */
	    XmTextSetString(widget, file_buffer);
	    XtVaGetValues(widget, XmNeditMode, &edit_mode, NULL);
	    if (edit_mode != XmMULTI_LINE_EDIT)
	       XmTextSetInsertionPosition(widget, kstrlen(file_buffer));

	    kfree(file_buffer);
	}
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_texttype()
|
|       Purpose: Gets the text type of the text object
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_TEXT_TYPE
|                calldata  - pointer to contents of return text type
|
|        Output: none
|    Written By: Danielle Argiro & John Salas
|          Date: April 12, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_texttype(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int    *type = (int *) calldata;
        klist  *textwidget_list;

        textwidget_list = klist_locate(xvw_textwidget_list, object);
        if (textwidget_list != NULL)
        {
           *type = (int) klist_clientdata(textwidget_list);
	   return(TRUE);
        }
        return(FALSE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_texttype()
|
|       Purpose: Sets the text type of the text object.
|
|         Input: object    - the object to have text type set
|                attribute - XVW_TEXT_TYPE
|                calldata  - pointer to contents of text type
|
|        Output: none
|    Written By: Danielle Argiro & John Salas
|          Date: April 12, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_texttype(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int *type = (int *) calldata;
	klist *textwidget_list;

	if ((*type != KTEXT_TYPE_STRING) && (*type != KTEXT_TYPE_FILE))
	{
	    kerror("xvwidgets", "xvw_set_texttype", "XVW_TEXT_TYPE may only be \
set to KTEXT_TYPE_STRING or KTEXT_TYPE_FILE");
	    return(FALSE);
	}

	textwidget_list = klist_locate(xvw_textwidget_list, object);
        if (textwidget_list != NULL)
        {
           textwidget_list->client_data = (kaddr) *type;
           return(TRUE);
        }
	return(FALSE);
}


/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_textmulti()
|
|       Purpose: Gets the current text multiline type of the text object
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_TEXT_MULTILINE
|                calldata  - pointer to contents of return text multiline type
|
|        Output: none
|    Written By: Danielle Argiro & John Salas
|          Date: April 12, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_textmulti(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int edit_mode;
	int *multiline = (int *) calldata;

	XtVaGetValues(xvw_widget(object), XmNeditMode, &edit_mode, NULL);
	if (edit_mode == XmMULTI_LINE_EDIT)
	    *multiline = TRUE;
	else *multiline = FALSE;
	return(TRUE);
}


/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_textmulti()
|
|       Purpose: Sets the current text multiline of the text object.
|
|         Input: object    - the object to have contents set
|                attribute - XVW_TEXT_MULTILINE
|                calldata  - pointer to contents of text multiline type
|
|        Output: none
|    Written By: Danielle Argiro & John Salas
|          Date: April 12, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_textmulti(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int    *multiline = (int *) calldata;

	if (*multiline)
	{
	   XtVaSetValues(xvw_widget(object),
		 	 XmNmarginWidth,      2,
		 	 XmNmarginHeight,     2,
		 	 XmNcursorPosition,   0,
		 	 XmNtopCharacter,     0,
		 	 XmNautoShowCursorPosition, TRUE,
		 	 XmNwordWrap,         TRUE,
		 	 XmNeditMode,         XmMULTI_LINE_EDIT,
			 NULL);
	   xvw_set_attribute(object, XVW_CHAR_MAX_HEIGHT, 100.0);
	}
	else
	{
	   XtVaSetValues(xvw_widget(object),
		 	 XmNmarginWidth,      1,
		 	 XmNmarginHeight,     1,
		 	 XmNcursorPosition,   0,
		 	 XmNtopCharacter,     0,
		 	 XmNautoShowCursorPosition, TRUE,
		 	 XmNeditMode,         XmSINGLE_LINE_EDIT,
			 NULL);
	   xvw_set_attribute(object, XVW_CHAR_MAX_HEIGHT, 1.0);
	}
	return(TRUE);
}

/* ARGSUSED */
void xvw_set_textfocus(
   xvobject text,
   kaddr    client_data,
   XEvent   *event)
{
	xvobject toplevel = xvw_toplevel(text);

	if (event->type == EnterNotify)
	{
	   XtSetKeyboardFocus(xvw_widget(toplevel), None);
	   XtSetKeyboardFocus(xvw_widget(toplevel), xvw_widget(text));
	}
}


/*################################################################
#
#   Scrollbar Utilities
#
################################################################*/

typedef struct _xvw_scrollbar
{
	double scroll_min;
	double scroll_max;
	double scroll_incr;
	double scroll_value;
	int    increment_set;
} xvw_scrollbar;

klist *scrollbar_list = NULL;

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_scrollbar()
|
|       Purpose: Allocates & returns the contents of a scrollbar
|	         scrollbar field => {min,value,incr,max} value
|	         for one of the attributes:
|
|			XVW_SCROLL_MIN
|			XVW_SCROLL_MAX
|			XVW_SCROLL_VALUE
|			XVW_SCROLL_INCR
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above value
|                calldata  - pointer to contents of return string
|
|        Output: none
|    Written By: Mark Young
|          Date: Nov 16, 1992 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_scrollbar(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int           tmpval;
	klist         *entry;
	xvw_scrollbar *temp;
	double        *retval = (double *) calldata;

	if ((entry = klist_locate(scrollbar_list, (kaddr) object)) == NULL)
	{
           kerror("xvobject", "xvw_get_scrollbar",
		   "Can't retrieve %s for %s", attribute, xvw_name(object));
	   return(FALSE);
	}
	temp = (xvw_scrollbar *) entry->client_data;

	if (kstrcmp(attribute, XVW_SCROLL_MIN) == 0)
	   *retval = temp->scroll_min;
	else if (kstrcmp(attribute, XVW_SCROLL_MAX) == 0)
	   *retval = temp->scroll_max;
	else if (kstrcmp(attribute, XVW_SCROLL_INCR) == 0)
	   *retval = temp->scroll_incr;
	else if (kstrcmp(attribute, XVW_SCROLL_VALUE) == 0)
	{
	   XtVaGetValues(xvw_widget(object), XmNvalue, &tmpval, NULL);
	   temp->scroll_value = (((temp->scroll_max - temp->scroll_min) *
				kmin(tmpval,100))/100.0) + temp->scroll_min;
	   *retval = temp->scroll_value;
	}
	else
	{
           kerror("xvobject", "xvw_get_scrollbar",
		  "Unknown attribute '%s' for scrollbar object %s",
		  attribute, xvw_name(object));
	   return(FALSE);
	}
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvw_set_scrollbar()
|
|       Purpose: Sets the contents of a scrollbar
|	         for one of the attributes:
|
|			XVW_SCROLL_MIN
|			XVW_SCROLL_MAX
|			XVW_SCROLL_VALUE
|			XVW_SCROLL_INCR
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above value
|                calldata  - pointer to contents of return string
|
|        Output: none
|    Written By: Mark Young & Danielle Argiro
|          Date: April 15, 1993 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_scrollbar(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	xvw_scrollbar *temp;
	klist         *entry;
	Arg           arg[KLENGTH];
	int           i, value = 0, incr = 1;
	Widget        widget = xvw_widget(object);
	double        diff, *setval = (double *) calldata;


	if ((entry = klist_locate(scrollbar_list, (kaddr) object)) == NULL)
	{
	   temp = (xvw_scrollbar *) kcalloc(1, sizeof(xvw_scrollbar));
	   temp->scroll_max   = 1.0;
	   temp->scroll_incr  = 0.1;
	   temp->scroll_min   =
	   temp->scroll_value = 0.0;
	   temp->increment_set = FALSE;
	   scrollbar_list = klist_add(scrollbar_list, (kaddr) object, 
				      (kaddr) temp);
	}
	else temp = (xvw_scrollbar *) entry->client_data;

	if (kstrcmp(attribute, XVW_SCROLL_MIN) == 0)
	{
	   temp->scroll_min = *setval;
	   if (!temp->increment_set)
	      temp->scroll_incr = (temp->scroll_max - temp->scroll_min)*0.1;
	}
	else if (kstrcmp(attribute, XVW_SCROLL_MAX) == 0)
	{
	   temp->scroll_max = *setval;
	   if (!temp->increment_set)
	      temp->scroll_incr = (temp->scroll_max - temp->scroll_min)*0.1;
	}
	else if (kstrcmp(attribute, XVW_SCROLL_VALUE) == 0)
	   temp->scroll_value = *setval;
	else if (kstrcmp(attribute, XVW_SCROLL_INCR) == 0)
	   temp->scroll_incr = *setval, temp->increment_set = TRUE;
	else
	{
           kerror("xvobject", "xvw_set_scrollbar",
		  "Unknown attribute '%s' for scrollbar object %s",
		  attribute, xvw_name(object));
	   return(FALSE);
	}
	diff = kabs(temp->scroll_max - temp->scroll_min);
	incr  = krange(1, (temp->scroll_incr/diff) * 100, 100);
	value = ((temp->scroll_value - temp->scroll_min)/diff) * 100;
	value = krange(0, value, 100);

	i = 0;
	XtSetArg(arg[i], XmNminimum,    0);       i++;
	XtSetArg(arg[i], XmNmaximum,    100+incr);i++;
	XtSetArg(arg[i], XmNsliderSize, incr);    i++;
	XtSetArg(arg[i], XmNvalue,      value);   i++;
	XtSetValues(widget, arg, i);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_update_scroll_value()
|
|       Purpose: Sets the double value of the scroll
|                according to the Motif widget set.
|
|         Input: object      - the scrollbar object
|                type        - the type of callback
|                call_data   - the structure passed in as the call_data
|                              for a scrollbar widget by MOTIF
|                value - scrolbar value 
|        Output: none
|       Returns: none
|    Written By: Mark Young & Danielle Argiro
|          Date: June 1, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_update_scroll_value(
    xvobject object,
    char     *type,
    kaddr    call_data,
    double   *value)
{
	xvw_get_attribute(object, XVW_SCROLL_VALUE, value);
}


/*--------------------------------------------------------------
|
|  Routine Name: xvw_set_orientation()
|
|       Purpose: Sets the orientation of a scrollbar
|	         for the attribute:
|
|			XVW_ORIENTATION
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above value
|                calldata  - pointer to contents of return string
|
|        Output: none
|    Written By: Mark Young & Danielle Argiro
|          Date: April 15, 1993 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_orientation(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int           *orientation = (int *) calldata;
	unsigned char current, new;

	if (*orientation == KSCROLLBAR_ORIENT_VERT) 
	    new = XmVERTICAL;
	else if (*orientation == KSCROLLBAR_ORIENT_HORIZ) 
            new = XmHORIZONTAL;
	else 
	{
           kerror("xvobject", "xvw_set_orientation",
	       "orientation must be one of KSCROLLBAR_ORIENT_VERT or KSCROLLBAR_ORIENT_HORIZ",
		attribute, xvw_name(object));
	   return(FALSE);
	}

	XtVaGetValues(xvw_widget(object), XmNorientation, &current, NULL);
	XtVaSetValues(xvw_widget(object), 
		      XmNorientation,         new, 
		      XmNprocessingDirection, XmMAX_ON_RIGHT,
		      NULL);
	return(TRUE);
}



/*################################################################
#
#    List Utilities
#
################################################################*/

/*-----------------------------------------------------------
|
|  Routine Name: xvw_update_list_struct()
|
|       Purpose: Sets the fields in the xvw_list_struct structure
|                according to the Motif widget set.
|
|         Input: list_struct - xvwidgets list structure to be set
|                call_data   - the structure passed in as the call_data
|                              for a list widget by Motif
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 21, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_update_list_struct(
    xvobject	    object,
    char	    *type,
    kaddr           call_data,
    int		    *double_click,
    xvw_list_struct *list_struct)
{
	char *string;
	XmListCallbackStruct *motif_list_struct;
	motif_list_struct = (XmListCallbackStruct *) call_data;

	*double_click = TRUE;
	XmStringGetLtoR(motif_list_struct->item, "charset1", &string);
        list_struct->string     = string;
	/*-- take one of index, since Motif starts at 1, we start at 9 --*/
	list_struct->list_index =  motif_list_struct->item_position - 1;
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_get_list_highlight()
|
|       Purpose: Gets the highlighted element of the list.
|
|         Input: object    - the object to have highlighted element obtained
|                attribute - should be XVW_LIST_HIGHLT_ELEM 
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_list_highlight(
	xvobject	object,
	char		*attribute,
	kaddr		calldata)
{
        int *indx = (int *) calldata;
	int *selected_list;
	int selected_count;

	if (kstrcmp(attribute, XVW_LIST_HIGHLT_ELEM) != 0)
	{
		kerror("xvobject", "xvw_get_list_highlight",
			"Unknown attribute '%s' for list object %s",
			attribute, xvw_name(object));
		return(FALSE);
	}
	/*-- subtract 1, since motif lists start at index 1, not 0 --*/
	if (XmListGetSelectedPos(xvw_widget(object),
		&selected_list, &selected_count))
	{
	   *indx = selected_list[0] - 1;
	}
	else
	   *indx = -1;

        return(TRUE);
}


/*------------------------------------------------------------
|
|  Routine Name: xvw_set_list_highlight()
|
|       Purpose: Sets the highlighted element of the list.
|
|         Input: object    - the object to have list element highlighted
|                attribute - should be XVW_LIST_HIGHLT_ELEM
|                calldata  - passes to index of element to be highlighted
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 15, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_list_highlight(
	xvobject	object,
	char		*attribute,
	kaddr		calldata)
{
	int      	*indx	= (int *) calldata;
	int		pos	= *indx;
	int		top, visible;


        if (kstrcmp(attribute, XVW_LIST_HIGHLT_ELEM) != 0)
        {
	   kerror("Motif", "xvw_set_list_highlight()",
		  "Unknown attribute '%s' for list object %s",
		  attribute, xvw_name(object));
	   return(FALSE);
        }
	pos++;	/*-- motif starts from 1, not zero --*/
	XmListSelectPos(xvw_widget(object), pos, FALSE);

	/*-- now we make sure highlighted item is visible --*/
	/*-- code adapted from O'Reilly Motif Programming Manual --*/
	XtVaGetValues(xvw_widget(object),
		XmNtopItemPosition,     &top,
		XmNvisibleItemCount,    &visible,
		NULL);

	if (pos < top)
	   XmListSetPos(xvw_widget(object), pos);
	else if (pos >= top+visible)
	   XmListSetBottomPos(xvw_widget(object), pos);
	/*-- end of code from O'Reilly --*/

        return(TRUE);
}


/*------------------------------------------------------------
|
|  Routine Name: xvw_set_list_unhighlight()
|
|       Purpose: Unhighlights an element of the list.
|
|         Input: object    - the object to have list element highlighted
|                attribute - should be XVW_LIST_UNHIGHLT_ELEM
|                calldata  - passes to index of element to be unhighlighted
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 15, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_list_unhighlight(
	xvobject	object,
	char		*attribute,
	kaddr		calldata)
{
	int		*indx	= (int *) calldata;
	int		pos	= *indx;

        if (kstrcmp(attribute, XVW_LIST_UNHIGHLT_ELEM) != 0)
        {
	   kerror("Motif", "xvw_set_list_unhighlight()",
		  "Unknown attribute '%s' for list object %s",
		  attribute, xvw_name(object));
	   return(FALSE);
        }
	/*-- increment position, since Motif starts at 1, not 0 --*/
	XmListDeselectPos(xvw_widget(object), ++pos);

        return(TRUE);
}

/*################################################################
#
#   Miscellaneous utilities
#
################################################################*/


/*------------------------------------------------------------
|
|  Routine Name: xvw_get_misc()
|
|       Purpose: Gets miscellaneous attributes of various xvobjects.
|
|         Input: object    - the object to have attribute obtained
|                attribute - one of the miscellaneous attributes, eg,
|                            XVW_TEXT_EDIT_TYPE
|        Output: calldata  - passes back the miscellaneous value to be set
|    Written By: Mark Young Danielle Argiro
|          Date: June 7, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_misc(
   xvobject object,
   char     *attribute,

   kaddr    calldata)
{
        int      *value = (int *) calldata;
	Widget   widget = xvw_widget(object);

        Boolean  motif_intval;
	unsigned char motif_ucval;

        if (kstrcmp(attribute, XVW_TEXT_EDIT_TYPE) == 0)
        {
            XtVaGetValues(widget, XmNeditable, &motif_intval, NULL);
            if (motif_intval == FALSE)
                *value = KTEXT_READ;
            else if (motif_intval == TRUE)
                *value = KTEXT_EDIT;
        }
        else if (kstrcmp(attribute, XVW_TEXT_WRAP) == 0)
        {
            XtVaGetValues(widget, XmNwordWrap, &motif_intval, NULL);
            if (motif_intval == TRUE)
                *value = KTEXT_WRAP_WORD;
            else if (motif_intval == FALSE)
                *value = KTEXT_WRAP_NEVER;
        }

        else if (kstrcmp(attribute, XVW_TEXT_SCROLL_VERT) == 0)
        {
            XtVaGetValues(widget, XmNscrollVertical, &motif_intval, NULL);
            if (motif_intval == TRUE)
                *value = KSCROLLBAR_ALWAYS;
            else if (motif_intval == FALSE)
                *value = KSCROLLBAR_NEVER;
        }

        else if (kstrcmp(attribute, XVW_TEXT_SCROLL_HORIZ) == 0)
        {
            XtVaGetValues(widget, XmNscrollHorizontal, &motif_intval, NULL);
            if (motif_intval == TRUE)
                *value = KSCROLLBAR_ALWAYS;
            else if (motif_intval == FALSE)
                *value = KSCROLLBAR_NEVER;
        }

        else if ((kstrcmp(attribute, XVW_BUTTON_TYPE) == 0) ||
		 (kstrcmp(attribute, XVW_LABEL_TYPE) == 0))
        {
            XtVaGetValues(widget, XmNlabelType, &motif_ucval, NULL);
            if (motif_ucval == XmSTRING)
                *value = KLABEL_TYPE_LABEL;
            else if (motif_ucval == XmPIXMAP)
                *value = KLABEL_TYPE_PIXMAP;
        }

        else if ((kstrcmp(attribute, XVW_LABEL_JUSTIFY) == 0) ||
                 (kstrcmp(attribute, XVW_PIXMAP_JUSTIFY) == 0))
        {
            XtVaGetValues(widget, XmNalignment, &motif_ucval, NULL);
            if (motif_ucval == XmALIGNMENT_BEGINNING)
                *value = KLABEL_JUSTIFY_LEFT;
            else if (motif_ucval == XmALIGNMENT_END)
                *value = KLABEL_JUSTIFY_RIGHT;
            else if (motif_ucval == XmALIGNMENT_CENTER)
                *value = KLABEL_JUSTIFY_CENTER;
        }
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_set_misc()
|
|       Purpose: Sets miscellaneous attributes of various xvobjects.
|
|         Input: object    - the object to have attribute set
|                attribute - one of the miscellaneous attributes, eg,
|                            XVW_TEXT_EDIT_TYPE
|                calldata  - passes in the miscellaneous value to be set
|
|        Output: none
|    Written By: Mark Young Danielle Argiro
|          Date: June 7, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_misc(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
        Widget widget = xvw_widget(object);
        int    *value = (int *) calldata;

        if (kstrcmp(attribute, XVW_TEXT_EDIT_TYPE) == 0)
        {

            if (*value == KTEXT_READ)
                XtVaSetValues(widget, XmNeditable, FALSE, NULL);
            else if (*value == KTEXT_EDIT)
                XtVaSetValues(widget, XmNeditable, TRUE, NULL);
            else if (*value == KTEXT_APPEND)
                XtVaSetValues(widget, XmNeditable, TRUE, NULL);
            else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_EDIT_TYPE");
                return(FALSE);
            }
        }

        else if (kstrcmp(attribute, XVW_TEXT_WRAP) == 0)
        {
            if (*value == KTEXT_WRAP_WORD)
                XtVaSetValues(widget, XmNwordWrap, TRUE, NULL);
            else if (*value == KTEXT_WRAP_LINE)
                XtVaSetValues(widget, XmNwordWrap, TRUE, NULL);
            else if (*value == KTEXT_WRAP_NEVER)
                XtVaSetValues(widget, XmNwordWrap, FALSE, NULL);
            else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_WRAP");
                return(FALSE);
            }

        }

        else if (kstrcmp(attribute, XVW_TEXT_SCROLL_VERT) == 0)
        {
            if (*value == KSCROLLBAR_ALWAYS)
                XtVaSetValues(widget, XmNscrollVertical, TRUE, NULL);
            else if (*value == KSCROLLBAR_IFNEEDED)
                XtVaSetValues(widget, XmNscrollVertical, TRUE, NULL);
            else if (*value == KSCROLLBAR_NEVER)
                XtVaSetValues(widget, XmNscrollVertical, FALSE, NULL);
            else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_SCROLL_VERT");
                return(FALSE);
            }
        }

        else if (kstrcmp(attribute, XVW_TEXT_SCROLL_HORIZ) == 0)
        {
            if (*value == KSCROLLBAR_ALWAYS)
                XtVaSetValues(widget, XmNscrollVertical, TRUE, NULL);
            else if (*value == KSCROLLBAR_IFNEEDED)
                XtVaSetValues(widget, XmNscrollVertical, TRUE, NULL);
            else if (*value == KSCROLLBAR_NEVER)
                XtVaSetValues(widget, XmNscrollVertical, FALSE, NULL);
            else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_SCROLL_HORIZ");
                return(FALSE);
            }
        }


        else if ((kstrcmp(attribute, XVW_BUTTON_TYPE) == 0) ||
		 (kstrcmp(attribute, XVW_LABEL_TYPE) == 0))
        {
	    if (*value == KLABEL_TYPE_LABEL)
		XtVaSetValues(widget, XmNlabelType, XmSTRING, NULL);
            else if (*value == KLABEL_TYPE_PIXMAP)
		XtVaSetValues(widget, XmNlabelType, XmPIXMAP, NULL);
            else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for '%s'", attribute);
                return(FALSE);
	    }
        }

        else if ((kstrcmp(attribute, XVW_LABEL_JUSTIFY) == 0) ||
                 (kstrcmp(attribute, XVW_PIXMAP_JUSTIFY) == 0))
        {
            if (*value == KLABEL_JUSTIFY_LEFT)
                XtVaSetValues(widget, XmNalignment, XmALIGNMENT_BEGINNING,NULL);
            else if (*value == KLABEL_JUSTIFY_RIGHT)
                XtVaSetValues(widget, XmNalignment, XmALIGNMENT_END, NULL);
            else if (*value == KLABEL_JUSTIFY_CENTER)
                XtVaSetValues(widget, XmNalignment, XmALIGNMENT_CENTER, NULL);
            else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for %s", attribute);
                return(FALSE);
            }
        }
	return(TRUE);
}


/*################################################################
#
#    Public Utilities - public headers in ../xvwidgets/widgets.c
#
################################################################*/

/*-----------------------------------------------------------
|
|  Routine Name: xvw_change_list()
|
|       Purpose: Changes the contents of a list object
|                while it is displayed
|
|         Input: object  - the list object to have contents changed
|                list    - new contents of list object
|                size    - size of the new list (# entries)
|                resize  - TRUE if list object should try to resize
|                          itself after the change, FALSE otherwise.
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 14, 1992 
| Modifications:
|
-------------------------------------------------------------------*/

void xvw_change_list(
   xvobject object,
   char     **list,
   int      size,
   int      resize)
{
        int           i, width;
	XmString      item, *items = NULL;
	Widget        widget = xvw_widget(object);

	/*
	 * resize backplane, if requested
	 */
	if (resize)
	{
            for (i = 0, width = 0; i < size; i++)
           	width  = kmax(width, kstrlen(list[i]));

            xvw_set_attributes(xvw_parent(object),
                        XVW_CHAR_WIDTH,  ((float) width),
                        XVW_CHAR_HEIGHT, ((float) size),
                        NULL);
	}

	/*
	 *  delete the old list
	 */
	XmListDeleteAllItems(widget);
	if (size == 0)
	   return;

	/*
	 * free the new list
	 */
	for (i = 0; i < size; i++)
	{
	    item = XmStringCreate(list[i], "charset1");
	    items = (XmString *) karray_add((char **) items, (kaddr) item, i);
	}
	XmListAddItems(widget, items, size, 1);
	for (i = 0; i < size; i++)
	   XmStringFree(items[i]);
	kfree(items);
}


/*------------------------------------------------------------
|
|  Routine Name: xvw_reverse_colors
|
|       Purpose: Reverses the foreground and background colors
|                on a button object to indicate whether it is selected.
|
|         Input: object - the button object on which to reverse colors
|                value  - TRUE if button should be changed to look selected,
|                         FALSE if button should be changed to look un-selected.
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 14, 1992 
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_reverse_colors(
   xvobject object,
   int      value)
{
        int   i;
        Arg   arg[MAX_ARGS];
        Pixel fg, bg, topshadow, bottomshadow;

        /*
         * get current fg & bg
         */
        i = 0;
        XtSetArg(arg[i], XmNarmColor,          &fg);            i++;
        XtSetArg(arg[i], XmNbackground,        &bg);            i++;
        XtSetArg(arg[i], XmNtopShadowColor,    &topshadow);  	i++;
        XtSetArg(arg[i], XmNbottomShadowColor, &bottomshadow);  i++;
        XtGetValues(xvw_widget(object), arg, i);

        /*
         * toggle current fg & bg
         */
        i = 0;
        XtSetArg(arg[i], XmNarmColor,          bg);    		i++;
        XtSetArg(arg[i], XmNbackground,        fg);    		i++;
        XtSetArg(arg[i], XmNtopShadowColor,    bottomshadow);  	i++;
        XtSetArg(arg[i], XmNbottomShadowColor, topshadow);  	i++;
        XtSetValues(xvw_widget(object), arg, i);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_retrieve_list
|
|       Purpose: Retrieve the actual list widget associated
|                with the compound list object.
|
|         Input: list - the compound object for which the 
|                       actual list object is desired
|        Output: none
|       Returns: The list object on success, NULL on failure.
|    Written By: Danielle Argiro
|          Date: April 21, 1993
| Modifications:
|
------------------------------------------------------------------*/

xvobject xvw_retrieve_list(
    xvobject parent)
{
        Widget list_widget;
        klist  *listwidget_list;

        listwidget_list = klist_locate(xvw_listwidget_list, parent);
        if (listwidget_list != NULL)
        {
           list_widget = (Widget) klist_clientdata(listwidget_list);
           return(xvw_object(list_widget));
        }
        else return(NULL);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_retrieve_menu
|
|       Purpose: Given a compound menubutton object, returns the
|                menu associated with the menubutton
|
|         Input: menubutton - the menubutton for which the menu is desired
|        Output:
|       Returns: The menu object on success, NULL on failure
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications:
|
------------------------------------------------------------------*/

xvobject xvw_retrieve_menu(
    xvobject menubutton)
{
        xvobject menu;
        klist    *menuwidget_list;

        menuwidget_list = klist_locate(xvw_menuwidget_list, menubutton);
        if (menuwidget_list != NULL)
        {
           menu = (xvobject) klist_clientdata(menuwidget_list);
           return(menu);
        }
        else return(NULL);
}

#else
/*
 * These are dummy function calls so that if the widget set is not supported
 * on the machine a message is printed out.
 */

/* ARGSUSED */
void xvw_change_list(
   xvobject object,
   char   **list,
   int    size,
   int    resize)
{
   kfprintf(kstderr,"\nThe Motif widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return;
}

/* ARGSUSED */
void xvw_reverse_colors(
   xvobject object,
   int    value)
{
   kfprintf(kstderr,"\nThe Motif widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return;
}

/* ARGSUSED */
xvobject xvw_retrieve_list(
    xvobject list)
{
   kfprintf(kstderr,"\nThe Motif widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return(NULL);
}

/* ARGSUSED */
xvobject xvw_retrieve_menu(
    xvobject menubutton)
{
   kfprintf(kstderr,"\nThe Motif widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return(NULL);
}
#endif /* KMOTIF_DEF */
/* Don't add after the endif */
